/*	$OpenBSD: rc4_test.c,v 1.6 2022/11/09 12:10:17 joshua Exp $ */
/*
 * Copyright (c) 2022 Joshua Sing <joshua@hypera.dev>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <openssl/evp.h>
#include <openssl/rc4.h>

#include <stdint.h>
#include <string.h>

struct rc4_test {
	const uint8_t key[32];
	const int key_len;
	const int len;
	const uint8_t in[512];
	const uint8_t out[512];
};

static const struct rc4_test rc4_tests[] = {
	/*
	 * Test vectors from RFC 6229, with 40 and 128-bit keys.
	 * Note that this only uses the first 32 bytes of each test vector due
	 * to stream offsets.
	 */
	{
		.key = {
			0x01, 0x02, 0x03, 0x04, 0x05,
		},
		.key_len = 5,
		.len = 32,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.out = {
			0xb2, 0x39, 0x63, 0x05, 0xf0, 0x3d, 0xc0, 0x27,
			0xcc, 0xc3, 0x52, 0x4a, 0x0a, 0x11, 0x18, 0xa8,
			0x69, 0x82, 0x94, 0x4f, 0x18, 0xfc, 0x82, 0xd5,
			0x89, 0xc4, 0x03, 0xa4, 0x7a, 0x0d, 0x09, 0x19,
		},
	},
	{
		.key = {
			0x83, 0x32, 0x22, 0x77, 0x2a,
		},
		.key_len = 5,
		.len = 32,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.out = {
			0x80, 0xad, 0x97, 0xbd, 0xc9, 0x73, 0xdf, 0x8a,
			0x2e, 0x87, 0x9e, 0x92, 0xa4, 0x97, 0xef, 0xda,
			0x20, 0xf0, 0x60, 0xc2, 0xf2, 0xe5, 0x12, 0x65,
			0x01, 0xd3, 0xd4, 0xfe, 0xa1, 0x0d, 0x5f, 0xc0,
		},
	},
	{
		.key = {
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
		},
		.key_len = 16,
		.len = 32,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.out = {
			0x9a, 0xc7, 0xcc, 0x9a, 0x60, 0x9d, 0x1e, 0xf7,
			0xb2, 0x93, 0x28, 0x99, 0xcd, 0xe4, 0x1b, 0x97,
			0x52, 0x48, 0xc4, 0x95, 0x90, 0x14, 0x12, 0x6a,
			0x6e, 0x8a, 0x84, 0xf1, 0x1d, 0x1a, 0x9e, 0x1c,
		},
	},
	{
		.key = {
			0xeb, 0xb4, 0x62, 0x27, 0xc6, 0xcc, 0x8b, 0x37,
			0x64, 0x19, 0x10, 0x83, 0x32, 0x22, 0x77, 0x2a,
		},
		.key_len = 16,
		.len = 32,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.out = {
			0x72, 0x0c, 0x94, 0xb6, 0x3e, 0xdf, 0x44, 0xe1,
			0x31, 0xd9, 0x50, 0xca, 0x21, 0x1a, 0x5a, 0x30,
			0xc3, 0x66, 0xfd, 0xea, 0xcf, 0x9c, 0xa8, 0x04,
			0x36, 0xbe, 0x7c, 0x35, 0x84, 0x24, 0xd2, 0x0b,
		},
	},

	/*
	 * Test vectors from the original cypherpunk posting of ARC4:
	 * https://groups.google.com/group/sci.crypt/msg/10a300c9d21afca0?pli=1
	 */
	{
		.key = {
			0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
		},
		.key_len = 8,
		.len = 8,
		.in = {
			0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
		},
		.out = {
			0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96,
		},
	},
	{
		.key = {
			0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
		},
		.key_len = 8,
		.len = 8,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.out = {
			0x74, 0x94, 0xc2, 0xe7, 0x10, 0x4b, 0x08, 0x79,
		},
	},
	{
		.key = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.key_len = 8,
		.len = 8,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.out = {
			0xde, 0x18, 0x89, 0x41, 0xa3, 0x37, 0x5d, 0x3a,
		},
	},
	{
		.key = {
			0xef, 0x01, 0x23, 0x45,
		},
		.key_len = 4,
		.len = 10,
		.in = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00,
		},
		.out = {
			0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			0xbd, 0x61,
		},
	},
	{
		.key = {
			0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
		},
		.key_len = 8,
		.len = 512,
		.in = {
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
		},
		.out = {
			0x75, 0x95, 0xc3, 0xe6, 0x11, 0x4a, 0x09, 0x78,
			0x0c, 0x4a, 0xd4, 0x52, 0x33, 0x8e, 0x1f, 0xfd,
			0x9a, 0x1b, 0xe9, 0x49, 0x8f, 0x81, 0x3d, 0x76,
			0x53, 0x34, 0x49, 0xb6, 0x77, 0x8d, 0xca, 0xd8,
			0xc7, 0x8a, 0x8d, 0x2b, 0xa9, 0xac, 0x66, 0x08,
			0x5d, 0x0e, 0x53, 0xd5, 0x9c, 0x26, 0xc2, 0xd1,
			0xc4, 0x90, 0xc1, 0xeb, 0xbe, 0x0c, 0xe6, 0x6d,
			0x1b, 0x6b, 0x1b, 0x13, 0xb6, 0xb9, 0x19, 0xb8,
			0x47, 0xc2, 0x5a, 0x91, 0x44, 0x7a, 0x95, 0xe7,
			0x5e, 0x4e, 0xf1, 0x67, 0x79, 0xcd, 0xe8, 0xbf,
			0x0a, 0x95, 0x85, 0x0e, 0x32, 0xaf, 0x96, 0x89,
			0x44, 0x4f, 0xd3, 0x77, 0x10, 0x8f, 0x98, 0xfd,
			0xcb, 0xd4, 0xe7, 0x26, 0x56, 0x75, 0x00, 0x99,
			0x0b, 0xcc, 0x7e, 0x0c, 0xa3, 0xc4, 0xaa, 0xa3,
			0x04, 0xa3, 0x87, 0xd2, 0x0f, 0x3b, 0x8f, 0xbb,
			0xcd, 0x42, 0xa1, 0xbd, 0x31, 0x1d, 0x7a, 0x43,
			0x03, 0xdd, 0xa5, 0xab, 0x07, 0x88, 0x96, 0xae,
			0x80, 0xc1, 0x8b, 0x0a, 0xf6, 0x6d, 0xff, 0x31,
			0x96, 0x16, 0xeb, 0x78, 0x4e, 0x49, 0x5a, 0xd2,
			0xce, 0x90, 0xd7, 0xf7, 0x72, 0xa8, 0x17, 0x47,
			0xb6, 0x5f, 0x62, 0x09, 0x3b, 0x1e, 0x0d, 0xb9,
			0xe5, 0xba, 0x53, 0x2f, 0xaf, 0xec, 0x47, 0x50,
			0x83, 0x23, 0xe6, 0x71, 0x32, 0x7d, 0xf9, 0x44,
			0x44, 0x32, 0xcb, 0x73, 0x67, 0xce, 0xc8, 0x2f,
			0x5d, 0x44, 0xc0, 0xd0, 0x0b, 0x67, 0xd6, 0x50,
			0xa0, 0x75, 0xcd, 0x4b, 0x70, 0xde, 0xdd, 0x77,
			0xeb, 0x9b, 0x10, 0x23, 0x1b, 0x6b, 0x5b, 0x74,
			0x13, 0x47, 0x39, 0x6d, 0x62, 0x89, 0x74, 0x21,
			0xd4, 0x3d, 0xf9, 0xb4, 0x2e, 0x44, 0x6e, 0x35,
			0x8e, 0x9c, 0x11, 0xa9, 0xb2, 0x18, 0x4e, 0xcb,
			0xef, 0x0c, 0xd8, 0xe7, 0xa8, 0x77, 0xef, 0x96,
			0x8f, 0x13, 0x90, 0xec, 0x9b, 0x3d, 0x35, 0xa5,
			0x58, 0x5c, 0xb0, 0x09, 0x29, 0x0e, 0x2f, 0xcd,
			0xe7, 0xb5, 0xec, 0x66, 0xd9, 0x08, 0x4b, 0xe4,
			0x40, 0x55, 0xa6, 0x19, 0xd9, 0xdd, 0x7f, 0xc3,
			0x16, 0x6f, 0x94, 0x87, 0xf7, 0xcb, 0x27, 0x29,
			0x12, 0x42, 0x64, 0x45, 0x99, 0x85, 0x14, 0xc1,
			0x5d, 0x53, 0xa1, 0x8c, 0x86, 0x4c, 0xe3, 0xa2,
			0xb7, 0x55, 0x57, 0x93, 0x98, 0x81, 0x26, 0x52,
			0x0e, 0xac, 0xf2, 0xe3, 0x06, 0x6e, 0x23, 0x0c,
			0x91, 0xbe, 0xe4, 0xdd, 0x53, 0x04, 0xf5, 0xfd,
			0x04, 0x05, 0xb3, 0x5b, 0xd9, 0x9c, 0x73, 0x13,
			0x5d, 0x3d, 0x9b, 0xc3, 0x35, 0xee, 0x04, 0x9e,
			0xf6, 0x9b, 0x38, 0x67, 0xbf, 0x2d, 0x7b, 0xd1,
			0xea, 0xa5, 0x95, 0xd8, 0xbf, 0xc0, 0x06, 0x6f,
			0xf8, 0xd3, 0x15, 0x09, 0xeb, 0x0c, 0x6c, 0xaa,
			0x00, 0x6c, 0x80, 0x7a, 0x62, 0x3e, 0xf8, 0x4c,
			0x3d, 0x33, 0xc1, 0x95, 0xd2, 0x3e, 0xe3, 0x20,
			0xc4, 0x0d, 0xe0, 0x55, 0x81, 0x57, 0xc8, 0x22,
			0xd4, 0xb8, 0xc5, 0x69, 0xd8, 0x49, 0xae, 0xd5,
			0x9d, 0x4e, 0x0f, 0xd7, 0xf3, 0x79, 0x58, 0x6b,
			0x4b, 0x7f, 0xf6, 0x84, 0xed, 0x6a, 0x18, 0x9f,
			0x74, 0x86, 0xd4, 0x9b, 0x9c, 0x4b, 0xad, 0x9b,
			0xa2, 0x4b, 0x96, 0xab, 0xf9, 0x24, 0x37, 0x2c,
			0x8a, 0x8f, 0xff, 0xb1, 0x0d, 0x55, 0x35, 0x49,
			0x00, 0xa7, 0x7a, 0x3d, 0xb5, 0xf2, 0x05, 0xe1,
			0xb9, 0x9f, 0xcd, 0x86, 0x60, 0x86, 0x3a, 0x15,
			0x9a, 0xd4, 0xab, 0xe4, 0x0f, 0xa4, 0x89, 0x34,
			0x16, 0x3d, 0xdd, 0xe5, 0x42, 0xa6, 0x58, 0x55,
			0x40, 0xfd, 0x68, 0x3c, 0xbf, 0xd8, 0xc0, 0x0f,
			0x12, 0x12, 0x9a, 0x28, 0x4d, 0xea, 0xcc, 0x4c,
			0xde, 0xfe, 0x58, 0xbe, 0x71, 0x37, 0x54, 0x1c,
			0x04, 0x71, 0x26, 0xc8, 0xd4, 0x9e, 0x27, 0x55,
			0xab, 0x18, 0x1a, 0xb7, 0xe9, 0x40, 0xb0, 0xc0,
		},
	},
};

#define N_RC4_TESTS (sizeof(rc4_tests) / sizeof(rc4_tests[0]))

static int
rc4_test(void)
{
	const struct rc4_test *rt;
	RC4_KEY key;
	EVP_CIPHER_CTX *ctx = NULL;
	const EVP_CIPHER *cipher;
	uint8_t out[512];
	int in_len, out_len, total_len;
	size_t i;
	int j;
	int failed = 1;

	if ((ctx = EVP_CIPHER_CTX_new()) == NULL) {
		fprintf(stderr, "FAIL: EVP_CIPHER_CTX_new() failed\n");
		goto failed;
	}

	for (i = 0; i < N_RC4_TESTS; i++) {
		rt = &rc4_tests[i];

		/* Encryption */
		memset(out, 0, sizeof(out));
		RC4_set_key(&key, rt->key_len, rt->key);
		RC4(&key, rt->len, rt->in, out);

		if (memcmp(rt->out, out, rt->len) != 0) {
			fprintf(stderr, "FAIL: encryption mismatch\n");
			goto failed;
		}

		/* Decryption */
		memset(out, 0, sizeof(out));
		RC4_set_key(&key, rt->key_len, rt->key);
		RC4(&key, rt->len, rt->out, out);

		if (memcmp(rt->in, out, rt->len) != 0) {
			fprintf(stderr, "FAIL: decryption mismatch\n");
			goto failed;
		}

		/*
		 * EVP tests
		 */
		if (rt->key_len == 5) {
			cipher = EVP_rc4_40();
		} else if (rt->key_len == 16) {
			cipher = EVP_rc4();
		} else {
			/* EVP does not support this key length */
			continue;
		}

		/* EVP encryption */
		total_len = 0;
		memset(out, 0, sizeof(out));
		if (!EVP_EncryptInit(ctx, cipher, rt->key, NULL)) {
			fprintf(stderr, "FAIL: EVP_EncryptInit failed\n");
			goto failed;
		}

		for (j = 0; j < rt->len;) {
			in_len = arc4random_uniform(rt->len / 2);
			if (in_len > rt->len - j)
				in_len = rt->len - j;

			if (!EVP_EncryptUpdate(ctx, out + total_len, &out_len,
			    rt->in + j, in_len)) {
				fprintf(stderr,
				    "FAIL: EVP_EncryptUpdate failed\n");
				goto failed;
			}

			j += in_len;
			total_len += out_len;
		}

		if (!EVP_EncryptFinal_ex(ctx, out + total_len, &out_len)) {
			fprintf(stderr, "FAIL: EVP_EncryptFinal_ex failed\n");
			goto failed;
		}
		total_len += out_len;

		if (!EVP_CIPHER_CTX_reset(ctx)) {
			fprintf(stderr, "FAIL: EVP_CIPHER_CTX_reset failed\n");
			goto failed;
		}

		if (total_len != rt->len) {
			fprintf(stderr,
			    "FAIL: EVP encryption length mismatch\n");
			goto failed;
		}

		if (memcmp(rt->out, out, rt->len) != 0) {
			fprintf(stderr, "FAIL: EVP encryption mismatch\n");
			goto failed;
		}

		/* EVP decryption */
		total_len = 0;
		memset(out, 0, sizeof(out));
		if (!EVP_DecryptInit(ctx, cipher, rt->key, NULL)) {
			fprintf(stderr, "FAIL: EVP_DecryptInit failed\n");
			goto failed;
		}

		for (j = 0; j < rt->len;) {
			in_len = arc4random_uniform(rt->len / 2);
			if (in_len > rt->len - j)
				in_len = rt->len - j;

			if (!EVP_DecryptUpdate(ctx, out + total_len, &out_len,
			    rt->in + j, in_len)) {
				fprintf(stderr,
				    "FAIL: EVP_DecryptUpdate failed\n");
				goto failed;
			}

			j += in_len;
			total_len += out_len;
		}

		if (!EVP_DecryptFinal_ex(ctx, out + total_len, &out_len)) {
			fprintf(stderr, "FAIL: EVP_DecryptFinal_ex failed\n");
			goto failed;
		}
		total_len += out_len;

		if (!EVP_CIPHER_CTX_reset(ctx)) {
			fprintf(stderr, "FAIL: EVP_CIPHER_CTX_reset failed\n");
			goto failed;
		}

		if (total_len != rt->len) {
			fprintf(stderr,
			    "FAIL: EVP decryption length mismatch\n");
			goto failed;
		}

		if (memcmp(rt->out, out, rt->len) != 0) {
			fprintf(stderr, "FAIL: EVP decryption mismatch\n");
			goto failed;
		}
	}

	failed = 0;

 failed:
	EVP_CIPHER_CTX_free(ctx);
	return failed;
}

int
main(int argc, char **argv)
{
	int failed = 0;

	failed |= rc4_test();

	return failed;
}
