/* Create directory and all its intermediaries.
   Copyright (C) 2025 Free Software Foundation, Inc.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General
   Public License along with this library.  If not, see
   <http://www.gnu.org/licenses/>. */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>

int
mkinterdir (char *dirname, mode_t mode, mode_t intermode)
{
  struct stat st;
  char *end = dirname + strlen (dirname);
  char *p;

  /* Find topmost existing directory component. */
  while ((p = strrchr (dirname, '/')) != NULL)
    {
      *p = 0;
      if (stat (dirname, &st) == 0)
	{
	  if (!S_ISDIR (st.st_mode))
	    return ENOTDIR;
	  break;
	}
      else if (errno != ENOENT)
	return errno;
    }

  /* Create intermediate directories. */
  while ((p += strlen (p)) < end)
    {
      *p = '/';
      if (mkdir (dirname, (p + strlen (p) == end ? mode : intermode)))
	return errno;
    }

  return 0;
}

int
mu_mkinterdir (char const *name, mode_t mode, mode_t intermode)
{
  char *dirname;
  struct stat st;
  int rc;

  if (stat (name, &st) == 0)
    return S_ISDIR (st.st_mode) ? 0 : ENOTDIR;
  else if (errno != ENOENT)
    return errno;

  dirname = strdup (name);
  if (!dirname)
    return -1;

  rc = mkinterdir (dirname, mode, intermode);

  free (dirname);

  return rc;
}
