/*
 * XercesSax.cpp --
 *
 *	Tcl Interface to the Xerces XML parser
 *
 * Copyright (c) 1998-2000 Ajuba Solutions.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * $Id: XercesSax.cpp,v 1.29 2000/09/29 00:47:25 awb Exp $
 */

#include "XercesSax.h"
#ifdef _WINDOWS
#include <memory>
#else
#include <memory.h>
#endif

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLEXPORT

#ifdef XERCESSAX_DEBUG
static Tcl_ThreadDataKey dataKey;
static int interpCount = 0;
static int haveObjects = 0;
static int threadCount = 0;
TCL_DECLARE_MUTEX(xercessaxMutex);
#include <crtdbg.h>
_CrtMemState memState;
#endif;

EXTERN int	Xercessax_Init _ANSI_ARGS_((Tcl_Interp *interp));

static int ParserParseCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int ParserCgetCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int ParserConfigureCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int ParserResetCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int ParserFreeCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int SetSaxHandlerCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int CreateParserCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static int SaxParserCmd (ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]);
static void XercesSAXDeleteProc(ClientData clientData);
static void ParserThreadDeleteProc(ClientData clientData); 

/*
 * Parser options that are readable and possibly writable
 */

static char *options[] = {
    "-elementstartcommand",
    "-elementendcommand", 
    "-characterdatacommand", 
    "-processinginstructioncommand", 
    "-externalentitycommand", 
    "-defaultcommand",
    "-unparsedentitydeclcommand", 
    "-notationdeclcommand", 
    "-unknownencodingcommand",
    "-commentcommand",
    "-notstandalonecommand",
    "-startcdatasectioncommand",
    "-endcdatasectioncommand",
    "-defaultexpandinternalentities",
    "-elementdeclcommand",
    "-attlistdeclcommand",
    "-startdoctypedeclcommand",
    "-enddoctypedeclcommand",
    "-paramentityparsing",
    "-baseurl", 
    "-final", 
    "-ignorewhitespace", 
    "-handlernamespace", 
    "-validator", 
    "-validate", 
    "-usedom", 
    "-location", 
    "-document",
    "-warningsareerrors",
    "-lasterrors", 
    "-lastwarnings",
    "-namespace",
    NULL
};
enum options {
    PC_ELEMENTSTART,
    PC_ELEMENTEND, 
    PC_CHARACTERDATA, 
    PC_PROCESSINGINSTRUCTION, 
    PC_EXTERNALENTITY, 
    PC_DEFAULT,
    PC_UNPARSEDENTITY, 
    PC_NOTATIONDECL, 
    PC_UNKNOWNENCODING,
    PC_COMMENT, 
    PC_NOTSTANDALONE, 
    PC_STARTCDATASECTION,
    PC_ENDCDATASECTION, 
    PC_DEFAULTEXPANDINTERNALENTITIES, 
    PC_ELEMENTDECL,
    PC_ATTLISTDECL, 
    PC_STARTDOCTYPEDECL, 
    PC_ENDDOCTYPEDECL, 
    PC_PARAMENTITYPARSING,
    PC_BASEURL, 
    PC_FINAL, 
    PC_IGNOREWHITESPACE, 
    PC_HANDLERNAMESPACE,
    PC_VALIDATOR, 
    PC_VALIDATE, 
    PC_USEDOM, 
    PC_LOCATION, 
    PC_DOCUMENT,
    PC_WARNINGSAREERRORS,
    PC_LASTERRORS,
    PC_LASTWARNINGS,
    PC_NAMESPACES
};

/*
 * Parser options that are readonly
 * NB: These options must also appear in the above list
 */

static char *readonlyOptions[] = {
    "-location", 
    "-document",
    "-lasterrors", 
    "-lastwarnings", 
    NULL
};
enum readonlyOptions {
    PCRO_LOCATION, 
    PCRO_DOMDOCUMENT,
    PCRO_LASTERRORS,
    PCRO_LASTWARNINGS
};



/*
 *--------------------------------------------------------------
 *
 * Xercessax_Init --
 *
 *      Does the actual work of initializing the XercesInt package.
 *
 * Results:
 *      Returns a standard Tcl completion code, and leaves an error
 *	    message in interp->result if an error occurs.
 *
 * Side effects:
 *      Initializes COM.
 *
 *--------------------------------------------------------------
 */
int
Xercessax_Init (
    Tcl_Interp *interp)     /* The interpreter for the extension */
{
    XercesIntData *xercesPtr;    /* Data for this interpreter */

#ifdef USE_TCL_STUBS
    if (Tcl_InitStubs(interp, "8.0", 0) == NULL) {
        return TCL_ERROR;
    }
#endif

    /*
     * Do once-per-extension initialization
     */ 
#ifdef XERCESSAX_DEBUG
    int *tsdPtr;
    Tcl_MutexLock(&xercessaxMutex);
    interpCount++;
    if (!haveObjects) {
        haveObjects = 1;
        _CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE );
	_CrtSetReportFile( _CRT_WARN, _CRTDBG_FILE_STDOUT );
	_CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE );
	_CrtSetReportFile( _CRT_ERROR, _CRTDBG_FILE_STDOUT );
	_CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE );
	_CrtSetReportFile( _CRT_ASSERT, _CRTDBG_FILE_STDOUT );
	// _CrtSetAllocHook( MyAllocHook );
        _CrtMemCheckpoint(&memState);
    }


    tsdPtr = (int*) Tcl_GetThreadData(&dataKey, sizeof(int));

    if (*tsdPtr == 0) {
	*tsdPtr = 1;
        threadCount++;
        Tcl_CreateThreadExitHandler(ParserThreadDeleteProc,
		(ClientData) NULL);
    }
    Tcl_MutexUnlock(&xercessaxMutex); 
#endif

   

    /*
     * Create data used by this interpreter
     */
    xercesPtr = new XercesIntData;
    memset(xercesPtr, 0, sizeof(XercesIntData));
    //sprintf(xercesPtr->_hint, "intd %d %d\n", threadCount, interpCount);
    // Tcl_SetAssocData(interp, PACKAGE_NAME, TclCom_InterpDataDeleteProc, interpDataPtr);

    try {
        XMLPlatformUtils::Initialize();
    }

    catch (const XMLException& catchInit) {
        Tcl_AppendResult(interp, "couldn't initialize XMLPlatformUtils:", catchInit.getMessage(), (char *) NULL);
        return TCL_ERROR;
    }

#ifdef XERCESSAX_DEBUG
    _CrtMemCheckpoint(&memState);
#endif


    Tcl_CreateObjCommand(interp, NAMESPACE "createParser", CreateParserCmd,
            (ClientData) xercesPtr, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateExitHandler(XercesSAXDeleteProc, xercesPtr);

#ifdef UNDEF
    Tcl_CreateObjCommand(interp, NAMESPACE "saxParse", SaxParseCmd,
            (ClientData) xercesPtr, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateObjCommand(interp, NAMESPACE "setSAXHandler", SetSaxHandlerCmd,
            (ClientData) xercesPtr, (Tcl_CmdDeleteProc *) NULL);
#endif
    
    return Tcl_PkgProvide(interp, PACKAGE_NAME, VERSION);
}


/*
 *----------------------------------------------------------------------------
 *
 * ParserDeleteCmd --
 *
 *	Called when a parser is deleted.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Memory structures are freed.
 *
 *----------------------------------------------------------------------------
 */

static void
ParserDeleteCmd(ClientData clientData)
{
    XercesParserData *parserPtr = (XercesParserData *) clientData;

    Tcl_DecrRefCount(parserPtr->parserHandle);

    if (parserPtr->elementStartCommand) {
	Tcl_DecrRefCount(parserPtr->elementStartCommand);
    }

    if (parserPtr->elementEndCommand) {
	Tcl_DecrRefCount(parserPtr->elementEndCommand);
    }

    if (parserPtr->characterDataCommand) {
	Tcl_DecrRefCount(parserPtr->characterDataCommand);
    }

    if (parserPtr->processingInstructionCommand) {
	Tcl_DecrRefCount(parserPtr->processingInstructionCommand);
    }

    if (parserPtr->externalEntityCommand) {
	Tcl_DecrRefCount(parserPtr->externalEntityCommand);
    }

    if (parserPtr->unknownEncodingCommand) {
	Tcl_DecrRefCount(parserPtr->unknownEncodingCommand);
    }

    if (parserPtr->commentCommand) {
	Tcl_DecrRefCount(parserPtr->commentCommand);
    }

    if (parserPtr->notStandaloneCommand) {
	Tcl_DecrRefCount(parserPtr->notStandaloneCommand);
    }

    if (parserPtr->startCdataSectionCommand) {
	Tcl_DecrRefCount(parserPtr->startCdataSectionCommand);
    }

    if (parserPtr->elementDeclCommand) {
	Tcl_DecrRefCount(parserPtr->elementDeclCommand);
    }

    if (parserPtr->attlistDeclCommand) {
	Tcl_DecrRefCount(parserPtr->attlistDeclCommand);
    }

    if (parserPtr->startDoctypeDeclCommand) {
	Tcl_DecrRefCount(parserPtr->startDoctypeDeclCommand);
    }

    if (parserPtr->endDoctypeDeclCommand) {
	Tcl_DecrRefCount(parserPtr->endDoctypeDeclCommand);
    }

    if (parserPtr->resolverPtr) {
	delete parserPtr->resolverPtr;
    }

    if (parserPtr->saxAdvHandler) {
	delete parserPtr->saxAdvHandler;
    }

    if (parserPtr->docTypeHandler) {
	delete parserPtr->docTypeHandler;
    }

    if (parserPtr->saxHandler) {
	delete parserPtr->saxHandler;
    }

    if (parserPtr->saxParserPtr) {
	delete parserPtr->saxParserPtr;
    }

    Tcl_DStringFree(&parserPtr->parseErrors);
    Tcl_DStringFree(&parserPtr->parseErrorsAndWarnings);

    ckfree ((char *) parserPtr);
}



/*
 *--------------------------------------------------------------
 *
 * CreateParserCmd --
 *
 *      This procedure implements ::xercessax::createParser
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
CreateParserCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    XercesIntData *xercesPtr = (XercesIntData *) clientData;
    XercesParserData *parserPtr;
    int doEscapes = 1;
    char handleBuffer[64];
   
    /*
     * Allocate a new parser
     */

    parserPtr = (XercesParserData *) ckalloc(sizeof(XercesParserData));
    memset(parserPtr, 0, sizeof(XercesParserData));
    Tcl_DStringInit(&parserPtr->parseErrors);
    Tcl_DStringInit(&parserPtr->parseErrorsAndWarnings);
    parserPtr->saxParserPtr = new SAXParser;
    parserPtr->saxParserPtr->setDoValidation(0);
    parserPtr->saxHandler = new SAXTclHandler(doEscapes?1:0, interp, parserPtr);
    parserPtr->saxParserPtr->setErrorHandler(parserPtr->saxHandler);
    parserPtr->saxParserPtr->setDocumentHandler(parserPtr->saxHandler);
    parserPtr->saxParserPtr->setDTDHandler(parserPtr->saxHandler);
    parserPtr->resolverPtr = new SAXTclResolver(interp, parserPtr);
    parserPtr->saxParserPtr->setEntityResolver(parserPtr->resolverPtr);
    parserPtr->saxAdvHandler = new SAXAdvTclHandler(interp, parserPtr);
    parserPtr->saxParserPtr->installAdvDocHandler(parserPtr->saxAdvHandler);
    parserPtr->docTypeHandler = new SAXTclDocTypeHandler(interp, parserPtr);

    DTDValidator& validator = (class DTDValidator &) parserPtr->saxParserPtr->getValidator();
    validator.setDocTypeHandler(parserPtr->docTypeHandler);

    sprintf(handleBuffer, NAMESPACE "xsp%x", parserPtr);

    Tcl_CreateObjCommand(interp, handleBuffer, SaxParserCmd,
            (ClientData) parserPtr, (Tcl_CmdDeleteProc *) ParserDeleteCmd);
    parserPtr->parserHandle = Tcl_NewStringObj(handleBuffer, -1);
    Tcl_IncrRefCount(parserPtr->parserHandle);

    Tcl_AppendResult(interp, handleBuffer, (char *) NULL);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ParserConfigureCmd --
 *
 *      This procedure implements ::tclcom::debug.
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
ParserConfigureCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    XercesParserData *parserPtr = (XercesParserData *) clientData;
    int i, index, stringLength;
    char *version;

    if (objc < 2) {
        Tcl_WrongNumArgs(interp, 2, objv, "<command>");
        return TCL_ERROR;
    }

    /*
     * Check for matched option / value pairs
     */

    if (objc & 1) {
        Tcl_AppendResult(interp, "missing option value", (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Check for valid options
     */

    for (i = 2; i < objc; i+=2) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, 
	        &index) != TCL_OK) {
	    return TCL_ERROR;
        }
        if (Tcl_GetIndexFromObj(interp, objv[i], readonlyOptions, "option", 0, 
	        &index) == TCL_OK) {
	    Tcl_AppendResult(interp, "option ", Tcl_GetString(objv[i]), "\" is readonly", 
                (char *) NULL);
	    return TCL_ERROR;
        } else {
            Tcl_ResetResult(interp);
        }
    }

    for (i = 2; i < objc; i++) {
        if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, 
	        &index) != TCL_OK) {
	    return TCL_ERROR;
        }
	switch (index) {
            case PC_ELEMENTSTART:
                i++;
                if (parserPtr->elementStartCommand) {
                    Tcl_DecrRefCount(parserPtr->elementStartCommand);
                    parserPtr->elementStartCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->elementStartCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->elementStartCommand);
                }
                break;

            case PC_ELEMENTEND:
                i++;
                if (parserPtr->elementEndCommand) {
                    Tcl_DecrRefCount(parserPtr->elementEndCommand);
                    parserPtr->elementEndCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->elementEndCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->elementEndCommand);
                }
                break;

             case PC_CHARACTERDATA:
                i++;
                if (parserPtr->characterDataCommand) {
                    Tcl_DecrRefCount(parserPtr->characterDataCommand);
                    parserPtr->characterDataCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->characterDataCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->characterDataCommand);
                }
                break;

	    case PC_PROCESSINGINSTRUCTION:
                i++;
                if (parserPtr->processingInstructionCommand) {
                    Tcl_DecrRefCount(parserPtr->processingInstructionCommand);
                    parserPtr->processingInstructionCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->processingInstructionCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->processingInstructionCommand);
                }
                break;

	    case PC_EXTERNALENTITY:
                i++;
                if (parserPtr->externalEntityCommand) {
                    Tcl_DecrRefCount(parserPtr->externalEntityCommand);
                    parserPtr->externalEntityCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->externalEntityCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->externalEntityCommand);
                }
                break;

	    case PC_DEFAULT:
                i++;
                if (parserPtr->defaultCommand) {
                    Tcl_DecrRefCount(parserPtr->defaultCommand);
                    parserPtr->defaultCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->defaultCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->defaultCommand);
                }
                break;

            case PC_UNPARSEDENTITY:
                i++;
                if (parserPtr->unparsedEntityCommand) {
                    Tcl_DecrRefCount(parserPtr->unparsedEntityCommand);
                    parserPtr->unparsedEntityCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->unparsedEntityCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->unparsedEntityCommand);
                }
                break;

            case PC_NOTATIONDECL:
                i++;
                if (parserPtr->notationDeclarationCommand) {
                    Tcl_DecrRefCount(parserPtr->notationDeclarationCommand);
                    parserPtr->notationDeclarationCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->notationDeclarationCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->notationDeclarationCommand);
                }
                break;

            case PC_UNKNOWNENCODING:
                i++;
                if (parserPtr->unknownEncodingCommand) {
                    Tcl_DecrRefCount(parserPtr->unknownEncodingCommand);
                    parserPtr->unknownEncodingCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->unknownEncodingCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->unknownEncodingCommand);
                }
                break;
            
            case PC_COMMENT:
                i++;
                if (parserPtr->commentCommand) {
                    Tcl_DecrRefCount(parserPtr->commentCommand);
                    parserPtr->commentCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->commentCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->commentCommand);
                }
                break;
             
            case PC_NOTSTANDALONE:
                i++;
                if (parserPtr->notStandaloneCommand) {
                    Tcl_DecrRefCount(parserPtr->notStandaloneCommand);
                    parserPtr->notStandaloneCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->notStandaloneCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->notStandaloneCommand);
                }
                break;

            case PC_STARTCDATASECTION:
                i++;
                if (parserPtr->startCdataSectionCommand) {
                    Tcl_DecrRefCount(parserPtr->startCdataSectionCommand);
                    parserPtr->startCdataSectionCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->startCdataSectionCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->startCdataSectionCommand);
                }
                break;

            case PC_ENDCDATASECTION:
                i++;
                if (parserPtr->endCdataSectionCommand) {
                    Tcl_DecrRefCount(parserPtr->endCdataSectionCommand);
                    parserPtr->endCdataSectionCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->endCdataSectionCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->endCdataSectionCommand);
                }
                break;

            case PC_ELEMENTDECL:
                i++;
                if (parserPtr->elementDeclCommand) {
                    Tcl_DecrRefCount(parserPtr->elementDeclCommand);
                    parserPtr->elementDeclCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->elementDeclCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->elementDeclCommand);
                }
                break;

            case PC_ATTLISTDECL:
                i++;
                if (parserPtr->attlistDeclCommand) {
                    Tcl_DecrRefCount(parserPtr->attlistDeclCommand);
                    parserPtr->attlistDeclCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->attlistDeclCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->attlistDeclCommand);
                }
                break;

            case PC_STARTDOCTYPEDECL:
                i++;
                if (parserPtr->startDoctypeDeclCommand) {
                    Tcl_DecrRefCount(parserPtr->startDoctypeDeclCommand);
                    parserPtr->startDoctypeDeclCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->startDoctypeDeclCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->startDoctypeDeclCommand);
                }
                break;

            case PC_ENDDOCTYPEDECL:
                i++;
                if (parserPtr->endDoctypeDeclCommand) {
                    Tcl_DecrRefCount(parserPtr->endDoctypeDeclCommand);
                    parserPtr->endDoctypeDeclCommand = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->endDoctypeDeclCommand = objv[i];
                    Tcl_IncrRefCount(parserPtr->endDoctypeDeclCommand);
                }
                break;

            case PC_BASEURL:
                i++;
                if (parserPtr->baseURL) {
                    Tcl_DecrRefCount(parserPtr->baseURL);
                    parserPtr->baseURL = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->baseURL = objv[i];
                    Tcl_IncrRefCount(parserPtr->baseURL);
                }
                break;

            case PC_HANDLERNAMESPACE:
                i++;
                if (parserPtr->handlerNamespace) {
                    Tcl_DecrRefCount(parserPtr->handlerNamespace);
                    parserPtr->handlerNamespace = NULL;
                }
                Tcl_GetStringFromObj(objv[i], &stringLength);
                if (stringLength) {
                    parserPtr->handlerNamespace = objv[i];
                    Tcl_IncrRefCount(parserPtr->handlerNamespace);
                }
                break;

	    case PC_FINAL:
                i++;
	        if (Tcl_GetBooleanFromObj(interp, objv[i], &parserPtr->final) != TCL_OK) {
	            return TCL_ERROR;
	        }
	        break;

            case PC_IGNOREWHITESPACE:
                i++;
	        if (Tcl_GetBooleanFromObj(interp, objv[i], &parserPtr->ignoreWhiteSpace) != TCL_OK) {
	            return TCL_ERROR;
	        }
	        break;

            case PC_VALIDATE:
                i++;
		if (Tcl_GetBooleanFromObj(interp, objv[i], &parserPtr->validate) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;

	    case PC_NAMESPACES:
                i++;
		if (Tcl_GetBooleanFromObj(interp, objv[i], &parserPtr->namespaces) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;

	    case PC_USEDOM:
                i++;
		if (Tcl_GetBooleanFromObj(interp, objv[i], &parserPtr->useDom) != TCL_OK) {
		    return TCL_ERROR;
		}
                if (parserPtr->useDom && !parserPtr->haveDomPackage) {
                    version = Tcl_PkgRequire(interp, "tcldompro", "1.1", 0);
                    if (version == NULL || (strlen(version) == 0)) {
                        return TCL_ERROR;
                    } else {
                        parserPtr->haveDomPackage = 1;
                    }
                }
	        break;

	    case PC_WARNINGSAREERRORS:
                i++;
		if (Tcl_GetBooleanFromObj(interp, objv[i], &parserPtr->treatWarningsAsErrors) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;
	}
    } 

    parserPtr->saxParserPtr->setDoValidation(parserPtr->validate?1:0);
    parserPtr->saxParserPtr->setDoNamespaces(parserPtr->namespaces?1:0);

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ParserCgetCmd --
 *
 *      This procedure implements $parser cget
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
ParserCgetCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    XercesParserData *parserPtr = (XercesParserData *) clientData;
    int index;

    if (objc != 3) {
        Tcl_WrongNumArgs(interp, 1, objv, "cget <option>");
        return TCL_ERROR;
    }  

    /*
     * Check for valid options
     */

	if (Tcl_GetIndexFromObj(interp, objv[2], options, "option", 0, 
        &index) != TCL_OK) {
        return TCL_ERROR;
    }

    switch (index) {
        case PC_ELEMENTSTART:
            if (parserPtr->elementStartCommand) {
                Tcl_SetObjResult(interp, parserPtr->elementStartCommand);
            }
            break;

        case PC_ELEMENTEND:
            if (parserPtr->elementEndCommand) {
                Tcl_SetObjResult(interp, parserPtr->elementEndCommand);
            }
            break;

        case PC_CHARACTERDATA:
            if (parserPtr->characterDataCommand) {
                Tcl_SetObjResult(interp, parserPtr->characterDataCommand);
            }
            break;

        case PC_PROCESSINGINSTRUCTION:
            if (parserPtr->processingInstructionCommand) {
                Tcl_SetObjResult(interp, parserPtr->processingInstructionCommand);
            }
            break;

        case PC_EXTERNALENTITY:
            if (parserPtr->externalEntityCommand) {
                Tcl_SetObjResult(interp, parserPtr->externalEntityCommand);
            }
            break;

        case PC_DEFAULT:
            if (parserPtr->defaultCommand) {
                Tcl_SetObjResult(interp, parserPtr->defaultCommand);
            }
            break;

        case PC_UNPARSEDENTITY:
            if (parserPtr->unparsedEntityCommand) {
                Tcl_SetObjResult(interp, parserPtr->unparsedEntityCommand);
            }
            break;

        case PC_NOTATIONDECL:
            if (parserPtr->notationDeclarationCommand) {
                Tcl_SetObjResult(interp, parserPtr->notationDeclarationCommand);
            }
            break;

        case PC_UNKNOWNENCODING:
            if (parserPtr->unknownEncodingCommand) {
                Tcl_SetObjResult(interp, parserPtr->unknownEncodingCommand);
            }
            break;

        case PC_BASEURL:
            if (parserPtr->baseURL) {
                Tcl_SetObjResult(interp, parserPtr->baseURL);
            }
            break;

        case PC_HANDLERNAMESPACE:
            if (parserPtr->handlerNamespace) {
                Tcl_SetObjResult(interp, parserPtr->handlerNamespace);
            }
            break;

        case PC_FINAL:
            Tcl_SetObjResult(interp, Tcl_NewBooleanObj(parserPtr->final));
	    break;

        case PC_IGNOREWHITESPACE:
            Tcl_SetObjResult(interp, Tcl_NewBooleanObj(parserPtr->ignoreWhiteSpace));
	    break;

        case PC_VALIDATE:
            Tcl_SetObjResult(interp, Tcl_NewBooleanObj(parserPtr->validate));
	    break;

	case PC_NAMESPACES:
            Tcl_SetObjResult(interp, Tcl_NewBooleanObj(parserPtr->namespaces));
	    break;

        case PC_USEDOM:
            Tcl_SetObjResult(interp, Tcl_NewBooleanObj(parserPtr->useDom));
	    break;

        case PC_LOCATION: {
            char workString[64];
            sprintf(workString, "%d %d", parserPtr->currentLine, parserPtr->currentColumn);
            Tcl_AppendResult(interp, workString, (char *) NULL);
	    break;
        }

        case PC_DOCUMENT: {
            Tcl_Obj *documentHandle;
            if (!parserPtr->useDom) {
                Tcl_AppendResult(interp, "dom not enabled; use -dom option", (char *) NULL);
                return TCL_ERROR;
            }
            if (parserPtr->currentDocument) {
                documentHandle = Tdp_GetDocumentObj(interp, parserPtr->currentDocument);
                if (documentHandle) {
                    Tcl_SetObjResult(interp, documentHandle);
                } else {
                    return TCL_ERROR;
                }
            }
	    break;
        }

	case PC_WARNINGSAREERRORS:
            Tcl_SetObjResult(interp, Tcl_NewBooleanObj(parserPtr->treatWarningsAsErrors));
	    break;

	case PC_LASTERRORS:
	    Tcl_DStringResult(interp, &parserPtr->parseErrors);
	    break;

	case PC_LASTWARNINGS:
	    Tcl_DStringResult(interp, &parserPtr->parseErrorsAndWarnings);
	    break;
    }

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ParserResetCmd --
 *
 *      This procedure implements ::tclcom::debug.
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
ParserResetCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    XercesParserData *parserPtr = (XercesParserData *) clientData;

    delete parserPtr->saxParserPtr;
    parserPtr->parseStatus = 0;
    parserPtr->saxParserPtr = new SAXParser;
    parserPtr->saxParserPtr->setDoValidation(parserPtr->validate?1:0);
    parserPtr->saxParserPtr->setDoNamespaces(parserPtr->namespaces?1:0);
    parserPtr->saxParserPtr->setErrorHandler(parserPtr->saxHandler);
    parserPtr->saxParserPtr->setDocumentHandler(parserPtr->saxHandler);
    parserPtr->saxParserPtr->setDTDHandler(parserPtr->saxHandler);
    parserPtr->saxParserPtr->setEntityResolver(parserPtr->resolverPtr);

    parserPtr->saxParserPtr->installAdvDocHandler(parserPtr->saxAdvHandler);

    /* Reset some state variables */
    parserPtr->currentNode = NULL;
    parserPtr->depth = 0;
    parserPtr->currentDocument = NULL;

    DTDValidator& validator = (class DTDValidator &) parserPtr->saxParserPtr->getValidator();
    validator.setDocTypeHandler(parserPtr->docTypeHandler);
    

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ParserFreeCmd --
 *
 *      This procedure deletes a parser object
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
ParserFreeCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    XercesParserData *parserPtr = (XercesParserData *) clientData;

    Tcl_DeleteCommand(interp, Tcl_GetString(parserPtr->parserHandle));
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ParserParseCmd --
 *
 *      This procedure implements $parser parse
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
ParserParseCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    char *xml;
    int xmlByteCount;
    XercesParserData *parserPtr = (XercesParserData *) clientData;
    MemBufInputSource *xmlSourcePtr;
    // TclMemoryBuffer *xmlSourcePtr;
    char workString[32];

    int doFirst = 0;
    int doNext = 0;
    int parseDone;
    int result = TCL_OK;

    if (objc < 3) {
        Tcl_WrongNumArgs(interp, 1, objv, "parse <xml>");
        return TCL_ERROR;
    }


    xml = Tcl_GetStringFromObj(objv[2], &xmlByteCount);
    parserPtr->parseStatus = 0;

    Tcl_DStringInit(&parserPtr->parseErrors);
    Tcl_DStringInit(&parserPtr->parseErrorsAndWarnings);

    xmlSourcePtr = new MemBufInputSource((const XMLByte *) xml, xmlByteCount, 
        "TestCase", 0);

#ifdef UNDEF
     xmlSourcePtr = new TclMemoryBuffer((const XMLByte *) xml, xmlByteCount,
        "TestCase", 0);
#endif

    try {
	if (doFirst) {
            parseDone = parserPtr->saxParserPtr->parseFirst(*xmlSourcePtr, parserPtr->parseToken, 0);
	} else if (doNext) {
	    parseDone = parserPtr->saxParserPtr->parseNext(parserPtr->parseToken);
	} else {
	    parserPtr->saxParserPtr->parse(*xmlSourcePtr, 0);
            parseDone = 1;
	}
    }

    catch (const SAXParseException& e) {
            Tcl_DString errorString;
	    Tcl_DString wholeError;
            Tcl_DStringInit(&errorString);
	    Tcl_DStringInit(&wholeError);
	    sprintf(workString, "line: %d ", e.getLineNumber());
	    Tcl_DStringAppend(&wholeError, workString, -1);
	    sprintf(workString, "column: %d ", e.getColumnNumber());
	    Tcl_DStringAppend(&wholeError, workString, -1);
            Tcl_UniCharToUtfDString(e.getMessage(), 
			XMLString::stringLen(e.getMessage()), &errorString);
	    Tcl_DStringAppend(&wholeError, Tcl_DStringValue(&errorString), -1);
	    Tcl_DStringResult(interp, &wholeError);
            Tcl_DStringFree(&errorString);
	    Tcl_DStringFree(&wholeError);

            result = TCL_ERROR;
        }

    delete xmlSourcePtr;

    if (result != TCL_OK) {
	return result;
    } else if (parserPtr->parseStatus != TCL_OK) {
        Tcl_DStringResult(interp, &parserPtr->parseErrors);
        return parserPtr->parseStatus;
    } else {
        Tcl_ResetResult(interp);
    }

    return result;
}


/*
 *--------------------------------------------------------------
 *
 * SaxParserCmd --
 *
 *      This procedure implements $parser <somecommand>
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
SaxParserCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    XercesParserData *parserPtr = (XercesParserData *) clientData;
    int index;

    static char *parserCommands[] = {
	    "cget", "configure", "free", "parse", "reset", NULL
    };
    enum parserCommands {
	    PARSER_CGET, PARSER_CONFIGURE, PARSER_FREE, PARSER_PARSE, PARSER_RESET
    };

    if (objc < 2) {
        Tcl_WrongNumArgs(interp, 2, objv, "<command>");
        return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], parserCommands, "command", 0, 
        &index) != TCL_OK) {
	    return TCL_ERROR;
	}

    switch (index) {
        case PARSER_CGET:
            return ParserCgetCmd(clientData, interp, objc, objv);
        case PARSER_CONFIGURE:
            return ParserConfigureCmd(clientData, interp, objc, objv);
        case PARSER_PARSE:
            return ParserParseCmd(clientData, interp, objc, objv);
	case PARSER_FREE:
	    return ParserFreeCmd(clientData, interp, objc, objv);
        case PARSER_RESET:
            return ParserResetCmd(clientData, interp, objc, objv);
        default:
            Tcl_AppendResult(interp, "unknown parser command", (char *) NULL);
            return TCL_ERROR;
    } 
   
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * SetSaxHandlerCmd --
 *
 *      This procedure implements ::tclcom::debug.
 *
 * Results:
 *      TCL_OK;
 *
 * Side effects:
 *      Depends on command option.
 *
 *--------------------------------------------------------------
 */

int
SetSaxHandlerCmd (
    ClientData clientData,	    /* Per interpreter data */
    Tcl_Interp *interp,		    /* Current interpreter. */
    int objc,			        /* Number of arguments. */
    Tcl_Obj *CONST objv[])	    /* The argument objects. */
{
    char *nameSpace;
    XercesParserData *parserPtr = (XercesParserData *) clientData;
    int doEscapes = 0;

    if (objc < 2) {
        Tcl_WrongNumArgs(interp, 2, objv, "<handleNamespace>");
        return TCL_ERROR;
    }

    nameSpace = Tcl_GetStringFromObj(objv[1], NULL);

#ifdef UNDEF

    if (parserPtr->saxHandler == NULL) {
        parserPtr->saxHandler = new SAXTclHandler(doEscapes?1:0);
    }
#endif

    parserPtr->saxHandler->setTclNamespace(interp, nameSpace);
  
   
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------------
 *
 * XercesSAXDeleteProc --
 *
 *	Called when the createParser command is deleted
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Memory structures are freed.
 *
 *----------------------------------------------------------------------------
 */

static void
XercesSAXDeleteProc(ClientData clientData)
{
    XercesIntData *xercesPtr = (XercesIntData *) clientData;
    delete xercesPtr;
}


/*
 *--------------------------------------------------------------
 *
 * ParsrThreadDeleteProc --
 *
 *      Thread delete handler. This procedure does nothing
 *      useful other than invoke the Windows memory dump code
 *      at point where all memory should have been freed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */
static void ParserThreadDeleteProc(
    ClientData clientData)          /* Thread-specific data for the interpreter */
{
    /*
     * Decrement the debug thread counter, and dump memory leaks
     * if this was the last thread to exit.
     */
#ifdef XERCESSAX_DEBUG
    Tcl_MutexLock(&xercessaxExtMutex);
    threadCount--;
    if (threadCount == 0) {
        _CrtMemDumpAllObjectsSince(&memState);
    }
    Tcl_MutexUnlock(&xercessaxExtMutex);
#endif
}

