/*
** @(#)$Id: tclsql.c,v 1.1.1.1 2000/03/01 17:53:41 stanton Exp $
**
** $Product: ISQLTCL Version 5.00a1 (1999-11-24) $
**
** Informix Database Support for Tcl/Tk
**
** Author: Kumar Srinivas
** Helper: Jonathan Leffler (jonathan.leffler@informix.com)
*/

#ifndef lint
static char const rcsid[] = "$Id: tclsql.c,v 1.1.1.1 2000/03/01 17:53:41 stanton Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tcl.h"
#include "isqltcl.h"

/* Tcl_PkgProvide is not const-correct so these are not const variables */
static char isql_version[] = "5.00";
static char isql_package[] = "isql";

/*
** Report general error in sql command.
** Either no sub-command or invalid sub-command.
*/
static int tcl_sql_usage(Tcl_Interp *interp)
{
    Tcl_AppendResult(interp, "wrong args to sql: should be one of open,\n\
    fetch,close,run,exists,reopen,explain,geterror,sqlca,sqlda,\n\
    sqld,database,getdatabase,finish,connect,disconnect,setconnection,\n\
    readblob,writeblob,begin,commit,rollback\n",
    (char *)NULL);
    return TCL_ERROR;
}

/* Usage error in sql sub-command */
static int tcl_sql_cmdusage(Tcl_Interp *interp, char *arg0, char *usage)
{
    Tcl_AppendResult(interp, "wrong # args: should be \"sql ", arg0, " ",
        usage, "\"", (char *) NULL);
    return TCL_ERROR;
}

/* Report SQL error */
static int tcl_sql_error(Tcl_Interp *interp, char *arg0)
{
    Tcl_AppendResult(interp, "\"sql ", arg0, ": ", sql_geterror(),
    "\"", (char *) NULL);
    return TCL_ERROR;
}

/* Report success at the Tcl/Tk level; return value (ret) may not be zero */
static int tcl_sql_ok(Tcl_Interp *interp, int ret)
{
    char buf[25];

    sprintf(buf, "%d", ret);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

/* Convert string numarg to integer or complain */
/* Alternatively, use strtol() */
static int tcl_cvt_intarg(Tcl_Interp *interp, char *cmd, char *argnum, char *numarg)
{
    int fd;
    char buf[25];

    if (sscanf(numarg, "%d%1s", &fd, buf) != 1)
    {
        Tcl_AppendResult(interp, "argument ", argnum, "to sql ", cmd,
            " not an integer ==> ", numarg, (char *) NULL);
        return TCL_ERROR;
    }
    return fd;
}

static int
tcl_sql_run(Tcl_Interp *interp, int argc, char **argv)
{
    char *stmt;
    char *arg0;
    int ret;

    if (argc < 2)
        return tcl_sql_cmdusage(interp, argv[0], "sql-stmt ?arg ...?");
    arg0 = argv[0];
    stmt = argv[1];
    ret = sql_run(stmt, argc - 2, argv + 2);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_open(Tcl_Interp *interp, int argc, char **argv)
{
    char *stmt;
    char *arg0;
    int ret;

    if (argc < 2)
        return tcl_sql_cmdusage(interp, argv[0], "sql-stmt ?arg ...?");
    arg0 = argv[0];
    stmt = argv[1];
    ret = sql_open(stmt, argc - 2, argv + 2);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_close(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int fd;
    int ret;

    if (argc != 2)
        return tcl_sql_cmdusage(interp, argv[0], "fd");
    arg0 = argv[0];
    if ((fd = tcl_cvt_intarg(interp, arg0, "", argv[1])) < 0)
        return TCL_ERROR;
    ret = sql_close(fd);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_fetch(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int fd;
    int ret;
    int retargc;
    int dostrip;
    char **retargv;

    if (argc < 2)
        return tcl_sql_cmdusage(interp, argv[0], "fd ?clip?");
    arg0 = argv[0];
    argc--;
    argv++;
    if ((fd = tcl_cvt_intarg(interp, arg0, "", argv[0])) < 0)
        return TCL_ERROR;
    if (argc >= 2 && argv[1] && sscanf(argv[1], "%d", &ret) == 1 && ret == 1)
        dostrip = 1;
    else
        dostrip = 0;
    ret = sql_fetch(fd);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    if (ret == 0) {
        retargv = sql_values(fd, &retargc, dostrip);
        if (!retargv)
            return tcl_sql_error(interp, arg0);
        interp->result = Tcl_Merge(retargc, retargv);
        interp->freeProc = (Tcl_FreeProc *) free;
        return TCL_OK;
    }
    /* What's the return value of ret?  SQLNOTFOUND? */
    return TCL_OK;
}

static int
tcl_sql_exists(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    char *table;
    char *column;
    char *value = NULL;
    char *where = NULL;
    int ret;

    if (argc < 3 || argc > 5)
        return tcl_sql_cmdusage(interp, argv[0], "table column ?value ?where?");
    arg0 = argv[0];
    table = argv[1];
    column = argv[2];
    if (argv[3]) {
        value = argv[3];
        if (argv[4])
            where = argv[4];
    }
    ret = sql_exists(table, column, value, where);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_reopen(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    int fd;

    if (argc < 2)
        return tcl_sql_cmdusage(interp, argv[0], "fd ?arg ...?");
    arg0 = argv[0];
    if ((fd = tcl_cvt_intarg(interp, arg0, "", argv[1])) < 0)
        return TCL_ERROR;
    ret = sql_reopen(fd, argc - 2, argv + 2);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_explain(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    int flag;

    if (argc != 2)
        return tcl_sql_cmdusage(interp, argv[0], "arg");
    arg0 = argv[0];
    if ((flag = tcl_cvt_intarg(interp, arg0, "", argv[1])) < 0)
        return TCL_ERROR;
    ret = sql_explain(ret);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_geterror(Tcl_Interp *interp, int argc, char **argv)
{
    char *p;

    /* Validate zero arguments? */
    p = sql_geterror();
    if (!p)
        p = "";
    Tcl_SetResult(interp, p, TCL_VOLATILE);
    return TCL_OK;
}

static int
tcl_sqlca(Tcl_Interp *interp, int argc, char **argv)
{
    int retargc;
    char **retargv;
    char *arg0 = argv[0];

    /* Validate zero arguments? */
    retargv = sql_sqlca(&retargc);
    if (!retargv)
        return tcl_sql_error(interp, arg0);
    interp->result = Tcl_Merge(retargc, retargv);
    interp->freeProc = (Tcl_FreeProc *) free;
    return TCL_OK;
}

static int
tcl_sqld(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    int fd;
    int type_ld;

    if (argc != 3)
        return tcl_sql_cmdusage(interp, argv[0], "fd in");
    arg0 = argv[0];
    if ((fd = tcl_cvt_intarg(interp, arg0, " 1", argv[1])) < 0)
        return TCL_ERROR;
    if ((type_ld = tcl_cvt_intarg(interp, arg0, " 2", argv[2])) < 0)
        return TCL_ERROR;
    ret = sql_sqld(fd, type_ld);
    if (ret == -2)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sqlda(Tcl_Interp *interp, int argc, char **argv)
{
    int retargc;
    char **retargv;
    char *arg0;
    int fd;
    int num;
    int type_ld;

    if (argc < 4)
        return tcl_sql_cmdusage(interp, argv[0], "fd in num");
    arg0 = argv[0];
    if ((fd = tcl_cvt_intarg(interp, arg0, " 1", argv[1])) < 0)
        return TCL_ERROR;
    if ((type_ld = tcl_cvt_intarg(interp, arg0, " 2", argv[2])) < 0)
        return TCL_ERROR;
    if ((num = tcl_cvt_intarg(interp, arg0, " 3", argv[3])) < 0)
        return TCL_ERROR;
    retargv = sql_sqlda(fd, type_ld, num, &retargc);
    if (!retargv)
        return tcl_sql_error(interp, arg0);
    interp->result = Tcl_Merge(retargc, retargv);
    interp->freeProc = (Tcl_FreeProc *) free;
    return TCL_OK;
}

static int
tcl_sql_sqlcols(Tcl_Interp *interp, int argc, char **argv)
{
    int retargc;
    char **retargv;
    char *arg0;
    int fd;
    int type_ld;

    if (argc != 3)
        return tcl_sql_cmdusage(interp, argv[0], "fd in");
    arg0 = argv[0];
    if ((fd = tcl_cvt_intarg(interp, arg0, " 1", argv[1])) < 0)
        return TCL_ERROR;
    if ((type_ld = tcl_cvt_intarg(interp, arg0, " 2", argv[2])) < 0)
        return TCL_ERROR;
    retargv = sql_sqlcols(fd, type_ld, arg0, &retargc);
    if (!retargv)
        return tcl_sql_error(interp, arg0);
    interp->result = Tcl_Merge(retargc, retargv);
    interp->freeProc = (Tcl_FreeProc *) free;
    return TCL_OK;
}

static int
tcl_sql_getdatabase(Tcl_Interp *interp, int argc, char **argv)
{
    char *p;

    /* Validate zero arguments? */
    p = sql_getdatabase();
    if (!p) p = "";
    Tcl_SetResult(interp, p, TCL_VOLATILE);
    return TCL_OK;
}

static int
tcl_sql_finish(Tcl_Interp *interp, int argc, char **argv)
{
    int ret;
    char *arg0;

    /* Validate zero arguments? */
    arg0 = argv[0];

    ret = sql_finish();
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_begin_work(Tcl_Interp *interp, int argc, char **argv)
{
    int ret;
    char *arg0;

    /* Validate zero arguments? */
    arg0 = argv[0];
    ret = sql_begin();
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_commit(Tcl_Interp *interp, int argc, char **argv)
{
    int ret;
    char *arg0;

    /* Validate zero arguments? */
    arg0 = argv[0];

    ret = sql_commit();
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_rollback(Tcl_Interp *interp, int argc, char **argv)
{
    int ret;
    char *arg0;

    /* Validate zero arguments? */
    arg0 = argv[0];

    ret = sql_rollback();
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_database(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char *dbname = NULL;
    int exclusive = 0;

    if (argc < 2 || argc > 3)
        return tcl_sql_cmdusage(interp, argv[0], "dbname ?exclusive?");
    arg0 = argv[0];
    dbname = argv[1];
    if (argc == 3) {
        /* Test too simple? */
        if (argv[2][0] == 'e' || argv[2][0] == '1')
            exclusive = 1;
    }
    ret = sql_database(dbname, exclusive);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_readblob(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char *table = NULL;
    char *column = NULL;
    char *rowidclause = NULL;
    char *intofile = NULL;
    int appendmode = 0;

    arg0 = argv[0];
    if (argc < 5 || argc > 6)
        return tcl_sql_cmdusage(interp, argv[0],
                    "table column rowid intofile ?append");
    table = argv[1];
    column = argv[2];
    rowidclause = argv[3];
    intofile = argv[4];
    if (argc == 6) {
        /* Test too simple? */
        if (argv[5][0] == 'a' || argv[5][0] == '1')
            appendmode = 1;
    }
    ret = sql_readblob(table, column, rowidclause, intofile, appendmode);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_writeblob(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char *table = NULL;
    char *column = NULL;
    char *rowidclause = NULL;
    char *fromfile = NULL;
    int fromsize = -1;

    arg0 = argv[0];
    if (argc < 5 || argc > 6)
        return tcl_sql_cmdusage(interp, argv[0],
                  "table column rowid fromfile|null ?size");
    table = argv[1];
    column = argv[2];
    rowidclause = argv[3];
    fromfile = argv[4];
    if (argc == 6)
        fromsize = atoi(argv[5]);
    ret = sql_writeblob(table, column, rowidclause, fromfile, fromsize);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sqlsigcmd(Tcl_Interp *interp, int argc, char **argv)
{
    int ret;
    char *arg0;
    char *p;

    /* Validate zero arguments? */
    arg0 = p = argv[0];
    if (p) {    /* Canonicalize names ?dubious? */
    if (strncmp(p, "sqlbreak", 4) == 0) p = "sqlbreak";
    else if (strncmp(p, "sqlexit", 4) == 0) p = "sqlexit";
    else if (strncmp(p, "sqldone", 5) == 0) p = "sqldone";
    else if (strncmp(p, "sqldetach", 5) == 0) p = "sqldetach";
    }

    ret = sql_sigcmd(p);
    if (ret == -1)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_connect(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char *dbname = NULL;
    char *uid = NULL;
    char *pw = NULL;
    char *cname = NULL;
    int i;
    int with_contrans = 0;

    /**
    ** Argument validation is complex, but there can be at most 9 args:
    ** CONNECT dbname AS conname USER username PASSWORD password WCT
    ** Also note that ISQLTCL drops the 'TO' which ESQL/C requires.
    */
    arg0 = argv[0];
    if (argc >= 2) dbname = argv[1];
    for (i = 2; i < argc; i++)
    {
        if (strncmp(argv[i], "user", 3) == 0) {
            uid = argv[i+1];
            i++;
        } else if (strncmp(argv[i], "password",1) == 0 ||
            strncmp(argv[i], "using",3) == 0) {
            pw = argv[i+1];
            i++;
        } else if (strncmp(argv[i], "with_concurrent_transaction",1) == 0) {
            with_contrans = 1;
        } else if (strncmp(argv[i], "as",1) == 0) {
            cname = argv[i+1];
            i++;
        }
    }

    /*
    if (argc >= 3) uid = argv[2];
    if (argc >= 4) pw = argv[3];
    if (argc >= 5) cname = argv[4];
    if (argc >= 6) {
    if (argv[5][0] == 'w' || argv[5][0] == '1')
        with_contrans = 1;
    }
    */

    ret = sql_connect(dbname, uid, pw, cname, with_contrans);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_disconnect(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char *conn = NULL;

    arg0 = argv[0];
    if (argc != 2)
        return tcl_sql_cmdusage(interp, arg0, "connection");
    conn = argv[1];
    ret = sql_disconnect(conn);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

static int
tcl_sql_setconnection(Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char *conn = NULL;
    int dormant = 0;

    arg0 = argv[0];
    if (argc < 2 || argc > 3)
        return tcl_sql_cmdusage(interp, arg0, "connection ?dormant?");
    conn = argv[1];
    if (argc == 3) {
        /* Test too simple? */
        if (argv[2][0] == 'd' || argv[2][0] == '1') {
            dormant = 1;
        }
    }
    ret = sql_setconnection(conn, dormant);
    if (ret < 0)
        return tcl_sql_error(interp, arg0);
    return tcl_sql_ok(interp, ret);
}

#ifdef TCL_SQL_INCLUDE_TCL_SYSTEM
static int
tcl_system(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
    char *arg0;
    int ret;
    char buf[25];

    if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
        " arg \"", (char *) NULL);
    return TCL_ERROR;
    }
    arg0 = argv[0];
    argc--;
    argv++;
    ret = system(argv[0]);
    return tcl_sql_ok(interp, ret);
}
#endif /* TCL_SQL_INCLUDE_TCL_SYSTEM */

#define DIM(x)  (sizeof(x)/sizeof(*(x)))

typedef struct cmd
{
    char *name;
    int (*func)(Tcl_Interp *, int, char **);
} cmd;

static cmd const cmd_list[] =
{
    {   "begin",            tcl_sql_begin_work      },
    {   "close",            tcl_sql_close           },
    {   "colcharlen",       tcl_sql_sqlcols         },
    {   "coldblen",         tcl_sql_sqlcols         },
    {   "colnames",         tcl_sql_sqlcols         },
    {   "coltypes",         tcl_sql_sqlcols         },
    {   "commit",           tcl_sql_commit          },
    {   "connect",          tcl_sql_connect         },
    {   "database",         tcl_sql_database        },
    {   "disconnect",       tcl_sql_disconnect      },
    {   "exists",           tcl_sql_exists          },
    {   "explain",          tcl_sql_explain         },
    {   "fetch",            tcl_sql_fetch           },
    {   "finish",           tcl_sql_finish          },
    {   "getdatabase",      tcl_sql_getdatabase     },
    {   "geterror",         tcl_sql_geterror        },
    {   "open",             tcl_sql_open            },
    {   "readblob",         tcl_sql_readblob        },
    {   "reopen",           tcl_sql_reopen          },
    {   "rollback",         tcl_sql_rollback        },
    {   "run",              tcl_sql_run             },
    {   "setconnection",    tcl_sql_setconnection   },
    {   "sqlbreak",         tcl_sqlsigcmd           },
    {   "sqlca",            tcl_sqlca               },
    {   "sqld",             tcl_sqld                },
    {   "sqlda",            tcl_sqlda               },
    {   "sqldetach",        tcl_sqlsigcmd           },
    {   "sqldone",          tcl_sqlsigcmd           },
    {   "sqlexit",          tcl_sqlsigcmd           },
    {   "writeblob",        tcl_sql_writeblob       },
};

static int cmd_compare(void const *p1, void const *p2)
{
    cmd *c1 = (cmd *)p1;
    cmd *c2 = (cmd *)p2;
    return(strcmp(c1->name, c2->name));
}

static int
tcl_informix_cmds(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
    cmd     in;
    cmd *sql;

    if (!argv[1])
        return tcl_sql_usage(interp);
    in.name = argv[1];

    sql = bsearch(&in, cmd_list, DIM(cmd_list), sizeof(cmd), cmd_compare);

    if (sql)
        return (*sql->func)(interp, argc - 1, argv + 1);
    else
        return tcl_sql_usage(interp);
}

int Isql_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "sql", tcl_informix_cmds,
        (ClientData)0, (void (*)())0);
#ifdef TCL_SQL_INCLUDE_TCL_SYSTEM
    Tcl_CreateCommand(interp, "tcl_system", tcl_system,
        (ClientData)0, (void (*)())0);
#endif /* TCL_SQL_INCLUDE_TCL_SYSTEM */
    Tcl_PkgProvide(interp, isql_package, isql_version);
    return TCL_OK;
}
