/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

/*
 * Return a pointer to linked list of fitsegs representing dataset.
 * If small is false each fitseg contains at most data from one run.
 * If small is true, each fitseg contains (in order of precedence)
 *  data from at most one run
 *  at least BREAK_POINTS_MIN valid points
 *  (or (at most BREAK_POINTS_MAX valid points)
 *      (at most BREAK_TIME_MAX time span))
 */
fitseg_t *
Piecewise_fitseg_list(FILE *finfo, dataset_t *ds, int clock, int small)
{
  fitseg_t *head;		/* points to first fitseg */
  fitseg_t *tail;		/* points to last completed fitseg */
  fitseg_t *working;		/* working != NULL ==> head, tail nonNULL */
  int workingtime = -1;
  int workingrun = -1;
  int workingvalid = 0;

  datapoint_t *dp;
  int i;

  /* begin with empty list of fitsegs */
  head = tail = working = NULL;

  /* iterate over data points, creatings fitsegs */
  for ( i = ds->extent_begin ; i < ds->extent_end; i++ )
  {
    dp = &ds->dptr[i];

    /* if this point doesn't fit in current fitseg, get a new one */
    if ( working == NULL ||	/* no current, so doesn't fit */
	workingrun != dp->run || /* different run */
	(small &&
	 /* need MIN POINTS */
	 workingvalid >= BREAK_POINTS_MIN &&
	 (workingvalid >= BREAK_POINTS_MAX ||	/* enough points */
	  workingtime + BREAK_TIME_MAX < dp->time) /* long enough */
	 ))
      {
	/* if we have a current fitseg, prepare to move on */
	if ( working != NULL )
	  {
	    tail->next = working;
	    tail = working;
	    working = NULL;	/* not needed */
	  }

	if ( working != NULL )
	  panic("NULL working");

	/* get new fitseg */
	working = fitseg_new();
	working->sp = segment_new();
	working->sp->clock = clock;
	working->sp->run = dp->run;
	working->sp->ds = ds;
	working->sp->first = i;
	working->sp->n = 0;

	if ( head == NULL )
	  head = tail = working;

	workingtime = dp->time;
	workingrun = dp->run;
	workingvalid = 0;
      }

    working->sp->n = i - working->sp->first + 1;
    if ( dp->valid == VALID_OK )
      workingvalid++;
  }
  if ( working != NULL && tail != working )
    tail->next = working;
  return head;
}

double
piecewise_median_deviation(FILE *finfo, fitseg_t *head)
{
  fitseg_t *fs;
  median_t *m;
  double mediandev;

  m = median_init();

  for (fs = head; fs; fs = fs->next)
    median_add(m, fs->fq->sdev);

  mediandev = median_extract(m, 0.5);

  fprintf(finfo, "\nRMS ERRORS of %d segments (LOW %.3f ms, HIGH %.3f ms):\n",
	  median_n(m),
	  median_extract(m, 0.0) * 1E3,
	  median_extract(m, 1.0) * 1E3);

  fprintf(finfo,
	  "1%%\t5%%\t10%%\t25%%\t50%%\t75%%\t85%%\t90%%\t95%%\t99%%\n");
  fprintf(finfo,
	  "%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\n",
	  median_extract(m, 0.01) * 1E3,
	  median_extract(m, 0.05) * 1E3,
	  median_extract(m, 0.10) * 1E3,
	  median_extract(m, 0.25) * 1E3,
	  median_extract(m, 0.50) * 1E3,
	  median_extract(m, 0.75) * 1E3,
	  median_extract(m, 0.85) * 1E3,
	  median_extract(m, 0.90) * 1E3,
	  median_extract(m, 0.95) * 1E3,
	  median_extract(m, 0.99) * 1E3);

  median_free(m);

  return mediandev;
}

/*
 * split data into small pieces, and attempt to combine segments
 * print results, and clear CLOCK_INTEGRATED_VALID bit if not
 * suitable for integrated
 */
fitseg_t *
Piecewise(FILE *finfo, dataset_t *ds, int cl, int rsadjn)
{
  fitseg_t *head;
  fitseg_t *fs1;
  int nvalid, ninvalid;
  int round, mround, print;
  double dev, mindev, mindev_prev, maxcombinedev;
  double mediandev, validdev, limitdev;
  double limit;
  int ncomb, nmig, nskip, runcombok, migrateok;

  /* break up data into a linked list of fitsegs */
  head = Piecewise_fitseg_list(finfo, ds, cl, 1);

  if ( head == NULL )
    {
      clock_state[cl].flags &= ~ CLOCK_VALID_INTEGRATED;
      fprintf(finfo, "NO POINTS ==> NOT USED IN INTEGRATED\n");
      return NULL;
    }

  /* compute fit for each member of list */
  for ( fs1 = head; fs1; fs1 = fs1->next )
    {
      dofit_fitseg(finfo, fs1);

      if ( flags.debug >= 1 )
	fitseg_print(finfo, fs1);
    }

  nvalid = fitseg_count_valid(head, 1);
  if ( nvalid == 0 ) goto PUNT;

  /* compute median of rms errors */
  mediandev = piecewise_median_deviation(finfo, head);

  /* discard data from high-delay segments */
  nvalid = fitseg_count_valid(head, 1);
  ninvalid = 0;

  if ( nvalid == 0 ) goto PUNT;

  validdev = mediandev > 0 ? HIGHDEV_RATIO * mediandev : HUGEDEV;
  fprintf(finfo,
	  "INVALIDATING HIGH-DEVIATION SEGMENTS (limit %f ms):\n",
	  validdev * 1E3);
	
  if ( validdev > 0 )
    for ( fs1 = head; fs1; fs1 = fs1->next )
      if ( fs1->fq->sdev >= validdev )
	{
	  ninvalid += Invalidate_all(finfo, fs1, VALID_BUNCH);
	  dofit_fitseg(finfo, fs1);
	}

  fprintf(finfo, "HIGH-GROUP: ORIG %d OK %d INVALID %d (%0.3f)\n",
	  nvalid, nvalid - ninvalid, ninvalid,
	  ninvalid * 1.0 / nvalid);
  if ( cl >= 0 && cl <= CLOCK_MAX() )
    Invalid_adjweight(finfo, cl, ninvalid * 1.0 / nvalid);

  /* toss out outlyers from each segment and fit again */
  nvalid = fitseg_count_valid(head, 1);
  ninvalid = 0;
  
  if ( nvalid == 0 ) goto PUNT;

  for ( fs1 = head; fs1; fs1 = fs1->next )
    {
      int n = Invalidate_self(finfo, fs1, OUTLIER_SELFS_RATIO, VALID_SELFS);
      ninvalid += n;
      if ( n > 0 )
	dofit_fitseg(finfo, fs1); /* recompute if changed */
    }
  fprintf(finfo, "\nSELF_PIECEWISE: ORIG %d OK %d INVALID %d\n",
	  nvalid, nvalid - ninvalid, ninvalid);

  if ( cl >= 0 && cl <= CLOCK_MAX() )
    Invalid_adjweight(finfo, cl, ninvalid * 1.0 / nvalid);

  /* calculate combining limit */
  if ( flags.predict )
    limitdev = CONFIG_COMBINE_RATIO_PREDICTION * mediandev;
  else
    limitdev = CONFIG_COMBINE_RATIO_ANALYSIS * mediandev;
  fprintf(finfo, "\nCOMBINING LIMIT DEVIATION: %f (ms)\n", limitdev * 1E3);

  /* attempt to merge, first at small limits */
  round = 0;
  print = 0;
  runcombok = migrateok = 0;
  maxcombinedev = 0;
  mindev_prev = HUGEDEV;
  for ( limit = 0.000001; limit <= HUGEDEV ; fflush(finfo) )
    {
      mround = 0;
      do			/* until no more combine */
	{
	  ncomb = nmig = nskip = 0;
	  mindev = HUGEDEV;
	  round++;
	
	  /* iterate over list, and combine segments that fit ok */
	  for ( fs1 = head; fs1; fs1 = fs1->next )
	    {
	      if ( fs1->next == NULL )
		break;		/* no next, so done with this iteration */

	      /* obtain dev of combined segment */
	      dev = fitseg_joinp(finfo, fs1, fs1->next, runcombok, print);

	      /* keep track of minimum */
	      if ( dev < mindev ) mindev = dev;

	      if ( dev <= limit )
		{		/* ok to join */
		  ncomb++;
		  if ( dev > maxcombinedev ) maxcombinedev = dev;
		  fitseg_join(finfo, fs1, fs1->next, 1);
		}
	      else
		{		/* not ok, try migrating instead */
		  if ( migrateok && fitseg_migrate(finfo, fs1, fs1->next) )
		    nmig++;
		  nskip++;
		}
	    }

	  if ( ! (flags.simulate && flags.predict) )
	    fprintf(finfo,
		    "ROUND %i limit=%.3e mindev=%.3e: ncomb=%d, nmig=%d, nskip=%d\n",
		    round, limit, mindev, ncomb, nmig, nskip);

	  if ( nskip == 0 && ncomb == 0 )
	    goto DONECOMBINE;
	  if ( limit > limitdev / 10.0 )
	    migrateok = 1;
	  if ( nskip + ncomb < 10 || limit > limitdev / 2.0 )
	    {
	      runcombok = migrateok = 1;
	      if ( ! (flags.simulate && flags.predict ) )
		print = 1;
	    }
	  fflush(finfo);
	} while ( ncomb >=1 || (nmig > 2 && mround++ < 5) );
      /* no more combinable at this limit level */

      if ( mindev == HUGEDEV && runcombok == 0 )
	{
	  mindev = 0;
	  runcombok = 1;
	}

      mindev_prev = mindev;

      /* just combined at chosen limit?  then finished */
      if ( limit >= limitdev )
	{
	  limit = HUGEDEV;
	  break;
	}

      if ( CONFIG_OPTIMIZE_PIECEWISE )
	/* higher of 10% increase or smallest attempted fit */
	limit = limit * 1.1 > mindev ? limit * 1.1 : mindev;
      else
	limit = mindev;

      /* can't try combining above limit */
      if ( limit >= limitdev )
	limit = limitdev;
    }

 DONECOMBINE:
  fprintf(finfo, "DONE COMBINE");
  if ( limitdev > 0 )
    {
      fprintf(finfo, " maxcombine %.0f us %.2f",
	  maxcombinedev * KK, maxcombinedev/limitdev);
      if ( mindev < HUGEDEV )
	fprintf(finfo, " mindev %.0f %.2f\n",
		mindev * KK, mindev/limitdev);
      else
	fprintf(finfo, " ALLCOMBINED");
    }
  fputc('\n', finfo); fflush(finfo);

#ifdef DIAGNOSTIC
  if ( head->sp == NULL || head->sp->ds == NULL)
    panic("Piecewise head->sp or head->sp->ds");
#endif

/* #define PIECEWISE_VERBOSE */
#ifdef PIECEWISE_VERBOSE
  fprintf(finfo, "\nBefore SELF_LARGE:\n");
  /* iterate over all segments */
  for ( fs1 = head; fs1; fs1 = fs1->next )
    fitseg_print(finfo, fs1);
#endif

  /* Invalidate high-deviation points from completed fit, recompute */
  nvalid = fitseg_count_valid(head, 1);
  ninvalid = 0;
  if ( nvalid == 0 ) goto PUNT;
  for ( fs1 = head; fs1; fs1 = fs1->next )
    {
      int n = Invalidate_self(finfo, fs1, OUTLIER_SELFL_RATIO, VALID_SELFL);
      ninvalid += n;
      if ( n > 0 )
	dofit_fitseg(finfo, fs1); /* recompute if changed */
    }
  fprintf(finfo, "\nSELF_LARGE: ORIG %d OK %d INVALID %d\n",
	  nvalid, nvalid - ninvalid, ninvalid);
  if ( cl >= 0 && cl <= CLOCK_MAX() )
    Invalid_adjweight(finfo, cl, ninvalid * 1.0 / nvalid);

  fputc('\n', finfo);
  for ( fs1 = head; fs1; fs1 = fs1->next )
    {
      if ( flags.noaging == 2 )
	doaging_check_fitseg(finfo, fs1);
      fitseg_print(finfo, fs1);
    }

  /* record time intervals; check validity for integrated */
  if ( cl != CLOCK_NTP  )
    {
      ntp_t newendrun;
      time_t newbegin, newend;
      int check_n;		/* number of (XXX valid) points */
      ntp_t check_t_extent;	/* points cover long enough? */
      ntp_t check_t_recent;	/* points close to current stop time? */


#ifdef DIAGNOSTIC
      if ( cl < 0 || cl > CLOCK_MAX() )
	panic("illegal cl in Piecewise update clock_state");
#endif

      fs1 = fitseg_last(head);
      newbegin = Trun2ntp(segment_begin_time(fs1->sp));
      newendrun = segment_end_time(fs1->sp, 1);
      newend = Trun2ntp(newendrun);

      /* if valid and same beginning as saved attempt, incr count */
      if ( clock_state[cl].valid && clock_state[cl].newbegin == newbegin )
	clock_state[cl].count++;
      else
	{
	  clock_state[cl].newbegin = newbegin;
	  clock_state[cl].count = 1;
	}

      if ( newend < clock_state[cl].end )
	{
	  clock_state[cl].valid = 0;
	  fprintf(finfo,
		  "\nCLOCK_STATE newend=(%s) < end(%s)!!\n  ==> resetting.\n",
		  mytime(Tntp2unix(newend)),
		  mytime(Tntp2unix(clock_state[cl].end)));
	}

      /* always move end forward */
      clock_state[cl].end = newend;

      if (clock_state[cl].valid == 0 ||
	  ! flags.predict ||
	  newbegin >= clock_state[cl].begin ||
	  (newbegin == clock_state[cl].newbegin &&
	   clock_state[cl].count >= HOURLY_EARLIER_COUNT))
	{
	  clock_state[cl].begin = newbegin;
	  clock_state[cl].newbegin = newbegin;
	  clock_state[cl].count = 0;
	  clock_state[cl].valid = 1;
	}
      fputc('\n', finfo);
      clock_state_print(finfo, cl);

      fs1 = fitseg_last(head);

      /* number of valid points */
      check_n = fs1->fq->n;
      /* length of time covered by segment */
      check_t_extent =
	segment_end_time(fs1->sp, 1)
	- segment_begin_time(fs1->sp);
      /* time from now back to last valid point */
      check_t_recent = globals.stop - newendrun;

      if (check_n < HOURLY_INTEGRATED_LIMIT_N ||
	  check_t_extent < HOURLY_INTEGRATED_LIMIT_TE ||
	  check_t_recent > HOURLY_INTEGRATED_LIMIT_TR )
	{
	  fprintf(finfo,
		  "%d POINTS, %u TIME, %u OLD ==> NOT USED IN INTEGRATED\n",
		  check_n, check_t_extent, check_t_recent);
	  clock_state[cl].flags &= ~ CLOCK_VALID_INTEGRATED;
	}
      else
	/* clock is valid for integrated */
	{
	  clock_state[cl].flags |= CLOCK_VALID_INTEGRATED;
	  fprintf(finfo, "PASSED TESTS ==> USED IN INTEGRATED\n");
	  if ( CONFIG_SDEV_ADJ )
	    sdev_adjweight(finfo, cl, fs1->fq->sdev);
	}
    }

  /* estimate runs that are missing (because point are invalid) */
  for ( fs1 = head; fs1; fs1 = fs1->next )
    fit_param_do_missing_runs(finfo, fs1->sp, fs1->fp);

  if ( CONFIG_AUTOCORRELATION )
    /* compute autocorrelation */
    if ( ! flags.predict )
      {
	fprintf(finfo, "Autocorrelation:\n");
	fs1 = fitseg_last(head);
	Autocorr_segment(finfo, fs1->sp, rsadjn);
      }

PUNT:
  return head;
}


/*
 * Do fit on all data in dataset
 */

fitseg_t *
Overall(FILE *finfo, dataset_t *ds, int clock)
{
  fitseg_t *head;

  fprintf(finfo, "\nOVERALL:\n");

  /* get list */
  head = Piecewise_fitseg_list(finfo, ds, clock, 0);

  if ( head == NULL )
    return NULL;

  /* join runs */
  while ( head->next )
    fitseg_join(finfo, head, head->next, 0);

  dofit_fitseg(finfo, head);

  /* estimate runs that are missing (because point are invalid) */
  fit_param_do_missing_runs(finfo, head->sp, head->fp);

  if ( flags.noaging == 2 )
    doaging_check_fitseg(finfo, head);
  fitseg_print(finfo, head);

  return head;
}
