/*
 * $Id: hashtable.c,v 1.6 1995/06/24 07:12:41 coleman Exp coleman $
 *
 * hashtable.c - hash tables
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#include "zsh.h"

/********************************/
/* Generic Hash Table functions */
/********************************/

/* Generic hash function */

/**/
unsigned
hasher(char *str)
{
    unsigned hashval = 0;

    while (*str)
	hashval += (hashval << 5) + ((unsigned) *str++);

    return hashval;
}

/* Get a new (empty) hash table */

/**/
HashTable
newhashtable(int size)
{
    HashTable ht;

    ht = (HashTable) zcalloc(sizeof *ht);
    ht->hsize = size;
    ht->nodes = (HashNode *) zcalloc(size * sizeof(HashNode));
    return ht;
}

/* Add a node to a hash table.                          *
 * nam is the key to use in hashing.  dat is a pointer  *
 * to the node to add.  If there is already a node in   *
 * the table with the same key, it is first freed, and  *
 * then the new node is added.  If the number of nodes  *
 * is now greater than twice the number of hash values, *
 * the table is then expanded.                          */

/**/
void
addhashnode(HashTable ht, char *nam, void *dat)
{
    int hval;
    HashNode hn, hp, hq;

    hn = (HashNode) dat;
    hn->nam = nam;

    hval = ht->hash(nam) % ht->hsize;
    hp = ht->nodes[hval];

    /* check if this is the first node for this hash value */
    if (!hp) {
	hn->next = NULL;
	ht->nodes[hval] = hn;
	if (++ht->ct == ht->hsize * 2)
	    expandhashtable(ht);
	return;
    }

    /* else check if the first node contains the same key */
    if (!strcmp(hp->nam, nam)) {
	hn->next = hp->next;
	ht->nodes[hval] = hn;
	zsfree(hp->nam);
	if (!ht->freenode)
	    zerr("attempt to call NULL freenode", NULL, 0);
	else
	    ht->freenode(hp);
	return;
    }

    /* else run through the list and check all the keys */
    hq = hp;
    hp = hp->next;
    for (; hp; hq = hp, hp = hp->next) {
	if (!strcmp(hp->nam, nam)) {
	    hn->next = hp->next;
	    hq->next = hn;
	    zsfree(hp->nam);
	    if (!ht->freenode)
		zerr("attempt to call NULL freenode", NULL, 0);
	    else
		ht->freenode(hp);
	    return;
	}
    }

    /* else just add it at the front of the list */
    hn->next = ht->nodes[hval];
    ht->nodes[hval] = hn;
    if (++ht->ct == ht->hsize * 2)
        expandhashtable(ht);
}

/* Get an entry in a hash table.          *
 * If successful, it returns a pointer to *
 * the hashnode, else it returns NULL     */

/**/
void *
gethashnode(HashTable ht, char *nam)
{
    int hval;
    HashNode hp;

    hval = ht->hash(nam) % ht->hsize;
    for (hp = ht->nodes[hval]; hp; hp = hp->next) {
	if (!strcmp(hp->nam, nam))
	    return (void *) hp;
    }
    return NULL;
}

/* Remove an entry from a hash table.           *
 * If successful, it removes the node from the  *
 * table and returns a pointer to it.  If there *
 * is no such node, then it returns NULL        */

/**/
void *
removehashnode(HashTable ht, char *nam)
{
    int hval;
    HashNode hp, hq;

    hval = ht->hash(nam) % ht->hsize;
    hp = ht->nodes[hval];

    /* if no nodes at this hash value, return NULL */
    if (!hp)
	return NULL;

    /* else check if the key in the first one matches */
    if (!strcmp(hp->nam, nam)) {
	ht->nodes[hval] = hp->next;
	zsfree(hp->nam);
	ht->ct--;
	return (void *) hp;
    }

    /* else run through the list and check the rest of the keys */
    hq = hp;
    hp = hp->next;
    for (; hp; hq = hp, hp = hp->next) {
	if (!strcmp(hp->nam, nam)) {
	    hq->next = hp->next;
	    zsfree(hp->nam);
	    ht->ct--;
	    return (void *) hp;
	}
    }

    /* else it is not in the list, so return NULL */
    return NULL;
}

/**/
int
hnamcmp(struct hashnode **a, struct hashnode **b)
{
    return forstrcmp(&((*a)->nam), &((*b)->nam));
}

/* Scan nodes in the hash table and execute *
 * func on each node.                       */

/**/
void
scanhashtable(HashTable ht, HFunc func)
{
    int count;
    HashNode hn;

#ifndef HASHORDER

    int nhash;
    struct hashnode **hnsorttab, **htp;

    hnsorttab = (struct hashnode **)zalloc(ht->ct * sizeof(HashNode));

    for (htp = hnsorttab, count = ht->hsize - 1; count >= 0; count--)
	for (hn = ht->nodes[count]; hn; hn = hn->next)
	    *htp++ = hn;

    qsort((void *) & hnsorttab[0], ht->ct, sizeof(HashNode),
	           (int (*) _((const void *, const void *)))hnamcmp);

    for (htp = hnsorttab, nhash = 0; nhash < ht->ct; nhash++, htp++)
	func((*htp)->nam, (char *)(*htp));

    free(hnsorttab);

#else

    for (count = ht->hsize - 1; count >= 0; count--)
	for (hn = ht->nodes[count]; hn; hn = hn->next)
	    func(hn->nam, (char *)hn);

#endif
}

/* Expand hash tables when they get too many entries. *
 * The new size is 4 times the previous size.         */

/**/
void
expandhashtable(HashTable ht)
{
    struct hashnode **onodes, **ha, *hn, *hp;
    int i, osize;

    osize = ht->hsize;
    onodes = ht->nodes;

    ht->hsize = osize * 4;
    ht->nodes = (HashNode *) zcalloc(ht->hsize * sizeof(HashNode));
    ht->ct = 0;

    /* scan through the old list of nodes, and *
     * rehash them into the new list of nodes  */
    for (i = 0, ha = onodes; i < osize; i++, ha++) {
	for (hn = *ha; hn;) {
	    hp = hn->next;
	    ht->addnode(ht, hn->nam, (void *) hn);
	    hn = hp;
	}
    }
    zfree(onodes, osize * sizeof(HashNode));
}

/* Free all the memory used by a hash table  */

/**/
void
freehashtable(HashTable ht)
{
    struct hashnode **ha, *hn, *hp;
    int i;

    ha = ht->nodes;
    for (i = 0; i < ht->hsize; i++, ha++) {
	for (hn = *ha; hn;) {
	    hp = hn->next;
	    zsfree(hn->nam);
	    ht->freenode(hn);
	    hn = hp;
	}
    }
    zfree(ht->nodes, ht->hsize * sizeof(HashNode));
    zfree(ht, sizeof(struct hashtable));

#ifdef ZSH_HASH_DEBUG
    zsfree(ht->tablename);
#endif
}

/* Print info about hash table */

#ifdef ZSH_HASH_DEBUG

#define MAXDEPTH 7

/**/
void
printhashtabinfo(HashTable ht)
{
    HashNode hn;
    int chainlen[MAXDEPTH + 1];
    int i, tmpcount, total;

    printf("name of table   : %s\n",   ht->tablename);
    printf("size of nodes[] : %d\n",   ht->hsize);
    printf("number of nodes : %d\n\n", ht->ct);

    memset(chainlen, 0, sizeof(chainlen));

    /* count the number of nodes just to be sure */
    total = 0;
    for (i = 0; i < ht->hsize; i++) {
	tmpcount = 0;
	for (hn = ht->nodes[i]; hn; hn = hn->next)
	    tmpcount++;
	if (tmpcount >= MAXDEPTH)
	    chainlen[MAXDEPTH]++;
	else
	    chainlen[tmpcount]++;
	total += tmpcount;
    }

    for (i = 0; i < MAXDEPTH; i++)
	printf("number of hash values with chain of length %d  : %4d\n", i, chainlen[i]);
    printf("number of hash values with chain of length %d+ : %4d\n", MAXDEPTH, chainlen[MAXDEPTH]);
    printf("total number of nodes                         : %4d\n", total);
}
#endif


/********************************/
/* Command Hash Table Functions */
/********************************/

/* Create a new command hash table */
 
/**/
void
newcmdnamtable(void)
{
    cmdnamtab = newhashtable(201);
    cmdnamtab->hash       = hasher;
    cmdnamtab->addnode    = addhashnode;
    cmdnamtab->getnode    = gethashnode;
    cmdnamtab->removenode = removehashnode;
    cmdnamtab->freenode   = freecmdnamnode;

#ifdef ZSH_HASH_DEBUG
    cmdnamtab->printinfo = printhashtabinfo;
    cmdnamtab->tablename = ztrdup("cmdnamtab");
#endif

    addbuiltins();
    pathchecked = path;
}

/* Restart the command hash table.  This creates a new command   *
 * hash table, but retains any shell functions.  It also retains *
 * the DISABLED flag on builtins which are marked as disabled.   */

/**/
void
restartcmdnamtable(HashTable ht)
{
    HashTable oldtable;
    Cmdnam cn, hn;
    int i;
 
    oldtable = cmdnamtab;
    cmdnamtab = newhashtable(201);
    cmdnamtab->hash = hasher;
    cmdnamtab->addnode = addhashnode;
    cmdnamtab->getnode = gethashnode;
    cmdnamtab->removenode = removehashnode;
    cmdnamtab->freenode = freecmdnamnode;

#ifdef ZSH_HASH_DEBUG
    cmdnamtab->printinfo = printhashtabinfo;
    cmdnamtab->tablename = ztrdup("cmdnamtab");
#endif

    addbuiltins();
    for (i = oldtable->hsize - 1; i >= 0; i--)
	for (hn = (Cmdnam) oldtable->nodes[i]; hn; hn = (Cmdnam) hn->next)
	    if (hn->flags & (SHFUNC | DISABLED)) {
		cn = (Cmdnam) zcalloc(sizeof *cn);
		*cn = *hn;
		hn->u.list = NULL;
		ht->addnode(cmdnamtab, ztrdup(hn->nam), cn);
	    }

    freehashtable(oldtable);
    pathchecked = path;
}

/**/
void
freecmdnamnode(void *a)
{
    struct cmdnam *c = (struct cmdnam *)a;
 
    if (c->flags & SHFUNC) {
	if (c->u.list)
	    freestruct(c->u.list);
    } else if ((c->flags & HASHCMD) == HASHCMD)
	zsfree(c->u.cmd);
 
    zfree(c, sizeof(struct cmdnam));
}

/**************************************/
/* Reserved Word Hash Table Functions */
/**************************************/

/* Build the hash table containing zsh's reserved words. */

/**/
void
newreswdtable(void)
{
    static char *reswds[] =
    {
	"do", "done", "esac", "then", "elif", "else", "fi", "for", "case",
	"if", "while", "function", "repeat", "time", "until", "exec", "command",
	"select", "coproc", "noglob", "-", "nocorrect", "foreach", "end", NULL
    };
    int i;

    /* Twenty four reserved words; any advance on the following? */
    reswdtab = newhashtable(23);
    reswdtab->hash       = hasher;
    reswdtab->addnode    = addhashnode;
    reswdtab->getnode    = gethashnode;
    reswdtab->removenode = NULL;
    reswdtab->freenode   = NULL;

#ifdef ZSH_HASH_DEBUG
    reswdtab->printinfo = printhashtabinfo;
    reswdtab->tablename = ztrdup("reswdtab");
#endif

    /* Add the actual words, not copies, to the table, *
     * as we do not expect to modify the table again.  */
    for (i = 0; reswds[i]; i++) {
	struct reswd *ptr = (struct reswd *) zcalloc(sizeof *ptr);
	ptr->cmd = i + DO;
	reswdtab->addnode(reswdtab, reswds[i], ptr);
    }
}


/********************************/
/* Aliases Hash Table Functions */
/********************************/

/* Create new hash table for aliases */

/**/
void
newaliastable(void)
{
    aliastab = newhashtable(23);
    aliastab->hash       = hasher;
    aliastab->addnode    = addhashnode;
    aliastab->getnode    = gethashnode;
    aliastab->removenode = removehashnode;
    aliastab->freenode   = freealiasnode;

#ifdef ZSH_HASH_DEBUG
    aliastab->printinfo = printhashtabinfo;
    aliastab->tablename = ztrdup("aliastab");
#endif

    /* add the default aliases */
    aliastab->addnode(aliastab, ztrdup("run-help"), createaliasnode(ztrdup("man"), 1));
    aliastab->addnode(aliastab, ztrdup("which-command"), createaliasnode(ztrdup("whence"), 1));
}

/**/
void
freealiasnode(void *a)
{
    Alias c = (Alias)a;
 
    zsfree(c->text);
    zfree(c, sizeof(struct alias));
}

/**/
Alias
createaliasnode(char *txt, int cmdflag)
{
    Alias al;

    al = (Alias) zcalloc(sizeof *al);
    al->text = txt;
    al->cmd = cmdflag;
    al->inuse = 0;
    return al;
}


/**********************************/
/* Parameter Hash Table Functions */
/**********************************/

/**/
void
freeparamnode(void *a)
{
    Param pm = (Param) a;
 
    zfree(pm, sizeof(struct param));
}


/********************************/
/* Compctl Hash Table Functions */
/********************************/

/**/
void
newcompctltable(void)
{
    compctltab = newhashtable(23);
    compctltab->hash       = hasher;
    compctltab->addnode    = addhashnode;
    compctltab->getnode    = gethashnode;
    compctltab->removenode = removehashnode;
    compctltab->freenode   = freecompctlp;

#ifdef ZSH_HASH_DEBUG
    compctltab->printinfo = printhashtabinfo;
    compctltab->tablename = ztrdup("compctltab");
#endif
}

/**/
void
freecompctl(void *a)
{
    Compctl cc = (Compctl) a;

    if (cc == &cc_default ||
 	cc == &cc_first ||
	cc == &cc_compos ||
	--cc->refc > 0)
	return;

    zsfree(cc->keyvar);
    zsfree(cc->glob);
    zsfree(cc->str);
    zsfree(cc->func);
    zsfree(cc->explain);
    zsfree(cc->prefix);
    zsfree(cc->suffix);
    zsfree(cc->hpat);
    zsfree(cc->subcmd);
    if (cc->cond)
	freecompcond(cc->cond);
    if (cc->ext) {
	Compctl n, m;

	n = cc->ext;
	do {
	    m = (Compctl) (n->next);
	    freecompctl(n);
	    n = m;
	}
	while (n);
    }
    if (cc->xor && cc->xor != &cc_default)
	freecompctl(cc->xor);
    zfree(cc, sizeof(struct compctl));
}

/**/
void
freecompctlp(void *a)
{
    Compctlp ccp = (Compctlp) a;

    freecompctl(ccp->cc);
    zfree(ccp, sizeof(struct compctlp));
}


/*******************************************/
/* Emacs Key Bindings Hash Table Functions */
/*******************************************/

/**/
void
newemkeybindtable(void)
{
    emkeybindtab = newhashtable(67);
    emkeybindtab->hash       = hasher;
    emkeybindtab->addnode    = addhashnode;
    emkeybindtab->getnode    = gethashnode;
    emkeybindtab->removenode = removehashnode;
    emkeybindtab->freenode   = freekey;

#ifdef ZSH_HASH_DEBUG
    emkeybindtab->printinfo = printhashtabinfo;
    emkeybindtab->tablename = ztrdup("emkeybindtab");
#endif
}


/****************************************/
/* Vi Key Bindings Hash Table Functions */
/****************************************/

/**/
void
newvikeybindtable(void)
{
    vikeybindtab = newhashtable(20);
    vikeybindtab->hash       = hasher;
    vikeybindtab->addnode    = addhashnode;
    vikeybindtab->getnode    = gethashnode;
    vikeybindtab->removenode = removehashnode;
    vikeybindtab->freenode   = freekey;

#ifdef ZSH_HASH_DEBUG
    vikeybindtab->printinfo = printhashtabinfo;
    vikeybindtab->tablename = ztrdup("vikeybindtab");
#endif
}
