/*

	lex.c - lexical analysis

	This file is part of zsh, the Z shell.

   zsh is free software; no one can prevent you from reading the source
   code, or giving it to someone else.
   This file is copyrighted under the GNU General Public License, which
   can be found in the file called COPYING.

   Copyright (C) 1990, 1991 Paul Falstad

   zsh is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY.  No author or distributor accepts
   responsibility to anyone for the consequences of using it or for
   whether it serves any particular purpose or works at all, unless he
   says so in writing.  Refer to the GNU General Public License
   for full details.

   Everyone is granted permission to copy, modify and redistribute
   zsh, but only under the conditions described in the GNU General Public
   License.   A copy of this license is supposed to have been given to you
   along with zsh so you can know your rights and responsibilities.
   It should be in a file named COPYING.

   Among other things, the copyright notice and this notice must be
   preserved on all copies.

*/

#include "zsh.h"
#include "y.tab.h"

/* lexical state */

static int ignl;

static int xignl,xlsep,xincmd,xincond,xinfunc,xinredir,xincase;
static int dbparens,xdbparens,xalstat;
static char *xhlastw;

static int xisfirstln, xisfirstch, xhistremmed, xhistdone,
	xspaceflag, xstophist, xlithist, xalstackind,xhlinesz;
static char *xhline, *xhptr;

static char *tokstr;

/* save the lexical state */

/* is this a hack or what? */

void lexsave() /**/
{
	xignl = ignl;
	xlsep = lsep;
	xincmd = incmd;
	xincond = incond;
	xinredir = inredir;
	xinfunc = infunc;
	xincase = incase;
	xdbparens = dbparens;
	xalstat = alstat;
	xalstackind = alstackind;
	xisfirstln = isfirstln;
	xisfirstch = isfirstch;
	xhistremmed = histremmed;
	xhistdone = histdone;
	xspaceflag = spaceflag;
	xstophist = stophist;
	xlithist = lithist;
	xhline = hline;
	xhptr = hptr;
	xhlastw = hlastw;
	xhlinesz = hlinesz;
}

/* restore lexical state */

void lexrestore() /**/
{
	ignl = xignl;
	lsep = xlsep;
	incmd = xincmd;
	incond = xincond;
	inredir = xinredir;
	infunc = xinfunc;
	incase = xincase;
	dbparens = xdbparens;
	alstat = xalstat;
	isfirstln = xisfirstln;
	isfirstch = xisfirstch;
	histremmed = xhistremmed;
	histdone = xhistdone;
	spaceflag = xspaceflag;
	stophist = xstophist;
	lithist = xlithist;
	hline = xhline;
	hptr = xhptr;
	hlastw = xhlastw;
	clearalstack();
	alstackind = xalstackind;
	hlinesz = xhlinesz;
	lexstop = eofseen = errflag = 0;
}

int yylex() /**/
{
int x;

	for (;;)
		{
		do
			x = gettok();
		while (x != ENDINPUT && exalias(&x));
		if (x == NEWLIN && ignl)
			continue;
		if (x == SEMI || x == NEWLIN)
			{
			if (lsep)
				continue;
			x = SEPER;
			lsep = 1;
			}
		else
			lsep = (x == AMPER);
		break;
		}
	ignl = 0;
	switch (x)
		{
		case OUTPAR: infunc = incmd = incase = 0; break;
		case INPAR:case INBRACE:case DBAR:case DAMPER:case DO:
		case THEN:case ELIF:case BAR:case BARAMP:case IF:case WHILE:
		case ELSE:ignl = 1;infunc = 0;
			case INOUTPAR: case SEPER:
			case AMPER:incmd = nocorrect = 0; break;
		case ESAC: incase = 0; break;
		case STRING: case ENVARRAY:
			if (!inredir && !infunc) incmd = 1; inredir = 0; break;
		case FOR: case SELECT: incmd = 1; break;
		case OUTANG:case OUTANGBANG:case DOUTANG:case INANG:
		case DINANG:case TRINANG:case INANGAMP:case OUTANGAMP:case OUTANGAMPBANG:
		case DOUTANGAMP:case DOUTANGAMPBANG: inredir = 1; break;
		case FUNC:infunc = 1;break;
		case DINBRACK: incond = 1; break;
		case DOUTBRACK: incond = 0; break;
		case DSEMI: ignl = incase = 1; incmd = 0; break;
		case CASE: incmd = incase = 1; break;
		}
	return x;
}

#define LX1_BKSLASH 0
#define LX1_COMMENT 1
#define LX1_NEWLIN 2
#define LX1_SEMI 3
#define LX1_BANG 4
#define LX1_AMPER 5
#define LX1_BAR 6
#define LX1_INPAR 7
#define LX1_OUTPAR 8
#define LX1_INBRACE 9
#define LX1_OUTBRACE 10
#define LX1_INBRACK 11
#define LX1_OUTBRACK 12
#define LX1_INANG 13
#define LX1_OUTANG 14
#define LX1_OTHER 15

#define LX2_BREAK 0
#define LX2_OUTPAR 1
#define LX2_BAR 2
#define LX2_STRING 3
#define LX2_INBRACK 4
#define LX2_OUTBRACK 5
#define LX2_TILDE 6
#define LX2_INPAR 7
#define LX2_INBRACE 8
#define LX2_OUTBRACE 9
#define LX2_OUTANG 10
#define LX2_INANG 11
#define LX2_EQUALS 12
#define LX2_BKSLASH 13
#define LX2_QUOTE 14
#define LX2_DQUOTE 15
#define LX2_BQUOTE 16
#define LX2_OTHER 17

unsigned char lexact1[256],lexact2[256],lextok2[256];

void initlextabs() /**/
{
int t0;
static char *lx1 = "\\q\n;!&|(){}[]<>xx";
static char *lx2 = "x)|$[]~({}><=\\\'\"`x";

	for (t0 = 0; t0 != 256; t0++)
		{
		lexact1[t0] = LX1_OTHER;
		lexact2[t0] = LX2_OTHER;
		lextok2[t0] = t0;
		}
	for (t0 = 0; lx1[t0]; t0++)
		if (lx1[t0] != 'x')
			lexact1[lx1[t0]] = t0;
	for (t0 = 0; lx2[t0]; t0++)
		if (lx2[t0] != 'x')
			lexact2[lx2[t0]] = t0;
	lexact2[';'] = LX2_BREAK;
	lexact2['&'] = LX2_BREAK;
	lextok2[','] = Comma;
	lextok2['*'] = Star;
	lextok2['?'] = Quest;
	lextok2['{'] = Inbrace;
	lextok2['['] = Inbrack;
	lextok2['$'] = String;
}

/* initialize lexical state */

void lexinit() /**/
{
	ignl = lsep = incmd = incond = infunc = inredir = incase =
		nocorrect = dbparens = alstat = lexstop = 0;
	if (isset(EXTENDEDGLOB))
		{
		lextok2['#'] = Pound;
		lextok2['^'] = Hat;
		}
	else
		{
		lextok2['#'] = '#'; 
		lextok2['^'] = '^';
		}
}

int len = 0,bsiz = 256;
char *bptr;

/* add a char to the string buffer */

void add(c) /**/
int c;
{
	*bptr++ = c;
	if (bsiz == ++len)
		{
		int newbsiz;

		newbsiz = bsiz * 8;
		while (newbsiz < inbufct)
			newbsiz *= 2;
		bptr = len+(tokstr = hrealloc(tokstr,bsiz,newbsiz));
		bsiz = newbsiz;
		}
}

int gettok() /**/
{
int bct = 0,pct = 0,brct = 0;
int c,d,intpos = 1;
int peekfd = -1,peek,incm;

beginning:
	hlastw = NULL;
	tokstr = NULL;
	incm = incmd || incond || inredir || incase;
	while (iblank(c = hgetc()) && !lexstop);
	isfirstln = 0;
	wordbeg = inbufct;
	hwbegin();
	hwaddc(c);
	if (dbparens)	/* handle ((...)) */
		{
		pct = 2;
		peek = STRING;
		len = dbparens = 0;
		bptr = tokstr = ncalloc(bsiz = 256);
		for (;;)
			{
			if (c == '(')
				pct++;
			else if (c == ')')
				pct--;
			else if (c == '\n')
				{
				zerr("parse error: )) expected",NULL,0);
				peek = LEXERR;
				return peek;
				}
			else if (c == '$')
				c = Qstring;
			if (pct >= 2)
				add(c);
			if (pct)
				c = hgetc();
			else
				break;
			}
		*bptr = '\0';
		yylval.str = tokstr;
		return peek;
		}
	if (idigit(c))	/* handle 1< foo */
		{
		d = hgetc();
		hungetc(d);
		lexstop = 0;
		if (d == '>' || d == '<')
			{
			peekfd = c-'0';
			c = hgetc();
			}
		}

	/* chars in initial position in word */

	if (c == hashchar &&
			(isset(INTERACTIVECOMMENTS) ||
			(!zleparse && (!interact || unset(SHINSTDIN) || strin))))
		{
		while ((c = hgetch()) != '\n' && !lexstop);
		if (c == '\n')
			peek = NEWLIN;
		else
			{
			peek = (errflag) ? LEXERR : ENDINPUT;
			errflag = 1;
			}
		return peek;
		}
	if (lexstop)
		return (errflag) ? LEXERR : ENDINPUT;
	switch (lexact1[(unsigned char) c])
		{
		case LX1_BKSLASH:
			d = hgetc();
			if (d == '\n')
				goto beginning;
			hungetc(d);
			break;
		case LX1_NEWLIN: return NEWLIN;
		case LX1_SEMI:
			d = hgetc();
			if (d != ';')
				{
				hungetc(d);
				return SEMI;
				}
			return DSEMI;
		case LX1_BANG:
			d = hgetc();
			hungetc(d);
			if (!inblank(d))
				break;
			if (!incm || incond)
				return BANG;
			break;
		case LX1_AMPER:
			d = hgetc();
			if (d != '&')
				{
				hungetc(d);
				return AMPER;
				}
			return DAMPER;
		case LX1_BAR:
			d = hgetc();
			if (d == '|')
				return DBAR;
			else if (d == '&')
				return BARAMP;
			hungetc(d);
			return BAR;
		case LX1_INPAR:
			d = hgetc();
			if (d == '(' && !incm)
				{
				yylval.str = tokstr = strdup("let");
				dbparens = 1;
				return STRING;
				}
			else if (d == ')')
				return INOUTPAR;
			hungetc(d);
			if (incm && !incond)
				break;
			return INPAR;
		case LX1_OUTPAR: return OUTPAR;
		case LX1_INBRACE: if (incm) break; return INBRACE;
		case LX1_OUTBRACE: return OUTBRACE;
		case LX1_INBRACK:
			if (incm)
				break;
			d = hgetc();
			if (d == '[')
				return DINBRACK;
			hungetc(d);
			break;
		case LX1_OUTBRACK:
			if (!incond)
				break;
			incond = 0;
			d = hgetc();
			if (d == ']')
				return DOUTBRACK;
			hungetc(d);
			break;
		case LX1_INANG:
			d = hgetc();
			if ((incmd && d == '(') || incase)
				{
				hungetc(d);
				break;
				}
			else if (d == '<')
				{
				int e = hgetc();

				if (e == '(')
					{
					hungetc(e);
					hungetc(d);
					peek = INANG;
					}
				else if (e == '<')
					peek = TRINANG;
				else if (e == '-')
					peek = DINANGDASH;
				else
					{
					hungetc(e);
					peek = DINANG;
					}
				}
			else if (d == '&')
				peek = INANGAMP;
			else
				{
				peek = INANG;
				hungetc(d);
				}
			yylval.fds.fd1 = peekfd;
			return peek;
		case LX1_OUTANG:
			d = hgetc();
			if (d == '(')
				{
				hungetc(d);
				break;
				}
			else if (d == '&')
				{
				d = hgetc();
				if (d == '!')
					peek = OUTANGAMPBANG;
				else
					{
					hungetc(d);
					peek = OUTANGAMP;
					}
				}
			else if (d == '!')
				peek = OUTANGBANG;
			else if (d == '>')
				{
				d = hgetc();
				if (d == '&')
					{
					d = hgetc();
					if (d == '!')
						peek = DOUTANGAMPBANG;
					else
						{
						hungetc(d);
						peek = DOUTANGAMP;
						}
					}
				else if (d == '!')
					peek = DOUTANGBANG;
				else if (d == '(')
					{
					hungetc(d);
					hungetc('>');
					peek = OUTANG;
					}
				else
					{
					hungetc(d);
					peek = DOUTANG;
					}
				}
			else
				{
				hungetc(d);
				peek = OUTANG;
				}
			yylval.fds.fd1 = peekfd;
			return peek;
		}

	/* we've started a string, now get the rest of it, performing
		tokenization */

	peek = STRING;
	len = 0;
	bptr = tokstr = ncalloc(bsiz = 256);
	for(;;)
		{
		int act;
		int d;
		
		if (inblank(c))
			act = LX2_BREAK;
		else
			{
			act = lexact2[(unsigned char) c];
			c = lextok2[(unsigned char) c];
			}
		switch (act)
			{
			case LX2_BREAK: goto brk;
			case LX2_OUTPAR:
				if (!pct)
					goto brk;
				c = Outpar;
				pct--;
				break;
			case LX2_BAR:
				if (!pct && !incase)
					goto brk;
				c = Bar;
				break;
			case LX2_STRING:
				d = hgetc();
				if (d == '[')
					{
					add(String);
					add(Inbrack);
					while ((c = hgetc()) != ']' && !lexstop)
						add(c);
					c = Outbrack;
					}
				else if (d == '(')
					{
					add(String);
					skipcomm();
					c = Outpar;
					}
				else
					hungetc(d);
				break;
			case LX2_INBRACK: brct++; break;
			case LX2_OUTBRACK:
				if (incond && !brct)
					goto brk;
				brct--;
				c = Outbrack;
				break;
			case LX2_TILDE: if (intpos) c = Tilde; break;
			case LX2_INPAR:
				d = hgetc();
				hungetc(d);
				if (d == ')' || !incm)
					goto brk;
				pct++;
				c = Inpar;
				break;
			case LX2_INBRACE: bct++; break;
			case LX2_OUTBRACE:
				if (!bct)
					goto brk;
				bct--;
				c = Outbrace;
				break;
			case LX2_OUTANG:
				d = hgetc();
				if (d != '(')
					{
					hungetc(d);
					goto brk;
					}
				add(Outang);
				skipcomm();
				c = Outpar;
				break;
			case LX2_INANG:
				d = hgetc();
				if (!(idigit(d) || d == '-' || d == '>' || d == '(' || d == ')'))
					{
					hungetc(d);
					goto brk;
					}
				c = Inang;
				if (d == '(')
					{
					add(c);
					skipcomm();
					c = Outpar;
					}
				else if (d == ')')
					hungetc(d);
				else
					{
					add(c);
					c = d;
					while (c != '>' && !lexstop)
						add(c),c = hgetc();
					c = Outang;
					}
				break;
			case LX2_EQUALS:
				if (intpos)
					{
					d = hgetc();
					if (d != '(')
						{
						hungetc(d);
						c = Equals;
						}
					else
						{
						add(Equals);
						skipcomm();
						c = Outpar;
						}
					}
				else if (peek != ENVSTRING && !incm)
					{
					d = hgetc();
					if (d == '(' && !incm)
						{
						*bptr = '\0';
						yylval.str = tokstr;
						return ENVARRAY;
						}
					hungetc(d);
					peek = ENVSTRING;
					intpos = 2;
					}
				break;
			case LX2_BKSLASH:
				c = hgetc();
				if (c == '\n')
					{
					c = hgetc();
					continue;
					}
				add(c);
				c = hgetc();
				continue;
			case LX2_QUOTE:
				add(Nularg);

				/* we add the Nularg to prevent this:

				echo $PA'TH'

				from printing the path. */

				while ((c = hgetc()) != '\'' && !lexstop)
					{
					add(c);
					if (isset(CSHJUNKIEQUOTES) && c == '\n')
						break;
					}
				if (c != '\'')
					{
					zerr("unmatched \'");
					peek = LEXERR;
					goto brk;
					}
				c = Nularg;
				break;
			case LX2_DQUOTE:
				add(Nularg);
				while ((c = hgetc()) != '\"' && !lexstop)
					if (c == '\\')
						{
						c = hgetc();
						if (c != '\n')
							{
							if (c != '$' && c != '\\' && c != '\"' && c != '`')
								add('\\');
							add(c);
							}
						}
					else if (isset(CSHJUNKIEQUOTES) && c == '\n')
						break;
					else
						{
						if (c == '$')
							{
							d = hgetc();
							if (d == '(')
								{
								add(Qstring);
								skipcomm();
								c = Outpar;
								}
							else if (d == '[')
								{
								add(String);
								add(Inbrack);
								while ((c = hgetc()) != ']' && !lexstop)
									add(c);
								c = Outbrack;
								}
							else
								{
								c = Qstring;
								hungetc(d);
								}
							}
						else if (c == '`')
							c = Qtick;
						add(c);
						}
				if (c != '\"')
					{
					zerr("unmatched \"");
					peek = LEXERR;
					goto brk;
					}
				c = Nularg;
				break;
			case LX2_BQUOTE:
				add(Tick);
				while ((c = hgetc()) != '`' && !lexstop)
					if (c == '\\')
						{
						c = hgetc();
						if (c != '\n')
							{
							if (c != '`' && c != '\\' && c != '$')
								add('\\');
							add(c);
							}
						}
					else if (isset(CSHJUNKIEQUOTES) && c == '\n')
						break;
					else
						add(c);
				if (c != '`')
					{
					zerr("unmatched `");
					peek = LEXERR;
					goto brk;
					}
				c = Tick;
				break;
			}
		add(c);
		c = hgetc();
		if (intpos)
			intpos--;
		if (lexstop)
			break;
		}
brk:
	hungetc(c);
	*bptr = '\0';
	yylval.str = tokstr;
	return peek;
}

/* expand aliases, perhaps */

int exalias(pk) /**/
int *pk;
{
struct alias *an;
char *s,*t;
int ic;

	s = yytext = hwadd();
	for (t = s; *t && *t != HISTSPACE; t++);
	if (!*t)
		t = NULL;
	else
		*t = '\0';
	ic = incmd || incond || inredir || incase || infunc;
	if (interact && isset(SHINSTDIN) && !strin && !incase && *pk == STRING &&
		(isset(CORRECTALL) || (isset(CORRECT) && !ic)) && !nocorrect)
			spckword(&yylval.str,&s,ic,1);
	if (zleparse && !alstackind)
		gotword(s);
	an = gethnode(s,aliastab);
	if (t)
		*t = HISTSPACE;
	if (alstackind != MAXAL && an && !an->inuse)
		if (!(an->cmd && ic && alstat != ALSTAT_MORE))
			{
			if (an->cmd < 0)
				{
				*pk = DO-an->cmd-1;
				return 0;
				}
			else
				{
				an->inuse = 1;
				hungets(ALPOPS);
				hungets((alstack[alstackind++] = an)->text);
				alstat = 0;
				/* remove from history if it begins with space */
				if (isset(HISTIGNORESPACE) && an->text[0] == ' ')
				    remhist();
				return 1;
				}
			}
		else if (incase && DO-an->cmd-1 == ESAC)
			*pk = ESAC;
	return 0;
}

/* skip (...) */

void skipcomm() /**/
{
int pct = 1,c;

	c = Inpar;
	do
		{
		add(c);
		c = hgetc();
		if (itok(c) || c == EOF)
			break;
		else if (c == '(') pct++;
		else if (c == ')') pct--;
		else if (c == '\\')
			{
			add(c);
			c = hgetc();
			}
		else if (c == '\'')
			{
			add(c);
			while ((c = hgetc()) != '\'' && !lexstop)
				add(c);
			}
		else if (c == '\"')
			{
			add(c);
			while ((c = hgetc()) != '\"' && !lexstop)
				if (c == '\\')
					{
					add(c);
					add(hgetc());
					}
				else add(c);
			}
		else if (c == '`')
			{
			add(c);
			while ((c = hgetc()) != '`' && !lexstop)
				if (c == '\\') add(c), add(hgetc());
				else add(c);
			}
		}
	while(pct);
}

