/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file donau-httpd_charity_patch.c
 * @brief Handle request to update a charity entry.
 * @author Bohdan Potuzhnyi
 */
#include <donau_config.h>
#include <gnunet/gnunet_json_lib.h>
#include <gnunet/gnunet_util_lib.h>
#include <jansson.h>
#include <microhttpd.h>
#include <taler/taler_json_lib.h>
#include <taler/taler_mhd_lib.h>
#include <taler/taler_util.h>
#include "donau-httpd_charity.h"
#include "donau-httpd_db.h"


MHD_RESULT
DH_handler_charity_patch (struct DH_RequestContext *rc,
                          const json_t *root,
                          const char *const args[])
{
  struct DONAU_CharityPublicKeyP charity_pub;
  uint64_t charity_id;
  char dummy;
  const char *charity_name = NULL;
  const char *charity_url = NULL;
  struct TALER_Amount max_per_year;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_fixed_auto ("charity_pub",
                                 &charity_pub),
    GNUNET_JSON_spec_string ("charity_name",
                             &charity_name),
    GNUNET_JSON_spec_string ("charity_url",
                             &charity_url),
    TALER_JSON_spec_amount ("max_per_year",
                            DH_currency,
                            &max_per_year),
    GNUNET_JSON_spec_end ()
  };

  if ( (NULL == args[0]) ||
       (1 != sscanf (args[0],
                     "%lu%c",
                     &charity_id,
                     &dummy)) )
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "charity_id");
  }

  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (rc->connection,
                                     root,
                                     spec);
    if (GNUNET_SYSERR == res)
      return MHD_NO; /* hard failure */
    if (GNUNET_NO == res)
    {
      GNUNET_break_op (0);
      return MHD_YES; /* failure */
    }
  }

  {
    struct DONAUDB_CharityMetaData meta;
    enum GNUNET_DB_QueryStatus qs;

    qs = DH_plugin->lookup_charity (DH_plugin->cls,
                                    charity_id,
                                    &meta);
    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
    case GNUNET_DB_STATUS_SOFT_ERROR:
      GNUNET_break (0);
      return TALER_MHD_reply_with_error (rc->connection,
                                         MHD_HTTP_INTERNAL_SERVER_ERROR,
                                         TALER_EC_GENERIC_DB_FETCH_FAILED,
                                         "lookup_charity");
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      return TALER_MHD_reply_with_error (rc->connection,
                                         MHD_HTTP_NOT_FOUND,
                                         TALER_EC_DONAU_CHARITY_NOT_FOUND,
                                         args[0]);
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      break;
    }

    if (0 < TALER_amount_cmp (&meta.receipts_to_date,
                              &max_per_year))
    {
      GNUNET_break_op (0);
      GNUNET_free (meta.charity_name);
      GNUNET_free (meta.charity_url);
      return TALER_MHD_reply_with_error (rc->connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "max_per_year must NOT be SMALLER than receipts_to_date");
    }

    qs = DH_plugin->update_charity (DH_plugin->cls,
                                    charity_id,
                                    &charity_pub,
                                    charity_name,
                                    charity_url,
                                    &max_per_year);
    GNUNET_free (meta.charity_name);
    GNUNET_free (meta.charity_url);

    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
    case GNUNET_DB_STATUS_SOFT_ERROR:
      GNUNET_break (0);
      return TALER_MHD_reply_with_error (rc->connection,
                                         MHD_HTTP_INTERNAL_SERVER_ERROR,
                                         TALER_EC_GENERIC_DB_STORE_FAILED,
                                         "update_charity");
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      return TALER_MHD_reply_with_error (rc->connection,
                                         MHD_HTTP_NOT_FOUND,
                                         TALER_EC_DONAU_CHARITY_NOT_FOUND,
                                         args[0]);
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      return TALER_MHD_reply_static (rc->connection,
                                     MHD_HTTP_OK,
                                     NULL,
                                     NULL,
                                     0);
    }
  }

  GNUNET_break (0);
  return TALER_MHD_reply_with_error (rc->connection,
                                     MHD_HTTP_INTERNAL_SERVER_ERROR,
                                     TALER_EC_GENERIC_INTERNAL_INVARIANT_FAILURE,
                                     "charity_patch");
}


/* end of donau-httpd_charity_patch.c */
