/*
 * S.F.S. v. 0.5alpha-linux
 *
 * chsecure     sets the secure extended attribute of a file on the SFS-server.
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@mikonos.dia.unisa.it>
 *		Angelo Celentano, <angcel@mikonos.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@mikonos.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@mikonos.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@mikonos.dia.unisa.it>
 *
 * Permission to use, copy, and modify this software without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies of any software which is or includes a copy or
 * modification of this software and in all copies of the supporting
 * documentation for such software.
 *
 * This software maybe be used for any purpose provided
 * the above copyright notice is retained.  It is supplied
 * as is, with no warranty expressed or implied.
 *
 */

/* Including system library */
#include <stdio.h>
#include <rpc/rpc.h>

/* Including parsing library */
#include <unistd.h>
extern char *optarg;                        /* Variables to parse the command*/
extern int optind, opterr, optopt;          /* line options                 */

/* Including SFS library */
#include <tcfs_prot.h>
#include "chsecure.h"

/* Usage function */
void usage(char **argv, int err_code)
{
  switch (err_code)
    {
    case ERR_SFSREG:
      printf("\n%s ERROR: User not registered\n",argv[0]);
      exit(ERR_SFSREG);

    case ERR_USE:
      printf ("\n%s usage: %s -s|-u filename [filename]\n",argv[0],argv[0]);
      printf ("Set-up the secure flag.\n");
      printf ("Parameters:\n");
      printf ("    -s         set the flag to Secure mode.\n");
      printf ("    -u         set che flag to Unsecure mode.\n");
      printf ("    filename   file to set-up.\n");
      exit (ERR_USE);
    }
}

/* Function to build the absolute path of file */
char *makepath(char *fname)
{
  char *absfname;                                /* Absolute pathname        */

  if(*fname=='/') return(fname);

  absfname=malloc(MAX_PATHLEN);
  getcwd(absfname, MAX_PATHLEN);
  if (!strcmp(fname,"."))
    return(absfname);
  if(absfname[strlen(absfname)-1]!="/")
    strcat(absfname, "/");
  strcat(absfname, fname);
  return(absfname);
}

/* Main function to make change secure */
int chsecure(setextatargs *in) 
{
  enum clnt_stat cs;                        /* RPC-Call return value        */
  unsigned int out;                         /* Result from the SFS-Server   */

  /* Make and check the RPC-Call to the SFS-Server */
  switch(cs=callrpc("localhost", NFS_PROGRAM, NFS_VERSION, NFSPROC_SETEXTAT, (xdrproc_t)xdr_setextatargs, (char*)in, (xdrproc_t)xdr_int, (char*)&out))
    {
    default:                                     /* RPC-Call failed         */
      clnt_perrno(cs);
      return(RPC_FAIL);
      
    case RPC_SUCCESS:
      switch(out)
        {
	default:
	  return(0);

	case ERR_IOCTL:                          /* IOCTL Error              */
	   fprintf (stderr,"%s \t --- \t Cannot chenge flag\n",in->file);
	   return(ERR_IOCTL);

	case ERR_STICK:                          /* Error in stick table     */
	  fprintf (stderr,"%s \t --- \t Entry not found in mountd-stick-table\n",in->file);
	  return(ERR_STICK);

	case ERR_SFSREG:                         /* User not registered      */
	  fprintf (stderr,"User not registered\n");
	  exit(ERR_SFSREG);
        }
    }
}

/* Chsecure main code */
int main (int argc, char **argv)
{
  setextatargs in;                         /* Structure to pass to the       *
                                            * SFS-Server                     */
  int opt=0;                               /* Option to parse                */

  if(argc<3) usage(argv, ERR_USE);         

  while(opt!=ERR_USE)                      
    {
      opt=getopt(argc, argv, OPTIONS);
      switch (opt)
	{
	case 's':                          /* Secure mode setting-up         */
	  if (*optarg=='-') usage(argv, ERR_USE); 
	  in.uid=geteuid();
	  in.file=makepath(optarg);
	  in.mode=SECURE;
	  chsecure(&in);
	  for(;optind<argc && *argv[optind]!='-';optind++)
	    {
	      in.file=makepath(argv[optind]);
	      chsecure(&in);
	    }
	  break;

	case 'u':                          /* Unsecure mode setting-up       */
	  if (*optarg=='-') usage(argv, ERR_USE);
	  in.uid=geteuid();
	  in.file=makepath(optarg);
	  in.mode=UNSECURE;
	  chsecure(&in);
	  for(;optind<argc && *argv[optind]!='-';optind++)
	    {
	      in.file=makepath(argv[optind]);
	      chsecure(&in);
	    }
	  break;

	case '?':                          /* Unrecognized parameter         */
	  fprintf(stderr,"\nUnrecognized parameter %c\n",opt);
	  usage(argv, ERR_USE);
	}
    }
  return(0);
}

