#ifndef lint
static char *baseid = "sliplogin.c,v 1.15 89/09/25 22:22:09 rayan Release ";
#endif

/*
 * sliplogin.c
 *
 * This program initializes its own tty port to be an async TCP/IP interface.
 * It merely sets up the SLIP module all by its lonesome on the STREAMS stack,
 * initializes the network interface, and pauses forever waiting for hangup.
 *
 * It is a remote descendant of several similar programs with incestuous ties:
 * - Kirk Smith's slipconf, modified by Richard Johnsson @ DEC WRL.
 * - slattach, probably by Rick Adams but touched by countless hordes.
 * - the original sliplogin for 4.2bsd, Doug Kingston the mover behind it.
 * - a simple slattach-like program used to test the STREAMS SLIP code.
 *
 * Doug Kingston 8810??		- logging + first pass at adding I_STR ioctl's
 * Rayan Zachariassen 881011	- version for SunOS STREAMS SLIP
 * Greg Earle 890331		- added tip/UUCP style LCK lock file
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stropts.h>
#include <sys/termios.h>
#include <sys/ttold.h>
#include <sys/sockio.h>
#include <sys/file.h>
#include <sys/syslog.h>

#include <sys/slip.h>

#include <netinet/in.h>
#include <net/if.h>

#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>

#include <signal.h>
#include <strings.h>
#include <pwd.h>
#ifdef BSD >= 43
#include <ttyent.h>
#endif

#define	DCD_CHECK_INTERVAL 0	/* if > 0, time between automatic DCD checks */
#define	DCD_SETTLING_TIME 1	/* time between DCD change and status check */

#ifndef	HOSTS_SLIP
#define	HOSTS_SLIP	"/etc/hosts.slip"
#endif	/* HOSTS_SLIP */

int gotalarm = 0;
int gothup = 0;
int timeleft = DCD_CHECK_INTERVAL;

void
alarm_handler()
{
	if (timeleft > DCD_SETTLING_TIME)
		(void) alarm((u_int)(timeleft-DCD_SETTLING_TIME));
	else
		(void) alarm((u_int)DCD_CHECK_INTERVAL);
	gotalarm = 1;
	timeleft = 0;
}

void
hup_handler()
{
	gothup = 1;
}

#if	defined(SIGDCD) && SIGDCD > 0
void
dcd_handler()
{
#if	DCD_SETTLING_TIME > 0
	timeleft = alarm(DCD_SETTLING_TIME);
#else
	gotalarm = 1;
#endif	/* DCD_SETTLING_TIME */
}
#endif

/* Test for existence of DCD on the port of the passed descriptor */

int
dcd(fd)
	int fd;
{
	int mbits;

	if (ioctl(fd, TIOCMGET, (caddr_t)&mbits) < 0)
		return 0;	/* port is dead, we die */
	return (mbits & TIOCM_CAR);
}

char	*Accessfile = HOSTS_SLIP;

extern char *malloc(), *ttyname(), *sprintf();
extern struct passwd *getpwuid();

char	*dstaddr, *localaddr, *netmask;
char	*termname;

#define	SLIPFD 0

main(argc, argv)
	int argc;
	char *argv[];
{
	int		fd, s, unit;
#ifdef	LCKDIR
	int		lockfd = -1;
	char		*lockfile = NULL;
#endif	/* LCKDIR */
	struct termios	tios;
	struct ifreq	ifr;
	extern void	findid();

	s = getdtablesize();
	for (fd = 3 ; fd < s ; fd++)
		(void) close(fd);

	openlog("sliplogin", LOG_PID, LOG_DAEMON);

	if (getuid() == 0) {
		if (argc <= 1) {
			(void) fprintf(stderr,
				       "Usage: %s loginname\n", argv[0]);
			(void) fprintf(stderr,
				       "   or: %s dstaddr localaddr [mask]\n",
				       argv[0]);
			exit(1);
		} else if (argc == 2) {
			findid(argv[1]);
			(void) fprintf(stderr, "local %s remote %s mask %s\n",
					       localaddr, dstaddr, netmask);
		} if (argc > 2) {
			if (argc < 3 || argc > 4) {
				(void) fprintf(stderr,
					"Usage: %s dstaddr localaddr [mask]\n",
					argv[0]);
				exit(1);
			}
			dstaddr = argv[1];
			localaddr = argv[2];
			if (argc == 4)
				netmask = argv[3];
			else
				netmask = "default";
		}
	} else
		findid((char *)0);

	/* ensure that the slip line is our controlling terminal */
	if ((fd = open("/dev/tty", O_RDONLY, 0)) >= 0) {
		(void) ioctl(fd, TIOCNOTTY, 0);
		(void) close(fd);
		termname = ttyname(SLIPFD);
		fd = open(termname, O_RDWR, 0);
		if (fd >= 0)
			(void) close(fd);
		(void) setpgrp(0, getpid());
	}

	(void) fchmod(SLIPFD, 0600);

#ifdef	LCKDIR
	termname = (rindex(termname, '/') + 1);
	lockfile = malloc((u_int)(sizeof "/LCK.."
					 + sizeof LCKDIR + strlen(termname)-1));
	(void) sprintf(lockfile, "%s/LCK..%s", LCKDIR, termname);
	lockfd = open(lockfile, O_EXCL | O_CREAT, 0600);
	if (lockfd == -1) {
		(void) fprintf(stderr,
			       "%s: can't get lock on SLIP device (%s)\n",
			       argv[0], lockfile);
		syslog(LOG_ERR, "Lock file exists\n");
		exit(1);
		/* NOTREACHED */
	}
#endif	/* LCKDIR */

	/* pop all streams modules */
	while (ioctl(SLIPFD, I_POP, 0) == 0)
		continue;

	/* set up the line parameters */
	if (ioctl(SLIPFD, TCGETS, (caddr_t)&tios) < 0) {
		syslog(LOG_ERR, "ioctl (TCGETS): %m");
		exit(1);
	}
	tios.c_cflag &= (CRTSCTS | 0xf);   /* only save the speed and CTSRTS */
	tios.c_cflag |= CS8|CREAD|HUPCL;
	tios.c_iflag = IGNBRK;
	if (ioctl(SLIPFD, TCSETS, (caddr_t)&tios) < 0) {
		syslog(LOG_ERR, "ioctl (TCSETS): %m");
		exit(1);
	}

	/* push the SLIP module */
	if (ioctl(SLIPFD, I_PUSH, "slip") < 0) {
		syslog(LOG_ERR, "ioctl (I_PUSH): %m");
		exit(1);
	}

	/* find out what unit number we were assigned */
	if (ioctl(SLIPFD, SLIOGUNIT, (caddr_t)&unit) < 0) {
		syslog(LOG_ERR, "ioctl (SLIOGUNIT): %m");
		exit(1);
	}

	syslog(LOG_NOTICE, "attaching slip%d: local %s remote %s mask %s\n",
		unit, localaddr, dstaddr, netmask);

	/* set the local and remote interface addresses */
	s = socket(AF_INET, SOCK_DGRAM, 0);

	if (getuid() != 0 || argc == 4) {
		bzero((char *)&ifr, sizeof(ifr));
		(void) sprintf(ifr.ifr_name, "%s%d", SLIPIFNAME, unit);
		in_getaddr(netmask, &ifr.ifr_addr);
		if (ioctl(s, SIOCSIFNETMASK, (caddr_t)&ifr) < 0) {
			syslog(LOG_ERR, "ioctl (SIOCSIFNETMASK): %m");
			exit(1);
		}
	}

	bzero((char *)&ifr, sizeof(ifr));
	(void) sprintf(ifr.ifr_name, "%s%d", SLIPIFNAME, unit);
	in_getaddr(dstaddr, &ifr.ifr_addr);
	if (ioctl(s, SIOCSIFDSTADDR, (caddr_t)&ifr) < 0) {
		syslog(LOG_ERR, "ioctl (SIOCSIFDSTADDR): %m");
		exit(1);
	}

	bzero((char *)&ifr, sizeof(ifr));
	(void) sprintf(ifr.ifr_name, "%s%d", SLIPIFNAME, unit);
	in_getaddr(localaddr, &ifr.ifr_addr);
	/* this has the side-effect of marking the interface up */
	if (ioctl(s, SIOCSIFADDR, (caddr_t)&ifr) < 0) {
		syslog(LOG_ERR, "ioctl (SIOCSIFADDR): %m");
		exit(1);
	}

	/* set up signal handlers */
#if	defined(SIGDCD) && SIGDCD > 0
	(void) signal(SIGDCD, dcd_handler);
#endif
	(void) sigblock(sigmask(SIGALRM));
	(void) signal(SIGALRM, alarm_handler);

	/* HUPs are sticky on a stream, so no point in testing line condition */
	(void) signal(SIGHUP, hup_handler);
	(void) signal(SIGTERM, hup_handler);

	(void) alarm((u_int)DCD_CHECK_INTERVAL);

	/* twiddle thumbs until we get a signal */
	while (1) {
		sigpause(0);
		(void) sigblock(sigmask(SIGALRM));
		if (gothup || (gotalarm && !dcd(SLIPFD)))
			break;
		gotalarm = 0;
	}

#ifdef	LCKDIR
	/* clean up and remove the tip/UUCP lockfile */
	(void) unlink(lockfile);
	(void) close(lockfd);
#endif	/* LCKDIR */

	/* closing the descriptor should pop the slip module */
	exit(0);
}

void
findid(name)
	char *name;
{
	char buf[BUFSIZ];
	static char mode[16];
	static char laddr[16];
	static char raddr[16];
	static char mask[16];
	char user[16];
	FILE *fp;
	struct passwd *pw;
	int n;

	if (name == NULL && (pw = getpwuid(getuid())) == NULL) {
		(void) fprintf(stderr, "Your UID (%d) is unknown\n", getuid());
		syslog(LOG_ERR, "UID (%d) is unknown\n", getuid());
		exit(1);
	} else if (name == NULL)
		name = pw->pw_name;
	if ((fp = fopen(Accessfile, "r")) == NULL) {
		perror(Accessfile);
		syslog(LOG_ERR, "%s: %m\n", Accessfile);
		exit(3);
	}
	while (fgets(buf, sizeof(buf) - 1, fp)) {
		if (ferror(fp))
			break;
		n = sscanf(buf, "%15s%*[ \t]%15s%*[ \t]%15s%*[ \t]%15s%*[ \t]%15s\n",
			user, mode, laddr, raddr, mask);
		if (user[0] == '#' || n != 5)
			continue;
		if (strcmp(user, name) == 0) {
			/* eventually deal with "mode" */
			localaddr = laddr;
			dstaddr = raddr;
			netmask = mask;
			(void) fclose(fp);
			return;
		}
		if (feof(fp))
			break;
	}
	fputs("SLIP access denied\n", stderr);
	syslog(LOG_ERR, "SLIP access denied for %s\n", name);
	exit(4);
	/* NOTREACHED */
}

in_getaddr(s, saddr)
	char *s;
	struct sockaddr *saddr;
{
	register struct sockaddr_in *sin = (struct sockaddr_in *)saddr;
	struct hostent *hp;
	struct netent *np;
	u_long val;
	extern struct in_addr inet_makeaddr();
	extern u_long inet_addr();
 
	bzero((caddr_t)saddr, sizeof *saddr);
	sin->sin_family = AF_INET;
	val = inet_addr(s);
	if (val != (u_long)-1) {
		sin->sin_addr.s_addr = val;
		return;
	}
	hp = gethostbyname(s);
	if (hp) {
		sin->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (char *)&sin->sin_addr, hp->h_length);
		return;
	}
	np = getnetbyname(s);
	if (np) {
		sin->sin_family = np->n_addrtype;
		sin->sin_addr = inet_makeaddr((int)np->n_net, (int)INADDR_ANY);
		return;
	}
	(void) fprintf(stderr, "sliplogin: %s: bad value\n", s);
	syslog(LOG_ERR, "%s: bad value\n", s);
	exit(1);
	/* NOTREACHED */
}
