/* fields.c -- routines to build the list of fields available from the server.
 *
 * Copyright (C) 1992, Bradley C. Spatz, bcs@ufl.edu
 * Last edited: Wed Apr 22 11:04:16 1992 by bcs (Bradley C. Spatz) on frenulum
 */

#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>

#include "fields.h"
#include "replies.h"
#include "global.h"

/* Declare some globals for this module related to the list. */
static field_node_t *field_root=NULL;
static int num_fields=0;
static String *field_name_list;
static int list_size=0;


int attr_to_mask(attr_str)
char *attr_str;
{
   int mask=0;
   char buf[80], *token;

   /* Make a copy of the source string and decode the words into a
    * a bit-coded integer.
    */
   strcpy(buf, attr_str);
   token = strtok(buf, " ");
   while (token != NULL) {
      if (strcmp(token, "Indexed") == 0) {
	 mask = mask | INDEXED;
      }
      else if (strcmp(token, "Lookup") == 0) {
	 mask = mask | LOOKUP;
      }	 
      else if (strcmp(token, "Public") == 0) {
	 mask = mask | PUBLIC;
      }	 
      else if (strcmp(token, "Default") == 0) {
	 mask = mask | DEFAULT;
      }	 
      else if (strcmp(token, "Change") == 0) {
	 mask = mask | CHANGE;
      }	 
      else if (strcmp(token, "Encrypt") == 0) {
	 mask = mask | ENCRYPT;
      }	 
      else if (strcmp(token, "NoPeople") == 0) {
	 mask = mask | NOPEOPLE;
      }	 
      else {
	 fprintf(stderr, "attribute_mask: %s attribute unknown!\n",
		 token);
      }
      token = strtok(NULL, " ");
   }

   return(mask);
}


/* Allocates, initializes, and returns a new node. */
field_node_t *new_field_node(name, max_length, attr_str, attr_mask,
			     description)
char *name;
int max_length;
char *attr_str;
int attr_mask;
char *description;
{
   field_node_t *new;

   new = (field_node_t *) malloc(sizeof(field_node_t));
   new->name = (char *) malloc(strlen(name) + 1);
   strcpy(new->name, name);
   new->max_length = max_length;
   new->attr_str = (char *) malloc(strlen(attr_str) + 1);
   strcpy(new->attr_str, attr_str);
   new->attr_mask = attr_mask;
   new->description = (char *) malloc(strlen(description) + 1);
   strcpy(new->description, description);
   new->left = (field_node_t *) NULL;
   new->right = (field_node_t *) NULL;

   return(new);
}


/* A basic recursive insert routine that returns the pointer to the
 * root sent as the first argument.  This routine should be optimized
 * at a later date to remove the recursion and use a better malloc()
 * scheme for performance.  But for right now, recursion is easy for me.
 */
field_node_t *insert_field(root, node)
field_node_t *root;
field_node_t *node;
{
   if (root == NULL) {
      root = node;
   }
   else if (strcmp(node->name, root->name) == 0) {
      fprintf(stderr, "insert_field: field '%s' already in tree!\n",
	      node->name);
   }
   else if (strcmp(node->name, root->name) < 0) {
      root->left = insert_field(root->left, node);
   }
   else {
      root->right = insert_field(root->right, node);
   }

   return(root);
}


#ifdef DEBUG
/* A debugging routine to print the contents of the tree. */
print_list(root, indent)
field_node_t *root;
int indent;
{
   int i;

   if (root == NULL) {
      return;
   }
   print_list(root->left, indent+2);
   for (i=0; i<indent; i++) {
      printf(" ");
   }
   printf("%s\n", root->description);
   print_list(root->right, indent+2);
}
#endif


field_node_t *find_entry(name)
String name;
{
   field_node_t *node;
   int done, compare;

   node = field_root;
   done = 0;
   while ((! done) && (node != NULL)) {
      compare = strcmp(node->name, name);
      if (compare > 0) {
	 node = node->left;
      }
      else if (compare == 0) {
	 done = 1;
      }
      else {
	 node = node->right;
      }
   }
   
   return(node);
}


/* Ask the server for a list of fields.  Store this information
 * for posterity, dude.
 */
static void grok_fields()
{
   int done, code, fid, max_length, attr_mask;
   char name[80], attr_str[80], description[80];
   field_node_t *new_node;

   if (field_root == NULL) {
      write_ns("fields\n");
      done = 0;
      while (! done) {
	 read_ns(line);
	 strcpy(attr_str, "");
	 attr_mask = 0;
	 sscanf(line, "%d:%d:%[^:]:max %d %s", &code, &fid, name,
		&max_length, attr_str);
	 switch (atoi(line)) {
	    case LR_OK:
	       done = 1;
	       break;
            case -(LR_OK):
	       /* We probably violate the protocol here, but we process lines
		* from the server in pairs, and hope this works.  I apologize
		* for the poor parsing here, but this *is* a prototype.  ;-)
		* We should have parsed what we need from the first line,
		* except for the attributes, which we check below.
		*/
	       if (strcmp(attr_str, "") != 0) {
		  /* We need to further parse the attributes.*/
		  strcpy(attr_str, word(word(line, ':', 3), ' ', 2));
		  attr_mask = attr_to_mask(attr_str);
	       }
	       read_ns(line);
	       strcpy(description, word(line, ':', 3));
	       new_node = new_field_node(name, max_length, attr_str,
					 attr_mask, description);
	       field_root = insert_field(field_root, new_node);
	       num_fields++;
	       break;
            default:
	       fprintf(buf, "grok_fields():%s\n", word(line, ':', 1));
	       break;
         }
      }

      /* Now allocate some space for a list of field names for a list widget.
       * We'll use that later. 
       */
      field_name_list = (char **) malloc((num_fields+1) * sizeof(char *));
   }
}


/* This routine accepts a mask of field attributes, that can be OR'd
 * together, that i used to generate a list of field names.  We
 * return a pointer to our field_name_list.  We hide the recursion
 * by calling the recursive here.
 */
String *field_list(extract_mask, want_mask)
int extract_mask;
int want_mask;
{
   grok_fields();
   list_size = 0;
   make_field_list(field_root, extract_mask, want_mask);
   field_name_list[list_size] = NULL;
   return(field_name_list);
}


make_field_list(root, extract_mask, want_mask)
field_node_t *root;
int extract_mask;
int want_mask;
{
   if (root == NULL) {
      return;
   }
   make_field_list(root->left, extract_mask, want_mask);
   
   /* Check the attributes of the current node.  If the desired attributes
    * are present in the entry, add the field name to the list and increment
    * our count.
    */
   if ((root->attr_mask & extract_mask) == want_mask) {
      field_name_list[list_size] = (char *) root->name;
      list_size++;
   }

   make_field_list(root->right, extract_mask, want_mask);
}


/* This routine send a formatted list of fields into the main
 * text window.  We get this information from the field list.
 */
do_field_info()
{
   XawAsciiSaveTextPosition(text);
   grok_fields();
   print_field_info(field_root);
   XawAsciiAppend(text, "\n");
   XawAsciiRestoreTextPosition(text);
}

print_field_info(root)
field_node_t *root;
{
   if (root == NULL) return;

   print_field_info(root->left);

   /* Print the current node. */
   sprintf(buf, "%s:\n", root->name);
   XawAsciiAppend(text, buf);
   sprintf(buf, "   Description: %s\n", root->description);
   XawAsciiAppend(text, buf);
   sprintf(buf, "    Max Length: %d\n", root->max_length);
   XawAsciiAppend(text, buf);
   sprintf(buf, "    Attributes: %s\n", root->attr_str);
   XawAsciiAppend(text, buf);
   
   print_field_info(root->right);
}


#ifdef DEBUG
/* A debugging routine to print a field name list. */
print_field_name_list(list)
String *list;
{
   int i=0;

   while (list[i] != NULL) {
      fprintf(stderr, "%3d: %s\n", i, list[i]);
      i++;
   }
}
#endif


/* Return TRUE/FALSE if field is a "default" field. */
int is_default_field(name)
char *name;
{
   field_node_t *node;

   node = find_entry(name);

   return(((node->attr_mask & DEFAULT) == DEFAULT) ? 1 : 0);
}
