########################################################################
# housekeeping
########################################################################

use v6.d;

unit module ProcStats:ver<0.1.1>:auth<CPAN:lembark>;

########################################################################
# subroutines
########################################################################

sub dump-rusage
(
    Bool()      :$final = False ,
    Bool()      :$first = $final,
    Bool()      :$force = $final,
    Stringy()   :$label 
                = $final 
                ?? 'Final' 
                !! ''
)
is export( :DEFAULT )
{
    my $wtime       = now.Num;

    constant FIELDS =
    <
        maxrss  ixrss   idrss   isrss    
        minflt  majflt  nswap   
        inblock oublock msgsnd
        msgrcv  nsignals nvcsw   nivcsw
    >;  
    constant IGNORE =
    <
        ixrss idrss isrss
        nswap
        msgsnd msgrcv nsignals
    >;  
    constant COMPARE =
    <
        maxrss
        majflt  minflt
        inblock oublock
    >;
    constant MICRO = 10 ** -6;
    state Int $passes   = 0;

    use nqp;
    nqp::getrusage( my int @raw );

    my ( $user_s, $user_us, $syst_s, $syst_us )
    = splice @raw, 0, 4;

    my %sample = FIELDS Z=> @raw;
    %sample{ IGNORE } :delete;

    my $utime   = ( $user_s + $user_us / 1_000_000  ).round( MICRO );
    my $stime   = ( $syst_s + $syst_us / 1_000_000  ).round( MICRO );

    # on first pass "$first" makes no difference.

    state %last  =
    state %first =
    (
        |%sample,
        :$wtime,
        :$utime,
        :$stime,
    );

    my %prior
    = $first
    ?? %first 
    !! %last
    ;

    my %curr
    = ( $force || ! $passes )
    ?? %sample
    !! do
    {
        my @diffs
        = COMPARE.grep( { %sample{$_} != %prior{$_} } );

        @diffs Z=> %sample{ @diffs }
    }
    ;

    sub write-stat ( Pair $p )  
    {
        note
        sprintf
            '%-*s : %s',
            once {FIELDS».chars.max},
            $p.key,
            $p.value
        ;
        return
    }
    
    sub write-diff ( Pair $p )
    {
        my $k   = $p.key;
        my $v   = ( $p.value - %first{ $k } ).round( MICRO );

        write-stat $k => $v;
    }

    state &write    = &write-stat;

    for %curr.sort -> $stat
    {
        FIRST
        {
            note '---';
            write-stat ( output => $++  );
            write-stat ( :$passes       );
            write-stat ( :$label        ) if $label;

            # compose pairs before passing the values
            # as parameters. write( ... ) without the 
            # space treats them as named parameters
            # and fails.

            write (:$wtime) ;
            write (:$utime) ;
            write (:$stime) ;
        }

        # stat is already a pair, no parens needed.

        write $stat ;
    }

    once { &write = &write-diff };

    %last = %sample;

    ++$passes;

    return
}

=finish

=begin pod

=head1 NAME

ProcStats - dump rusage process statistics with optional label.

=head1 SYNOPSIS

    # print rusage output to stderr.
    # "sample" is incremented with each output.
    #
    # first pass ("sample 0") dumps all stats.
    # successive ones only list changes.
    #
    # system time (stime) invariably increases due to 
    # rusage call and is only output if another field
    # causes output.
    #
    # format is YAML hash with one document per sample.


    dump-rusage( label => "$*PROGRAM" );

    ---
    sample   : 0
    label    : rand-dictonary
    stime    : 724655
    idrss    : 129224
    inblock  : 0
    isrss    : 0
    ixrss    : 39741
    majflt   : 0
    maxrss   : 0
    minflt   : 0
    msgrcv   : 48
    msgsnd   : 0
    nivcsw   : 0
    nsignals : 0
    nswap    : 32466
    nvcsw    : 0
    oublock  : 0
    utime    : 0

    # force output even if there are no changes.

    dump-rusage( label => 'Initial users', :force );
    ---
    sample   : 1
    label    : Initial users
    stime    : 748975
    idrss    : 129620
    nswap    : 32639

    dump-rusage;

    ---
    sample   : 2
    stime    : 769168
    idrss    : 129864
    nswap    : 32732

    dump-rusage( :force, :start );

    ---
    sample   : 92
    label    : First
    stime    : 126712
    idrss    : 136936
    inblock  : 0
    isrss    : 0
    ixrss    : 70682
    majflt   : 0
    maxrss   : 0
    minflt   : 0
    msgrcv   : 48
    msgsnd   : 0
    nivcsw   : 0
    nsignals : 0
    nswap    : 34532
    nvcsw    : 0
    oublock  : 0
    utime    : 1


=head2 Notes

    man 2 rusage;

=head2 Example Code

=item bare-for-loop

Run the output by itself to see what overhead the usage has 
on the local system:

    $ examples/bare-for-loop 2>&1 | tail -14;
    ---
    output   : 141
    passes   : 1000
    label    : Final
    wtime    : 0.536303
    utime    : 0.703809
    stime    : 0.000958
    inblock  : 0
    majflt   : 0
    maxrss   : 11448
    minflt   : 3501
    nivcsw   : 4
    nvcsw    : 175
    oublock  : 0

The example code is simply a loop with the ":final" option
(which is captured by tail -14).

=item rand-user-table

This performs a set of random operations on a hash by adding
keys, dropping keys, and incrementing the stored values. 

It calls dump-rusage every 1000 iterations of the trial:

    for 1 .. 1000
    {
        constant weighted_operations =
        (
            &user-add  => 0.10,
            &user-drop => 0.10,
            &user-op   => 0.80,
        ).Mix;

        weighted_operations.roll( 1_000 )».();

 *      dump-rusage( label => 'Keys: ' ~ %user-data.elems );
    }

    ./exmaple/rand-user-table 2>&1 | tail -14;
    ---
    output   : 517
    passes   : 1001
    label    : Final
    wtime    : 18.543969
    utime    : 18.854329
    stime    : 0.03012
    inblock  : 0
    majflt   : 0
    maxrss   : 18196
    minflt   : 11679
    nivcsw   : 56
    nvcsw    : 6404
    oublock  : 0

Note that 18.54s > 18.85s: wallclock is less than user time due to
threading! Use "wtime" for elapsed, not utime.

The "maxrss" value shows that memory grew 18_196 KiB during execution.


=head1 SEE ALSO


=item getrusage(3) getrusage(3p)

getrusage is POSIX.

    struct rusage 
    {
       struct timeval ru_utime; /* user CPU time used */
       struct timeval ru_stime; /* system CPU time used */
       long   ru_maxrss;        /* maximum resident set size */
       long   ru_ixrss;         /* integral shared memory size */
       long   ru_idrss;         /* integral unshared data size */
       long   ru_isrss;         /* integral unshared stack size */
       long   ru_minflt;        /* page reclaims (soft page faults) */
       long   ru_majflt;        /* page faults (hard page faults) */
       long   ru_nswap;         /* swaps */
       long   ru_inblock;       /* block input operations */
       long   ru_oublock;       /* block output operations */
       long   ru_msgsnd;        /* IPC messages sent */
       long   ru_msgrcv;        /* IPC messages received */
       long   ru_nsignals;      /* signals received */
       long   ru_nvcsw;         /* voluntary context switches */
       long   ru_nivcsw;        /* involuntary context switches */
    };

