#!/usr/bin/env perl6
use v6.d;

#
# Copyright © 2018 Joelle Maslak
# All Rights Reserved - See License
#

use Net::BGP;
use Net::BGP::IP;
use Net::BGP::Time;

my subset Port of UInt where ^2¹⁶;
my subset Asn  of UInt where ^2¹⁶;

sub MAIN(
    Bool:D               :$passive = False,
    Int:D                :$port = 179,
    Int:D                :$my-asn,
    Int                  :$max-log-messages,
    Net::BGP::IP::ipv4:D :$my-bgp-id,
    Int:D                :$batch-size = 32,
    Str                  :$cidr-filter,
    Str                  :$announce,
    Bool:D               :$short-format = False,
    *@args is copy
) {
    my $bgp = Net::BGP.new(
        :$port,
        :$my-asn,
        :identifier(ipv4-to-int($my-bgp-id)),
    );

    # Add peers
    while @args {
        my $peer-ip  = @args.shift;
        if ! @args.elems { die("Must specify peer ASN after the peer IP"); }
        my $peer-asn = @args.shift;
       
        my $md5; 
        if @args.elems {
            if @args[0] ~~ m/^ '--md5='/ {
                $md5 = S/^ '--md5='// given @args.shift;
                $bgp.add-md5($peer-ip, $md5);
            }
        }

        $bgp.peer-add( :$peer-asn, :$peer-ip :$passive );
    }

    # Build CIDR filter
    my @cidr-str = $cidr-filter.split(',') if $cidr-filter.defined;
    my @cidr-filter = gather {
        for @cidr-str -> $cidr {
            take Net::BGP::CIDR.from-str($cidr);
        }
    }

    # Build the announcements
    my @announce-str = $announce.split(',') if $announce.defined;
    my @announcements = @announce-str.map: -> $info {
        my @parts = $info.split('-');
        if @parts.elems ≠ 2 { die("Ammouncement must be in format <ip>-<nexthop>") }
        Net::BGP::Message.from-hash(
            {
                message-name => 'UPDATE',
                as-path      => '',             # XXX We do something different for eBGP
                local-pref   => 100,            # XXX Set localpref
                origin       => 'I',
                next-hop     => @parts[1],
                nlri         => @parts[0],
            },
            :asn32,     # Should change depending on host
        );
    }

    # Start the TCP socket
    $bgp.listen();
    lognote("Listening") unless $short-format;
    short-format-output(short-line-header) if $short-format;

    my $channel = $bgp.user-channel;

    my $messages-logged = 0;
    my $start = monotonic-whole-seconds;

    react {
        my %sent-connections;

        whenever $channel -> $event is copy {
            my @stack;

            my uint32 $cnt = 0;
            repeat {
                if $event ~~ Net::BGP::Event::BGP-Message {
                    if $event.message ~~ Net::BGP::Message::Open {
                        if %sent-connections{ $event.connection-id }:!exists {
                            for @announcements -> $bgpmsg {
                                $bgp.send-bgp( $event.connection-id, $bgpmsg );
                            }
                            %sent-connections{ $event.connection-id } = True;
                        }
                    }
                }

                @stack.push: $event;
                if $cnt++ ≤ 8*2*$batch-size {
                    $event = $channel.poll;
                } else {
                    $event = Nil;
                }
            } while $event.defined;

            if @stack.elems == 0 { next; }

            my @str;
            if (@stack.elems > $batch-size) {
                @str = @stack.hyper(
                    :degree(8), :batch((@stack.elems / 8).ceiling)
                ).grep(
                    { is-filter-match($^a, :@cidr-filter) }
                ).map({ $short-format ?? short-lines($^a) !! $^a.Str }).flat;
            } else {
                @str = @stack.map: { $^a.Str }
                @str = @stack.grep(
                    { is-filter-match($^a, :@cidr-filter) }
                ).map({ $short-format ?? short-lines($^a) !! $^a.Str }).flat;
            }

            for @str -> $event {
                if $short-format {
                    short-format-output($event);
                } else {
                    logevent($event);
                }

                $messages-logged++;
                if $max-log-messages.defined && ($messages-logged ≥ $max-log-messages) {
                    if ! $short-format {
                        log('*', "RUN TIME: " ~ (monotonic-whole-seconds() - $start) );
                    }
                    exit;
                }
            }
            @str.list.sink;
        }
    }
}

multi is-filter-match(Net::BGP::Event::BGP-Message:D $event, :@cidr-filter -->Bool:D ) {
    if $event.message ~~ Net::BGP::Message::Update {
        if ! @cidr-filter.elems { return True }

        my @nlri = @( $event.message.nlri );
        for @cidr-filter.grep( { $^a.ip-version == 4 } ) -> $cidr {
            if @nlri.first( { $cidr.contains($^a) } ).defined { return True; }
        }

        my @withdrawn = @( $event.message.withdrawn );
        for @cidr-filter.grep( { $^a.ip-version == 4 } ) -> $cidr {
            if @withdrawn.first( { $cidr.contains($^a) } ).defined { return True; }
        }

        my @nlri6 = @( $event.message.nlri6 );
        for @cidr-filter.grep( { $^a.ip-version == 6 } ) -> $cidr {
            if @nlri6.first( { $cidr.contains($^a) } ).defined { return True; }
        }

        my @withdrawn6 = @( $event.message.withdrawn6 );
        for @cidr-filter.grep( { $^a.ip-version == 6 } ) -> $cidr {
            if @withdrawn6.first( { $cidr.contains($^a) } ).defined { return True; }
        }

        return False;
    } else {
        return True;
    }
}
multi is-filter-match($event, :@cidr-filter -->Bool:D) { True }

multi get-str($event, :@cidr-filter -->Str) { $event.Str }

sub logevent(Str:D $event) {
    state $counter = 0;
    lognote("«" ~ $counter++ ~ "» " ~ $event);
}

sub lognote(Str:D $msg) {
    log('N', $msg);
}

sub log(Str:D $type, Str:D $msg) {
    say "{DateTime.now.Str} [$type] $msg";
}

sub short-format-output(Str:D $line -->Nil) {
    say $line;
}

multi short-lines(Net::BGP::Event::BGP-Message:D $event -->Array[Str:D]) {
    my Str:D @out;

    my $bgp = $event.message;
    if $bgp ~~ Net::BGP::Message::Open {
        push @out, short-line-open($event.peer);
    } elsif $bgp ~~ Net::BGP::Message::Update {
        if $bgp.nlri.elems {
            for @($bgp.nlri) -> $prefix {
                push @out, short-line-announce($prefix, $event.peer, $bgp);
            }
        } elsif $bgp.nlri6.elems {
            for @($bgp.nlri6) -> $prefix {
                push @out, short-line-announce6($prefix, $event.peer, $bgp);
            }
        } elsif $bgp.withdrawn.elems {
            for @($bgp.withdrawn6) -> $prefix {
                push @out, short-line-withdrawn($prefix, $event.peer);
            }
        } elsif $bgp.withdrawn6.elems {
            for @($bgp.withdrawn6) -> $prefix {
                push @out, short-line-withdrawn($prefix, $event.peer);
            }
        }
    } else {
        # Do nothing for other types of messgaes
    }

    return @out;
}

multi short-lines($event -->Array[Str:D]) { return Array[Str:D].new; }

sub short-line-header(-->Str:D) {
    return join("|",
        "Type",
        "Date",
        "Peer",
        "Prefix",
        "Next-Hop",
        "Path",
        "Communities",
    );
}

sub short-line-announce(
    Net::BGP::CIDR $prefix,
    Str:D $peer,
    Net::BGP::Message::Update $bgp
    -->Str:D
) {
    return join("|",
        "A",
        DateTime.now.posix,
        $peer,
        $prefix,
        $bgp.next-hop,
        $bgp.path,
        $bgp.community-list.join(" "),
    );
}

sub short-line-announce6(
    Net::BGP::CIDR $prefix,
    Str:D $peer,
    Net::BGP::Message::Update $bgp
    -->Str:D
) {
    return join("|",
        "A",
        DateTime.now.posix,
        $peer,
        $prefix,
        $bgp.next-hop6,
        $bgp.path,
        $bgp.community-list.join(" "),
    );
}

sub short-line-withdrawn(
    Net::BGP::CIDR $prefix,
    Str:D $peer,
    -->Str:D
) {
    return join("|",
        "W",
        DateTime.now.posix,
        $peer,
        $prefix,
    );
}

sub short-line-open(
    Str:D $peer,
    -->Str:D
) {
    return join("|",
        "O",
        DateTime.now.posix,
        $peer
    );
}

