/*
*+
*  Name:
*     palRdplan

*  Purpose:
*     Approximate topocentric apparent RA,Dec of a planet

*  Language:
*     Starlink ANSI C

*  Type of Module:
*     Library routine

*  Invocation:
*     void palRdplan( double date, int np, double elong, double phi,
*                     double * ra, double * dec, double * diam );

*  Arguments:
*     date = double (Given)
*        MJD of observation (JD-2400000.5) in TDB. For all practical
*        purposes TT can be used instead of TDB, and for many applications
*        UT will do (except for the Moon).
*     np = int (Given)
*        Planet: 1 = Mercury
*                2 = Venus
*                3 = Moon
*                4 = Mars
*                5 = Jupiter
*                6 = Saturn
*                7 = Uranus
*                8 = Neptune
*             else = Sun
*     elong = double (Given)
*        Observer's east longitude (radians)
*     phi = double (Given)
*        Observer's geodetic latitude (radians)
*     ra = double * (Returned)
*        RA (topocentric apparent, radians)
*     dec = double * (Returned)
*        Dec (topocentric apparent, radians)
*     diam = double * (Returned)
*        Angular diameter (equatorial, radians)

*  Description:
*     Approximate topocentric apparent RA,Dec of a planet, and its
*     angular diameter.

*  Authors:
*     TIMJ: Tim Jenness (JAC, Hawaii)
*     {enter_new_authors_here}

*  Notes:
*     - Unlike with slaRdplan, Pluto is not supported.
*     - The longitude and latitude allow correction for geocentric
*       parallax.  This is a major effect for the Moon, but in the
*       context of the limited accuracy of the present routine its
*       effect on planetary positions is small (negligible for the
*       outer planets).  Geocentric positions can be generated by
*       appropriate use of the routines palDmoon and iauPlan94.

*  History:
*     2012-03-07 (TIMJ):
*        Initial version, with some documentation from SLA/F.
*     {enter_further_changes_here}

*  Copyright:
*     Copyright (C) 2012 Science and Technology Facilities Council.
*     All Rights Reserved.

*  Licence:
*     This program is free software; you can redistribute it and/or
*     modify it under the terms of the GNU General Public License as
*     published by the Free Software Foundation; either version 3 of
*     the License, or (at your option) any later version.
*
*     This program is distributed in the hope that it will be
*     useful, but WITHOUT ANY WARRANTY; without even the implied
*     warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
*     PURPOSE. See the GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with this program; if not, write to the Free Software
*     Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
*     MA 02110-1301, USA.

*  Bugs:
*     {note_any_bugs_here}
*-
*/

#include <math.h>

#include "pal.h"
#include "palmac.h"
#include "sofa.h"

void palRdplan( double date, int np, double elong, double phi,
                double * ra, double * dec, double * diam ) {

  /* AU in km */
  const double AUKM = 1.49597870e8;

  /* Equatorial radii (km) */
  const double EQRAU[] = {
    696000.0, /* Sun */
      2439.7,
      6051.9,
      1738,
      3397,
     71492,
     60268,
     25559,
     24764
  };

  /* Local variables */
  int i, j;
  double stl;
  double vgm[6];
  double v[6];
  double rmat[3][3];
  double vse[6];
  double vsg[6];
  double vsp[6];
  double vgo[6];
  double dx,dy,dz,r,tl;

  /* Classify np */
  if (np < 0 || np > 8 ) np=0;  /* Sun */

  /* Approximate local sidereal time */
  stl = palGmst( date - palDt( palEpj(date)) / 86400.0) + elong;

  /* Geocentre to Moon (mean of date) */
  palDmoon( date, v );

  /* Nutation to true of date */
  palNut( date, rmat );
  iauRxp( rmat, v, vgm );
  iauRxp( rmat, &(v[3]), &(vgm[3]) );

  /* Moon? */
  if (np == 3) {

    /* geocentre to Moon (true of date) */
    for (i=0; i<6; i++) {
      v[i] = vgm[i];
    }

  } else {

    /* Not moon: precession/nutation matrix J2000 to date */
    palPrenut( 2000.0, date, rmat );

    /* Sun to Earth-Moon Barycentre (J2000) */
    palPlanet( date, 3, v, &j );

    /* Precession and nutation to date */
    iauRxp( rmat, v, vse );
    iauRxp( rmat, &(v[3]), &(vse[3]) );

    /* Sun to geocentre (true of date) */
    for (i=0; i<6; i++) {
      vsg[i] = vse[i] - 0.012150581 * vgm[i];
    }

    /* Sun ? */
    if (np == 0) {

      /* Geocentre to Sun */
      for (i=0; i<6; i++) {
        v[i] = -vsg[i];
      }

    } else {

      /* Sun to Planet (J2000) */
      palPlanet( date, np, v, &j );

      /* Precession and nutation to date */
      iauRxp( rmat, v, vsp );
      iauRxp( rmat, &(v[3]), &(vsp[3]) );

      /* Geocentre to planet */
      for (i=0; i<6; i++) {
        v[i] = vsp[i] - vsg[i];
      }

    }

  }

  /* Refer to origina at the observer */
  palPvobs( phi, 0.0, stl, vgo );
  for (i=0; i<6; i++) {
    v[i] -= vgo[i];
  }

  /* Geometric distance (AU) */
  dx = v[0];
  dy = v[1];
  dz = v[2];
  r = sqrt( dx*dx + dy*dy + dz*dz );

  /* Light time */
  tl = PAL__CR * r;

  /* Correct position for planetary aberration */
  for (i=0; i<3; i++) {
    v[i] -= tl * v[i+3];
  }

  /* To RA,Dec */
  iauC2s( v, ra, dec );
  *ra = iauAnp( *ra );

  /* Angular diameter (radians) */
  *diam = 2.0 * asin( EQRAU[np] / (r * AUKM ) );

}
