functor UtilFun
  (structure Data :DATA
   structure Pathname :PATHNAME
  ) :UTIL = struct

structure SysIO = System.Unsafe.SysIO

fun stringEqual (x:string, y:string) = (x=y)

fun printSep (sep :string) ([] :string list) :unit = ()
  | printSep sep (a::[]) = print a
  | printSep sep (a::(rest as (b::c))) =
      (print a; print sep; printSep sep rest)

fun wn (lst :string list) :unit =
  (print "% "; map print lst; print "\n")
fun err (lst :string list) :'a =
  (print "? "; map print lst; print "\n"; raise Data.CompilingError)

val zeroTime :System.Timer.time = System.Timer.TIME{sec=0,usec=0}
fun isZeroTime (System.Timer.TIME {sec,usec}) = (sec=0) andalso (usec=0)

val currentTime :unit -> System.Timer.time =
      System.Unsafe.CInterface.c_function "timeofday"

fun seconds (System.Timer.TIME{sec,...}) = sec
fun microSeconds (System.Timer.TIME{usec,...}) = usec
fun newer (x,y) = System.Timer.earlier (y, x)
fun newAs (x,y) = not (System.Timer.earlier (x, y))

fun modtime (filename:string) :System.Timer.time =
  if filename = "" then zeroTime
    else ((SysIO.mtime (SysIO.PATH filename)) handle _ => zeroTime)

fun absoluteName (directory:string) (filename:string) = 
  if substring(filename,0,1) = "/"
    then filename
    else Pathname.clearPath (Pathname.mergeDirFile directory filename)

fun trim (path:string) :string =
  let val parts = Pathname.explodePath path
      val len = length parts
      val strip' = len - (!System.Control.Print.pathnames) - 1
      val strip = if strip'<=1 then 0 else if strip'>len then len else strip'
      val showParts' = nthtail (parts, strip)
      val showParts = if strip>0 then ("..."::showParts') else showParts'
  in
    Pathname.implodePath showParts
  end

fun timeStamp () =
  let val System.Timer.TIME {sec,usec} = currentTime ()
  in (makestring sec)^"#"^(makestring usec) end

fun getwd () =
  let val pwdFile = "/tmp/SG-" ^ (timeStamp ())
      val status = System.Unsafe.CInterface.system("pwd > " ^ pwdFile ^ chr(0))
      val _ = if (status = 0) then () else
                (err ["Could not determine current working directory"])
      val instream = open_in pwdFile
  in
    let val line = input_line instream
        val dirname = substring (line, 0, (size line)-1)
    in
      close_in instream;
      System.Unsafe.SysIO.unlink pwdFile;
      dirname
    end
      handle e =>
        (print "? Could not determine current working directory\n";
         close_in instream;
         System.Unsafe.SysIO.unlink pwdFile;
         raise e)
  end

fun remDuplicates (x:string list) (y:string list) =
  let fun rmdup (([]:string list), (x:string list)) = ([], x)
        | rmdup (x, []) = (x, [])
        | rmdup (d as (dh::dt), r as (rh::rt)) =
            if dh = rh
                then let val (d',r')=rmdup(dt,rt) in (dh::d',r') end
              else if dh < rh
                then let val (d',r')=rmdup(dt,r) in (dh::d',r') end
                else let val (d',r')=rmdup(d,rt) in (d',rh::r') end
  in rmdup (x,y) end

fun handleInterrupt (operation :unit -> unit) =
  let exception Done
      val old'handler = System.Signals.inqHandler(System.Signals.SIGINT)
      fun reset'handler () =
        System.Signals.setHandler(System.Signals.SIGINT, old'handler)
  in (callcc (fn k =>
         (System.Signals.setHandler(System.Signals.SIGINT,SOME(fn _ => k)) ;
          operation ();
          raise Done));
        raise Interrupt)
      handle Done => (reset'handler ())
           | exn  => (reset'handler (); raise exn)
  end

val toolIsDefined :(string * int -> bool) ref = ref (fn (s,i) => false)
end
