/*
 * tclstartup.c --
 *
 * Startup code for the Tcl shell and other interactive applications.  Also
 * create special commands used just by Tcl shell features.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"

extern char * getenv ();

extern char *optarg;
extern int   optind, opterr;

typedef struct tclParms_t {
    int    execFile;      /* Run the specified file. (no searching)        */
    int    execCommand;   /* Execute the specified command.                */
    int    quickStartup;  /* Quick startup.                                */
    char  *execStr;       /* Command file or command to execute.           */
    char **tclArgv;       /* Arguments to pass to tcl script.              */
    int    tclArgc;       /* Count of arguments to pass to tcl script.     */
    char  *programName;   /* Name of program (less path).                  */
    } tclParms_t;

/*
 * Prototypes of internal functions.
 */
void
DumpTclError _ANSI_ARGS_((Tcl_Interp  *interp));

void
ParseCmdArgs _ANSI_ARGS_((int          argc,
                          char       **argv,
                          tclParms_t  *tclParmsPtr));

int
FindDefaultFile _ANSI_ARGS_((Tcl_Interp  *interp,
                             char        *defaultFile));

void
ProcessDefaultFile _ANSI_ARGS_((Tcl_Interp  *interp,
                                char        *defaultFile));


/*
 *----------------------------------------------------------------------
 *
 * Tcl_SourcepartCmd --
 *
 *	This procedure is invoked to process the "sourcepart" Tcl command:
 *          sourcepart fileName offset length
 *      which evaluates a range of a file.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
Tcl_SourcepartCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Interp       *iPtr = (Interp *) interp;
    long          fileOffset;
    int           bytesToRead;
    int           fileId, result = TCL_ERROR;
    struct stat   statBuf;
    char         *oldScriptFile;
    char         *fileName, *cmdBuffer = NULL, *end;

    if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " fileName offset length\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (Tcl_GetLong (interp, argv[2], &fileOffset) != TCL_OK)
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv[3], &bytesToRead) != TCL_OK)
        return TCL_ERROR;

    fileName = argv [1];
    if (fileName [0] == '~')
        if ((fileName = Tcl_TildeSubst (interp, fileName)) == NULL)
            return TCL_ERROR;

    fileId = open (fileName, O_RDONLY, 0);
    if (fileId < 0) {
	Tcl_AppendResult (interp, "open failed on: ", argv [1], ": ",
                          Tcl_UnixError (interp), (char *) NULL);
        return TCL_ERROR;
    }
    if (fstat(fileId, &statBuf) == -1) {
	Tcl_AppendResult (interp, "stat failed on: ", argv [1], ": ",
                          Tcl_UnixError (interp), (char *) NULL);
        goto exitPoint;
    }
    if (statBuf.st_size < fileOffset + bytesToRead) {
	Tcl_AppendResult (interp, "file not big enough for requested range: ",
                          argv [1], (char *) NULL);
        goto exitPoint;
    }
    if (lseek (fileId, fileOffset, 0) < 0) {
	Tcl_AppendResult (interp, "seek failed on: ", argv [1], ": ",
                          Tcl_UnixError (interp), (char *) NULL);
        goto exitPoint;
    }

    cmdBuffer = (char *) ckalloc((unsigned) bytesToRead+1);
    if (read(fileId, cmdBuffer, (int) bytesToRead) != bytesToRead) {
	Tcl_AppendResult (interp, "read failed on: ", argv [1], ": ",
                          Tcl_UnixError (interp), (char *) NULL);
        goto exitPoint;
    }
    close(fileId);
    fileId = -1;  /* Mark as closed */

    cmdBuffer[bytesToRead] = '\0';

    oldScriptFile = iPtr->scriptFile;
    iPtr->scriptFile = fileName;

    result = Tcl_Eval (interp, cmdBuffer, 0, &end);

    iPtr->scriptFile = oldScriptFile;
    if (result == TCL_RETURN) {
	result = TCL_OK;
    }
    /*
     * Record information telling where the error occurred.

     */

    if (result == TCL_ERROR) {
        char buf [100];
        sprintf (buf, "\n    (file \"%.50s\" line %d)", argv [1],
                 interp->errorLine);
	Tcl_AddErrorInfo(interp, buf);
    }
exitPoint:
    if (cmdBuffer != NULL)
        ckfree((char *)cmdBuffer);
    if (fileId >= 0)
        close (fileId);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * DumpTclError --
 *
 * Display error information and abort when an error is returned in the
 * interp->result.
 *
 * Parameters:
 *     o interp - A pointer to the interpreter, should contain the
 *       error message in `result'.
 *----------------------------------------------------------------------
 */
static void
DumpTclError (interp)
    Tcl_Interp  *interp;
{
    char *errorStack;

    fflush (stdout);
    fprintf (stderr, "Error: %s\n", interp->result);

    errorStack = Tcl_GetVar (interp, "errorInfo", 1);
    if (errorStack != NULL)
        fprintf (stderr, "%s\n", errorStack);
    exit (1);
}

/*
 *----------------------------------------------------------------------
 *
 * ParseCmdArgs --
 *
 * Parse the arguments passed to the Tcl shell
 *
 * Parameters:
 *     o argc, argv - Arguments passed to main.
 *     o tclParmsPtr - Results of the parsed Tcl shell command line.
 *----------------------------------------------------------------------
 */
static void
ParseCmdArgs (argc, argv, tclParmsPtr)
    int          argc;
    char       **argv;
    tclParms_t  *tclParmsPtr;
{
    char   *scanPtr, *programName;
    int     programNameLen;
    int     option;

    tclParmsPtr->execFile = FALSE;
    tclParmsPtr->execCommand = FALSE;
    tclParmsPtr->quickStartup = FALSE;
    tclParmsPtr->execStr = NULL;

    /*
     * Determine file name (less directories) that the Tcl interpreter is
     * being run under.
     */
    scanPtr = programName = argv[0];
    while (*scanPtr != '\0') {
        if (*scanPtr == '/')
            programName = scanPtr + 1;
        scanPtr++;
    }
    tclParmsPtr->programName = programName;
    programNameLen = strlen (programName);
    
    /*
     * Scan arguments looking for flags to process here rather than to pass
     * on to the scripts.  The '-c' or '-f' must also be the last option to
     * allow for script arguments starting with `-'.
     */
    while ((option = getopt (argc, argv, "qc:f:u")) != -1) {
        switch (option) {
            case 'q':
                if (tclParmsPtr->quickStartup)
                    goto usageError;
                tclParmsPtr->quickStartup = TRUE;
                break;
            case 'c':
                tclParmsPtr->execCommand = TRUE;
                tclParmsPtr->execStr = optarg;
                goto exitParse;
            case 'f':
                tclParmsPtr->execFile = TRUE;
                tclParmsPtr->execStr = optarg;
                goto exitParse;
            case 'u':
            default:
                goto usageError;
        }
    }
    exitParse:
  
    /*
     * If neither `-c' nor `-f' were specified and at least one parameter
     * is supplied, then if is the file to execute.  The rest of the arguments
     * are passed to the script.  Check for '--' as the last option, this also
     * is a terminator for the file to execute.
     */
    if ((!tclParmsPtr->execCommand) && (!tclParmsPtr->execFile) &&
        (optind != argc) && !STREQU (argv [optind-1], "--")) {
        tclParmsPtr->execFile = TRUE;
        tclParmsPtr->execStr = argv [optind];
        optind++;
    }

    tclParmsPtr->tclArgv = &argv [optind];
    tclParmsPtr->tclArgc = argc - optind;
    return;

usageError:
    fprintf (stderr, "usage: %s %s\n", argv [0],
             "[-qu] [[-f] script]|[-c command] [args]");
    exit (1);
}

/*
 *----------------------------------------------------------------------
 * FindDefaultFile --
 *
 *   Find the Tcl default file.  If is looked for in the following order:
 *       o A environment variable named `TCLDEFAULT'.
 *       o A file named `TCLDEFAULT'.
 *       o The specified defaultFile (which normally has an version number
 *         appended.
 *   A tcl variable `TCLDEFAULT', will contain the path of the default file
 *   to use after this procedure is executed, or a null string if it is not
 *   found.
 * Parameters
 *     o interp (I) - A pointer to the interpreter.
 *     o defaultFile (I) - The file name of the default file to use, it
 *       normally contains a version number.
 * Returns:
 *     TCL_OK if all is ok, TCL_ERROR if a error occured.
 *----------------------------------------------------------------------
 */
static int
FindDefaultFile (interp, defaultFile)
    Tcl_Interp  *interp;
    char        *defaultFile;
{
    char        *defaultFileToUse;
    struct stat  statBuf;

    if ((defaultFileToUse = getenv ("TCLDEFAULT")) == NULL) {
        defaultFileToUse = "TCLDEFAULT";
        if (stat (defaultFileToUse, &statBuf) < 0) {
            defaultFileToUse = defaultFile;
        }
    }
    if (stat (defaultFileToUse, &statBuf) < 0)
        defaultFileToUse = "";
    if (Tcl_SetVar (interp, "TCLDEFAULT", defaultFileToUse,
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        return TCL_ERROR;
    else
        return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * ProcessDefaultFile --
 *
 *   Process the Tcl default file and TclInit files.  The default file
 * is the only file at a fixed path. It is a script file that usaually 
 * defines a variable "TCLINIT", which has the path of the  full
 * initialization file. The default file can also set things such as path
 * variables.  If the TCLINIT variable is set, that file is then evaluated.
 * If usually does the full Tcl initialization.
 *
 * Parameters
 *     o interp  (I) - A pointer to the interpreter.
 *     o defaultFile (I) - The file name of the default file to use, it
 *       normally contains a version number.
 *----------------------------------------------------------------------
 */
static void
ProcessDefaultFile (interp, defaultFile)
    Tcl_Interp  *interp;
    char        *defaultFile;
{
    char *defaultFileToUse, *initFile;

    defaultFileToUse = Tcl_GetVar (interp, "TCLDEFAULT", 1);
    if (*defaultFileToUse == '\0') {
        fflush (stdout);
        fprintf (stderr, "Can't access Tcl default file,\n");
        fprintf (stderr, "  Located in one of the following ways:\n");
        fprintf (stderr, "    Environment variable: `%s',\n", "TCLDEFAULT");
        fprintf (stderr, "    File in current directory: `TCLDEFAULT', or\n");
        fprintf (stderr, "    File `%s'.\n", defaultFile);
        exit (1);
    }
    if (Tcl_EvalFile (interp, defaultFileToUse) != TCL_OK)
        goto errorAbort;
    Tcl_ResetResult (interp);

    initFile = Tcl_GetVar (interp, "TCLINIT", 1);
    if (initFile != NULL) {
        if (Tcl_EvalFile (interp, initFile) != TCL_OK)
            goto errorAbort;
        }
    Tcl_ResetResult (interp);
    return;

errorAbort:
    DumpTclError (interp);
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_Startup --
 *
 *      Initializes the Tcl extended environment.  This function runs the
 *      TclInit.tcl command file and optionally creates an interactive 
 *      command loop. See the user documentation for a complete description
 *      of how this procedure works.
 *
 * Parameters
 *     o interp - A pointer to the interpreter.
 *     o argc, argv - Arguments passed to main.
 *     o defaultFile (I) - The file name of the default file to use, it
 *       normally contains a version number.
 * Returns:
 *   TCL_OK if all is ok, TCL_ERROR if an error occured.
 *----------------------------------------------------------------------
 */
void
Tcl_Startup (interp, argc, argv, defaultFile)
    Tcl_Interp  *interp;
    int          argc;
    CONST char **argv;
    CONST char  *defaultFile;
{
    int         result;
    char       *args, *cmdBuf;
    tclParms_t  tclParms;

    /*
     * Initialize special commands needed by the shell.
     */    
    Tcl_CreateCommand (interp, "sourcepart", Tcl_SourcepartCmd,
                      (ClientData)NULL, (void (*)())NULL);

    /*
     * Process the arguments.
     */
    ParseCmdArgs (argc, (char **) argv, &tclParms);

    /*
     * Set Tcl variables based on the arguments parsed.
     */    
    if (Tcl_SetVar (interp, "programName", tclParms.programName, 
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        goto errorAbort;


    if (Tcl_SetVar (interp, "interactiveSession", 
                    (tclParms.execStr == NULL ? "1" : "0"), 
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        goto errorAbort;

    args = Tcl_Merge (tclParms.tclArgc, tclParms.tclArgv);
    if (Tcl_SetVar (interp, "argv", args,
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        result = TCL_ERROR;
    else
        result = TCL_OK;
    ckfree (args);
    if (result != TCL_OK)
        goto errorAbort;

    if (Tcl_SetVar (interp, "scriptName", 
                    tclParms.execFile ? tclParms.execStr : "", 
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        goto errorAbort;

    /*
     * Locate the default file and save in Tcl var TCLDEFAULT.
     */
    if (FindDefaultFile (interp, (char *) defaultFile) != TCL_OK)
        goto errorAbort;

    /*
     * If not quick startup, process the Tcl default file and execute the 
     * Tcl initialization file.
     */
    if (!tclParms.quickStartup) 
        ProcessDefaultFile (interp, (char*) defaultFile);

    /*
     * If the invoked tcl interactively, give the user an interactive session,
     * otherwise, source the command file or execute the specified command.
     */
    if (tclParms.execFile) {
        result = Tcl_EvalFile (interp, tclParms.execStr);
        if (result != TCL_OK)
            goto errorAbort;
        Tcl_ResetResult (interp);
    } else if (tclParms.execCommand) {
        result = Tcl_Eval (interp, tclParms.execStr, 0, NULL);
        if (result != TCL_OK)
            goto errorAbort;
        Tcl_ResetResult (interp);
    } else
        Tcl_CommandLoop (interp, stdin, stdout, TRUE);
    return;

errorAbort:
    DumpTclError (interp);
}

