/* 
 * list.c --
 *
 *      TCL extend list commands.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"

/*
 * Prototypes of internal functions.
 */
int
CompareKeyListField _ANSI_ARGS_((Tcl_Interp   *interp,
                                 char         *fieldName,
                                 char         *field,
                                 char        **valuePtr,
                                 int          *valueSizePtr));

int
FindKeyListField _ANSI_ARGS_((Tcl_Interp  *interp,
                              char        *fieldName,
                              int          listArgc,
                              char       **listArgv,
                              int         *listIdxPtr));

int
Tcl_GetKeyedListField _ANSI_ARGS_((Tcl_Interp  *interp,
                                   CONST char  *fieldName,
                                   CONST char  *keyedList,
                                   char       **fieldValuePtr));

char *
Tcl_SetKeyedListField _ANSI_ARGS_((Tcl_Interp  *interp,
                                   CONST char  *fieldName,
                                   CONST char  *fieldvalue,
                                   CONST char  *keyedList));

char *
Tcl_DeleteKeyedListField _ANSI_ARGS_((Tcl_Interp  *interp,
                                      CONST char  *fieldName,
                                      CONST char  *keyedList));

/*
 *----------------------------------------------------------------------
 *
 * CompareKeyListField --
 *   Compare a field name to a field (keyword/value pair) to determine if
 * the field names match.
 *
 * Parameters:
 *   o interp (I/O) - Error message will be return in result if there is an
 *     error.
 *   o fieldName (I) - Field name to compare against field.
 *   o field (I) - Field to see if its name matches.
 *   o valuePtr (O) - If the field names match, a pointer to value part is
 *     returned.
 *   o valueSizePtr (O) - If the field names match, the length of the value
 *     part is returned here.
 * Results:
 *    TCL_OK - If the field names match.
 *    TCL_BREAK - If the fields names don't match.
 *    TCL_ERROR -  If the list has an invalid format.
 *----------------------------------------------------------------------
 */
static int
CompareKeyListField (interp, fieldName, field, valuePtr, valueSizePtr)
    Tcl_Interp   *interp;
    char         *fieldName;
    char         *field;
    char        **valuePtr;
    int          *valueSizePtr; 
{
    char *elementPtr, *nextPtr;
    int   fieldNameSize, elementSize;

    if (field [0] == '\0') {
        Tcl_AppendResult (interp, "invalid keyed list format: ",
                          "list contains an empty field entry",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (TclFindElement (interp, field, &elementPtr, &nextPtr, 
                        &elementSize, NULL) != TCL_OK)
        return TCL_ERROR;
    if (elementSize == 0) {
        Tcl_AppendResult (interp, "invalid keyed list format: ",
                          "list contains an empty field name",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (nextPtr[0] == '\0') {
        Tcl_AppendResult (interp, "invalid keyed list format: ",
                          "no value associated with field \"",
                          elementPtr, "\"", (char *) NULL);
        return TCL_ERROR;
    }

    fieldNameSize = strlen (fieldName);
    if (!((elementSize == fieldNameSize) && 
            STRNEQU (elementPtr, fieldName, fieldNameSize)))
        return TCL_BREAK;   /* Names do not match */

    /*
     * Extract the value from the list.
     */
    if (TclFindElement (interp, nextPtr, &elementPtr, &nextPtr, &elementSize, 
                        NULL) != TCL_OK)
        return TCL_ERROR;
    if (nextPtr[0] != '\0') {
        Tcl_AppendResult (interp, "invalid keyed list format: ",
                          "trailing data following value in field: \"",
                          elementPtr, "\"", (char *) NULL);
        return TCL_ERROR;
    }
    *valuePtr = elementPtr;
    *valueSizePtr = elementSize;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FindKeyListField --
 *   Locate a field (key/value pair) in a key list that has been broken
 * into an argv.
 *
 * Parameters:
 *   o interp (I/O) - Error message will be return in result if there is an
 *     error.
 *   o fieldName (I) - The name of the field to find, should have all 
 *     subsequent parts (seperated by `.'), the pointer to the next part will
 *     be returned as part of the parseResult.
 *   o listArgc/listArgv (I) - The keyed list, split into an argv.
 *   o listIdxPtr (O) - The argv index containing the list entry that matches
 *     the field name, or -1 if the key was not found.
 * Results:
 *   Standard Tcl result.
 *----------------------------------------------------------------------
 */
static int
FindKeyListField (interp, fieldName, listArgc, listArgv, listIdxPtr)
    Tcl_Interp  *interp;
    char        *fieldName;
    int          listArgc;
    char       **listArgv;
    int         *listIdxPtr;
{
    int   idx, result, valueSize;
    char *value;

    for (idx = 0; idx < listArgc; idx++) {
        result = CompareKeyListField (interp, fieldName, listArgv [idx],
                                      &value, &valueSize);
        if (result != TCL_BREAK)
            break;  /* Found or error */
    }
    if (result == TCL_ERROR)
        return TCL_ERROR;
    if (idx >= listArgc)
        *listIdxPtr = -1;  /* Not found */
    else
        *listIdxPtr = idx;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetKeyedListField --
 *   Retrieve a field value from a keyed list.  The list is walked rather than
 * converted to a argv for increased performance.
 *
 * Parameters:
 *   o interp (I/O) - Error message will be return in result if there is an
 *     error.
 *   o fieldName (I) - The name of the field to extract.
 *   o keyedList (I) - The list to search for the field.
 *   o fieldValuePtr (O) - If the field is found, a pointer to a dynamicly
 *     allocated string containing the value is returned here.  If NULL is
 *     specified, then only the presence of the field is validated, the
 *     value is not returned.
 * Results:
 *   TCL_OK - If the field was found.
 *   TCL_BREAK - If the field was not found.
 *   TCL_ERROR - If an error occured.
 *----------------------------------------------------------------------
 */
int
Tcl_GetKeyedListField (interp, fieldName, keyedList, fieldValuePtr)
    Tcl_Interp  *interp;
    CONST char  *fieldName;
    CONST char  *keyedList;
    char       **fieldValuePtr;
{
    char *scanPtr;
    char *value;
    int   valueSize, result;

    /*
     * Walk the list looking for a field name that matches.
     */
    scanPtr = (char *) keyedList;
    result = TCL_OK;
    while (*scanPtr != '\0') {
        char *fieldPtr;
        int   fieldSize;
        char  saveChar;

        result = TclFindElement (interp, scanPtr, &fieldPtr, &scanPtr, 
                                 &fieldSize, NULL);
        if (result != TCL_OK)
            break;

        saveChar = fieldPtr [fieldSize];
        fieldPtr [fieldSize] = '\0';

        result = CompareKeyListField (interp, (char *) fieldName, fieldPtr,
                                      &value, &valueSize);
        fieldPtr [fieldSize] = saveChar;
        if (result != TCL_BREAK)
            break;  /* Found or an error */
    }

    if (result != TCL_OK)
        return result;   /* Not found or an error */

    if (fieldValuePtr != NULL) {
        char *fieldValue;

        fieldValue = ckalloc (valueSize + 1);
        strncpy (fieldValue, value, valueSize);
        fieldValue [valueSize] = '\0';
        *fieldValuePtr = fieldValue;
    }
    return TCL_OK;  /* Found! */

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SetKeyedListField --
 *   Set a field value in keyed list.
 *
 * Parameters:
 *   o interp (I/O) - Error message will be return in result if there is an
 *     error.
 *   o fieldName (I) - The name of the field to set.
 *   o fieldValue (I) - The value to set for the field.
 *   o keyedList (I) - The keyed list to set a field value in, may be an
 *     NULL or an empty list to create a new keyed list.
 * Results:
 *   A pointer to a dynamically allocated string, or NULL if an error
 *   occured.
 *----------------------------------------------------------------------
 */
char *
Tcl_SetKeyedListField (interp, fieldName, fieldValue, keyedList)
    Tcl_Interp  *interp;
    CONST char  *fieldName;
    CONST char  *fieldValue;
    CONST char  *keyedList;
{
    char   *newField, *newList;
    int     listArgc, fieldIdx;
    char  **listArgv = NULL;
    char   *newArgv [2];

    if (fieldName == '\0') {
        Tcl_AppendResult (interp, "null key not allowed", (char *) NULL);
        return NULL;
    }

    /*
     * Build a list out of the new key/value pair, we may need it soon.
     */
    newArgv [0] = (char *) fieldName;
    newArgv [1] = (char *) fieldValue;
    newField = Tcl_Merge (2, newArgv);

    if (keyedList == NULL)
        keyedList = "";

    /*
     * Parse the keyed list into an argv and search for the key/value pair.
     */
    if (Tcl_SplitList (interp, (char *) keyedList, &listArgc, 
                       &listArgv) != TCL_OK)
        goto errorExit;

    if (FindKeyListField (interp, (char *) fieldName, listArgc, listArgv, 
                          &fieldIdx) != TCL_OK)
        goto errorExit;

    /*
     * If the field does not current exist in the keyed list, append it,
     * otherwise replace it.
     */
    if (fieldIdx == -1) {
        fieldIdx = listArgc;
        listArgc++;
    }

    listArgv [fieldIdx] = newField;
    newList = Tcl_Merge (listArgc, listArgv);

    ckfree ((char *) newField);
    ckfree ((char *) listArgv);
    return newList;

errorExit:
    ckfree ((char *) newField);
    if (listArgv != NULL)
        ckfree ((char *) listArgv);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_DeleteKeyedListField --
 *   Delete a field value in keyed list.
 *
 * Parameters:
 *   o interp (I/O) - Error message will be return in result if there is an
 *     error.
 *   o fieldName (I) - The name of the field to set.
 *   o fieldValue (I) - The value to set for the field.
 *   o keyedList (I) - The keyed list to set a field value in, may be an
 *     NULL or an empty list to create a new keyed list.
 * Results:
 *   A pointer to a dynamically allocated string, or NULL if an error
 *   occured.
 *----------------------------------------------------------------------
 */
char *
Tcl_DeleteKeyedListField (interp, fieldName, keyedList)
    Tcl_Interp  *interp;
    CONST char  *fieldName;
    CONST char  *keyedList;
{
    char  *newList;
    int    listArgc, fieldIdx, idx;
    char **listArgv;

    if (fieldName == '\0') {
        Tcl_AppendResult (interp, "null key not allowed", (char *) NULL);
        return NULL;
    }

    if (Tcl_SplitList (interp, (char *) keyedList, &listArgc, 
                       &listArgv) != TCL_OK)
        return NULL;

    if (FindKeyListField (interp, (char *) fieldName, listArgc, listArgv, 
                          &fieldIdx) != TCL_OK)
        goto errorExit;

    if (fieldIdx == -1) {
        Tcl_AppendResult (interp, "field name not found: \"",  fieldName, 
                          "\"", (char *) NULL);
        goto errorExit;
    }

    /*
     * Move all entries in the argv following the one being deleted, up one
     * spot.
     */
    for (idx = fieldIdx; idx < listArgc; idx++)
        listArgv [idx] = listArgv [idx + 1];
    
    newList = Tcl_Merge (listArgc - 1, listArgv);

    ckfree ((char *) listArgv);
    return newList;

errorExit:
    ckfree ((char *) listArgv);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_KeyldelCmd --
 *     Implements the TCL keyldel command:
 *         keyldel listvar key
 *
 * Results:
 *    Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_KeyldelCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char  *keyedList, *newList;
    int    listArgc, fieldIdx, idx;
    char **listArgv;
    char  *varPtr;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " listvar key", (char *) NULL);
        return TCL_ERROR;
    }

    keyedList = Tcl_GetVar (interp, argv[1], TCL_LEAVE_ERR_MSG);
    if (keyedList == NULL)
        return TCL_ERROR;

    newList = Tcl_DeleteKeyedListField (interp, argv [2], keyedList);
    if (newList == NULL)
        return TCL_ERROR;

    varPtr = Tcl_SetVar (interp, argv [1], newList, TCL_LEAVE_ERR_MSG);
    ckfree ((char *) newList);

    return (varPtr == NULL) ? TCL_ERROR : TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_KeylgetCmd --
 *     Implements the TCL keylget command:
 *         keylget listvar key [retvar | {}]
 *
 * Results:
 *    Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_KeylgetCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char   *keyedList;
    char   *fieldValue;
    char  **fieldValuePtr;
    int     result;

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " listvar key [retvar | {}]", (char *) NULL);
        return TCL_ERROR;
    }
    if (argv [2] == '\0') {
        Tcl_AppendResult (interp, "null key not allowed", (char *) NULL);
        return TCL_ERROR;
    }
    keyedList = Tcl_GetVar (interp, argv[1], TCL_LEAVE_ERR_MSG);
    if (keyedList == NULL)
        return TCL_ERROR;

    /*
     * Recursively extract the field (or sub-field) value.  First determine
     * if we actually need a value.
     */
    if ((argc == 4) && (argv [3][0] == '\0'))
        fieldValuePtr = NULL;
    else
        fieldValuePtr = &fieldValue;

    result = Tcl_GetKeyedListField (interp, argv [2], keyedList,
                                    fieldValuePtr);
    if (result == TCL_ERROR)
        return TCL_ERROR;

    /*
     * Handle field name not found.
     */
    if (result == TCL_BREAK) {
        if (argc == 3) {
            Tcl_AppendResult (interp, "key \"", argv [2], 
                              "\" not found in keyed list", (char *) NULL);
            return TCL_ERROR;
        } else {
            interp->result = "0";
            return TCL_OK;
        }
    }

    /*
     * Handle field name found and return in the result.
     */
    if (argc == 3) {
        Tcl_SetResult (interp, fieldValue, TCL_DYNAMIC);
        return TCL_OK;
    }

    /*
     * Handle null return variable specified and key was found.
     */
    if (argv [3][0] == '\0') {
        interp->result = "1";
        return TCL_OK;
    }

    /*
     * Handle returning the value to the variable.
     */
    if (Tcl_SetVar (interp, argv [3], fieldValue, TCL_LEAVE_ERR_MSG) == NULL)
        result = TCL_ERROR;
    else
        result = TCL_OK;
    ckfree (fieldValue);
    interp->result = "1";
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_KeylsetCmd --
 *     Implements the TCL keylset command:
 *         keylset listvar key value
 *
 * Results:
 *    Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_KeylsetCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char *keyedList, *newList;
    char *varPtr;

    if (argc != 4) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " listvar key value", (char *) NULL);
        return TCL_ERROR;
    }

    keyedList = Tcl_GetVar (interp, argv[1], 0);

    newList = Tcl_SetKeyedListField (interp, argv [2], argv [3], keyedList);
    if (newList == NULL)
        return TCL_ERROR;
    
    varPtr = Tcl_SetVar (interp, argv [1], newList, TCL_LEAVE_ERR_MSG);
    ckfree ((char *) newList);

    return (varPtr == NULL) ? TCL_ERROR : TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_LvarpopCmd --
 *     Implements the TCL replace command:
 *         lvarpop var [index [string]]
 *
 * Results:
 *      Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_LvarpopCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int        myargc, result;
    char     **myargv;
    char      *varcontents;
    unsigned   listIdx, idx;
    char      *resultList;

    if ((argc < 2) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " var [index [string]]", (char *) NULL);
        return TCL_ERROR;
    }

    varcontents = Tcl_GetVar (interp, argv[1], TCL_LEAVE_ERR_MSG);
    if (varcontents == NULL)
        return TCL_ERROR;

    if (Tcl_SplitList (interp, varcontents, &myargc, &myargv) == TCL_ERROR) {
        result = TCL_ERROR;
        goto exitPoint;
    }
    if (argc == 2) 
        listIdx = 0;
    else
        if (Tcl_GetUnsigned (interp, argv[2], &listIdx) != TCL_OK) {
            result = TCL_ERROR;
            goto exitPoint;
        }

    /*
     * This is dangerous, but this is like the standard Tcl commands.
     */
    if (listIdx >= myargc) {
        result = TCL_OK;
        goto exitPoint;
    }
    Tcl_SetResult (interp, myargv[listIdx], TCL_VOLATILE);

    if (argc == 4)
        myargv [listIdx] = argv[3];
    else {
        myargc--;
        for (idx = listIdx; idx < myargc; idx++)
            myargv [idx] = myargv[idx+1];
    }

    resultList = Tcl_Merge(myargc, myargv);
    if (Tcl_SetVar (interp, argv[1], resultList, TCL_LEAVE_ERR_MSG) == NULL)
        result = TCL_ERROR;
    else
        result = TCL_OK;
    ckfree (resultList);

exitPoint:
    ckfree((char *) myargv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_LemptyCmd --
 *     Implements the strcat TCL command:
 *         lempty list
 *
 * Results:
 *     Standard TCL result.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_LemptyCmd (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    char *scanPtr;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " list",
                          (char *) NULL);
        return TCL_ERROR;
    }

    scanPtr = argv [1];
    while ((*scanPtr != '\0') && (isspace (*scanPtr)))
        scanPtr++;
    sprintf (interp->result, "%d", (*scanPtr == '\0'));
    return TCL_OK;

} /* Tcl_LemptyCmd */
