/* 
 * cmdloop --
 *
 *   Interactive command loop, C and Tcl callable.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"


/*
 * Pointer to eval procedure to use.  This way bring in the history module
 * from a library can be made optional.  This only works because the calling
 * sequence of Tcl_Eval is a superset of Tcl_RecordAndEval.  This defaults
 * to no history, set this variable to Tcl_RecordAndEval to use history.
 */

int (*tclShellCmdEvalProc) () = Tcl_Eval;

/*
 * Prototypes of internal functions.
 */
int
IsSetVarCmd _ANSI_ARGS_((Tcl_Interp *interp,
                         char       *command));

void
OutStr _ANSI_ARGS_((FILE *filePtr,
                    char *string));

void
OutFlush _ANSI_ARGS_((FILE *filePtr));

void
Tcl_PrintResult _ANSI_ARGS_((FILE   *fp,
                             int     returnval,
                             char   *resultText));

void
OutputPrompt _ANSI_ARGS_((Tcl_Interp *interp,
                          FILE       *outFP,
                          int         topLevel));

int
SetPromptVar _ANSI_ARGS_((Tcl_Interp  *interp,
                          char        *hookVarName,
                          char        *newHookValue,
                          char       **oldHookValuePtr));


/*
 *----------------------------------------------------------------------
 *
 * IsSetVarCmd --
 *
 *      Determine if the current command is a `set' command that set
 *      a variable (i.e. two arguments).  This routine should only be
 *      called if the command returned TCL_OK, due to it calling
 *      Tcl_SplitList, which might alter the interpreter in the result
 *      buffer if the command is not a valid list.
 *
 *----------------------------------------------------------------------
 */
static int
IsSetVarCmd (interp, command)
    Tcl_Interp *interp;
    char       *command;
{
    int    cmdArgc;
    char **cmdArgv;
    int    isSet;

    if ((!STRNEQU (command, "set", 3)) || (!isspace (command [3])))
        return FALSE;  /* Quick check */
    if (Tcl_SplitList (interp, command, &cmdArgc, &cmdArgv) != TCL_OK)
       return FALSE;
    isSet = STREQU (cmdArgv[0], "set") && (cmdArgc >= 3);
    ckfree ((char *) cmdArgv);
    return isSet;

}

/*
 *----------------------------------------------------------------------
 *
 * OutStr --
 *
 *   Print a string to the specified file handle and check for errors.
 *
 *----------------------------------------------------------------------
 */
static void
OutStr (filePtr, string)
    FILE *filePtr;
    char *string;
{
    int stat;

    stat = fputs (string, filePtr);
    if (stat == EOF)
        panic ("command loop: error writting to output file: %s\n",
               strerror (errno));
}

/*
 *----------------------------------------------------------------------
 *
 * OutFlush --
 *
 *   Flush a stdio file and check for errors.
 *
 *----------------------------------------------------------------------
 */
static void
OutFlush (filePtr)
    FILE *filePtr;
{
    int stat;

    stat = fflush (filePtr);
    if (stat == EOF)
        panic ("command loop: error flushing output file: %s\n",
               strerror (errno));
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_PrintResult --
 *
 *      Print a Tcl result
 *
 * Results:
 *
 *      Takes an open file pointer, a return value and some result
 *      text.  Prints the result text if the return value is TCL_OK,
 *      prints "Error:" and the result text if it's TCL_ERROR,
 *      else prints "Bad return code:" and the result text.
 *
 *----------------------------------------------------------------------
 */
static void
Tcl_PrintResult (fp, returnval, resultText)
    FILE   *fp;
    int     returnval;
    char   *resultText;
{

    if (returnval == TCL_OK) {
        if (resultText [0] != '\0') {
            OutStr (fp, resultText);
            OutStr (fp, "\n");
        }
    } else {
        OutFlush (fp);
        OutStr (stderr, (returnval == TCL_ERROR) ? "Error" : 
                                                   "Bad return code");
        OutStr (stderr, ": ");
        OutStr (stderr, resultText);
        OutStr (stderr, "\n");
    }
}

/*
 *----------------------------------------------------------------------
 *
 * OutputPrompt --
 *     Outputs a prompt by executing either the command string in
 *     TCLENV(topLevelPromptHook) or TCLENV(downLevelPromptHook).
 *
 *----------------------------------------------------------------------
 */
static void
OutputPrompt (interp, outFP, topLevel)
    Tcl_Interp *interp;
    FILE       *outFP;
    int         topLevel;
{
    char *hookName;
    char *promptHook;
    int   result;
    int   promptDone = FALSE;

    hookName = topLevel ? "topLevelPromptHook"
                        : "downLevelPromptHook";
    if (((promptHook = Tcl_GetVar2 (interp, "TCLENV", hookName, 1)) != 
          NULL) && (*promptHook != '\0')) {

        result = Tcl_Eval(interp, promptHook, 0, (char **)NULL);

        if (!((result == TCL_OK) || (result == TCL_RETURN))) {
            OutStr (stderr, "Error in prompt hook: ");
            OutStr (stderr, interp->result);
            OutStr (stderr, "\n");
            Tcl_PrintResult (outFP, result, interp->result);
        } else {
            OutStr (outFP, interp->result);
            promptDone = TRUE;
        }
    } 
    if (!promptDone) {
        if (topLevel)
            OutStr (outFP, "%");
        else
            OutStr (outFP, ">");
    }
    OutFlush (outFP);

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_CommandLoop --
 *
 *      Run a Tcl command loop
 *
 * Results:
 *
 *      Takes an interpreter, in and out file handles and an
 *      interactive flag and reads and executes everything
 *      it reads from input.
 *
 *----------------------------------------------------------------------
 */
void
Tcl_CommandLoop (interp, in, out, interactive)
    Tcl_Interp *interp;
    FILE       *in;
    FILE       *out;
    int         interactive;
{
    Tcl_CmdBuf cmdBuf;
    char       inputBuf[256];
    int        topLevel = TRUE;
    int        result;
    char      *cmd;

    cmdBuf = Tcl_CreateCmdBuf();

    while (TRUE) {

        clearerr(in);
        clearerr(out);
        OutputPrompt (interp, out, topLevel);
        errno = 0;
        if (fgets(inputBuf, sizeof(inputBuf), in) == NULL) {
            if (!feof(in) && interactive && (errno == EINTR)) {
                Tcl_ResetSignals ();
                putchar('\n');
                continue;  /* Go get the next command */
            }
            if (ferror (in))
                panic ("command loop: error on input file: %s\n",
                       strerror (errno));
            goto endOfFile;
        }
        cmd = Tcl_AssembleCmd(cmdBuf, inputBuf);

        if (cmd == NULL)
            topLevel = FALSE;
        else {
            result = (*tclShellCmdEvalProc) (interp, cmd, 0, (char **)NULL);
            if ((result != TCL_OK) || 
                   (interactive && !IsSetVarCmd (interp, cmd)))
                Tcl_PrintResult (out, result, interp->result);

            topLevel = TRUE;
        }
    }
endOfFile:
    Tcl_DeleteCmdBuf(cmdBuf);
}

/*
 *----------------------------------------------------------------------
 *
 * SetPromptVar --
 *     Set one of the prompt hook variables, saving a copy of the old
 *     value, if it exists.
 *
 * Parameters:
 *   o hookVarName (I) - The name of the prompt hook, which is an element
 *     of the TCLENV array.  One of topLevelPromptHook or downLevelPromptHook.
 *   o newHookValue (I) - The new value for the prompt hook.
 *   o oldHookValuePtr (O) - If not NULL, then a pointer to a copy of the
 *     old prompt value is returned here.  NULL is returned if there was not
 *     old value.  This is a pointer to a malloc-ed string that must be
 *     freed when no longer needed.
 * Result:
 *   TCL_OK if the hook variable was set ok, TCL_ERROR if an error occured.
 *----------------------------------------------------------------------
 */
static int
SetPromptVar (interp, hookVarName, newHookValue, oldHookValuePtr)
    Tcl_Interp *interp;
    char       *hookVarName;
    char       *newHookValue;
    char      **oldHookValuePtr;
{
    char *hookValue;    
    char *oldHookPtr = NULL;

    if (oldHookValuePtr != NULL) {
        hookValue = Tcl_GetVar2 (interp, "TCLENV", hookVarName, 
                                 TCL_GLOBAL_ONLY);
        if (hookValue != NULL) {
            oldHookPtr = ckalloc (strlen (hookValue) + 1);
            strcpy (oldHookPtr, hookValue);
        }
    }
    if (Tcl_SetVar2 (interp, "TCLENV", hookVarName, newHookValue, 
                     TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL) {
        if (oldHookPtr != NULL)
            ckfree (oldHookPtr);
        return TCL_ERROR;
    }    
    if (oldHookValuePtr != NULL)
        *oldHookValuePtr = oldHookPtr;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_CommandloopCmd --
 *     Implements the TCL commandloop command:
 *       commandloop prompt prompt2
 *
 * Results:
 *     Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_CommandloopCmd(clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char *oldTopLevelHook  = NULL;
    char *oldDownLevelHook = NULL;
    int   result = TCL_ERROR;

    if (argc > 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0],
                          " [prompt] [prompt2]", (char *) NULL);
        return TCL_ERROR;
    }
    if (argc > 1) {
        if (SetPromptVar (interp, "topLevelPromptHook", argv[1],
                          &oldTopLevelHook) != TCL_OK)
            goto exitPoint;
    }
    if (argc > 2) {
        if (SetPromptVar (interp, "downLevelPromptHook", argv[2], 
                          &oldDownLevelHook) != TCL_OK)
            goto exitPoint;
    }

    Tcl_CommandLoop (interp, stdin, stdout, TRUE);

    if (oldTopLevelHook != NULL)
        SetPromptVar (interp, "topLevelPromptHook", oldTopLevelHook, NULL);
    if (oldDownLevelHook != NULL)
        SetPromptVar (interp, "downLevelPromptHook", oldDownLevelHook, NULL);
        
    result = TCL_OK;
exitPoint:
    if (oldTopLevelHook != NULL)
        ckfree (oldTopLevelHook);
    if (oldDownLevelHook != NULL)
        ckfree (oldDownLevelHook);
    return result;
}
