/************************************************************************\
*                                                                        *
*  GS: A Generalized, stochastic petri net Simulator                     *
*      V0.01      March 1989      Andreas Nowatzyk (agn@unh.cs.cmu.edu)  *
*      Carnegie-Mellon Univerity, School of Computer Science             *
*      Schenley Park, Pittsburgh, PA 15213                               *
*                                                                        *
\************************************************************************/

/* Converts GreatSPN's net-files into a more resonable format */

#include <stdio.h>

#define TTY_IMM		0			/* transition types		    */
#define TTY_EXP		1
#define	TTY_DET		2

char	*trns_types[3] = {"imm", "exp", "det"};

/******************** Data structures ********************/

struct place {					/* A place is... */
    char		*name;
    int			tokens;
};

struct place_list {				/* list of places */
    struct place	*pl;
    struct place_list	*next;
};

struct transition {				/* A transition is... */
    char		*name;
    unsigned char	type;
    double		rate;
    int			dep;			/* enabling dependency */

    struct place_list	*inpts;
    struct place_list	*outpts;
    struct place_list	*inhibs;
};

struct parameter {				/* parameters */
    char		*name;
    double		deflt;
};

/******************** Net storage ********************/

struct place		*PLACES;
int			n_PL;

struct transition	*TRANSITIONS;
int			n_TR;

struct parameter	*RATES, *MARKS;
int			n_RT, n_MK;


main(argc, argv)				/* boring stuff */
    int	argc;
    char *argv[];
{
     read_net();
     print_net();
}

read_net ()					/* read a network */
{
    register int    i, j;
    int             n_GR, k, l, m;
    register struct place_list *t;
    char            buf[1024], tmp[1024];

    while (gets(buf))				/* skip preamble */
	if (buf[0] == '|' && buf[1] == 0)
	    break;

    if (!gets(buf) || 5 != sscanf(buf, "%*s%d%d%d%d%d", &n_MK, &n_PL, &n_RT, &n_TR, &n_GR) ||
	n_MK < 0 || n_PL < 1 || n_RT < 0 || n_TR < 1 || n_GR < 0)
	err("Bogus parameters");

						/* allocate storrage */
    if (n_MK) MARKS = (struct parameter *) malloc(n_MK * sizeof(struct parameter));
    if (n_RT) RATES = (struct parameter *) malloc(n_RT * sizeof(struct parameter));
    PLACES = (struct place *) malloc(n_PL * sizeof(struct place));
    TRANSITIONS = (struct transition *) malloc(n_TR * sizeof(struct transition));

    for (i = 0; i < n_MK; i++) {		/* read mark-parameters */
	if (!gets(buf)) err("Premature end of file");
	if (2 != sscanf(buf, "%s%lf", tmp, &(MARKS[i].deflt)))
	    err("Param problem");
	strcpy(MARKS[i].name = (char *) malloc(strlen(tmp) + 1), tmp);
    }

    for (i = 0; i < n_PL; i++) {		/* read places */
	if (!gets(buf)) err("Premature end of file");
	if (2 != sscanf(buf, "%s%d", tmp, &(PLACES[i].tokens)))
	    err("Place def problem");
	strcpy(PLACES[i].name = (char *) malloc(strlen(tmp) + 1), tmp);
    }

    for (i = 0; i < n_RT; i++) {		/* read rate-parameters */
	if (!gets(buf)) err("Premature end of file");
	if (2 != sscanf(buf, "%s%lf", tmp, &(RATES[i].deflt)))
	    err("Param problem");
	strcpy(RATES[i].name = (char *) malloc(strlen(tmp) + 1), tmp);
    }

    for (i = 0; i < n_GR; i++)			/* skip groups */
	if (!gets(buf)) err("Premature end of file");

    for (i = 0; i < n_TR; i++) {/* read transitions */
	if (!gets(buf)) err("Premature end of file");
	if (5 != sscanf(buf, "%s%lf%d%d%d", tmp, &(TRANSITIONS[i].rate), &(TRANSITIONS[i].dep), &k, &l) || l < 1)
	    err("Transition def problem");

	switch (k) {
	case 0:
	    TRANSITIONS[i].type = TTY_EXP;
	    break;
	case 1:
	    TRANSITIONS[i].type = TTY_IMM;
	    break;
	case 127:
	    TRANSITIONS[i].type = TTY_DET;
	    if (TRANSITIONS[i].rate < 1e-10)
		err("0-delay, deterministic transition");
	    TRANSITIONS[i].rate = 1.0 / TRANSITIONS[i].rate;
	    break;
	default:
	    err("Unknown transition type");
	}

	strcpy(TRANSITIONS[i].name = (char *) malloc(strlen(tmp) + 1), tmp);
	TRANSITIONS[i].inpts = 0;
	TRANSITIONS[i].outpts = 0;
	TRANSITIONS[i].inhibs = 0;

	for (j = l; j--;) {			/* read inputs */
	    if (!gets(buf)) err("Premature end of file");
	    if (3 != sscanf(buf, "%d%d%d", &k, &l, &m) || !k || --l < 0 || l >= n_PL || m < 0)
		err("Transition input def problem");
	    if(k < 0) k = -k;

	    while (k--) {
	    	t = (struct place_list *) malloc (sizeof(struct place_list));
		t->pl = &PLACES[l];
		t->next = TRANSITIONS[i].inpts;
		TRANSITIONS[i].inpts = t;
	    }

	    while (m--)				/* skip geo-info */
		if (!gets(buf)) err("Premature end of file");
	}

	if (!gets(buf)) err("Premature end of file");
	if (1 != sscanf (buf, "%d", &k) || k < 1) err("TR output problem");
	for (j = k; j--;) {			/* read outputs */
	    if (!gets(buf)) err("Premature end of file");
	    if (3 != sscanf(buf, "%d%d%d", &k, &l, &m) || !k || --l < 0 || l >= n_PL || m < 0)
		err("Transition output def problem");
	    if(k < 0) k = -k;

	    while (k--) {
	    	t = (struct place_list *) malloc (sizeof(struct place_list));
		t->pl = &PLACES[l];
		t->next = TRANSITIONS[i].outpts;
		TRANSITIONS[i].outpts = t;
	    }

	    while (m--)				/* skip geo-info */
		if (!gets(buf)) err("Premature end of file");
	}

	if (!gets(buf)) err("Premature end of file");
	if (1 != sscanf (buf, "%d", &k) || k < 0) err("TR inhibt problem");
	for (j = k; j--;) {			/* read inhibits */
	    if (!gets(buf)) err("Premature end of file");
	    if (3 != sscanf(buf, "%d%d%d", &k, &l, &m) || !k || --l < 0 || l >= n_PL || m < 0)
		err("Transition inhibit def problem");

	    if (k) {
	    	t = (struct place_list *) malloc (sizeof(struct place_list));
		t->pl = &PLACES[l];
		t->next = TRANSITIONS[i].inhibs;
		TRANSITIONS[i].inhibs = t;
	    }

	    while (m--)				/* skip geo-info */
		if (!gets(buf)) err("Premature end of file");
	}
    }
}

err(s)
   char *s;
{
    fprintf (stderr, "%s\n", s);
    exit(1);
}

print_net ()					/* print the garbage */
{
    register int i, j;
    register struct place_list *t;

    for (i = 0; i < n_MK; i++)
	printf ("PARAM %s %.10e;\n", MARKS[i].name,  MARKS[i].deflt);

    for (i = 0; i < n_RT; i++)
	printf ("PARAM %s %.10e;\n", RATES[i].name,  RATES[i].deflt);

    for (i = 0; i < n_PL; i++) {
	j = PLACES[i].tokens;
	if (j < 0) {
	    j = -1 - j;
	    if (j >= n_MK)
		err ("Marking parameter problem");
	    printf ("PLACE %s #marks= %s;\n", PLACES[i].name, MARKS[j].name);
	} else
	    printf ("PLACE %s #marks= %d;\n", PLACES[i].name, j);
    }

    for (i = 0; i < n_TR; i++) {
	if (TRANSITIONS[i].rate < 0) {
	    j = -0.5 - TRANSITIONS[i].rate;
	    if (j >= n_RT)
		err ("Rate parameter problem");
	    printf ("TRANS %s %s rate= %s, dep= %d : ", TRANSITIONS[i].name,
		    trns_types[TRANSITIONS[i].type], RATES[j].name, TRANSITIONS[i].dep);
	} else
	    printf ("TRANS %s %s rate= %e, dep= %d : ", TRANSITIONS[i].name,
		    trns_types[TRANSITIONS[i].type], TRANSITIONS[i].rate, TRANSITIONS[i].dep);

	for (t = TRANSITIONS[i].inpts; t; t = t->next)
	    printf ("%s ", t->pl->name);

	for (t = TRANSITIONS[i].inhibs; t; t = t->next)
	    printf ("!%s ", t->pl->name);

	printf ("-->");

	for (t = TRANSITIONS[i].outpts; t; t = t->next)
	    printf (" %s", t->pl->name);

	printf(";\n");
    }
    printf("#end\n");
}
