
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	startup.c
 *
 *	Exec more pvmds.  It's good for you.
 *
$Log$
 */

#ifdef IMA_TITN
#include <bsd/sys/types.h>
#else
#include <sys/types.h>
#endif
#include <sys/time.h>
#ifdef IMA_RS6K
#include <sys/select.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#ifdef	SYSVSTR
#include <string.h>
#define	CINDEX(s,c)	strchr(s,c)
#else
#include <strings.h>
#define	CINDEX(s,c)	index(s,c)
#endif
#include <netdb.h>
#include <signal.h>

#include "global.h"
#include "fromlib.h"
#include "ddpro.h"
#include "tdpro.h"
#include "protoglarp.h"
#include "myalloc.h"
#include "mesg.h"
#include "host.h"
#include "listmac.h"

#ifndef	RSHCOMMAND
#define	RSHCOMMAND	"/usr/ucb/rsh"
#endif

#ifndef	PVMDFILE
#define	PVMDFILE	"pvm3/lib/pvmd"
#endif

#ifndef	DEFBINDIR
#define	DEFBINDIR	"pvm3/bin/%"
#endif

#ifndef	max
#define	max(a,b)	((a)>(b)?(a):(b))
#endif

#ifndef	min
#define	min(a,b)	((a)<(b)?(a):(b))
#endif


/***************
 **  Globals  **
 **           **
 ***************/

extern void bailout();
extern char *inadport_hex();

extern int debugmask;				/* from pvmd.c */
extern char **epaths;				/* from pvmd.c */
extern struct htab *filehosts;		/* from pvmd.c */
extern struct htab *hosts;			/* from pvmd.c */
extern char *loclsnam;				/* from pvmd.c */
extern int loclsock;				/* from pvmd.c */
extern int log_fd;					/* from logging.c */
extern int log_how;					/* from logging.c */
extern char *myarchname;			/* from pvmd.c */
extern int mytid;					/* from pvmd.c */
extern int myunixpid;				/* from pvmd.c */
extern int netsock;					/* from pvmd.c */
extern struct htab *oldhosts;		/* from pvmd.c */
extern int ourudpmtu;				/* from pvmd.c */
extern int ppnetsock;				/* from pvmd.c */
extern int pprwid;					/* from pvmd.c */
extern int runstate;				/* from pvmd.c */
extern int tidhmask;				/* from pvmd.c */
extern int useruid;					/* from pvmd.c */
extern char *username;				/* from pvmd.c */


/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";
static char etext[1024];		/* scratch for error log */


/*	pathsep()
*
*	Break path string into one or more strings on a delimiter character.
*	Also do macro substitution on another character.
*	Return null-terminated array of strings, in new malloc'd space.
*/

char **
pathsep(s, bc, ms, mc)
	char *s;	/* the string to break up */
	char bc;	/* the char to break on */
	char *ms;	/* string to replace macro char */
	char mc;	/* macro char */
{
	char **els;
	int nel = 2;			/* length of els */
	char *p, *q, *r;
	char buf[1024];			/* XXX danger, static limit */

	for (p = s; p = CINDEX(p, bc); p++)
		nel++;
	els = TALLOC(nel, char*, "psep");

	nel = 0;
	for (p = s; p; p = q) {
		if (q = CINDEX(p, bc))
			*q++ = 0;
		for (r = buf; *p; p++) {
			if (*p == mc) {
				strcpy(r, ms);
				r += strlen(r);
			} else
				*r++ = *p;
		}
		*r = 0;
		els[nel++] = STRALLOC(buf);
	}
	els[nel] = 0;
	return els;
}


/*	master_config()
*
*	Master pvmd.  Config a host table with length 1.
*/

master_config(hn, hf)
	char *hn;			/* hostname or null */
	char *hf;			/* hostfile name or null */
{
	struct hostent *he;
	struct hostd *hp;
	char hna[128];
	int hh2;
	struct hostd *hp2;

	if (hf && *hf)
		filehosts = readhostfile(hf);

	hosts = ht_new(1);
	hosts->ht_serial = 1;
	hosts->ht_master = 1;
	hosts->ht_cons = 1;
	hosts->ht_local = 1;

	if (!hn || !*hn) {
		if (gethostname(hna, sizeof(hna)-1) == -1) {
			log_error("main() can't gethostname()\n");
			bailout(0);
		}
		hn = hna;
	}

	if (!(he = gethostbyname(hn))) {
		sprintf(etext, "main() %s: can't gethostbyname\n", hn);
		log_error(etext);
		bailout(0);
	}

	hp = hd_new(1);
	hp->hd_name = STRALLOC(hn);
	hp->hd_arch = STRALLOC(myarchname);
	hp->hd_mtu = UDPMAXLEN;
	bcopy(he->h_addr_list[0], (char*)&hp->hd_sad.sin_addr,
		sizeof(struct in_addr));
	ht_insert(hosts, hp);
	hd_unref(hp);

	hp = hd_new(0);
	hp->hd_name = STRALLOC("pvmd'");
	hp->hd_arch = STRALLOC(myarchname);
	hp->hd_mtu = UDPMAXLEN;
	bcopy(he->h_addr_list[0], (char*)&hp->hd_sad.sin_addr,
		sizeof(struct in_addr));
	ht_insert(hosts, hp);
	hd_unref(hp);

	if (mksocs())
		bailout(0);

	/* get epath from host file if available */

	hp = hosts->ht_hosts[1];
	hh2 = 0;
	if (filehosts) {
		for (hh2 = filehosts->ht_last; hh2 > 0; hh2--)
			if ((hp2 = filehosts->ht_hosts[hh2])
			&& !strcmp(hp2->hd_name, hp->hd_name))
				break;
	}
	if (hh2 > 0 && hp2->hd_epath)
		epaths = pathsep(hp2->hd_epath, ':', myarchname, '%');
	else
		epaths = pathsep(DEFBINDIR, ':', myarchname, '%');

	runstate = PVMDNORMAL;
}


/*	slave_config()
*
*	Slave pvmd being started by master.  Trade minimal config info
*	so we can send packets back and forth.
*/

slave_config()
{
	int lh;			/* local host index */
	int mh;			/* master host index */
	char buf[256];
	struct hostd *hp;
	char *av[10];
	int ac;
	int i;

	printf("ddpro<%d>\n", DDPROTOCOL);
	fflush(stdout);

	ac = sizeof(av)/sizeof(av[0]);
	if (!fgets(buf, sizeof(buf)-1, stdin)
	|| crunchzap(buf, &ac, av)
	|| ac != 6) {
		log_error("slave_config() expected params\n");
		bailout(0);
	}

	lh = atoi(av[1]);
	mh = atoi(av[3]);
	hosts = ht_new(1);
	hosts->ht_serial = 1;
	hosts->ht_master = mh;
	hosts->ht_cons = mh;
	hosts->ht_local = lh;

	hp = hd_new(mh);
	hp->hd_name = STRALLOC("?");
	hex_inadport(av[4], &hp->hd_sad);
	hp->hd_mtu = atoi(av[5]);
	ht_insert(hosts, hp);
	hd_unref(hp);

	hp = hd_new(0);
	hp->hd_name = STRALLOC("pvmd'");
	hp->hd_arch = STRALLOC(myarchname);
	hp->hd_mtu = UDPMAXLEN;
	hex_inadport(av[2], &hp->hd_sad);
	ht_insert(hosts, hp);
	hd_unref(hp);

	hp = hd_new(lh);
	hp->hd_name = STRALLOC(av[0]);
	hp->hd_arch = STRALLOC(myarchname);
	hp->hd_mtu = UDPMAXLEN;
	hex_inadport(av[2], &hp->hd_sad);
	ht_insert(hosts, hp);
	hd_unref(hp);

	if (mksocs())
		bailout(0);

	printf("arch<%s> ip<%s> mtu<%d>\n",
		myarchname,
		inadport_hex(&hp->hd_sad),
		ourudpmtu);
	fflush(stdout);

	ac = sizeof(av)/sizeof(av[0]);
	if (!fgets(buf, sizeof(buf)-1, stdin)
	|| crunchzap(buf, &ac, av)
	|| ac != 1) {
		log_error("slave_config() expected params2\n");
		bailout(0);
	}
	epaths = pathsep(av[0], ':', myarchname, '%');

	fputs("ok\n", stdout);
	fflush(stdout);
	if (i = fork()) {
		if (i == -1)
			log_perror("slave_config() fork");
		exit(0);
	}

	for (i = getdtablesize(); --i >= 0; )
/* XXX don't like this - hard to maintain */
		if (i != netsock && i != loclsock && i != log_fd)
			(void)close(i);

	log_how &= ~1;

	if ((myunixpid = getpid()) == -1) {
		log_error("main() can't getpid()\n");
		bailout(0);
	}

	runstate = PVMDSTARTUP;
}


/*	start_slaves()
*
*	This is the main function for pvmd'.  It attempts to start slave
*	pvmds on a list of hosts.
*/

start_slaves(htp)
	struct htab *htp;		/* table of new slaves */
{
	struct mesg *mp;		/* to inform parent pvmd when finished */
	struct hostent *he;
	struct hostd *hp, *hp2;
	int hh, hh2;

	/*
	* for each host in table without errors, get ipaddr
	*/

	for (hh = 1; hh <= htp->ht_last; hh++) {
		hp = htp->ht_hosts[hh];
		if (!hp || hp->hd_err)
			continue;

	/*
	* first check table from host file if available
	*/

		hh2 = 0;
		if (filehosts) {
			for (hh2 = filehosts->ht_last; hh2 > 0; hh2--)
				if ((hp2 = filehosts->ht_hosts[hh2])
				&& !strcmp(hp2->hd_name, hp->hd_name))
					break;
		}
		if (hh2 > 0) {
			hp->hd_flag |= hp2->hd_flag;
			if (hp2->hd_login)
				hp->hd_login = STRALLOC(hp2->hd_login);
			if (hp2->hd_dpath)
				hp->hd_dpath = STRALLOC(hp2->hd_dpath);
			if (hp2->hd_epath)
				hp->hd_epath = STRALLOC(hp2->hd_epath);
			bcopy((char*)&hp2->hd_sad.sin_addr, (char*)&hp->hd_sad.sin_addr,
					sizeof(struct in_addr));

		} else {
			if (he = gethostbyname(hp->hd_name)) {
				bcopy(he->h_addr_list[0], (char*)&hp->hd_sad.sin_addr,
						sizeof(struct in_addr));

			} else {
				if (debugmask & PDMSTARTUP) {
					sprintf(etext, "start_slaves() can't gethostbyname: %s\n",
							hp->hd_name);
					log_error(etext);
				}
				hp->hd_err = PvmNoHost;
			}
		}

	/*
	* check that it's not already configured
	*/

		if (!hp->hd_err) {
			for (hh2 = oldhosts->ht_last; hh2 > 0; hh2--) {
				if ((hp2 = oldhosts->ht_hosts[hh2])
				&& (hp2->hd_sad.sin_addr.s_addr == hp->hd_sad.sin_addr.s_addr))
					hp->hd_err = PvmDupHost;
			}
		}
	}

	/*
	* check that new hosts aren't duplicated
	*/

	for (hh = htp->ht_last; hh > 0; hh--) {
		if (!(hp = htp->ht_hosts[hh]) || hp->hd_err)
			continue;
		for (hh2 = hh; --hh2 > 0; )
			if ((hp2 = htp->ht_hosts[hh2])
			&& !hp2->hd_err
			&& (hp2->hd_sad.sin_addr.s_addr == hp->hd_sad.sin_addr.s_addr))
				hp->hd_err = PvmDupHost;
	}

	/*
	* try to start each host that doesn't have an error
	*/


	if (debugmask & PDMSTARTUP) {
		log_error("start_slaves() using host table:\n");
		ht_dump(htp);
	}
	for (hh = 1; hh <= htp->ht_last; hh++) {
		hp = htp->ht_hosts[hh];
		if (!hp || hp->hd_err)
			continue;
		slave_exec(hp);
	}

	if (debugmask & PDMSTARTUP) {
		log_error("start_slaves() result host table:\n");
		ht_dump(htp);
	}

	/*
	* send host table with addresses and errors back to parent
	*/

	mp = mesg_new(0);
	mp->m_dst = hosts->ht_hosts[hosts->ht_master]->hd_hostpart | TIDPVMD;
	mp->m_cod = DM_STARTACK;
	mp->m_wid = pprwid;
	pkint(mp, htp->ht_cnt);
	for (hh = 1; hh <= htp->ht_last; hh++) {
		if (hp = htp->ht_hosts[hh]) {
			pkint(mp, hh);
			pkint(mp, hp->hd_err);
			if (!hp->hd_err) {
				pkstr(mp, hp->hd_arch);
				pkstr(mp, inadport_hex(&hp->hd_sad));
				pkint(mp, hp->hd_mtu);
			}
		}
	}
	if (debugmask & PDMSTARTUP)
		log_error("start_slaves() pvmd' sending back host table\n");
	sendmessage(mp);

	work();
	return 0;	/* not reached */
}


/*	eread()
*
*	Read an fd until \n is reached.  Also pass input from a second fd
*	to stderr.
*/

int
eread(rfd, buf, len, efd, hname)
	int rfd;			/* fd to read from */
	char *buf;			/* buffer for data */
	int len;			/* buffer length */
	int efd;			/* alt fd to pass to stderr or -1 if none */
	char *hname;		/* hostname we're talking to */
{
	int got = 0;			/* length written into buf */
	fd_set rfds, the_fds;
	struct timeval tout;
	char lbuf[256];			/* for std err */
	int n;
	int i;
	int mfd = max(rfd, efd) + 1;

	FD_ZERO(&the_fds);
	FD_SET(rfd, &the_fds);
	if (efd >= 0)
		FD_SET(efd, &the_fds);
	tout.tv_sec = 30;	/* max 30 secs to get response */
	tout.tv_usec = 0;

	while (1) {
		rfds = the_fds;
		if ((n = select(mfd, &rfds, (fd_set*)0, (fd_set*)0, &tout)) == -1) {
			log_perror("eread() select");
			return -1;
		}
		if (!n) {
	/*
	* select timeout - no response from slave pvmd within 30 seconds
	*/
			sprintf(etext, "eread() no response from pvmd@%s in 30 seconds\n",
					hname);
			log_error(etext);
			return 0;
		}

		if (FD_ISSET(rfd, &rfds)) {
			if ((n = read(rfd, buf, len)) == -1) {
				log_perror("eread() read");
				return 0;
			}
			if (!n)
				return got;
			got += n;
			for (i = 0; i < n; i++)
				if (buf[i] == '\n')
					return got;
			buf += n;
			len -= n;
		}

		if (efd >= 0 && FD_ISSET(efd, &rfds)) {
	/*
	* response on standard error.  log prefixed by remote's host name.
	*/
			if ((n = read(efd, lbuf, sizeof(lbuf)-1)) > 0) {
				char *p = lbuf, *q, c;

				lbuf[n] = 0;
				sprintf(etext, "eread() pvmd@%s: ", hname);
				q = etext + strlen(etext);
				while (c = *p++ & 0x7f) {
					if (isprint(c))
						*q++ = c;

					else {
						*q++ = '^';
						*q++ = (c + '@') & 0x7f;
					}
				}
				*q++ = '\n';
				*q = 0;
				log_error(etext);
			}
		}
	}
}


/*	slave_exec()
*
*	Exec a new slave pvmd and do first-stage configuration.
*	Returns with hp->hd_err set if error occurred, else
*	other fields of hp filled in with config info.
*/

slave_exec(hp)
	struct hostd *hp;
{
	int wfd = -1;			/* slave fd[0] */
	int rfd = -1;			/* slave fd[1] */
	int efd = -1;			/* slave fd[2] */
	char *hn = hp->hd_name;
	char *av[16];			/* for rsh args and reply parsing */
	int ac;
	char buf[256];
	int n;
	int ver;
	int byrsh = !(hp->hd_flag & HF_ASKPW);
	char debflg[8];
	int pid = -1;			/* pid of rsh */

#ifndef NOREXEC
	struct servent *se;
	static u_short execport = 0;

	if (!execport) {
		if (!(se = getservbyname("exec", "tcp"))) {
			sprintf(etext, "slave_exec() can't getservbyname(): %s\n", "exec");
			bailout(0);
		}
		execport = se->s_port;
		endservent();
	}
#endif

/* XXX gratuitous hack... this is if we can't use rexec or rsh
   XXX some day we'll have real network auth functions */

	if (hp->hd_flag & HF_MANUAL) {
		fprintf(stderr, "*** manual startup ***\n");
		fprintf(stderr, "login to \"%s\" and type \"%s -s\"\n",
				hp->hd_name,
				(hp->hd_dpath ? hp->hd_dpath : PVMDFILE));

	/* get version */

		fprintf(stderr, "response=? ");
		if (!(fgets(buf, sizeof(buf), stdin))) {
			sprintf(etext, "host %s read error\n", hn);
			log_error(etext);
			goto oops;
		}
		ac = sizeof(av)/sizeof(av[0]);
		if (crunchzap(buf, &ac, av) || ac != 1) {
			log_error("slave_exec() expected version\n");
			goto oops;
		}
		ver = atoi(av[0]);
		if (ver != DDPROTOCOL) {
			sprintf(etext, "slave_exec() host %s d-d protocol mismatch (%d/%d)\n",
					hn, ver, DDPROTOCOL);
			log_error(etext);
			hp->hd_err = PvmBadVersion;
			goto done;
		}

	/* send it name, ipaddrs, mtu */

		fprintf(stderr, "ok, type:\n\"name<%s> n<%d> ip<%s>",
				hn, ((hp->hd_hostpart & tidhmask) >> (ffs(tidhmask) - 1)),
				inadport_hex(&hp->hd_sad));
		fprintf(stderr, " n<%d> ip<%s> mtu<%d>\"\n",
				hosts->ht_master,
				inadport_hex(&hosts->ht_hosts[hosts->ht_master]->hd_sad),
				hosts->ht_hosts[hosts->ht_master]->hd_mtu);

	/* get its arch, ipaddr, mtu */

		fprintf(stderr, "response=? ");
		if (!(fgets(buf, sizeof(buf), stdin))) {
			sprintf(etext, "host %s read error\n", hn);
			log_error(etext);
			goto oops;
		}
		if (debugmask & PDMSTARTUP)
			log_error(buf);
		ac = sizeof(av)/sizeof(av[0]);
		if (crunchzap(buf, &ac, av) || ac != 3) {
			log_error("slave_exec() expected params\n");
			goto oops;
		}

		hp->hd_arch = STRALLOC(av[0]);
		hex_inadport(av[1], &hp->hd_sad);
		hp->hd_mtu = atoi(av[2]);

	/* send it epath */

		fprintf(stderr, "ok, type: \"ep<%s>\"\n",
				(hp->hd_epath ? hp->hd_epath : DEFBINDIR));
		fprintf(stderr, "response=? ");
		fgets(buf, sizeof(buf), stdin);
		goto done;
	}

/* XXX end hack */

	if (byrsh) {			/* use rsh to start */
		int wpfd[2], rpfd[2], epfd[2];
		int i;

		if (debugmask & PDMSTARTUP) {
			sprintf(etext, "slave_exec() trying rsh to %s\n", hn);
			log_error(etext);
		}

	/* fork an rsh to startup the slave pvmd */

		if (pipe(wpfd) == -1 || pipe(rpfd) == -1 || pipe(epfd) == -1) {
			log_perror("slave_exec() pipe");
			goto oops;
		}
		if (debugmask & PDMSTARTUP) {
			sprintf(etext, "slave_exec() pipes: %d %d %d %d %d %d\n",
					wpfd[0], wpfd[1], rpfd[0], rpfd[1], epfd[0], epfd[1]);
			log_error(etext);
		}

		if ((pid = fork()) == -1) {
			log_perror("slave_exec() fork");
			bailout(0);
		}
		if (!pid) {
			(void)dup2(wpfd[0], 0);
			(void)dup2(rpfd[1], 1);
			(void)dup2(epfd[1], 2);
			for (i = getdtablesize(); --i > 2; )
				(void)close(i);
			ac = 0;
			av[ac++] = RSHCOMMAND;
			av[ac++] = hn;
			if (hp->hd_login) {
				av[ac++] = "-l";
				av[ac++] = hp->hd_login;
			}
			av[ac++] = hp->hd_dpath ? hp->hd_dpath : PVMDFILE;
			av[ac++] = "-s";
			(void)sprintf(debflg, "-d%x", debugmask);
			av[ac++] = debflg;
			av[ac++] = 0;
			execvp(av[0], av);
			fputs("slave_exec() execvp failed\n", stderr);
			fflush(stderr);
			_exit(1);
		}
		(void)close(wpfd[0]);
		(void)close(rpfd[1]);
		(void)close(epfd[1]);
		wfd = wpfd[1];
		rfd = rpfd[0];
		efd = epfd[0];

	} else {		/* use rexec to start */

		if (debugmask & PDMSTARTUP) {
			sprintf(etext, "slave_exec() trying rexec to %s\n", hn);
			log_error(etext);
		}

#ifdef NOREXEC
		sprintf(etext, "slconfg() sorry, no rexec()\n");
		log_error(etext);
		goto oops;
#else
		sprintf(buf, "%s -s", hp->hd_dpath ? hp->hd_dpath : PVMDFILE);
		if ((wfd = rfd = rexec(&hn, execport,
				(hp->hd_login ? hp->hd_login : username),
				(char*)0, buf, &efd))
		== -1) {
			sprintf(etext, "slave_exec() rexec failed for host %s\n", hn);
			log_error(etext);
			goto oops;
		}
#endif
	}

	/* get its version */

	if ((n = eread(rfd, buf, sizeof(buf)-1, efd, hn)) == -1) {
		sprintf(etext, "host %s read error\n", hn);
		log_error(etext);
		goto oops;
	}
	buf[n] = 0;
	if (debugmask & PDMSTARTUP)
		log_error(buf);
	ac = sizeof(av)/sizeof(av[0]);
	if (crunchzap(buf, &ac, av) || ac != 1) {
		log_error("slave_exec() expected version\n");
		goto oops;
	}

	ver = atoi(av[0]);
	if (ver != DDPROTOCOL) {
		sprintf(etext, "slave_exec() host %s d-d protocol mismatch (%d/%d)\n",
				hn, ver, DDPROTOCOL);
		log_error(etext);
		hp->hd_err = PvmBadVersion;
		goto done;
	}

	/* send it name, ipaddrs, mtu */

	(void)sprintf(buf, "name<%s> n<%d> ip<%s>",
			hn, ((hp->hd_hostpart & tidhmask) >> (ffs(tidhmask) - 1)),
			inadport_hex(&hp->hd_sad));
	(void)sprintf(buf + strlen(buf), " n<%d> ip<%s> mtu<%d>\n",
			hosts->ht_master,
			inadport_hex(&hosts->ht_hosts[hosts->ht_master]->hd_sad),
			hosts->ht_hosts[hosts->ht_master]->hd_mtu);
	if (debugmask & PDMSTARTUP)
		log_error(buf);
	(void)write(wfd, buf, strlen(buf));

	/* get its arch, ipaddr, mtu */

	if ((n = eread(rfd, buf, sizeof(buf)-1, efd, hn)) == -1) {
		sprintf(etext, "host %s read error\n", hn);
		log_error(etext);
		goto oops;
	}
	buf[n] = 0;
	if (debugmask & PDMSTARTUP)
		log_error(buf);
	ac = sizeof(av)/sizeof(av[0]);
	if (crunchzap(buf, &ac, av) || ac != 3) {
		log_error("slave_exec() expected params\n");
		goto oops;
	}

	hp->hd_arch = STRALLOC(av[0]);
	hex_inadport(av[1], &hp->hd_sad);
	hp->hd_mtu = atoi(av[2]);

	/* send it epath */

	sprintf(buf, "ep<%s>\n", (hp->hd_epath ? hp->hd_epath : DEFBINDIR));
	if (debugmask & PDMSTARTUP)
		log_error(buf);
	(void)write(wfd, buf, strlen(buf));

	/* get ok */

	if ((n = eread(rfd, buf, sizeof(buf)-1, efd, hn)) == -1) {
		sprintf(etext, "host %s read error\n", hn);
		log_error(etext);
		goto oops;
	}
	goto done;

oops:
	if (pid != -1)
		(void)kill(pid, SIGTERM);
	hp->hd_err = PvmCantStart;
done:
	close(wfd);
	close(rfd);
	close(efd);
	wait(0);
	return;
}


/*	crunchzap()
*
*	Parse a string into words delimited by <> pairs.
*	Max number of words is original value of *acp.
*
*	Trashes out the original string.
*	Returns 0 with av[0]..av[*acp - 1] pointing to the words.
*	Returns 1 if too many words.
*/

crunchzap(s, acp, av)
	char *s;		/* the string to parse */
	int *acp;		/* max words in, ac out */
	char **av;		/* pointers to args */
{
	register int ac;
	register char *p = s;
	register n = *acp;

	/* separate out words of command */

	ac = 0;
	while (*p) {
		while (*p && *p++ != '<');
		if (*p) {
			if (ac >= n) {
	/* command too long */
				*acp = ac;
				return 1;
			}
			av[ac++] = p;
			while (*p && *p != '>') p++;
			if (*p) *p++ = 0;
		}
	}
	*acp = ac;
	return 0;
}


/*	mksocs()
*
*	Make UDP sockets netsock and ppnetsock.  Make TCP master socket
*	loclsock.
*
*	Returns 0 if ok else 1.
*/

int
mksocs()
{
	struct hostd *hp = hosts->ht_hosts[hosts->ht_local];
	struct hostd *hp0 = hosts->ht_hosts[0];
	struct sockaddr_in sin;
	char buf[128];
	int d;
#ifndef NOSOCKOPT
	int bsz;
#endif
	unsigned short portno;
	char *p;

	/* make pvmd-pvmd socket */

	if ((netsock = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
		log_perror("mksocs() socket netsock");
		return 1;
	}

	for (portno = IPPORT_USERRESERVED; portno < LASTPORT; portno++) {
		bcopy((char*)&hp->hd_sad, (char*)&sin, sizeof(sin));
		sin.sin_port = htons(portno);
		if (!bind(netsock, (struct sockaddr*)&sin, sizeof(sin)))
			break;
	}
	if (portno >= LASTPORT) {
		log_perror("mksocs() bind netsock");
		return 1;
	}
	hp->hd_sad.sin_port = htons(portno);

	/* make pvmd-pvmd' socket */

	if ((ppnetsock = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
		log_perror("mksocs() socket ppnetsock");
		return 1;
	}

	while (++portno < LASTPORT) {
		bcopy((char*)&hp0->hd_sad, (char*)&sin, sizeof(sin));
		sin.sin_port = htons(portno);
		if (!bind(ppnetsock, (struct sockaddr*)&sin, sizeof(sin)))
			break;
	}
	if (portno >= LASTPORT) {
		log_perror("mksocs() bind ppnetsock");
		return 1;
	}
	hp0->hd_sad.sin_port = htons(portno);

	/* make pvmd-local task socket */

	if ((loclsock = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		log_perror("mksocs() socket loclsock");
		return 1;
	}
	while (++portno < LASTPORT) {
		sin.sin_family = AF_INET;
		sin.sin_port = htons(portno);
		sin.sin_addr.s_addr = htonl(0x7f000001);
		if (!bind(loclsock, (struct sockaddr*)&sin, sizeof(sin)))
			break;
	}
	if (portno >= LASTPORT) {
		log_perror("mksocs() bind loclsock");
		return 1;
	}
	if (listen(loclsock, SOMAXCONN) == -1) {
		log_perror("mksocs() listen loclsock");
		return 1;
	}
	(void)sprintf(buf, TDSOCKNAME, useruid);
	if ((d = open(buf, O_RDONLY, 0)) != -1) {
		(void)sprintf(etext, "mksocs() %s: pvmd already running?\n", buf);
		log_error(etext);
		return 1;
	}
	(void)close(d);
	(void)unlink(buf);
	if ((d = open(buf, O_CREAT|O_WRONLY|O_TRUNC, 0600)) == -1) {
		log_perror(buf);
		return 1;
	}
	loclsnam = STRALLOC(buf);
	p = inadport_hex(&sin);
	(void)write(d, p, strlen(p));
	(void)close(d);

#ifndef NOSOCKOPT
	bsz = ourudpmtu * 2;
	if (setsockopt(netsock, SOL_SOCKET, SO_SNDBUF,
			(char*)&bsz, sizeof(bsz)) == -1
	|| setsockopt(netsock, SOL_SOCKET, SO_RCVBUF,
			(char*)&bsz, sizeof(bsz)) == -1
	|| setsockopt(ppnetsock, SOL_SOCKET, SO_SNDBUF,
			(char*)&bsz, sizeof(bsz)) == -1
	|| setsockopt(ppnetsock, SOL_SOCKET, SO_RCVBUF,
			(char*)&bsz, sizeof(bsz)) == -1) {
		log_perror("mksocs() setsockopt");
		return 1;
	}
#endif /*NOSOCKOPT*/
	return 0;
}


