/****************************************************************
**                                                             **
**                        SPIKE_PARMS                          **
**                                                             **
**          By Upinder S. Bhalla, May 1990. Caltech            **
**                                                             **
** This file contains routines for extracting a number of      **
** parameters of a spike that may be useful for comparing      **
** different spike traces. The objective is to get a set of    **
** such parameters which will enable one to evaluate the       **
** validity of a neuronal simulation and to constrain the      **
** parameters used in the simulation.                          **
**                                                             **
****************************************************************/
#include <stdio.h>
#include <math.h>

#define MAXPTS 2000
#define MAXSPIKES 200
#define ENDT 10.0
#define PEAKRANGE 0.005	/* 5 mV should be plenty */
#define DEFAULTMAX -0.01	/* -10 mV as a lower acceptable spike */
#define DEFAULTLO -0.05	/* -50 mV as an upper acceptable AHP */

char *do_spike_parms(argc,argv)
	int argc;
	char	**argv;
{
	float x[MAXPTS],y[MAXPTS];
	float dydx[MAXPTS],d2y[MAXPTS];
	int peak[MAXSPIKES],valley[MAXSPIKES];
	int	peakno = 0,valleyno=0;
	int	npts = 0;
	int	plotno = 0;
	float t,dt;
	float dy,dx;
	float startt = 0,endt = ENDT;
	float peakest = DEFAULTMAX,hyperpolest = DEFAULTLO;
	int	i,k;
	int	RecoveryIndex,KneeIndex,RiseIndex;
	float yp1=0,ypn=0;	/* derivatives at ends of graph,
						** assumed zero since their effect does not
						** extend inwards too far.
						*/

	/* Variables to be evaluated */
	float	PTP=0;
	float	ISI = 0, SpikeWidth = 0, MaxV = 0;
	float	LoT = 0, LoV = 0, LoCurve = 0; 
	float	RiseSlope=0, FallSlope=0;
	float	RecoverySlope=0,RecoveryT=0,RecoveryV=0;
	float	InflectionSlope=0,InflectionT=0,InflectionV=0;
	float	KneeT=0,KneeV=0,KneeCurve=0;
	float	ThreshT=0,ThreshV=0,ThreshCurve=0;


	if (argc < 2) {
		printf("usage : %s filename [-plot plotno] [-start time] [-end time]\n",argv[0]);
		printf("[-peak V] [-hyperpol V]\n",argv[0]);
		printf("The command returns :\n");
		printf("InterSpikeInterval SpikeWidth MaxV LoT LoV LoCurve RiseSlope FallSlope\n");
		printf("RecoverySlope RecoveryT RecoveryV InflectionSlope InflectionT InflectionV\n");
		printf("KneeT KneeV KneeCurve ThreshT ThreshV ThreshCurve\n");
		return;
	}
	for (i = 2 ; i < argc ; i++) {
		if (strcmp(argv[i],"-plot") == 0) {
			i++; plotno = atoi(argv[i]);
		}
		if (strcmp(argv[i],"-start") == 0) {
			i++; startt = atof(argv[i]);
		}
		if (strcmp(argv[i],"-end") == 0) {
			i++; endt = atof(argv[i]);
		}
		if (strcmp(argv[i],"-peak") == 0) {
			i++; peakest = atof(argv[i]);
		}
		if (strcmp(argv[i],"-hyperpol") == 0) {
			i++; hyperpolest = atof(argv[i]);
		}
	}

	if (!read_plot2(argv[1],plotno,&dt,y,MAXPTS,&npts,startt,endt)) {
		printf("Read plot failed\n");
		return("failed");
	}
	for (i = 0, t = startt ; i < npts ; i++, t+=dt)
		x[i] = t;
	endt = t - dt;
/*
** Construct a smoothed dy/dx table
** Construct a d2y/dx2 table
** Run through looking for the Min and Max, identifying peaks and ISI
** Run through identifying the first order parameters and getting their 
**	 values
** Run through for the second order terms.
*/
/* Constructing the d2y/dx2 table using the spline routine */
	spline(x,y,npts,yp1,ypn,d2y);
/* Constructing the dy/dx table using the spline deriv info */
	for (i = 0 ; i < (npts - 1) ; i++) {
		dx = x[i+1]-x[i];
		dy = y[i+1]-y[i];
		dydx[i] = dy/dx - dx * d2y[i]/3.0 - dx * d2y[i+1]/6.0;
	}
	/* Finding the peaks */
	for (i = 1 ; i < (npts - 1) ; i++) {
		if (y[i] > y[i-1] && y[i] > y[i+1] && y[i] > (peakest-PEAKRANGE)){
			peak[peakno] = i;
			MaxV += y[i];
			peakest = y[i];
			peakno++;
		}
		if (y[i] < y[i-1] && y[i] < y[i+1] && y[i] < (hyperpolest+PEAKRANGE)){
			valley[valleyno] = i;
			LoV += y[i];
			hyperpolest = y[i];
			LoT += x[i] - x[peak[valleyno]];
			valleyno++;
		}
	}
	if (peakno == 0 || valleyno == 0) {
		printf("No complete spikes were found : only %d peaks and %d valleys\n",peakno,valleyno);
		return;
	}
	if (peakno != valleyno) {
		printf("An incomplete spike : %d peaks and %d valleys\n",peakno,valleyno);
		return;
	}


	for (i = 0 ; i < peakno ; i++) {
		PTP += y[peak[i]] - y[valley[i]];
		LoCurve += d2y[valley[i]];
	}

	for (k = 0 ; k < peakno ; k++) {
	/* Rise Slope */
		for (i = peak[k] ; i > 0 ; i--) {
			if (dydx[i] > dydx[i - 1] && dydx[i] > dydx[i + 1]) {
				RiseSlope += dydx[i];
				RiseIndex = i;
				SpikeWidth -= x[i];
				break;
			}
		}
	/* Fall Slope */
		for (i = peak[k] ; i < valley[k] ; i++) {
			if (dydx[i] < dydx[i - 1] && dydx[i] < dydx[i + 1]) {
				FallSlope += dydx[i];
				SpikeWidth += x[i];
				break;
			}
		}
	/* Thresh Curve */
		for (i = peak[k] ; i > valley[k-1] ; i--) {
			if (d2y[i] > d2y[i - 1] && d2y[i] > d2y[i + 1]) {
				ThreshCurve += d2y[i];
				ThreshT += x[peak[k]] - x[i];
				ThreshV += y[i];
				break;
			}
		}
	}
	for (k = 0 ; k < (peakno - 1) ; k++) {
	/* Recovery Slope */
		for (i = valley[k] ; i < peak[k + 1] ; i++) {
			if (dydx[i] > dydx[i - 1] && dydx[i] > dydx[i + 1]) {
				RecoveryIndex = i;
				RecoverySlope += dydx[i];
				RecoveryT += x[i] - x[peak[k]];
				RecoveryV += y[i];
				break;
			}
		}
	/* Knee Curve */
	/* If it fails, it will find the peak */
		for (i = valley[k] ; i < peak[k+1] ; i++) {
			if (d2y[i] < d2y[i - 1] && d2y[i] < d2y[i + 1]) {
				KneeIndex = i;
				KneeCurve += d2y[i];
				KneeT += x[i] - x[peak[k]];
				KneeV += y[i];
				break;
			}
		}
	/* Inflection Slope */
	/* If the knee failed, so will this, and all values will be zero */
		for (i = KneeIndex ; i < peak[k+1] ; i++) {
			if (d2y[i] <= 0.0 && d2y[i+1] > 0.0) {
				InflectionSlope += dydx[i];
				InflectionT += x[i] - x[peak[k]];
				InflectionV += y[i];
				break;
			}
		}
	}


	ISI = (x[peak[peakno-1]] - x[peak[0]])/((float)(peakno - 1));
	PTP /= (float)peakno;
	SpikeWidth /= (float)peakno;
	MaxV /= (float)peakno;
	LoT /= (float)peakno;
	LoV /= (float)peakno;
	LoCurve /= (float)peakno;
	RiseSlope /= (float)peakno;
	FallSlope /= (float)peakno;
	ThreshCurve /= (float)peakno;
	ThreshT /= (float)peakno;
	ThreshV /= (float)peakno;
	if (peakno > 1) {
		RecoverySlope /= (float)(peakno - 1);
		RecoveryT /= (float)(peakno - 1);
		RecoveryV /= (float)(peakno - 1);
		KneeCurve /= (float)(peakno - 1);
		KneeT /= (float)(peakno - 1);
		KneeV /= (float)(peakno - 1);
		InflectionSlope /= (float)(peakno - 1);
		InflectionT /= (float)(peakno - 1);
		InflectionV /= (float)(peakno - 1);
	}



	printf("%d peaks, %d valleys, over %d pts and from %f to %f sec\n",
		peakno,valleyno,npts,startt,endt);
	printf("ISI = %f, PTP = %f, SpikeWidth = %f, MaxV = %f\n",
		ISI, PTP, SpikeWidth, MaxV);
	printf("LoT = %f, LoV = %f, LoCurve = %f\n", 
		LoT, LoV, LoCurve); 
	printf("RiseSlope=%f, FallSlope=%f\n",
		RiseSlope, FallSlope);
	printf("RecoverySlope=%f,RecoveryT=%f,RecoveryV=%f\n",
		RecoverySlope,RecoveryT,RecoveryV);
	printf("InflectionSlope=%f,InflectionT=%f,InflectionV=%f\n",
		InflectionSlope,InflectionT,InflectionV);
	printf("KneeT=%f,KneeV=%f,KneeCurve=%f\n",
		KneeT,KneeV,KneeCurve);
	printf("ThreshT=%f,ThreshV=%f,ThreshCurve=%f\n",
		ThreshT,ThreshV,ThreshCurve);

	return("didit");
}

/*
Fix knee & inflection tendency to find a place too near the trough.
Fix thresh to find initial curvature point of spike, rather than
some higher up point.
*/
