#ifndef lint
static char SCCSid[] = "@(#) ./vectors/dvecb.c 07/23/93";
#endif

/*
   This file defines the vector operations in the simplest way possible
   (that still allows for general user contexts):  The first field in the
   user context contains size of the vector, and the actual vector type
   is no more than a pointer to the data 

   These routines are for standard double precision serial vectors.

   This file uses the level-1 BLAS instead of inline code where possible.
 */

#include "tools.h"
#include "system/flog.h"
#include "inline/dot.h"
#include "inline/vmult.h"
#include "inline/setval.h"
#include "inline/copy.h"
#include "inline/axpy.h"

#include <math.h>
#include "vectors/vector.h"              /*I "vectors/vector.h" I*/
#include "vectors/dvec.h"                /*I "vectors/dvec.h" I*/

#if defined(cray)
#define ddot_   SDOT
#define dnrm2_  SNRM2
#define dscal_  SSCAL
#define dcopy_  SCOPY
#define dswap_  SSWAP
#define daxpy_  SAXPY
#elif defined(FORTRANCAPS)
#define ddot_   DDOT
#define dnrm2_  DNRM2
#define dscal_  DSCAL
#define dcopy_  DCOPY
#define dswap_  DSWAP
#define daxpy_  DAXPY
#elif !defined(FORTRANUNDERSCORE)
#define ddot_   ddot
#define dnrm2_  dnrm2	
#define dscal_  dscal
#define dcopy_  dcopy
#define dswap_  dswap
#define daxpy_  daxpy
#endif

/*@
   DVBdot - Dot product for serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to form dot product of

   Output Parameter:
.  z     - result
 @*/
void DVBdot( N, x, y, z )
VEDefaultUsrCntx *N;
double *x, *y, *z;
{
int n = N->n, one = 1;
double ddot_();

*z = ddot_( &n, x, &one, y, &one );
}
/*@
   DVBnorm - Two norm for serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to form 2-norm of

   Output Parameter:
.  z     - result
 @*/
void DVBnorm( N, x, z )
VEDefaultUsrCntx *N;
double           *x, *z;
{
int n = N->n, one = 1;
double dnrm2_();

*z = dnrm2_( &n, x, &one );
}

/*@
   DVBscal - Multiply a serial double precision vector by a scalar.

   Input Parameters:
.  N     - Vector context
.  alpha - scale factor
.  x     - vectors to scale
 @*/
void DVBscal( N, alpha, x )
VEDefaultUsrCntx *N;
double           alpha, *x;
{
int n = N->n, one = 1;

dscal_( &n, &alpha, x, &one );
}

/*@
   DVBcopy - Copy a serial double precision vector.

   Input Parameters:
.  N     - Vector context
.  x     - vector to copy

   Output Parameter:
.  y     - copy of x
 @*/
void DVBcopy( N, x, y )
VEDefaultUsrCntx *N;
double           *x, *y;
{
int n = N->n, one = 1;

dcopy_( &n, x, &one, y, &one );
}

/*@
   DVBswap - Swap two serial double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to swap
 @*/
void DVBswap( N, x, y )
VEDefaultUsrCntx *N;
double           *x, *y;
{
int n = N->n, one = 1;

dswap_( &n, x, &one, y, &one );
}

/*@
   DVBaxpy -  y = alpha*x + y where x and y are serial double 
   precision vectors. 

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  x,y   - vectors
 @*/
void DVBaxpy( N, alpha, x, y )
VEDefaultUsrCntx *N;
double           alpha, *x, *y;
{
int n = N->n, one = 1;

daxpy_( &n, &alpha, x, &one, y, &one );
}

/*@
  DVBSetDefaultFunctions - Set vector operations for serial 
  double precision vectors, using the BLAS where possible.
 
  Input Parameters:
.      vopP  - vector context obtained by VECreate()

  Notes:
  Functions not provided by the level-1 BLAS use the DV routines.
 @*/
void DVBSetDefaultFunctions( vopP )
VECntx *vopP;
{
vopP->dot    = DVBdot;
vopP->norm   = DVBnorm;
vopP->max    = DVmax;
vopP->tdot   = DVBdot;
vopP->scal   = DVBscal;
vopP->copy   = DVBcopy;
vopP->set    = DVset;
vopP->axpy   = DVBaxpy;
vopP->aypx   = DVaypx;
vopP->swap   = DVBswap;
vopP->waxpy  = DVwaxpy;
vopP->pmult  = DVpmult;
vopP->pdiv   = DVpdiv;
vopP->gather = DVgather;
vopP->scatter= DVscatter;
vopP->obtain_vectors = DVobtain_vectors;
vopP->release_vectors= DVrelease_vectors;
vopP->create_vector  = DVCreateVector;
vopP->destroy_vector = DVDestroyVector;
}
