/*
     Two routines to perform Backsolve and Forward solve on a 
   SpMatSplit using an SSOR relaxation factor. 
*/
#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"
/*+
   SpiForwardsolve - For use inside the SSOR preconditioner. Performs 
   a forwardsolve with a relaxation factor omega.

   Input Parameters:
.  BB - Split matrix
.  omega - SSOR parameter
.  b     - right-hand side vector

   Output Parameter:
.  x     - solution vector
 +*/
void SpiForwardsolve( BB, omega, b ,x)
SpMatSplit  *BB;
double      *x, *b, omega;
{
  int      i, n, *nzs;
  register double       *v, sum;
  register int          *vi, nz;
  SpMat    *B = BB->factor;
  SpVec    *row, **rs;
  SpRowMat *R = (SpRowMat *)B->data;
  
  n      = B->rows;
  nzs    = BB->nzl;
  rs     = R->rs;
  for (i=0; i<n; i++) {
    row  = *rs++;
    v    = row->v;
    vi   = row->i;
    nz   = *nzs;
    sum  = *b++;
    SPARSEDENSEMDOT(sum,x,v,vi,nz);
    x[i] = omega*sum/row->v[*nzs++];
  }
}
/*------------------------------------------------------------------*/
/*+
   SpiBacksolve - For use inside the SSOR preconditioner. Performs 
   a backsolve with a relaxation factor omega.

   Input Parameters:
.  BB - Split matrix
.  omega - SSOR parameter
.  b     - right-hand side vector

   Output Parameter:
.  x     - solution vector
 +*/
void SpiBacksolve( BB, omega, b, x)
SpMatSplit  *BB;
double      *x, *b, omega;
{
  int      i, n, *nzs;
  double   *vv;
  register double       *v, sum;
  register int          *vi, nz;
  SpMat    *B = BB->factor;
  SpVec    *row, **rs;
  SpRowMat *R = (SpRowMat *)B->data;

  n      = B->rows;
  nzs    = BB->nzl + n - 1;
  rs     = R->rs + n - 1;
  for (i=n-1; i>=0; i--) {
    nz   = *nzs-- + 1;
    row  = *rs--;
    vv = v = row->v + nz;
    vi   = row->i + nz;
    nz   = row->nz - nz;
    sum  = b[i];
    SPARSEDENSEMDOT(sum,x,v,vi,nz);
    x[i] = omega*sum / *(vv-1);
  }
}

