#ifndef lint
static char SCCSid[] = "@(#) ./sparse/row/findblks.c 07/23/93";
#endif

/*
   This file contains routines to find runs of indices that may
   be compressed into a skyline-like format, consisting of a number
   of ranges of consequetively numbered entries
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include <stdio.h>                /*I <stdio.h> I*/

/* This needs to be larger (should, in fact, be dynamic) */
#define MAX_OFFSETS 200
/*@
     SpFindBlocks - Optimize the sparse format for matrices with blocks
     
     Input Parameters:
.    mat       - matrix to find blocks for
.    threshold - minimum number of rows to be considered a block

     Notes:
     This really looks for the case where n elimination rows can be combined
     into a single block that can be used to eliminate a range of columns at a
     time.
     
     Returns the number of non-zeros that are within blocks.
     All blocks have at least threshold rows.
     
     We should also identify common rows (not just elimination blocks)
 @*/
int SpFindBlocks( mat, threshold )
SpMat *mat;
int   threshold;
{
int      n;
SpRowMat *R;
SpVec    *x;
register int k, *ix, *iy, l, d, i, j, nz, ny;
int      totinblock = 0;
/* To simplify the process of getting the blocks, we keep a buffer of the
   maximum size that we'll put into a single block */
int      offsets[MAX_OFFSETS], offsetcnt;
/* Preallocate storage of the size of the number of rows */
int      *startrow, *ninblock, **offs, *offsavail;
int      nblocks, nn;

R        = GETROWMAT(mat);
n        = mat->rows;
if (threshold < 2) threshold = 2;
/* There can be at most ceil(n/threshold) blocks */
nn       = (n+threshold-1) / threshold;
startrow = (int *)MALLOC( nn * sizeof(int) );        CHKPTRV(startrow,0);
ninblock = (int *)MALLOC( nn * sizeof(int) );        CHKPTRV(ninblock,0);
offs     = (int **)MALLOC( nn * sizeof(int*) );      CHKPTRV(offs,0);
/* We use offsavail to reduce the amount of MALLOCs we do */
offsavail= (int *)MALLOC( n * sizeof(int) );         CHKPTRV(offsavail,0);
nblocks  = 0;

for (i=0; i<n; i++) {
    x   = R->rs[i];
    nz  = x->nz;
    ix  = x->i;
    /* Find the diagonal element.  If this is a Split matrix, we don't
       have to look for it. */
    d = 0;
    while (d < nz) {
	if (ix[d] >= i) break;
	d++;
	}
    
    /* For each row below this, see if the rows have the same structure.
       We do this the hard way for now. */
    offsetcnt = 0;
    offsets[offsetcnt++] = d;
    for (j=i+1; j<n; j++) {
	if (offsetcnt >= MAX_OFFSETS) break;
	iy = R->rs[j]->i;
	ny = R->rs[j]->nz;
	/* Find the element that corresponds to column i */
	k = 0;
	while (k < ny) {
	    if (iy[k] >= ix[d]) break;
	    k++;
	    }
	
	/* Compare remainders of rows */
	if (nz - d != ny - k) break;
	for (l=0; l<(nz-d); l++) 
	    if (iy[k+l] != ix[d+l]) break;
	if (l < (nz-d)) break;
	offsets[offsetcnt++] = k;
	}
    if (j > i + threshold - 1) {
	/* printf( "[%d] to [%d] form a block with %d cols\n", 
	   i, j-1, nz-d ); */
	totinblock        += (nz-d) * (j - i);
	startrow[nblocks] = i;
	ninblock[nblocks] = (j-i);
	offs[nblocks]     = offsavail;
	for (k=0; k<(j-i); k++) {
	    R->rs[i+k]->blki = nblocks;
	    offsavail[k]     = offsets[k];
	    }
	offsavail         += (j-i);
	nblocks++;
	i = j - 1;   /* -1 since the for loop will increment i */
	}
    }
/* printf( "Total number in blocks is %d\n", totinblock ); */
/* Add structure to the matrix */
if (nblocks > 0) { 
    if ((offsavail - offs[0]) > n) printf( "Problem with offset array\n" );
    R->blks = NEW(SpRowBlock);     CHKPTRV(R->blks,0);
    R->blks->nblocks   = nblocks;
    R->blks->startrows = startrow;
    R->blks->blocksize = ninblock;
    R->blks->offsets   = offs;
    }
else {
    FREE( offsavail ); FREE( offs ); FREE( ninblock ); FREE( startrow );
    }
return totinblock;
}

/*@
     SpFindRowBlocks - Optimize the sparse format for matrices with blocks
     
     Input Parameters:
.    mat       - matrix to find blocks for
.    threshold - minimum number of rows to be considered a block

     Returns:
     The number of non-zeros that are within blocks.
     All blocks have at least threshold rows.
     
     Notes:
     (These are entire rows that have identical structure)
     Note that row blocks don't have an offset structure.
 @*/
int SpFindRowBlocks( mat, threshold )
SpMat *mat;
int   threshold;
{
int      n;
SpRowMat *R;
SpVec    *x;
register int *ix, *iy, l, i, j, k, nz, ny;
int      totinblock = 0;
/* To simplify the process of getting the blocks, we keep a buffer of the
   maximum size that we'll put into a single block */
/* Preallocate storage of the size of the number of rows */
int      *startrow, *ninblock;
int      nblocks, nn;

R        = GETROWMAT(mat);
n        = mat->rows;
if (threshold < 2) threshold = 2;
/* There can be at most ceil(n/threshold) blocks */
nn       = (n+threshold-1) / threshold;
startrow = (int *)MALLOC( nn * sizeof(int) );         CHKPTRV(startrow,0);
ninblock = (int *)MALLOC( nn * sizeof(int) );         CHKPTRV(ninblock,0);
nblocks  = 0;

for (i=0; i<n; i++) {
    x   = R->rs[i];
    nz  = x->nz;
    ix  = x->i;
    
    /* For each row below this, see if the rows have the same structure.
       We do this the hard way for now. */
    for (j=i+1; j<n; j++) {
	iy = R->rs[j]->i;
	ny = R->rs[j]->nz;
	/* Compare remainders of rows */
	if (nz != ny) break;
	for (l=0; l<nz; l++) 
	    if (iy[l] != ix[l]) break;
	if (l < nz) break;
	}
    if (j > i+threshold-1) {
	/* printf( "[%d] to [%d] form a block with %d cols\n", 
	   i, j-1, nz-d ); */
	totinblock        += nz * (j - i);
	startrow[nblocks] = i;
	ninblock[nblocks] = (j-i);
	for (k=0; k<(j-i); k++) {
	    R->rs[i+k]->blki = nblocks;
	    }
	nblocks++;
	i = j - 1;   /* -1 since the for loop will increment i */
	}
    }
/* printf( "Total number in blocks is %d\n", totinblock ); */
/* Add structure to the matrix */
if (nblocks > 0) { 
    R->rblks = NEW(SpRowBlock);     CHKPTRV(R->blks,0);
    R->rblks->nblocks   = nblocks;
    R->rblks->startrows = startrow;
    R->rblks->blocksize = ninblock;
    R->rblks->offsets   = 0;
    }
else {
    FREE( ninblock ); FREE( startrow );
    }
return totinblock;
}

/* 
   Recover the row-mat structure 
 */
SpDestroyBlk( R )
SpRowMat *R;
{
if (R->blks) {
    FREE( R->blks->startrows );
    FREE( R->blks->blocksize );
    FREE( R->blks->offsets[0] );
    FREE( R->blks->offsets );
    FREE( R->blks );
    }
if (R->rblks) {
    FREE( R->rblks->startrows );
    FREE( R->rblks->blocksize );
    /* Offsets not used in rblks */
    FREE( R->rblks );
    }
}

/*@
   SpPrintBlks - Print the blocks that the SpFindBlocks routine discovered

   Input Parameters:
.  fp  - pointer to FILE
.  B   - matrix
@*/
void SpPrintBlks( fp, B )
FILE  *fp;
SpMat *B;
{
int          i, sr, nr, j, *off;
SpRowMat     *R = (SpRowMat *)B->data;
/* This is the blocked row structure, if it exists */
SpRowBlock   *blks= R->blks;

fprintf( fp, "Number of blocks is %d\n", blks->nblocks );
for (i=0; i<blks->nblocks; i++) {
    sr = blks->startrows[i];
    nr = blks->blocksize[i];
    fprintf( fp, "Rows %d to %d\n", sr, sr + nr - 1 );
    fprintf( fp, "offsets for each row in this block are\n    " );
    off = blks->offsets[i];
    for (j=0; j<nr; j++) {
	fprintf( fp, " %d", off[j] );
	}
    fprintf( fp, "\n" );
    }
}
