#ifndef lint
static char SCCSid[] = "@(#) ./sparse/pool.c 07/23/93";
#endif

/*
    Allocate Storage Pools (primarily for Sparse Matrix Operations).

    This needs to be generalize so that we don't need to allocate the
    sparse vectors all at once (for example, we may want to allocate
    the column indices ONLY, then allocate the rows as complex or
    in blocks for matrix-matrix and matrix-vector operations.

    Also, we may be allocating something else, like pointers to blocks.
 */

#include "tools.h"
#include "sparse/pool.h"
#define MERGE_FREE

void *SpiChunckMalloc();
void SpiChunckFree();

/*
   Basic routines to manipulate the storage pool.
   Question: why not init the pool with some storage?
 */
void SPiInitPool( pool )
SPPool *pool;
{
SPBlock        *nb;

TRPUSH(SPTRID + TRIDPOOL);
nb = (SPBlock *) MALLOC( sizeof(SPBlock) ); CHKPTR(nb);
nb->next          = 0;
nb->last          = nb;
nb->n             = sizeof(SPBlock);
nb->nalloc        = 0;
pool->free        = 0;
pool->alloc       = nb;
pool->pool.ptr    = 0;
pool->pool.n      = 0;
pool->pool.next   = 0;
pool->chunck_size = BASIC_CHUNCK;
TRPOP;
}

/* Free all of the data in the pool */
void SpiPoolFree( pool )
SPPool *pool;
{
SPBlock *nb, *nnb;

nb = pool->alloc;
while (nb) {
    nnb = nb->next;
    if (nb->n == BASIC_CHUNCK_IN_BYTES) 
	SpChunckRelease( nb );
    else {
	FREE( nb );
	}
    nb  = nnb;
    }
SpChunckFlush();
}


/* 
   This is a special routine that allocates a double and an int storage
   area.  To handle other data types, just change call a different routine.
 */
void SPiPoolAllocNV( pool, n, v, i )
SPPool *pool;
double **v;
int    n, **i;
{
long   m;
double *SPiPoolAlloc( );

m = n * (sizeof(double) + sizeof(int));
*v = SPiPoolAlloc( pool, m );  CHKPTR(*v);
*i = (int *)(*v + n);
}

/* 
  Get n bytes of space and return a pointer to it.  Space is always
  alligned on doubles, so we return that type
 */
double *SPiPoolAlloc( pool, n )
SPPool *pool;
int    n;
{
unsigned int m;
SPBlock      *nb;
double       *p;
unsigned int csize;   /* Actual chunck size to allocate */

TRPUSH(SPTRID + TRIDPOOL + 1);

/* Convert n into sizeof(double) */
n = (n + sizeof(double)-1) / sizeof(double);

/* If there isn't space in the pool, we allocated it */
if (n > pool->pool.n) {
    if (n >= pool->chunck_size)
	csize = n;
    else
	csize = pool->chunck_size;

    /* Could search Free Blocks for a possible match */

    /* Allocate chunck_size locations PLUS space to hold the Block pointer. */
    m              = sizeof(SPBlock) + csize * (sizeof(double) + sizeof(int));
    /* This tries to adjust the size to a multiple of doubles; it isn't
       correct for int's of sizes other than double or double/2 */
    if (sizeof(int) < sizeof(double)) m += sizeof(int);
    if (m == BASIC_CHUNCK_IN_BYTES) nb = SpiChunckMalloc(m); 
    else                   nb = (SPBlock *)MALLOC( m );
    CHKPTRV(nb,0);
    nb->next       = pool->alloc;
    nb->last       = nb + ((m - 1)/sizeof(SPBlock));
    nb->n          = m;
    nb->nalloc     = 0;
    pool->alloc    = nb;
    pool->pool.n   = csize;
    pool->pool.ptr = (double *)(nb + 1);
    }
/* Now, allocate space */
p = pool->pool.ptr;

/* Advance the pool pointer.  */
pool->pool.ptr += n;
pool->pool.n   -= n;  /* WRONG; the macro in sppriv.h wants this to be the 
			 number of elements, not bytes (at least this is
			 conservative) */
pool->alloc->nalloc++;
TRPOP;
return p;
}

/*ARGSUSED*/
/* This is a routine to free storage.  Note that if block_size > 1, 
   the space is allocated modulo block_size.
   n here is the number of ELEMENTS, not bytes.
 */
void SPiPoolFreeNV( pool, n, v, i )
SPPool *pool;
double *v;
int    n, *i;
{
SPBlock *nb, *b, **prev;

nb  = (SPBlock *)v;
/* Merge/sort blocks.  This is a linear search algorithm, and could
   be a source of poor performance if there are many blocks.  This is
   less likely to be a problem for these routines than for a general
   malloc routine, since the intent is to have a separate pool for each
   object (sparse matrix).  If this turns out to be a problem, then
   we can form the blocks in a tree for fast lookup. */
#ifdef MERGE_FREE
b   = pool->alloc;
prev= &pool->alloc;
while (b) {
    if (nb >= b && nb < b->last) {
	b->nalloc--;
	if (b->nalloc == 0) {
	    *prev = b->next;
        if (n > BASIC_CHUNCK) {
	    FREE(b);
	    }
        else SpiChunckFree(b);
	    }
	return;
	}
    prev = &b->next;
    b    = b->next;
    }
#endif
/* If we got here, we could not free the space */
nb->next   = pool->free;
pool->free = nb;
}

/*+
   SPiPoolSize - Return the amount of space currently allocated by the pool 

   Input Parameters:
.  pool - pointer to the storage pool
 +*/
long SPiPoolSize( pool )
SPPool *pool;
{
SPBlock *nb;
long    size = 0;

nb = pool->alloc;
while (nb) {
    size += nb->n;
    nb   = nb->next;
    }
return size;
}



