#ifndef lint
static char SCCSid[] = "@(#) ./sparse/fblock/fbksolve.c 07/23/93";
#endif

/*
    This file contains a simple sparse solve routine.
    These are intended for problems where the matrix is re-ordered
    to improve performance (e.g., numerical stability, reduction of fill)

    These routines currently duplicate exactly the operations of the
    blocksize = 1 case (non-blocked rows).
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"
#include "inline/blas2.h"
#include "sparse/fblock/spfbpriv.h"

#if defined(intelnx)
#undef DMV
#undef DVPMV
#undef DVPAMV
#define DMV(a,b,nr,nc,c)   DMV2aIL(a,b,nr,nc,c)
#define DVPMV(a,b,nr,nc,c) DVPMV2aIL(a,b,nr,nc,c)
#define DVPAMV(a,b,nr,nc,al,c) DVPAMVIL(a,b,nr,nc,al,c)
#else 
#undef DMV
#undef DVPMV
#undef DVPAMV
#define DMV(a,b,nr,nc,c)   DMVIL(a,b,nr,nc,c)
#define DVPMV(a,b,nr,nc,c) DVPMVIL(a,b,nr,nc,c)
#define DVPAMV(a,b,nr,nc,al,c) DVPAMVIL(a,b,nr,nc,al,c)
#endif

void SpFBiSolvePerm(), SpFBiSolveBase(), SpFBiSolvePermBase();

/*@ 
  SpFBSolve - Solve a system of equations

  Input Parameters:
.  BB - Split matrix already processed by Factor
.  b  - Right-hand-side
.  x  - solution
 @*/
void SpFBSolve( BB, b, x )
SpMatSplit *BB;
double     *b, *x;
{
if (BB->factor->map) 
    SpFBiSolvePerm( BB, x, b );
else
    SpFBiSolveBase( BB, x, b );
}

/*
        [(R A C) Cinv] x = R b
 */
void SpFBiSolvePerm( BB, x, b )
SpMatSplit *BB;
double      *x, *b;
{
double *tmp;
tmp    = (double *)SPAllocTemp(BB->factor->rows * sizeof(double)); CHKPTR(tmp);
SpFBiSolvePermBase( BB, x, b, tmp );
SPFreeTemp( tmp );
}

void SpFBiSolvePermBase( BB, x, b, Tmp )
SpMatSplit *BB;
double      *x, *b, *Tmp;
{
int      bsize, b2, i, j, k, n, *nzs, ib, jb;
register double       *v, sum, *vv, *tmp = Tmp;
register int          *vi, nz;
SpMat    *B = BB->factor;
SpFBVec   *row, **rs;
int      *r = B->map->rowmap, *c = B->map->colmap;
SpFBRowMat *R = (SpFBRowMat *)B->data;
double   mone = -1.0;

n      = R->nfbvecs;
bsize  = R->bsize;
b2     = bsize * bsize;
/* forward solve the lower triangular */
nzs    = BB->nzl;
rs     = R->rs;
/* Could preload tmp with b, then use a single backsolve routine */
ib     = 0;
for (i=0; i<n; i++) {
    row  = *rs++;
    v    = row->v;
    vi   = row->i;
    nz   = *nzs;  nzs += bsize;
    for (j=0; j<bsize; j++) tmp[ib+j] = b[*r++];
    jb   = 0;
    for (j=0; j<nz; j++) {
	DVPAMV( tmp + ib, v + j*b2, bsize, bsize, mone, tmp + vi[jb] );
	jb += bsize;
        }
    /* Do the diagonal block (lower triangle; assume unit diagonal) */
    vv = v + nz * bsize;
    for (j=0; j<bsize; j++) {
	sum = 0.0;
	for (k=1; k<j; k++) {
	    sum -= vv[k*bsize+j] * tmp[ib+k];
	    }
	tmp[ib+j] += sum;
	}
    ib += bsize;
    }

/* backward solve the upper triangular */
nzs -=bsize; rs--; ib -= bsize;
c = c + (n*bsize-1);
for (i=n-1; i>=0; i--) {
    nz   = *nzs + 1; nzs -= bsize;
    row  = *rs--;
    v    = row->v + nz;
    vi   = row->i + nz;
    nz   = row->nz - nz;
    jb   = 0;
    for (j=0; j<nz; j++) {
	DVPAMV( tmp + ib, v + j*b2, bsize, bsize, mone, tmp + vi[jb] );
	jb  += bsize;
        }
    /* Do the diagonal block (upper triangle) */
    vv = v + nz * bsize;
    for (j=bsize-1; j>=0; j--) {
	sum = tmp[ib+j];
	for (k=j+1; k<bsize; k++) {
	    sum -= vv[k*bsize+j] * tmp[ib+k];
	    }
	x[*c--] = tmp[ib+j] = sum * vv[j*(bsize+1)];
	}
    ib -= bsize;
    }
}

/*
   This is a solve that does not involve any mappings 
 */
void SpFBiSolveBase( BB, x, b )
SpMatSplit  *BB;
double      *x, *b;
{
int      bsize, b2, i, j, k, n, *nzs, ib, jb;
double   *vv;
register double       *v, sum;
register int          *vi, nz;
SpMat    *B = BB->factor;
SpFBVec    *row, **rs;
SpFBRowMat *R = (SpFBRowMat *)B->data;
double    mone = -1.0;

n      = R->nfbvecs;
bsize  = R->bsize;
b2     = bsize * bsize;
/* forward solve the lower triangular */
nzs    = BB->nzl;
rs     = R->rs;
ib     = 0;
for (i=0; i<n; i++) {
    row  = *rs++;
    v    = row->v;
    vi   = row->i;
    nz   = *nzs / bsize;  nzs += bsize;
    for (j=0; j<bsize; j++) x[ib+j] = *b++;
    jb = 0;
    for (j=0; j<nz; j++) {
	DVPAMV( x + ib, v+j*b2, bsize, bsize, mone, x + vi[jb] );
	jb += bsize;
        }
    /* Do the diagonal block (lower triangle; assume unit diagonal) */
    vv = v + nz * bsize;
    for (j=0; j<bsize; j++) {
	sum = 0.0;
	for (k=0; k<j; k++) {
	    sum -= vv[k*bsize+j] * x[ib+k];
	    }
	x[ib+j] += sum;
	}
    ib += bsize;
    }

/* backward solve the upper triangular */
nzs -= bsize; rs--; ibsize -= bsize;
for (i=n-1; i>=0; i--) {
    nz   = (*nzs) / bsize + 1; nzs -= bsize;
    row  = *rs--;
    v    = row->v + nz*b2;
    vi   = row->i + nz;
    nz   = row->nz - nz;
    jb   = 0;
    for (j=0; j<nz; j++) {
	DVPAMV( x + ib, v + j * b2, bsize, bsize, mone, x + vi[jb] );
	jb += bsize;
        }
    /* Do the diagonal block (upper triangle) */
    vv = v;
    for (j=bsize-1; j>=0; j--) {
	sum = x[ib+j];
	for (k=j+1; k<bsize; k++) {
	    sum -= vv[k*bsize+j] * x[ib+k];
	    }
	x[ib+j] = sum * vv[j*(bsize+1)];
	}
    ib -= ibsize;
    }
}


