/*     @(#) ./solvers/svctx.h 07/23/93
      Contains the context variable for SERIAL linear system
   solver package, SV. See svcreate.c and the readme file for full
   details.
*/

#if !defined(__SVCTX)
#define __SVCTX
#include "tools.h"
#include "sparse/spmat.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "system/system.h"

/* Choices for solvers */
typedef enum { SVLU, SVJacobi, SVSSOR, SVILU, SVICC, SVICCJP, SVBDD, SVOSM,
	       SVNOPRE } 
        SVMETHOD;
typedef struct {
  int          nv, cnst_v;
  double       **v;
  } SVnullSpace;
  
typedef struct {
  int          size;             /* size of linear system */
  int          setupcalled;
  int          solvecalled;
  int          symmetric;

  SVMETHOD     type;             /* SVLU, SVJacobi, SVSSOR, SVILU ... */
  SpMat        *mat;
  SpMat        *amat;
  void         (*setup)();
  int          (*solve)();       /* returns number of iterations */
  void         *(*BuildSolution)();
  void         (*destroy)();
  void         *private;
  ITMETHOD     method;           /* ITGMRES, ITCG, ITQMR, .... */
  ITCntx       *itctx;
  int          its;
  int          flops;
  int          memory;
  double       t_solve, t_setup; /* Times for solve and setup */
  int          nz;     
  int          nzorig;
  int          natural_blocksize;
  int          use_initial_guess;
  SVnullSpace  *ns;
  /* Since iterative methods are a large part of these solvers, we
     include additional information on the computation */
  int          namult, nbinv;
  /* In order to handle vendor-supplied iterative and direct methods, 
     we need to separate the iterative data from the other data. 
     To start with, we indicate whether iter information should be
     accepted */
  int          is_iter;
} SVctx;
  

typedef struct {
  SpMatSplit   *split;   
  SPORDERTYPE  ordering;         /* ORDER_ND, ORDER_QMD, ORDER_RCM ...*/
  int          threshold;        /* minimum block size for block factor */
  PIVOTTYPE    pivoting;
} SVLUctx;

typedef struct {
  double       *diag;
} SVJacobictx;

typedef struct {
  SpMatSplit   *split;   
  double       omega;  
  double       *diag;
  void         **work;
} SVSSORctx;

typedef struct {
  SpMatSplit   *split;   
  double       *diag;
  void         **work;
  int          fill;
  double       rtol;        /* Drop tolerance */
  PIVOTTYPE    pivoting;
} SVILUctx;

typedef struct {
  SpMatSplit   *split;   
  double       *diag;
  void         **work;
  int          failures;
  double       alpha;
} SVICCctx;

/* From here out, add methods with there own include file */

/* Block diagonal decomposition solvers SVBDD */
#include "solvers/bdd.h"
/* Overlapping Schwarz */
#include "solvers/osm.h"
/* No preconditioning */
#include "solvers/nopre.h"

void  SVRegister();
void  SVRegisterAll();
void  SVRegisterDestroy();

SVctx *SVCreate();
/*
void  SVSetUp();
int   SVSolve();
int   SVSolveAdd();
int   SVSolveTranspose(); 
void  SVDestroy();
 */

/* Should these be in a private context somewhere? */
void  SViCreateLU();
void  SViSetupLU();
int   SViSolveLU();
void  SViDestroyLU();
void  SViCreateJacobi();
void  SViSetupJacobi();
int   SViSolveJacobi();
void  SViDestroyJacobi();
void  SViCreateSSOR();
void  SViSetupSSOR();
int   SViSolveSSOR();
void  SViDestroySSOR();
void  SViCreateILU();
void  SViSetupILU();
int   SViSolveILU();
void  SViDestroyILU();
void  SViCreateICC();
void  SViCreateICCJP();
void  SViSetupICC();
void  SViSetupICCJP();
int   SViSolveICC();
void  SViDestroyICC();

/* Solver database sample structure */
#include <stdio.h>
typedef struct {
    FILE *dir, 
         *history;
    char *title, *hname;
    }  SVDB;
extern SVDB *SVDBOpen();
extern void SVDBClose(), SVDBHeader(), SVDBHistory();

/* This checks that an iterative context exists.  The use is
   SVCheckIter(ctx,name){code if context exists;}
 */
#define SVCheckIter(ctx,name) \
    if (!(ctx)->itctx) {char buf[200];\
	sprintf(buf,"Must call SVSetUp before calling %s",name);SETERRC(1,buf);}else
/*M
    SVSetPrecondMat - Sets the matrix to be used for the preconditioning.

    Input Parameters:
.   ctx - solver context
.   bmat - matrix to be used for preconditioning.

    Notes:
    By default, the matrix provided in SVCreate is used as the basis for
    the preconditioning.  Using SVSetPrecondMat allows the user to
    specify a different matrix for use in specifying the preconditioning.

    Synopsis:
    void SVSetPrecondMat( ctx, bmat )
    SVctx *ctx;
    SpMat *bmat;
M*/
#define SVSetPrecondMat(ctx,bmat) (ctx)->mat = bmat

/*

  Add soon...
    SVSetPrecond( ctx, binv, bctx )
    replace the preconditioner entirely.  Note that the solver
method would then be something like SVUSER.  

    SVSetAmult( ctx, amult, actx )

to allow the amult operation to be replaced.
 */

/*M
     SVSetUp - Called after a call to SVCreate(), allocates space
     that will be needed later in the call to SVSolve().

     Synopsis:
     void SVSetUp(ctx)
     SVctx *ctx;
M*/
#define SVSetUp(ctx)       (*(ctx)->setup)(ctx);

/*M
     SVSolve - Solves the linear system.

     Input Parameters:
.       ctx -  solver context
.       b - the right hand side 

     Output Parameters:
.       x - the solution

     Synopsis:
     int SVSolve(ctx, b, x)
     SVctx *ctx;
     void  *b,*x;

     Returns:
     Number of iterations (iterative methods) or 0 (direct methods) on 
     success.  A negative value is returned on failure; for iterative methods,
     this is (roughly) the negative of the number of iterations completed.

     Notes:
     See SVCreate and SVSetUp to see how the solver context is defined
     and initialized before calling this routine.
M*/
#define SVSolve(ctx,x,y)   (*(ctx)->solve)(ctx,x,y);

/*M
     SVDestroy - Destroys a solver context created by SVCreate().

     Synopsis:
     void SVDestroy(ctx)
     SVctx *ctx;
M*/
#define SVDestroy(ctx)     (*(ctx)->destroy)(ctx);

/*M
     SVSetLUOrdering - Sets the order type to be used for the 
     factorization in the linear system solve using LU.

     Input Parameters:
.      ctx - solver context
.      ordering - ORDER_ND, ORDER_QMD, ORDER_RCM for nested 
                 dissection, Quotient mimimum degree or Reverse
                 Cuthill-McGee.

     Synopsis:
     void SVSetLUOrdering(ctx, ordering)
     SVctx *ctx;
     int   ordering;

     Notes:
     Call after SVCreate() but before SVSetUp().
M*/
#define SVSetLUOrdering(C,a)  \
  if ((C)->type == SVLU) {((SVLUctx *)(C)->private)->ordering = a;} 

/*M
     SVSetLUThreshold - Sets the minimum block size to use
     in the LU factorization. 

     Input Parameters:
.      ctx - solver context
.      threshold - identical rows will only be clustered together
                   if there are at least threshold of them.

     Synopsis:
     void SVSetLUThreshold(ctx, threshold)
     SVctx *ctx;
     int   threshold;

     Notes:
     Call after SVCreate() but before SVSetUp().
     Four is generally a good number to use as the threshold.
M*/
#define SVSetLUThreshold(C,a)  \
  if ((C)->type == SVLU) {((SVLUctx *)(C)->private)->threshold = a;} 

/*M
     SVSetLUPivoting - Sets the pivoting type to be used for the 
     factorization in the linear system solve using LU.

     Input Parameters:
.      ctx - solver context
.      pivoting - PIVOT_NONE, PIVOT_PRE_SYM, PIVOT_PRE_NONSYM, PIVOT_ACTIVE
                  PIVOT_PRE_SYM and PIVOT_PRE_NONSYM are symmetric and
                  unsymmetric reorderings done based on the elements of the
                  matric before any factoring takes place.  Using these 
                  orderings does not guarentee that a zero pivot will not
                  be encountered during factorization.  PIVOT_ACTIVE (not
		  yet implemented) implements a pivoting strategy during
		  the actual factorization; a zero pivot will not be 
		  encountered unless the matrix is numerically singular.

     Synopsis:
     void SVSetLUPivoting(ctx, pivoting)
     SVctx *ctx;
     int   pivoting;

     Notes:
     Call after SVCreate() but before SVSetUp().
M*/
#define SVSetLUPivoting(C,a)  \
  if ((C)->type == SVLU) {((SVLUctx *)(C)->private)->pivoting = a;} 

/*M
     SVSetAccelerator - Sets the type of accelerator to use
     for the iterative process.

     Input Parameters:
.      ctx - solver context
.      accelerator - for instance, ITRICHARDSON, ITCHEBYCHEV, ITCG,
                    ITGMRES, ITTCQMR, ITBCGS

     Synopsis:
     void SVSetAccelerator(ctx, type)
     SVctx    *ctx;
     ITMETHOD type;

     Notes:
     Call after SVCreate() but before SVSetUp().
M*/
#define SVSetAccelerator(C,a) \
  if ((C)->is_iter) {(C)->method = a;}

/*MC
     SVSetMonitor - Sets the routine that monitors the residual
     at each iteration of the iterative method.

     Input Parameters:
.      ctx - solver context
.      monitor - function pointer.  If null, print the iteration and residual
.      mctx    - context for private data for the monitor routine (may be null)

     Synopsis:
     void SVSetMonitor(ctx,monitor,mctx)
     SVctx *ctx;
     void  (*monitor)(), *mctx;

     Notes:
     Call after SVSetUp().  The default simply prints the residual at each 
     iteration. Look in the iter directory for more information.
M*/
#define SVSetMonitor(C,a,b) \
  if ((C)->is_iter) { \
      SVCheckIter(C,"SVSetMonitor") {\
    if (a == 0) {(C)->itctx->usr_monitor = ITDefaultMonitor;} \
    else  {ITSetMonitor((C)->itctx,a,b);}}  \
  }

/*MC
     SVSetConvergenceTest - Sets the function that is to be used to determine
     convergence.  

     Input Parameters:
.      ctx - solver context
.      converge - function pointer
.      cctx    - context for private data for the convergence routine (may be 
                 null)

     Synopsis:
     void SVSetConvergenceTest( ctx, converge, cctx )
     SVctx *ctx;
     int  (*converge)();
     void  *cctx;

     Notes:
     The default convergence test is a combination of relative and absolute
     tolerances.  The residual value that is tested may be an approximation;
     routines that need exact values should compute them.  Look in the iter 
     directory for more information.
     Call after SVSetUp().
M*/
#define SVSetConvergenceTest(C,a,b) \
  if ((C)->is_iter) { \
    SVCheckIter(C,"SVSetConvergenceTest") {\
    if (a == 0) {(C)->itctx->converged = ITDefaultConverged;} \
    else  {ITSetConvergenceTest((C)->itctx,a,b);}}  \
  }

/*M
    SVSaveResidualHistory - Specifies the array to hold residual histories

    Input Parameters:
.      ctx - solver context
.      p   - points to array to hold residual history
.      nmax - size of p

     Synopsis:
     void SVSaveResidualHistory( ctx, p, nmax )
     SVctx  *ctx;
     double *p;
     int    nmax;
M*/
#define SVSaveResidualHistory(C,a,b) \
  if ((C)->is_iter) { \
    SVCheckIter(C,"SVSaveResidualHistory") {\
    {ITSetResidualHistory((C)->itctx,a,b);}}  \
  }
  
/*M
    SVSetUseInitialGuess - Use the value in "x" as the initial guess for
                           iterative solvers.

    Input Parameters:
.   ctx - solver context
.   flag - 1 for use the initial guess, 0 for use zero as the initial guess

    Synopsis:
    void SVSetUseInitialGuess( ctx, flag )
    SVctx *ctx;
    int   flag;

    Note:
    By default, 0 is used as the initial guess.
M*/
#define SVSetUseInitialGuess( C, flag ) \
    (C)->use_initial_guess = flag

/*M
     SVSetSSOROmega - Sets the relaxation factor for SSOR.

     Input Parameters:
.      ctx - solver context
.      omega - the relaxation factor  (1.2 often works well)

     Synopsis:
     void SVSetSSOROmega(ctx,omega)
     SVctx  *ctx;
     double omega;

     Notes:
     The default value of omega is one.
     Call after SVCreate() but before SVSetUp().
M*/
#define SVSetSSOROmega(C,a)  \
  if ((C)->type == SVSSOR)  ((SVSSORctx *)(C)->private)->omega = a;

/*M
     SVSetILUFill - Sets the level of fill for the incomplete 
     LU preconditioner.

     Input Parameters:
.      ctx - solver context
.      fill - postive or zero integer

     Synopsis:
     void SVSetILUFill(ctx, fill)
     SVctx  *ctx;
     int    fill;

     Notes:
     The default is zero. The larger the value
     the faster the convergence, but the factors become more 
     dense and hence each iteration is slower.
     Call after SVCreate() but before SVSetUp().
M*/
#define SVSetILUFill(C,a) \
  if ((C)->type == SVILU)  ((SVILUctx *)(C)->private)->fill = a;

/*M
     SVSetILUDropTol - Sets the drop tolerance for the incomplete 
     LU preconditioner.

     Input Parameters:
.      ctx - solver context
.      tol - drop tolerance

     Synopsis:
     void SVSetILUDropTol(ctx, tol)
     SVctx  *ctx;
     double tol;

     Notes:
     The default is zero. The smaller the value
     the faster the convergence; but the factors become more 
     dense, and hence each iteration is slower.  Use a negative value
     to disable the drop tolerance value.
     Call after SVCreate() but before SVSetUp().

M*/
#define SVSetILUDropTol(C,a) \
  if ((C)->type == SVILU)  ((SVILUctx *)(C)->private)->rtol = a;
/*M
     SVSetILUPivoting - Sets the pivoting type to be used for the 
     factorization in the linear system solve using LU.

     Input Parameters:
.      ctx - solver context
.      pivoting - PIVOT_NONE, PIVOT_PRE_SYM, PIVOT_PRE_NONSYM, PIVOT_ACTIVE
                  PIVOT_PRE_SYM and PIVOT_PRE_NONSYM are symmetric and
                  unsymmetric reorderings done based on the elements of the
                  matric before any factoring takes place.  Using these 
                  orderings does not guarentee that a zero pivot will not
                  be encountered during factorization.  PIVOT_ACTIVE (not
		  yet implemented) implements a pivoting strategy during
		  the actual factorization.

     Synopsis:
     void SVSetILUPivoting(ctx, pivoting)
     SVctx *ctx;
     int   pivoting;

     Notes:
     Call after SVCreate() but before SVSetUp().
M*/
#define SVSetILUPivoting(C,a)  \
  if ((C)->type == SVILU) {((SVILUctx *)(C)->private)->pivoting = a;} 

/*M
     SVSetIts - Sets the maximum number of iterations allowed.

     Input Parameters:
.      ctx - solver context
.      max_its - maximum number of iterations to allow

     Synopsis:
     void SVSetIts(ctx, max_its)
     SVctx  *ctx;
     int    max_its;

     Notes:
     Call after SVSetUp().
M*/
#define SVSetIts(C,a) \
    if ((C)->is_iter) { \
if ((C)->method != -1) SVCheckIter(C,"SVSetIts")(C)->itctx->max_it = a;}

/*M
     SVSetRelativeTol - Sets the relative tolerance for convergence.

     Input Parameters:
.      ctx - solver context
.      tol - tolerance

     Synopsis:
     void SVSetRelativeTol(ctx, tol)
     SVctx  *ctx;
     double tol;

     Notes:
     Call after SVSetUp().  The two-norm of the residual (possibly estimated)
     is used for the test.
M*/
#define SVSetRelativeTol(C,a) \
  if ((C)->is_iter) { \
  if ((C)->method != -1)SVCheckIter(C,"SVSetRelativeTol")(C)->itctx->rtol = a;}

/*M
     SVSetAbsoluteTol - Sets the absolute tolerance for convergence.

     Input Parameters:
.      ctx - solver context
.      tol - tolerance

     Description:
     The default convergence test uses a relative tolerance test in 
     checking for convergence.  In many applications, an absolute test is
     also needed (for example, in transient problems where the initial guess
     can be fairly accurate).  This routine sets an absolute convergence
     tolerance on for the iterations.

     Warning: The interpretation of a absolute convergence test, and the
     choice of the value to use, can be very tricky.  For example, if
     the linear system comes from a multicomponent PDE, and the components
     are not similar in size, then even with a fairly tight absolute
     tolerance, the relative error in the component that is smaller in 
     magnitude may be large.

     Call after SVSetUp().

     Synopsis:
     void SVSetAbsoluteTol(ctx, tol)
     SVctx  *ctx;
     double tol;
M*/
#define SVSetAbsoluteTol(C,a) \
  if ((C)->is_iter) { \
  if ((C)->method != -1)SVCheckIter(C,"SVSetAbsoluteTol")(C)->itctx->atol = a;}

/*M
     SVSetGMRESRestart - Sets the number of iterations before using 
     a restart for GMRES.

     Input Parameters:
.      ctx - solver context
.      its - number of iterations before restart

     Synopsis:
     void SVSetGMRESRestart(ctx, its)
     SVctx  *ctx;
     int    its;

     Notes:
     Call after SVSetUp().
M*/
#define SVSetGMRESRestart(C,a) \
  {if ((C)->method==ITGMRES) SVCheckIter(C,"SVSetGMRESRestart"){\
					ITGMRESSetDirections((C)->itctx,a);}}
                            
/*MC
     SVGetICCFailures - Returns the number of shifts needed for
     incomplete Choleski factorization before positive definite
     preconditioner was found.

     Input Parameters:
.      ctx - solver context

     Output Parameters:
.      count - number of shifts before acceptable factor found

     Synopsis:
     void SVGetICCFailures(ctx, count)
     SVctx  *ctx;
     int    *count;

     Notes:
     Call after SVSolve.
M*/
#define SVGetICCFailures(C,a)  \
  if ((C)->type == SVICC || (C)->type == SVICCJP)  \
  {a = ((SVICCctx *)(C)->private)->failures;}
 
/*MC
     SVGetICCAlpha - Returns shift factor needed for
     incomplete Choleski factorization before positive definite
     preconditioner was found.

     Input Parameters:
.      ctx - solver context

     Output Parameters:
.      alpha - number of shifts before acceptable factor found

     Synopsis:
     void SVGetICCAlpha(ctx, alpha)
     SVctx  *ctx;
     double *alpha;

     Notes:
     Call after SVSolve().
M*/
#define SVGetICCAlpha(C,a)  \
  if ((C)->type == SVICC || (C)->type == SVICCJP)  \
  {a = ((SVICCctx *)(C)->private)->alpha;}

/*MC
     SVGetFlops - Returns number of flops used related to solver context
     since creation of solver context or since a call to SVSetFlopsZero().

     Input Parameters:
.      ctx - solver context

     Output Parameters:
.      flops - number of flops since SVCreate() or SVSetFlopsZero()

     Synopsis:
     void SVGetFlops(ctx, flops)
     SVctx  *ctx;
     int    *flops;
M*/
#define SVGetFlops(C,a)  \
  a = (C)->flops;
 
/*M
     SVSetFlopsZero - Resets the flop counter associated with a solver
     context.
     
     Input Parameters:
.      ctx - solver context

     Synopsis:
     void SVSetFlopsZero(ctx)
     SVctx  *ctx;

     Notes:
     See SVGetFlops.

M*/
#define SVSetFlopsZero(C)  \
  (C)->flops = 0;

/*MC
     SVGetMemory - Returns the amount of space used by the solver context.

     Input Parameters:
.      ctx - solver context

     Output Parameters:
.      mem - number of bytes of memory used

     Synopsis:
     void SVGetMemory( ctx, mem )
     SVctx  *ctx;
     int    *mem;
M*/
#define SVGetMemory(C,a)  \
  a = (C)->memory; 
 
/*M
     SVGetTimeSetup - Returns the CPU time in seconds used by SVSetUp

     Input Parameter:
.    ctx - solver context

     Returns:
.    time in setup in seconds

     Synopsis:
     double SVGetTimeSetup( ctx )
     SVctx  *ctx;
M*/
#define SVGetTimeSetup(C) ((C)->t_setup)

/*M
     SVGetTimeSolve - Returns the CPU time in seconds used by SVSolve

     Input Parameter:
.    ctx - solver context

     Returns:
.    time in solve in seconds

     Synopsis:
     double SVGetTimeSolve( ctx )
     SVctx  *ctx;
M*/
#define SVGetTimeSolve(C) ((C)->t_solve)

/*M
     SVResetTimes - Reset the times saved for solve and setup to zero.

     Input Parameter:
.    ctx - solver context

     Synopsis:
     void SVResetTimes( ctx )
     SVctx *ctx;
M*/
#define SVResetTimes(C) {(C)->t_solve = 0; (C)->t_setup = 0;}

#define SVGetNonzeros(C,a) \
  a = (C)->nz;

/* update the flops after an iterative method completes, given the cost for 
   a matrix mult (am), preconditioner (bi). */
#define SVGetITFlops(C,am,bi) SVCheckIter(C,"SVGetITFlops"){\
 (C)->flops += (am)*((C)->itctx->nmatop + \
 (C)->itctx->namult) + (bi)*((C)->itctx->nmatop + (C)->itctx->nbinv) + \
 (C)->itctx->nvectors * (C)->size + (C)->itctx->nscalar;\
 ITClearWorkCounts( (C)->itctx ); }
#endif

/*M
     SVUseVectorBLAS - Makes the iterative methods use the level-1 BLAS where
     possible.

     Input Parameters:
.      ctx - solver context

     Synopsis:
     void SVUseVectorBLAS( ctx )
     SVctx *ctx;

     Notes:
     Call after SVSetUp().  The default is to use vector operations that
     may be more efficient for small problems.  Also, a number of operations
     that are needed by iterative methods (such as y gets a y plus x) are not
     provided by the BLAS.
M*/
#define SVUseVectorBLAS(C) {\
  if ((C)->is_iter) { \
      SVCheckIter(C,"SVUseVectorBLAS") {\
    DVBSetDefaultFunctions((C)->itctx->vc);}}}

/*M
    SVSetIsSymmetric - Indicate that the problem is symmetric 

    Input Parameters:
.   ctx  - solver context
.   flag - degree of symmetry: 0 for unsymmetric, 1 for symmetric structure,
           2 for symmetric, and 3 for symmetric positive definite

    Synopsis:
    void SVSetIsSymmetric( ctx, flag )
    SVctx *ctx;
    int   flag;

    Notes:
    Call before SVSetUp().  The default is to assume unsymmetric.  This routine
    provides a hint to SLES that may (or may not) be used to improve the
    efficiency of the solver method.
M*/
#define SVSetIsSymmetric(C,flag) \
  (C)->symmetric = flag;
