#ifndef lint
static char SCCSid[] = "@(#) ./nonlin/newtr/newtr.c 07/23/93";
#endif

/*
     This implements Newton's Method with a trust region approach
   for Nonlinear systems of equations.

     This is intended as a model implementation, it does not 
   necessarily have many of the bells and whistles of other 
   implementations. The basic algorithm is taken from the 
   book "Sources and Development of Methematical Software" 
   Wayne Cowell, editor, pages 88-111 More', Sorensen, Garbow, Hillstrom
   "The Minpack Project"

     It is intended to be DATA-STRUCTURE NEUTRAL and can be called
   RECURSIVELY.

     The Newton code has two context variables. 

     1) NLCntx *nlP -- The nonlinear solver context which is 
                       created by NLCreate(NLNEWTONTRUST);
     2) void *usrP - The user context, the user sets this pointer to
                     point to a user define structure which contains
                     stuff which is used to evaluate the function, solve 
                     systems which involve approximations to the 
                     Jacobian, etc.

   The stepCompute routine needs to return two values: ynorm (the norm of
   the step) and gpnorm.  This is the predicted value for the function
   at the new point, assuming a local linearization.  The value is 0 if
   the step lies within the trust region and is > 0 otherwise.
*/
#include <math.h>
#include <stdio.h>
#include "tools.h"
#include "nonlin/nlall.h"

/*------------------------------------------------------------*/
/* Solves a nonlinear system of equations using Newton's      */
/* method and a trust region scheme.                          */
/*------------------------------------------------------------*/
int NLNewtonTRSolve(nlP,usrP)
NLCntx *nlP;
void    *usrP;
{
  NLNewtonTRCntx  *neP = (NLNewtonTRCntx *) nlP->MethodPrivate;
  void               *x = nlP->vec_sol,*f,*y,*g,**work,*tmp;
  int                N = nlP->max_it,i;
  double             rho,fnorm,gnorm,gpnorm,xnorm,delta;
  double             *res = nlP->residual_history, ynorm;
  FILE               *fp = nlP->fp;

  work = VGETVECS(nlP->vc,usrP,3); CHKPTRV(work,0);
  f = work[0]; y = work[1]; g = work[2];

  if (nlP->initial_guess) (*nlP->initial_guess)(nlP,usrP,x);  /* x <- x_0  */
  else VSET(nlP->vc,usrP,0.0,x);

  VNORM(nlP->vc,usrP,x,&xnorm); nlP->nvectors++;
   
  (*nlP->fun)(nlP,usrP,x,f);   nlP->nfunc++;      /* f <- function(x)   */
  VNORM(nlP->vc,usrP,f,&fnorm); nlP->nvectors++;   /* fnorm <- || f || */  
  if (res) *res++ = fnorm;
  delta = neP->delta0*fnorm; 
        
  if (nlP->usr_monitor) {
    (*nlP->usr_monitor)(nlP,usrP,x,f,fnorm);
  }

  for ( i=0; i<N; i++ ) {
    if (nlP->stepSetup) (*nlP->stepSetup)(nlP,usrP,x); /* form J */
    while(1) {                                             /* y <- -J\f  */
      (*nlP->stepCompute)(nlP,usrP,x,f,y,fnorm,delta,
                                      nlP->trunctol,&gpnorm,&ynorm);  
      VAXPY(nlP->vc,usrP,1.0,x,y);              /* y <- x + y         */
      (*nlP->fun)(nlP,usrP,y,g);                /* g = function(y)    */ 
      VNORM(nlP->vc,usrP,g,&gnorm);             /* gnorm <- || g ||   */ 
    
      if (fnorm == gpnorm) rho = 0.0;
      else  rho = (fnorm*fnorm - gnorm*gnorm)/(fnorm*fnorm - gpnorm*gpnorm); 

      /* update size of trust region */
      if      (rho < neP->mu)  delta *= neP->delta1;
      else if (rho < neP->eta) delta *= neP->delta2;
      else                     delta *= neP->delta3;

      if (fp) fprintf(fp,"%d f %g g %g ynorm %g ",i,fnorm,gnorm,ynorm);
      if (fp) fprintf(fp,"gpred %g rho %g delta %g\n",gpnorm,rho,delta);

      nlP->nfunc++; nlP->nsteps++; nlP->nvectors += 2;
      if (rho > neP->sigma)            break;
      if (delta < xnorm*neP->deltatol) return -i; /* not making progress*/
    }
    if (nlP->stepDestroy) (*nlP->stepDestroy)(nlP,usrP);

    fnorm = gnorm;
    if (res) *res++ = fnorm;
    tmp = f; f = g; g = tmp;
    tmp = x; x = y; y = tmp;
    VNORM(nlP->vc,usrP,x,&xnorm); nlP->nvectors++;

    if (nlP->usr_monitor) {
      (*nlP->usr_monitor)(nlP,usrP,x,f,fnorm);
    }

    /* Test for convergence */
    if ((*nlP->converged)(nlP,usrP,xnorm,ynorm,fnorm)) {
      /* make sure solution is in corect location */
      if (x != nlP->vec_sol) VCOPY(nlP->vc,usrP,x,nlP->vec_sol);
      break;
    } 
  }

  VFREEVECS(nlP->vc,usrP,work,3);
  return i+1;
}

/* -------------------------------------------------------------*/
NLCntx *NLNewtonTRCreate()
{
  NLCntx            *nlP;
  NLNewtonTRCntx *neP;

  nlP                    = NEW(NLCntx); CHKPTRN(nlP);
  NLSetDefaults( nlP );

  nlP->method            = NLNEWTONTR;

  nlP->setup             = NLNewtonTRSetUp;
  nlP->solver            = NLNewtonTRSolve;
  nlP->destroy           = NLNewtonTRDestroy;

  neP                    = NEW(NLNewtonTRCntx); CHKPTRN(neP);
  nlP->MethodPrivate     = (void *) neP;
  neP->mu                = .25;
  neP->eta               = .75;
  neP->delta0            = .2;
  neP->delta1            = .3;
  neP->delta2            = .75;
  neP->delta3            = 2.0;
  neP->sigma             = .0001;
  neP->deltatol          = 1.e-12;
  return nlP;
}
/*------------------------------------------------------------*/
/*ARGSUSED*/
void NLNewtonTRSetUp(nlP,usrP)
NLCntx *nlP;
void   *usrP;
{
  if (!nlP->stepCompute) {
    fprintf(stderr,"NLNewtonTRSetUp needs stepCompute!\n");
    SETERR(1);
  }
  if (!nlP->fun) {
    fprintf(stderr,"NLNewtonTRSetUp needs fun!\n");
    SETERR(1);
  }
  if (!nlP->vc) {
    fprintf(stderr,"NLNewtonTRSetUp needs vector ops!\n");
    SETERR(1);
  }
}
/*------------------------------------------------------------*/
void NLNewtonTRDestroy(nlP)
NLCntx *nlP;
{
  FREE(nlP->MethodPrivate);
  FREE(nlP);
}

