/*
   This example defines a processor set for the rows and for the columns of
   a mesh, then tests some global operation on these processor subsets.
   The subsets are themselves subsetted.

   This example is intended as more of a test of the processor subsets than
   as an example of a useful program.  Multiple global operations
   are done "back-to-back" to test the code for chosing message types
   so as to separate different uses of the global operations.

   The example here simulates a simple matrix factorization algorithm.
   It looks basically like

   for (row=0; row<nr; row++) {
       // get processors in this column and in remaining rows
       val = (myrow >= row) ? myrow : -1
       pscol = partition( val, ALLPROCS );
       // get processors by row and in remaining columns 
       val = (col >= row) ? myrow : -1;
       psrow  = partition( val, ALLPROCS );
       if (pscol) scatter
       if (psrow) scatter
       PSDestroy( pscol )
       PSDestroy( psrow )
       }

    Another approach is to FIRST form a processor set for the 
    row and column of this processor, then subset IT

    pscol = partition( mycol, ALLPROCS )
    psrow = partition( myrow, ALLPROCS )
    for (row = 0; row<nx; row++) {
       // get processors in this column and in remaining rows
       val = (myrow >= row) ? myrow : -1
       pscol1 = partition( val, pscol );
       // get processors by row and in remaining columns 
       val = (col >= row) ? myrow : -1;
       psrow1  = partition( val, psrow );
       if (pscol1) scatter
       if (psrow1) scatter
       PSDestroy( pscol1 )
       PSDestroy( psrow1 )
       }
   PSDestroy( pscol )
   PSDestroy( psrow )

   Both of these will be tried.
 */

#include <stdio.h>
#include "tools.h"
#include "comm/comm.h"

extern int main2();

int main(argc,argv)
int argc;
char **argv;
{
return PICall( main2, argc, argv );
}

int main2( argc, argv )
int argc;
char **argv;
{
int     row, col, myrow, mycol, nrows, ncols, i, val;
double  vrow, vcol, work, v;
ProcSet *prow, *pcol, *psrow1, *pscol1;

nrows = 2;
ncols = 2;
SYArgGetInt( &argc, argv, 1, "-r", &nrows );
SYArgGetInt( &argc, argv, 1, "-c", &ncols );
if (nrows * ncols != NUMNODES) {
    fprintf( stderr, "[%d,%d] doesn't fit in %d processors\n", nrows, ncols, 
	     NUMNODES );
    SYexitall("",1);
    }
/* PISetLogging( 3 ); */

myrow = MYPROCID % nrows;
mycol = (MYPROCID / nrows);
vrow = myrow;
vcol = mycol;

printf( "[%d] About to start First test\n", MYPROCID ); fflush( stdout );

/* First Test */
for (row=0; row<nrows; row++) {
    if (MYPROCID == 0)printf( "Row %d\n", row );
    /* get processors in this column and in remaining rows */
    val  = (myrow >= row) ? mycol : -1;
    pcol = PSPartition( val, ALLPROCS );
    /* printf( "[%d] pcol = %x(%d)\n", MYPROCID, pcol, val ); fflush( stdout );
        */
    /* get processors by row and in remaining columns */
    val  = (col >= row) ? ncols + myrow : -1;
    prow = PSPartition( val, ALLPROCS );
#ifdef FOO
    printf( "[%d] prow = %x(%d)\n", MYPROCID, prow, val ); fflush( stdout );
    {int i;
     for (i=0; i<=NUMNODES; i++){
	if (GTOKEN(ALLPROCS,i)) {
	    if (pcol) {printf( "[%d] col procset\n", i);
	    PSPrintProcset( pcol, 0, stdout );
		       fflush(stdout);}
	    else {
		printf( "[%d] empty col procset\n", i );
		fflush(stdout) ;
		}
	    }
	}
     for (i=0; i<=NUMNODES; i++){
	 if (GTOKEN(ALLPROCS,i)) {
	     if (prow) {printf( "[%d] row procset\n", i);
	    PSPrintProcset( prow, 0, stdout );
		       fflush(stdout);}
	    else {
		printf( "[%d] empty row procset\n", i );
		fflush(stdout) ;
		}
	    }
	 }
     }
#endif
    if (pcol) {
	v = vcol;
	/* printf( "[%d] about to do col scatter\n", MYPROCID ); 
	   fflush( stdout ); */
	GSCATTER( &v, sizeof(double), myrow == row, pcol, MSG_DBL );
	/* printf( "[%d] done with col scatter\n", MYPROCID ); 
	   fflush( stdout ); */
	/* Check for correct data */
	if (v != vcol)
	    fprintf( stderr, "[%d] column scatter got %d expected %d\n",
		     MYPROCID, (int)v, (int)vcol );
	}
    if (prow) { 
	v = vrow;
	/* printf( "[%d] about to do row scatter\n", MYPROCID ); 
	   fflush( stdout ); */
	GSCATTER( &v, sizeof(double), mycol == row, prow, MSG_DBL );
	/* printf( "[%d] done with  row scatter\n", MYPROCID ); 
	   fflush( stdout ); */
	/* Check for correct data */
	if (v != vrow)
	    fprintf( stderr, "[%d] row scatter got %d expected %d\n",
		     MYPROCID, (int)v, (int)vrow );
	}
    /* printf( "[%d] about to do destroy col\n", MYPROCID ); 
       fflush( stdout ); */
    if (pcol)
	PSDestroy( pcol );
    /* printf( "[%d] about to do destroy row\n", MYPROCID ); 
       fflush( stdout ); */
    if (prow)
	PSDestroy( prow );
    }

GSYNC(ALLPROCS);

/* Second Test */
printf( "[%d] About to start Second test\n", MYPROCID ); fflush( stdout );

pcol = PSPartition( mycol, ALLPROCS );
prow = PSPartition( ncols + myrow, ALLPROCS );
for (row = 0; row<nrows; row++) {
    if (MYPROCID == 0) printf( "Row %d\n", row );
    /* get processors in this column and in remaining rows */
    val    = (myrow >= row) ? mycol : -1;
    pscol1 = PSPartition( val + nrows*ncols, pcol );
    /* get processors by row and in remaining columns  */
    val    = (mycol >= row) ? ncols + myrow : -1;
    psrow1 = PSPartition( val + nrows*ncols, prow );
    if (pscol1) {
	v = vcol;
	GSCATTER( &v, sizeof(double), myrow == row, pscol1, MSG_DBL );
	/* Check for correct data */
	if (v != vcol)
	    fprintf( stderr, "[%d] column scatter got %d expected %d\n",
		     MYPROCID, (int)v, (int)vcol );
	}
    if (psrow1) { 
	v = vrow;
	GSCATTER( &v, sizeof(double), mycol == row, psrow1, MSG_DBL );
	/* Check for correct data */
	if (v != vrow)
	    fprintf( stderr, "[%d] row scatter got %d expected %d\n",
		     MYPROCID, (int)v, (int)vrow );
	}
    if (pscol1)
	PSDestroy( pscol1 );
    if (psrow1)
	PSDestroy( psrow1 );
    }
PSDestroy( pcol );
PSDestroy( prow );

fprintf( stderr, "[%d] End of run for example3 \n", MYPROCID );

return 0;
}    




