
/*
    Here are some definitions to make the parallel ports easier

    Use the value of ARCH to specify the TYPE of parallel machine
 */

#include "tools.h"

#ifndef __commh
#define __commh

#if defined(tc2000) || defined(encore)
#define SHARED_MEMORY
#endif
#if defined(intelnx) || defined(p4) || defined(pvm) || defined(picl) || \
    defined(eui)
#define DISTRIBUTED_MEMORY
#endif

#include "comm/nbr.h"
#include "comm/procset.h"

extern void PIBroadcastArgs();
extern int  PIcall();
extern void PImerge();
extern void PIArgsToResources();
extern void PIGetResourceLimits();
extern void PIChangeResourceLimits();
extern void PISetOption();
extern int  PIGetNbrsHC(), PIGetNbrsMSH();

/* Here we define some parameter checking routines. */
#ifdef DEBUG_ALL
#define COMMCHECKSEND(typ,len,to) {\
    if ((typ) < -1) SETERRC(1,"Invalid type in send");\
    if ((len) < 0)  SETERRC(1,"Invalid (negative) length in send");\
    if ((to) < 0 || (to)>=NUMNODES) SETERRC(1,"Invalid destination");}
/* maxlen field of CHECKRECV isn't used yet */
#define COMMCHECKRECV(typ,maxlen) {\
    if ((typ) < -1)   SETERRC(1,"Invalid type in recv");\
    if ((maxlen) < 0)  SETERRC(1,"Invalid (negative) length in recv");}
#else
#define COMMCHECKSEND(type,len,to)
#define COMMCHECKRECV(type,len)
#endif
#include "comm/logging.h"

/*
    Eventually, we want to cache the node # and number of nodes in
    these variables so that the cost of logging and other checks
    is as low as possible.  Since these need to be set, for those
    systems that can be used WITHOUT PIcall, we need to provide an
    "init" routine that must be called.  Just to make these slightly
    safer, they will be initialized to -1 so that an "unset" condition
    is easy to establish.
*/
#if defined(pvm) || defined(picl)
extern int __MYPROCID, __NUMNODES;
#endif

/*D
    MessagePassing -  Common message passing macros.  

    Description:
    These convert into the popular forms of Intel NX, P4, and others 
    (eventually PICL, PVM, and others as contributed)
 
    These include send, receive, and global operations.  To encourage
    truely portable use of global operations, they all support processor
    subsets; if the native routine can't do that, then an
    implementation-specific version must be provided.  They also support
    asynchronous and "force" operations.

    There are many possible combinations.  They are aranged as

    SEND/RECV SYNC/ASYNC <null>/NOMEM <null>/FORCE

    <null> means that the item may be omitted.  The meanings are

    SEND/RECV - send or receive.

    SYNC/ASYNC - synchronous or asynchronous.  When the synchronous version
    returns, the buffer may be reused.  On a receive, the buffer contains
    a message.  When the asynchronous version returns, the "id" argument
    is a context that must be waited on (see SEND/RECV WAIT) before
    the buffer may be used.  The "id" value returned by these is > 0 on
    success.

    NOMEM - In order to allow the most flexible use of message passing
    libraries, this interface allows message buffers to be specially 
    allocated (see MSGALLOC SEND/RECV).  If the NOMEM version of a 
    routine is selected, the buffer was NOT allocated with the MSGALLOC
    macros.  The buffer MAY be used, OR a local buffer may be created 
    and freed.

    FORCE - (applies only to SEND) Send without any handshaking with the
    receiver.  This amy be used when it is certain that the receiver
    is already waiting to receive a message.  This should be used with
    great care; if the receiver is not ready for the message, the message
    may be discarded (the behavior is undefined).

$    SENDASYNC(type,buffer,length,to,datatype,id)
$    RECVASYNC(type,buffer,maxlength,datatype,id)
$    SENDSYNC(type,buffer,length,to,datatype)
$    RECVSYNC(type,buffer,maxlength,datatype)

    xxxxxxxxNOMEM - same as SEND/RECV ASYNC/SYNC, except the buffers
    have not been allocated with the MSGALLOC SEND/RECV macrocs.
    xxxxxxxxxxxxxFORCE - same as SEND/RECV ASYNC/SYNC <null>/NOMEM, except
    the destination must be ready to receive the message.

$    SENDWAIT(same as sendasync)
$    RECVWAIT(same as recvasync)
$    SENDWAITNOMEM ... ect

    In addition, a receive that handles an arbitrary length message is 
    provided

    RECVSYNCUNSZ(type,bufp,size,datatype)

    returns a pointer to a buffer "bufp" that contains a message of
    size "size".  The buffer may be freed with the usual macro.

    Message inquiry functions    
$    RECVLEN()      - length in bytes of the most recently received message
$    RECVFROM()     - id of the sender of the most recently received message

    A simplified mechanism for probing the receive queue is provided.
    A Probe determines if a message of a given type is available.
    If the probe is successful, RECVFROM may be used to determine
    who the message is from (the value of RECVLEN is implementation
    dependent).
$    SYNCPROBE(type)
$    ASYNCPROBE(type)

    Because of some indetminincy in some implementations, if you want
    to receive the message that you have just probed, use
    RECVPROBED(type,buffer,maxlength,datatype)

$    GxSUM(val,n,work,procset)
$    GxMAX(val,n,work,procset)
$    GxMIN(val,n,work,procset)

    (There should be more; they can be provided as requested.  There is
    also a function for providing an arbitrary, user-defined combination
    operation).

$    GCOL(lbuf,lsize,gbuf,gsiz,glen,procset,datatype) - global collection
$    GCOLX(lbuf,gsizes,gbuf,procset,datatype)         - global collection in 
                                              node number order with
					      given sizes

$    GSCATTER(buf,siz,issrc,procset,datatype)- buf is sent from src to everyone
                                              else in the procset
$    GSCATTERSRC(buf,size,src,procset,datatype)- like GSCATTER, except from
                                              a particular source

    GTOKEN(procset,i) - returns 1 if the processor is the i'th member of
    the procset and 0 otherwise.  In addition, a partial ordering is
    enforced: processor i can not proceed until processor (i-1) sees the
    value i.  This should ONLY be used in a loop like

        for (i=0;i<=np;i++) {if (gtoken(procset,i)) <do-something>;}

    where np is the number of processors in the procset.

    The "G" routines are global routines; if "procset" is non-null, they
    are global over a subset of processors.

    System inquiry functions
$    NUMNODES   - Number of processors
$    MYPROCID   - id (in [0:(NUMNODES-1)] of this processor
$    MSGTYPERANGE(&low,&high) - sets low to the lowest permissibly value,
                               high to the highest.  This is implementation
                               dependent, and not all systems make this
                               information available.
$    MSGSIZES(&min,&max)      - sets min to the minimum message size and
                               max to the maximum message size, in bytes.

    The following are not yet implemented
    
$    MSGMODELTIME(&latency,&timeperbyte,&hoptime) -
    		Models the time to send data as
		(latency + n_bytes * timeperword + n_hops * hoptime),
		where n_hops is the "distance" and n_bytes is the number
		of bytes being sent.  This information may not be available;
		the fall-back values are (0,1,0).  Values are in ?
		(we will decide after designing the pipelined broadcasting
		code).
$    MSGPACKETSIZE - returns the natural size in bytes of a message packet.
		    Value represents a natural size for a message
                    (often, the actual size of a message).  If < 0, there
                    is no (known) natural size.
$    MSGDISTANCE(from,to) - yields the number of hops or connections
                           in sending a message from processor "from" to
                           processor "to".  Since this is not aways known,
                           a value of "1" is the default except for
                           from == to.
                           
    With these, we could add an ADDITIONAL send/recv option: pipelined
    (not clear we want this for async routines)
 D*/

#if defined(p4)
#include "comm/commp4.h"
#elif defined(picl)
#include "comm/commpicl.h"
#elif defined(pvm)
#include "comm/commpvm.h"
#elif defined(intelnx)
#include "comm/commnx.h"
#elif defined(cm5)
#include "comm/commcm5.h"
#elif defined(eui)
#include "comm/commeui.h"
#elif defined(ncube)
#include "comm/commn3.h"
#else
#include "comm/commdef.h"
#endif

/* Some standard defines for common cases */
#ifdef NO_ASYNC_SEND
typedef int ASYNCSendId_t;
#define SENDASYNC(type,buffer,length,to,datatype,id) \
        SENDSYNC(type,buffer,length,to,datatype)
#define SENDASYNCNOMEM(type,buffer,length,to,datatype,id) \
        SENDASYNC(type,buffer,length,to,datatype,id)
#define SENDWAIT(type,buffer,length,to,datatype,id) 
#define SENDWAITNOMEM(type,buffer,length,to,datatype,id)  
#define SENDCANCEL(id)
#endif

#ifdef NO_ASYNC_RECV
typedef int ASYNCRecvId_t;
#define RECVASYNC(type,buffer,length,datatype,id) {id=1;}   
#define RECVASYNCNOMEM(type,buffer,length,datatype,id) {id=1;}
#define RECVWAIT(type,buffer,length,datatype,id) \
        RECVSYNC(type,buffer,length,datatype)
#define RECVWAITNOMEM(type,buffer,length,datatype,id)     \
        RECVSYNCNOMEM(type,buffer,length,datatype)
#define RECVCANCEL(id)
#endif

#if defined(NO_ASYNC_RECV) || defined(NO_ASYNC_SEND)
#define ASYNCDONE(id) 1
#endif

#ifdef NO_FORCE
#define SENDASYNCFORCE(type,buffer,length,to,datatype,id) \
        SENDSYNC(type,buffer,length,to,datatype)
#define RECVASYNCFORCE(type,buffer,length,datatype,id) \
        RECVASYNC(type,buffer,length,datatype,id) 
#define SENDSYNCFORCE(type,buffer,length,to,datatype) \
        SENDSYNC(type,buffer,length,to,datatype)
#define RECVSYNCFORCE(type,buffer,length,datatype) \
        RECVSYNC(type,buffer,length,datatype)
#define SENDASYNCNOMEMFORCE(type,buffer,length,to,datatype,id) \
        SENDASYNCNOMEM(type,buffer,length,to,datatype,id)
#define RECVASYNCNOMEMFORCE(type,buffer,length,datatype,id) \
        RECVASYNCNOMEM(type,buffer,length,datatype,id) 
#define SENDSYNCNOMEMFORCE(type,buffer,length,to,datatype) \
        SENDSYNCNOMEM(type,buffer,length,to,datatype)
#define RECVSYNCNOMEMFORCE(type,buffer,length,datatype)  \
        RECVSYNCNOMEM(type,buffer,length,datatype)
#define SENDWAITFORCE(type,buffer,length,to,datatype,id) 
#define SENDWAITNOMEMFORCE(type,buffer,length,to,datatype,id) 
#define RECVWAITFORCE(type,buffer,length,datatype,id) \
        RECVSYNC(type,buffer,length,datatype)
#define RECVWAITNOMEMFORCE(type,buffer,length,datatype,id)     \
        RECVWAITNOMEM(type,buffer,length,datatype,id)
#endif

#if NO_NATIVE_SUBSET
/* Here go standard routines when global operations are supported,
   but not operations on subsets.  This should define only those that
   have global versions available.  To use these,
   #define G....GLOB(...) for the operations that are available.
 */
#if defined(GISUMGLOB)
#define GISUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);if(!(procset)) \
			GISUMGLOB(val,n,work);else\
                        gisumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if defined(GDSUMGLOB)
#define GDSUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);if(!(procset)) \
			   GDSUMGLOB(val,n,work);else\
                           gdsumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if defined(GFSUMGLOB)
#define GFSUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);if(!(procset)) \
			   GFSUMGLOB(val,n,work);else\
                           gfsumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if defined(GIMAXGLOB)
#define GIMAX(val,n,work,procset)  {LOGOPSTART(LOG_GMAXS);if(!(procset)) \
			  GIMAXGLOB(val,n,work);else\
			  gihighset(val,n,work,procset);LOGOPEND(LOG_GMAXE);}
#endif
#if defined(GIMINGLOB)
#define GIMIN(val,n,work,procset)  {LOGOPSTART(LOG_GMINS);if(!(procset)) \
			    GIMINGLOB(val,n,work);else\
                            gilowset(val,n,work,procset);LOGOPEND(LOG_GMINE);}
#endif
#if defined(GDMAXGLOB)
#define GDMAX(val,n,work,procset)  {LOGOPSTART(LOG_GMAXS);if(!(procset)) \
			  GDMAXGLOB(val,n,work);else\
			  gdhighset(val,n,work,procset);LOGOPEND(LOG_GMAXE);}
#endif
#if defined(GDMINGLOB)
#define GDMIN(val,n,work,procset)  {LOGOPSTART(LOG_GMINS);if(!(procset)) \
			    GDMINGLOB(val,n,work);else\
                            gdlowset(val,n,work,procset);LOGOPEND(LOG_GMINE);}
#endif
#if defined(GIORGLOB)
#define GIOR(val,n,work,procset)   {if (!(procset)) \
                            GIORGLOB(val,n,work);else\
                             giorset( val, n, work, procset );}
#endif
#if defined(GSYNCGLOB)
#define GSYNC(procset)      {LOGOPSTART(LOG_GSYNCS);if(!(procset))GSYNCGLOB();\
                             else gsyncset(procset);LOGOPEND(LOG_GSYNCE);}
#endif
#if defined(GCOLGLOB)
#define GCOL(lbuf,lsize,gbuf,gsiz,glen,procset,datatype) {\
      LOGOPSTART(LOG_GCOLS);\
      if(!(procset))GCOLGLOB(lbuf,lsize,gbuf,gsiz,glen,datatype);else\
      gcolset(lbuf,lsize,gbuf,gsiz,glen,procset,datatype);LOGOPEND(LOG_GCOLE);}
#endif
#if defined(GCOLXGLOB)
#define GCOLX(lbuf,gsizes,gbuf,procset,datatype) {LOGOPSTART(LOG_GCOLS);\
        if(!(procset))GCOLXGLOB(lbuf,gsizes,gbuf,datatype);else\
        gcolxset(lbuf,gsizes,gbuf,procset,datatype);LOGOPEND(LOG_GCOLE);}
#endif
/* #define GTOKEN(procset,i) gtokenset(procset,i)*/
#if defined(GSCATTERGLOB)
#define GSCATTER(buf,size,issrc,procset,datatype) {LOGOPSTART(LOG_GSCATS);\
        if (!(procset)) {GSCATTERGLOB(buf,size,issrc,datatype);}else\
        gscatterset(buf,size,issrc,procset,datatype);LOGOPEND(LOG_GSCATE);}
#endif
#if defined(GSCATTERSRCGLOB)
#define GSCATTERSRC(buf,size,src,procset,datatype) {LOGOPSTART(LOG_GSCATS);\
        if (!(procset)){GSCATTERSRCGLOB(buf,size,src,datatype);}else\
        gscattersetsrc(buf,size,src,procset,datatype);LOGOPEND(LOG_GSCATE);}
#endif

#endif /* NO_GLOBAL_SUBSET */

#ifdef NO_NATIVE_GLOBAL
extern void gcsumset(), gisumset(), gdsumset(), gfsumset();
extern void gchighset(), gihighset(), gfhighset(), gdhighset(),
            gclowset(),  gilowset(),  gflowset(),  gdlowset();
extern void gcorset(), giorset(), gcandset(), giandset();
extern void gopset(), gsyncset(),
	    gcolset(), gcolxset();
#if !defined(GCSUM)
#define GCSUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);\
			  gcsumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if !defined(GISUM)
#define GISUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);\
			  gisumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if !defined(GDSUM)
#define GDSUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);\
			  gdsumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if !defined(GFSUM)
#define GFSUM(val,n,work,procset)  {LOGOPSTART(LOG_GSUMS);\
			  gfsumset(val,n,work,procset);LOGOPEND(LOG_GSUME);}
#endif
#if !defined(GCMAX)
#define GCMAX(val,n,work,procset)  {LOGOPSTART(LOG_GMAXS);\
			  gchighset(val,n,work,procset);LOGOPEND(LOG_GMAXE);}
#endif
#if !defined(GIMAX)
#define GIMAX(val,n,work,procset)  {LOGOPSTART(LOG_GMAXS);\
			  gihighset(val,n,work,procset);LOGOPEND(LOG_GMAXE);}
#endif
#if !defined(GDMAX)
#define GDMAX(val,n,work,procset)  {LOGOPSTART(LOG_GMAXS);\
			  gdhighset(val,n,work,procset);LOGOPEND(LOG_GMAXE);}
#endif
#if !defined(GFMAX)
#define GFMAX(val,n,work,procset)  {LOGOPSTART(LOG_GMAXS);\
			  gfhighset(val,n,work,procset);LOGOPEND(LOG_GMAXE);}
#endif
#if !defined(GCMIN)
#define GCMIN(val,n,work,procset)  {LOGOPSTART(LOG_GMINS);\
			  gclowset(val,n,work,procset);LOGOPEND(LOG_GMINE);}
#endif
#if !defined(GIMIN)
#define GIMIN(val,n,work,procset)  {LOGOPSTART(LOG_GMINS);\
			  gilowset(val,n,work,procset);LOGOPEND(LOG_GMINE);}
#endif
#if !defined(GDMIN)
#define GDMIN(val,n,work,procset)  {LOGOPSTART(LOG_GMINS);\
			  gdlowset(val,n,work,procset);LOGOPEND(LOG_GMINE);}
#endif
#if !defined(GFMIN)
#define GFMIN(val,n,work,procset)  {LOGOPSTART(LOG_GMINS);\
			  gflowset(val,n,work,procset);LOGOPEND(LOG_GMINE);}
#endif

#if !defined(GCOR)
#define GCOR(val,n,work,procset)   \
    gcorset( val, n, work, procset )
#endif
#if !defined(GIOR)
#define GIOR(val,n,work,procset)   \
    giorset( val, n, work, procset )
#endif

#if !defined(GCAND)
#define GCAND(val,n,work,procset)   \
    gcandset( val, n, work, procset )
#endif
#if !defined(GIAND)
#define GIAND(val,n,work,procset)   \
    giandset( val, n, work, procset )
#endif

#if !defined(GSYNC)
#define GSYNC(procset)             {LOGOPSTART(LOG_GSYNCS);\
				gsyncset(procset);LOGOPEND(LOG_GSYNCE);}
#endif
#if !defined(GCOL)
#define GCOL(lbuf,lsize,gbuf,gsiz,glen,procset,datatype){\
      LOGOPSTART(LOG_GCOLS);\
      gcolset(lbuf,lsize,gbuf,gsiz,glen,procset,datatype);LOGOPEND(LOG_GCOLE);}
#endif
#if !defined(GCOLX)
#define GCOLX(lbuf,gsizes,gbuf,procset,datatype) {LOGOPSTART(LOG_GCOLS);\
        gcolxset(lbuf,gsizes,gbuf,procset,datatype);LOGOPEND(LOG_GCOLE);}
#endif
#if !defined(GTOKEN)
#define GTOKEN(procset,i) gtokenset(procset,i)
#endif
#if !defined(GSCATTER)
#define GSCATTER(buf,siz,issrc,procset,datatype) {LOGOPSTART(LOG_GSCATS);\
        gscatterset(buf,siz,issrc,procset,datatype);LOGOPEND(LOG_GSCATE);}
#endif
#if !defined(GSCATTERSRC)
#define GSCATTERSRC(buf,siz,src,procset,datatype) {LOGOPSTART(LOG_GSCATS);\
        gscattersetsrc(buf,siz,src,procset,datatype);LOGOPEND(LOG_GSCATE);}
#endif

#endif

/* Finally, include the "forward" compatibility routines */
#include "comm/newnames.h"

/* At this point, PImytid is defined */
#ifdef DEBUG_TRACEBACK
/* REPLACE the definitions of DEBUGTBF to include the process id */
#undef DEBUGTBF
#undef DEBUGTBCF
#define DEBUGTBF(line,file) fprintf( stderr, "[%d] Line %d in %s\n", PImytid, line, file )
#define DEBUGTBCF(line,file,c) \
    fprintf( stderr, "[%d] Line %d in %s: %s\n", PImytid, line, file, c )
#endif

/* Here begin definitions for the man pages of these macros */
/*M
     PInsendm - Start an asynchronous send

     Synopsis:
     void PInsendm(type, buffer, length, to, datatype, id)
     int  type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type     
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for PIwsendm.

     Note:
    "buffer" must have been allocated with MSGALLOCSEND.
M*/
/*M
     PInrecvm - Start an asynchronous receive

     Synopsis:
     void PInrecvm( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for PInwaitm.

     Note:
    "buffer" must have been allocated with MSGALLOCRECV.
M*/
/*M
     PIbsendm - Send a message to another processor

     Synopsis:
     void PIbsendm( type, buffer, length, to, datatype)
     int type, length, to, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer   
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
    "buffer" must have been allocated with MSGALLOCSEND.
M*/
/*M
     PIbrecvm - Receive a message from another processor

     Synopsis:
     void PIbrecvm( type, buffer, length, datatype)
     int type, length, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
    "buffer" must have been allocated with MSGALLOCRECV.
M*/
/*M
     PIwsendm - Complete an asynchronous send

     Synopsis:
     void PIwsendm( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to 
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInsendm.

     Note:
    "buffer" must have been allocated with MSGALLOCSEND.
M*/
/*M
     PIwrecvm - Complete an asynchronous receive

     Synopsis:
     void PIwrecvm( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer   
.    length   - length (in units of sizeof(char)) 
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInrecvm.

     Note:
    "buffer" must have been allocated with MSGALLOCRECV.
M*/
/*M
     PInsend - Start an asynchronous send

     Synopsis:
     void PInsend( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for Piwsend.

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCSEND
M*/
/*M
     PInrecv - Start an asynchronous receive

     Synopsis:
     void PInrecv( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for PIwrecv.

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCRECV
M*/
/*M
     PIbsend - Send a message to another processor

     Synopsis:
     void PIbsend( type, buffer, length, to, datatype)
     int type, length, to, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to 
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCSEND
M*/
/*M
     PIbrecv - Receive a message from another processor

     Synopsis:
     void PIbrecv( type, buffer, length, datatype)
     int type, length, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCRECV
M*/
/*M
     PIwsend - Complete an asynchronous send

     Synopsis:
     void PIwsend( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInsend.

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCSEND
M*/
/*M
     PIwrecv - Complete an asynchronous receive

     Synopsis:
     void PIwrecv( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInrecv.

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCRECV
M*/
/*M
     PInsendmrr - Start an asynchronous send

     Synopsis:
     void PInsendmrr( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for PIwsendmrr

     Note:
     "buffer" must have been allocated with MSGALLOCSEND.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PInrecvmrr - Start an asynchronous receive

     Synopsis:
     void PInrecvmrr( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for PIwrecvmrr

     Note:
     "buffer" must have been allocated with MSGALLOCRECV.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIbsendmrr - Send a message to another processor

     Synopsis:
     void PIbsendmrr( type, buffer, length, to, datatype)
     int type, length, to, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer   
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
     "buffer" must have been allocated with MSGALLOCSEND.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIbrecvmrr - Receive a message from another processor

     Synopsis:
     void PIbrecvmrr( type, buffer, length, datatype)
     int type, length, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
     "buffer" must have been allocated with MSGALLOCRECV.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIwsendmrr - Complete an asynchronous send

     Synopsis:
     void PIwsendmrr( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to (int)
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation for SENDWAIT

     Note:
     "buffer" must have been allocated with MSGALLOCSEND.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIwrecvmrr - Complete an asynchronous receive

     Synopsis:
     void PIwrecvmrr( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer   
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInrecvmrr

     Note:
     "buffer" must have been allocated with MSGALLOCRECV.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PInsendrr - Start an asynchronous send

     Synopsis:
     void PInsendrr( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for SENDWAITNOMEMFORCE

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCSEND.
     "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PInrecvrr - Start an asynchronous receive

     Synopsis:
     void PInrecvrr( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    id       - identifies operation for PIwrecvrr

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCRECV.
$    "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIbsendrr - Send a message to another processor

     Synopsis:
     void PIbsendrr( type, buffer, length, to, datatype)
     int type, length, to, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCSEND.
$    "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIbrecvrr - Receive a message from another processor

     Synopsis:
     void PIbrecvrr( type, buffer, length, datatype)
     int type, length, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  )
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCRECV.
$    "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIwsendrr - Complete an asynchronous send

     Synopsis:
     void PIwsendrr( type, buffer, length, to, datatype, id)
     int type, length, to, datatype;
     ASYNCSendId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    to       - processor to send to 
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInsendrr

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCSEND.
$    "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIwrecvrr - Complete an asynchronous receive

     Synopsis:
     void PIwrecvrr( type, buffer, length, datatype, id)
     int type, length, datatype;
     ASYNCRecvId_t id;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    buffer   - pointer to buffer  
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     
.    id       - identifies operation from PInrecvrr

     Note:
     This version should be used when "buffer" has NOT been allocated with
     MSGALLOCRECV.
$    "rr" means that the message will be sent and if the destination
     machine is not ready to receive it, it will be discarded.  This
     provides a way to access faster but potentially unreliable
     communications.
M*/
/*M
     PIbrecvUnsz - Receive a message of unknown length

     Synopsis:
     void PIbrecvUnsz( type, buffer, length, datatype)
     int type, length, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type       
.    length   - length (in units of sizeof(char)))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

     Output Parameter:
.    buffer   - pointer to buffer

     Note:
     This routine allocates a buffer of the length needed to receive the 
     message.
M*/

/*M
     PIbprobe - Block until a message of a given type is available

     Synopsis:
     void PIbprobe(type)
     int type;

     Input Parameter:
.    type    - message type 

     Note:
     This routine blocks until a message of type "type" is available.
M*/
/*M
     PInprobe - Test if a message of a given type is available

     Synopsis:
     int PInprobe(type)
     int type;

     Input Parameter:
.    type    - message type 

     Note:
     This routine returns 1 if a message of type "type" is available and
     0 otherwise
M*/
/*M
     PInstatus - Test if an asynchronous message has completed

     Synopsis:
     int PInstatus(id)
     ASYNCSendId_t id;

     Input Parameter:
.    id      - message id.  This is the id set in the nonblocking routines

     Note:
     This routine returns 1 i the message with "id" has completed, and 
     0 otherwise
M*/
/*M
     PIcrecv - Cancel a previously issued PInrecv...

     Synopsis:
     void PIcrecv(id)
     ASYNCRecvId_t id;

     Input Parameter:
.    id      - message id.  This is the id set in the nonblocking routines
M*/
/*M
     PIcsend - Cancel a previously issued PInsend...

     Synopsis:
     void PIcsend(id)
     ASYNCSendId_t id;

     Input Parameter:
.    id      - message id.  This is the id set in the nonblocking routines
M*/
/*M
     PIbrecvProbed - Receive a message that has been probed

     Synopsis:
     void PIbrecvProbed( type, buffer, length, datatype)
     int type, length, datatype;
     void *buffer;

     Input Parameters:
.    type     - message type      
.    buffer   - pointer to buffer
.    length   - length (in units of sizeof(char))
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

M*/

/*M
     PIsize - Return the length of a received message

     Synopsis:
     int PIsize()

     Note: 
     Returns the number of bytes received by the last PIxrecv...
M*/
/*M
     PIfrom - Return the processor that sent a received message

     Synopsis:
     int PIfrom()

     Note: 
     Returns the number of processor that sent the message most recently
     received.
M*/

/*M
     PItype - Return the type of the most recently received message

     Synopsis:
     int PItype()

M*/

/*M
     PIgisum - Global sum reduction

     Synopsis:
     void PIgisum( val, n, work, procset)
     int *val, n, *work;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the sum from all participating nodes.
M*/
/*M
     PIgdsum - Global sum reduction

     Synopsis:
     void PIgdsum( val, n, work, procset)
     double  *val, *work;
     int     n;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the sum from all participating nodes.
M*/
/*M
     PIgfsum - Global sum reduction

     Synopsis:
     void PIgfsum( val, n, work, procset)
     float   *val, *work;
     int     n;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the sum from all participating nodes.
M*/
/*M
     PIgimax - Global maximum reduction

     Synopsis:
     void PIgimax( val, n, work, procset)
     int *val, n, *work;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the maximum from all participating nodes.
M*/
/*M
     PIgdmax - Global maximum reduction

     Synopsis:
     void PIgdmax( val, n, work, procset)
     double  *val, *work;
     int     n;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the maximum from all participating nodes.
M*/

/*M
     PIcombine - Global user-defined combination (reduction to all nodes)

     Synopsis:
     void PIcombine( val, n, work, procset, elmsize, datatype, op )
     void    *val, *work;
     int     n, elmsize, datatype;
     ProcSet *procset;
     void    (*op)( );

     Input Parameters:
.    val     - value to combine
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)
.    elmsize - size in bytes of a single element of val
.    datatype- datatype of val (MSG_DBL etc)
.    op      - user-defined routine to combine elements (see below)

     Notes:
     On exit, val contains the combined value from all participating nodes.

     The routine "op" is defined as
$    void op( a, b, n )
$    void *a, *b;
$    int  n;

     with arguments
.    a - value to combine with b.  Contains combined result on output
.    b - value to combine with a.
.    n - number of elements (NOT bytes) to combine.

     For example, the routine used for a global sum of doubles is
$        void PIdsum( a, b, n )
$        double *a, *b;
$        int    n;
$        {
$        int i;
$        for (i=0; i<n; i++) a[i] += b[i];
$        }
$
     Currently, there is no Fortran version.

M*/

/*M
     PIgimin - Global minimum reduction

     Synopsis:
     void PIgimin( val, n, work, procset)
     int *val, n, *work;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the minimum from all participating nodes.
M*/
/*M
     PIgdmin - Global minimum reduction

     Synopsis:
     void PIgdmin( val, n, work, procset)
     double  *val, *work;
     int     n;
     ProcSet *procset;

     Input Parameters:
.    val     - value to sum
.    n       - number of elements in val
.    work    - work area of the same size as val
.    procset - processor set (in null, all processors are used)

     Note:
     On exit, val contains the minimum from all participating nodes.
M*/
/*M
     PIgsync - Global synchronize

     Synopsis:
     void PIgsync(procset)
     ProcSet *procset;

     Input Parameter:
.    procset - processor set (in null, all processors are used)

     Note:
     All processors in the processor set wait until they have all executed
     this statement.
M*/

/*M
     PIgcol - Global collection

     Synopsis:
     void PIgcol( lbuf, lsize, gbuf, gsiz, glen, procset, datatype)
     void    *lbuf, *gbuf;
     int     lsize, gsiz, *glen, datatype;
     ProcSet *procset;

     Input Parameters:
.    lbuf    - Pointer to local contibution
.    lsize   - size of local contribution  
.    procset - processor set (if null, all processors are used)
.    gsiz    - Size of gbuf
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT) 

     Output Parameters:
.    gbuf    - Pointer to collected data
.    glen    - Actual number of characters collected

     Note:
     Data is collected from all nodes; when this routine exits, all nodes
     have copies of all the data.  The data is present in "gbuf" in
     an implementation-dependent order; the only thing that is guarenteed is
     that each contribution is contiguous.
M*/
/*M
     PIgcolx - Global collection

     Synopsis:
     void PIgcolx( lbuf, gsizes, gbuf, procset,datatype)
     void    *lbuf, *gbuf;
     int     *gsizes,datatype;
     ProcSet *procset;

     Input Parameters:
.    lbuf    - Pointer to local contibution
.    gsizes  - size of all the contributions
.    procset - processor set (if null, all processors are used)
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT) 

     Output Parameters:
.    gbuf    - Pointer to collected data

     Note:
     Data is collected from all nodes; when this routine exits, all nodes
     have copies of all the data.  The data is present in node-number order.
M*/

/*M
     PIbcast - Broadcast data to all processors

     Synopsis:
     void PIbcast( buf, siz, issrc, procset, datatype) 
     void    *buf;
     int     siz, issrc, datatype;
     ProcSet *procset;

     Input Parameters:
.    buf      - pointer to data
.    siz      - size of buf in bytes
.    issrc    - 1 if this is the sending processor, 0 otherwise.  Exactly
                one processor may set issrc to 1.
.    procset  - processor set (in null, all processors are used)
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

M*/

/*M
     PIbcastSrc - Broadcast data to all processors

     Synopsis:
     void PIbcastSrc( buf, siz, src, procset, datatype ) 
     void    *buf;
     int     siz, src, datatype;
     ProcSet *procset;

     Input Parameters:
.    buf      - pointer to data
.    siz      - size of buf in bytes
.    src      - node number of the source processor.
.    procset  - processor set (in null, all processors are used)
.    datatype - type of data (MSG_INT, MSG_OTHER, MSG_DBL, or MSG_FLT)     

M*/

/*M
     PIgtoken - Pass a "token" among processors

     Synopsis:
     int PIgtoken( procset, i)
     ProcSet *procset;
     int     i;

     Input Parameters:
.    procset - processor set (in null, all processors are used)
.    i       - Loop index (see below)

     Note:
     This routine should be used in a loop to insure that each processor
     can carry out an action while the others wait.  An example is for
     each processor to print out some information on what it is doing.
     An example is
$    for (i=0; i<=PInumtids; i++) 
$        if (PIgtoken(0,i)){ 
$            <do_something> 
$        }
$    Note that i runs from 0 through PInumtids; that is, the loop is
     executed PInumtids+1 times.  The last time serves to pass the "token"
     back to the 0'th processor.
M*/     

/*MC
     MSGALLOCSEND - Allocate storage for sending a message

     Synopsis:
     void MSGALLOCSEND( msg, max, type )
     (type *)msg;
     int     max;

     Input Parameters:
.    msg     - pointer to hold allocated message buffer
.    max     - length of message buffer
.    type    - type
M*/
/*MC
     MSGALLOCRECV - Allocate storage for receiving a message

     Synopsis:
     void MSGALLOCRECV( msg, max, type )
     (type *)msg;
     int     max;

     Input Parameters:
.    msg     - pointer to hold allocated message buffer
.    max     - length of message buffer
.    type    - type
M*/
/*M
     MSGFREESEND - Free storage for sending a message

     Synopsis:
     void MSGFREESEND(msg)
     void *msg;

     Input Parameters:
.    msg     - pointer to allocated message buffer
M*/
/*M
     MSGFREERECV - Free storage for receiving a message

     Synopsis:
     void MSGFREERECV(msg)
     void *msg;

     Input Parameters:
.    msg     - pointer to allocated message buffer
M*/
/*MC
     PInumtids - Return the number of processors

     Synopsis:
     int PInumtids

M*/
/*MC
     PImytid - Return my processor id

     Synopsis:
     int PImytid

     Note:
     PImytid is in the range [0,PInumtids-1]
M*/
/*M
     PIMsgSizes - Return the range of message sizes

     Synopsis:
     void PIMsgSizes( min, max)
     int *min, *max;

     Output parameters:
.    min,max - lengths of minimum and maximum messages, in bytes.

     Note:
     Message lengths in the range [min.max] may be used.  Regrettably, this
     information is not always available; in that case, this routine returns
     values that are believed to work (typical minimum is sizeof(int) and
     maximum is 16536).
M*/
/*M
     PITagRange - Return the range of value (user) message tags

     Synopsis:
     void PITagRange( low, high)
     int *low, *high;

     Output Parameters:
.    low,high - range of legal message types.

     Note:
     Message types in the range [low,high] may be used.  Regrettably, this
     information is not always available; in that case, this routine returns
     the minimum range that is known to work.     
M*/
/*M
     PIdistance - Return the number of hops between two processors
     
     Synopsis:
     int PIdistance( from, to)
     int from, to;

     Input Paramters:
.     from - id of sending processor
.     to   - id of receiving processor

     Note:
     The exact meaning of a "hop" depends on the implementation.       
M*/
/*MC
     PIdiameter - Return the maximum number of hops between two processors
     
     Synopsis:
     int PIdiameter

     Note:
     The exact meaning of a "hop" depends on the implementation.       
M*/

#endif

