/* mrandom.c 1.12 7/21/92 */
/*
 *		mrandom.c
 *
 *	Wrapper for random(), allowing safe restart.
 *
 *	Original Implementation:
 *		Clark Thomborson, September 1991
 *	Modifications:
 *	   - give uniform interface to other rngs
 *	   - remove bias from integer generator
 *	   - avoid possible infinite loop in mrandom(m)
 *	   - allow access to multiple simultaneous rngs
 *	   - add interface to nrand48()
 *	   - add interface to rand()
 *	   - tuned for speed
 *		Clark Thomborson, May 1992
 *	
 *
 *	This material is based upon work supported by the National
 *	Science Foundation under grant number MIP-9023238.  The
 *	Government has certain rights in this material.
 *
 *	Any opinions, findings, and conclusions or recommendations
 *	expressed in this material are those of the author and do
 *	not necessarily reflect the view of the National Science
 *	Foundation.
 *
 *	This code is neither copyrighted nor patented, although I am
 *	not sure about the status of some of the routines it calls.
 */

#include <values.h> /* we need MAXLONG */
#include <stdio.h>  /* we need FILE */
#include <math.h>  /* we need floor(), irint() */
#include "mrandom.h"
#include "bentley.h" /* Bentley's RNG is alg #2 */
#include "pcrand.h"  /* the portable combined RNG is alg #3 */

/* table size and output range for the trivial RNG and the RNGs in 4.3bsd */
#define RNGstatesize_0 1
#define	RNGstatesize_1 32
#define	RNGstatesize_4 2
#define	RNGstatesize_5 1
#define RNGrangem1_0 MAXLONG
#define RNGrangem1_1 MAXLONG
#define RNGrangem1_4 MAXLONG
/* according to my Sun man pages, system V unix has a different rand() */
#ifdef SYSV
#define RNGrangem1_5 MAXSHORT
#else
#define RNGrangem1_5 MAXLONG
#endif

/* a convenient modulus for the 32-bit RNGcount1 in our statefile */
#define BILLION	1000000000

/* RNGstate[0] for random(), table length 32 */
#define MRNGSTATE0	3

/* Format of our RNG statefile */
#define RNGfileLINE0	"RNG statefile for algorithm %ld "
#define RNGfileLINE0_0	"(trivial RNG: long state=seed1; state += seed2)\n"
#define RNGfileLINE0_1	"(4.3bsd random.c: non-linear additive feedback)\n"
#define RNGfileLINE0_4	"(4.3bsd nrand48.c: 48-bit multiplicative)\n"
#define RNGfileLINE0_5	"(4.3bsd rand.c: 32-bit multiplicative)\n"
#define RNGfileLINE1	"Initial seeds = %ld, %ld\n"
#define RNGfileLINE2	\
	"Number of calls to underlying RNG after seeding = %ld billion + %ld\n"
#define RNGfileLINE3	"RNG state table =\n"
#define RNGfileLINEn	"   %08lx %08lx %08lx %08lx\n"
#define RNGfileLINEn_4	"   %04hx %04hx %04hx\n"
#define RNGfileLINEnn	"   %08lx"
#define RNGfileLINE12	"Next value in this pseudorandom sequence = %08lx\n"

/* Convenient names for data in our RNGdata struct */
#define RNGalg		(rng->rngalg)
#define RNGseed1	(rng->rngseed1)
#define RNGseed2	(rng->rngseed2)
#define RNGcount1	(rng->rngcount1)
#define RNGcount2	(rng->rngcount2)
#define RNGnextval	(rng->rngnextval)
#define RNGstatesize	(rng->rngstatesize)
#define RNGrangem1	(rng->rngrangem1)
#define RNGrange	(rng->rngrange)
#define RNGstate	(rng->rngstate)

/* The most-recently initialized or restarted RNG */
RNGdata *mru_rng = 0;

/* setrng(rng) sets rng to point to a valid RNGdata structure, if any
 * can be found.  Otherwise it calls the error routine no_rng().
 * Coded as a macro for efficiency.
 */
#define setrng(rng)	if (rng == 0) { \
				if (mru_rng == 0) { \
					no_rng(); \
				} else { \
					rng = mru_rng; \
				} \
			}

/* Error routine called by setrng().
 */
void no_rng()
{
  fprintf(stderr, "RNG must be initialized or restarted before use!\n");
  fflush(stderr);
  exit(1);
}

/* A trivial RNG, which can be set up to return a constant or an
 * arithmetic progression.  Can come in handy for debugging. 
 */
long triv_rng(rng)
RNGdata *rng;
{
  RNGstate[0] += RNGseed2;
  RNGstate[0] &= MAXLONG; /* clear sign bit */
  return(RNGstate[0]);
}

/* Call the appropriate RNG n times, keeping track of the number of calls.
 * Writes a vector v[] of the long ints returned by these RNG calls.
 */
extern void xrandomrv(rng,n,v)
RNGdata *rng;
long n;
long v[];
{
  long i;
  RNGcount1 += n;
  if (RNGcount1 >= BILLION) {
    RNGcount1 -= BILLION;
    RNGcount2 += 1;
  }
  switch (RNGalg) {
    case 0:	for (i=0;i<n;i++) v[i]=triv_rng(rng); break;
    case 1:	for (i=0;i<n;i++) v[i]=random(); break;
    case 2:	lprandv(RNGstate,n,v); break;
    case 3:	for (i=0;i<n;i++) v[i]=lpcrand(RNGstate); break;
    case 4:	for (i=0;i<n;i++) v[i]=nrand48(RNGstate); break;
    case 5:	srand(RNGstate[0]); /* in case RNGs were switched */
		for (i=0;i<n;i++) v[i]=rand(); 
		RNGstate[0] = v[n-1]; /* save current RNG state */
		break;
    default:	for (i=0;i<n;i++) v[i]=0; break;
  }
}

/* set RNGalg, RNGstatesize, RNGrangem1, and RNGrange for the given algorithm.
 * print error message to stderr if alg is out of range.
 */
void setRNGparams(rng,alg)
RNGdata *rng;
long alg;
{
  RNGalg = alg;
  switch (alg) {
    case 0:
	RNGstatesize = RNGstatesize_0;
	RNGrangem1 = RNGrangem1_0;
	break;
    case 1: 
	RNGstatesize = RNGstatesize_1;
	RNGrangem1 = RNGrangem1_1;
	break;
    case 2:
	RNGstatesize = RNGstatesize_2;
	RNGrangem1 = RNGrangem1_2;
	break;
    case 3:
	RNGstatesize = RNGstatesize_3;
	RNGrangem1 = RNGrangem1_3;
	break;
    case 4:
	RNGstatesize = RNGstatesize_4;
	RNGrangem1 = RNGrangem1_4;
	break;
    case 5:
	RNGstatesize = RNGstatesize_5;
	RNGrangem1 = RNGrangem1_5;
	break;
    default:
	fprintf(stderr, "Undefined RNG algorithm.\n");
	fflush(stderr);
	break;
  }
  RNGrange = 1.0 + (double) RNGrangem1;
}

/* Examine the RNG data array to see if it looks ok.  Return 1 if ok, 0 if not
 */
int checkrandom(rng)
RNGdata *rng;
{
  long statesize, rangem1;
  double range;
  unsigned short *sarray;
  
  /* see if RNGalg is in range */
  if (RNGalg<0 || RNGalg>5) return(0);

  /* see if RNGstatesize, RNGrangem1, or RNGrange have been overwritten */
  statesize = RNGstatesize;
  rangem1 = RNGrangem1;
  range = RNGrange;
  setRNGparams(rng,RNGalg);
  if (statesize != RNGstatesize || 
	rangem1 != RNGrangem1 ||
	range != RNGrange ) {
    return(0);
  }

  /* now look at RNGstate (algorithm-specific tests) */
  switch (RNGalg) {
    case 0: return( RNGstate[0] >= 0 );
    case 1: return( RNGstate[0] == MRNGSTATE0 );
    case 2: return( ckrand(RNGstate) );
    case 3: return( checkran(RNGstate) );
    case 4: sarray = (unsigned short *) RNGstate;
	    return( sarray[3] == 0 );
    case 5: return( RNGstate[0] >= 0 );
    default: return(0);
  }
}

/* Write a RNG state identifier into the user-supplied string rngid,
 * which must be of length at least RNGIDSTRLEN.  If the user has not
 * initialized the rng with init_rng() or restart_rng(), abort with
 * an error message to stderr.  Otherwise return the value of rngid.
 */
extern char *describe_rng (rng,rngid)
RNGdata *rng;
char rngid[RNGIDSTRLEN];
{
  if (!rng) { /* avoid following null pointers */
    fprintf(stderr,
	 "Bad argument (rng==0) to describe_rng.\n");
    fflush(stderr);
    exit(1);
  }
  if (!checkrandom(rng)) {
    fprintf(stderr,
      "RNG has not been initialized, or else it has been overwritten!\n");
    fflush(stderr);
    exit(1);
  }
  sprintf(rngid, "RNG state identifier is (%ld: %ld, %ld; %ld, %ld)\n",
		 RNGalg, RNGseed1, RNGseed2, RNGcount1, RNGcount2);
  return(rngid);
}

/* Initialize the general-purpose rng data area so that it holds the
 * state and other data required for the selected algorithm "alg", where
 *	alg = 0 is a trivial generator that returns state += seed2,
 *	  where state is initially set to seed1.  Use for debugging only!
 *	alg = 1 is 4.3bsd random.c (non-linear additive feedback)
 *	alg = 2 is Bentley's remarkable RNG (lagged Fibonnacci)
 *	alg = 3 is the portable combined (multiplicative) RNG
 *	alg = 4 is 4.3bsd nrand48.c,
 *
 * Note: before calling init_rng(&myrng, ...), the user must allocate space
 * for this generator, for example, with the storage declaration
 *		RNGdata myrng;
 *
 * The seed1 parameter is used by all RNG algorithms to initialize its
 * state table.  The seed2 parameter is only used by algorithms 3 and 4.
 *
 * The count1 parameter indicates how many times the selected RNG algorithm
 * should be called prior to returning control to the calling routine.
 * We recommend a high value, at least 10000, for this parameter, to minimize
 * the ill effects of seeding.  The count2 parameter can be given a non-zero
 * value if you wish to "cycle" the generator a huge number of times: it 
 * will be called (count2*1e9 + count1) times.  Probably count2 should always
 * be 0 until computers become much, much faster than today.
 *
 * init_rng() returns 1 unless an out-of-range argument is detected
 * (rngalg >4 or <0; count1 >1e9 or <0; or count2 <0), in which case
 * it returns 0.
 *
 * Note: a single program may call init_rng() any number of times, to set up
 * multiple generators (possibly using more than one RNG algorithm), e.g with 
 *		RNGdata myrngs[8];
 *		long i,sum=0;
 *		for (i=0; i<7; i++) {
 *		  /* generator #i gets seed1 = i
 *		  init_rng(&myrngs[i],2,i,0,100000,0);
 *		}
 *		/* our eighth RNG uses algorithm #3
 *		init_rng(&myrngs[7],3,7,0,100000,0);
 *		/* add 1-bit numbers, one from each generator
 *		for (i=0; i<7; i++) {
 *		  sum += mrandom(&myrngs[i],2);
 *		}
 *
 * Warning: do not attempt to use multiple rngdata areas for algorithm #1.
 * The 4.3bsd random.c code has internal state that will not be modified
 * correctly when you switch generators (this was a bug in the original
 * implementation and it would be very difficult to fix here).
 * 
 * We recommend that init_rng() be used very sparingly.  Except when
 * replicating experiments or debugging, it is better to restart an
 * existing generator (stored in a statefile) than to initialize a new one.
 */
extern int init_rng (rng, alg, seed1, seed2, count1, count2)
RNGdata *rng;
long alg, seed1, seed2, count1, count2;
{
  unsigned short *sarray;
  long x[64];

  /* (In a future version, we could let rng==0 imply a default rng) */
  if (!rng) { /* avoid following null pointers */
    fprintf(stderr,
	 "Bad argument (rng==0) to init_rng.\n");
    fflush(stderr);
    exit(1);
  }

  /* keep track of the most-recently-used rng */
  mru_rng = rng;

  /* examine count1, count2: in range?  is count2 nonzero? */
  if (count1 >= BILLION || count1 < 0 || count2 < 0) {
    return(0);
  }
  if (count2 != 0) {
    fprintf(stderr, "Warning: this initialization will take a LONG time!\n");
    fflush(stderr);
  }

  setRNGparams(rng,alg); /* set RNGalg, RNGrangem1, RNGstatesize */
  RNGseed1 = seed1; /* keep a record of the seeds */
  RNGseed2 = seed2;
  RNGcount1 = 0;  /* mod-billion counter */
  RNGcount2 = 0;  /* div-billion counter */

  switch (RNGalg) {
    case 0:
      RNGstate[0] = RNGseed1;
      break;
    case 1: /* call the 4.3bsd random() initialization routine */
      initstate(seed1, RNGstate, RNGstatesize*4);
      break;
    case 2: /* call Bentley's init routine */
      sprand(seed1, RNGstate);
      break;
    case 3: /* call the portable combined multiplicative alg */
      setran(seed1, seed2, RNGstate);
      break;
    case 4: /* use nrand48() */
      /* copy seeds into the state array */
      sarray = (unsigned short *) RNGstate;
      sarray[0] = (RNGseed2)&0xFFFF; /* LSBs of seed */
      sarray[1] = (RNGseed1)&0xFFFF;
      sarray[2] = (RNGseed1>>16)&0xFFFF; /* MSBs of seed */
      sarray[3] = 0; /* stuff a 0 into the other half of RNGstate[1] */
      break;
    case 5: /* use rand() */
      RNGstate[0] = seed1; /* the srand() call will be made later */
      break;
    default: 
      return(0); /* error return */
  }

  /* Cycle the generator, in blocks of 64 (for speed) */
  for ( ; RNGcount2 < count2; ) { /* get to the right BILLION count */
    xrandomrv(rng,64,x);
  }
  for ( ; RNGcount1+64 < count1; ) { /* get close to the right count1 */
    xrandomrv(rng,64,x);
  }
  for ( ; RNGcount1 != count1; ) { /* and, finally, do the last few */
    xrandomrv(rng,1,x);
  }

  return(1); /* normal exit */

} /* end init_rng */

/* Return the next xrandomrv() output without disturbing the RNG state.
 */
long nextval(rng)
RNGdata *rng;
{
  long state[MAXRNGTABLESIZE];
  long i, r, retval, tcount1, tcount2;

  /* preserve old RNG state */
  tcount1 = RNGcount1;
  tcount2 = RNGcount2;
  for (i=0; i<RNGstatesize; i++) {
    state[i] = RNGstate[i];
  }

  /* find the next value in this pseudorandom sequence */
  xrandomrv(rng,1,&retval);

  /* restore RNG state */
  RNGcount1 = tcount1;
  RNGcount2 = tcount2;
  /* special fixup for 4.3bsd random()'s hidden state variables */
  if (RNGalg == 1) for (i=1; i<31; i++) {
    r = random();
  }
  for (i=0; i<RNGstatesize; i++) {
    RNGstate[i] = state[i];
  }

  return(retval);
}

/* Restart a generator from a statefile.  Print a message on stderr
 * and return 0 if the restart failed due to a garbled or nonexistent
 * statefile.  Otherwise return 1.
 */
extern int restart_rng(rng,filename)
RNGdata *rng;
char *filename;
{
  FILE *fp;
  unsigned short* sarray; /* for reading alg #4's statefile */
  long i, m, newstate[MAXRNGTABLESIZE], r; /* temps */
  int errflag; /* initially 0, becomes 1 if a problem is discovered */

  if (!rng) { /* avoid following null pointers */
    fprintf(stderr,
	 "Bad argument (rng==0) to restart_rng.\n");
    fflush(stderr);
    exit(1);
  }

  /* If mru_rng == 0, we assume random() hasn't been called yet,
   * so its internal state is at its startup values.
   * If mru_rng != 0 and alg=1, then we've used random() already, so we
   * must reset its internal state to its startup values.
   */
  if (mru_rng != 0 && mru_rng->rngalg == 1) {
    m = (mru_rng->rngcount1%31) + mru_rng->rngcount2*(BILLION%31);
    /* note: the hidden state variables are counters mod 31 */
    for ( ; (m%31) != 0; m++) {
      r = random();
    }
  }

  /* restore counter values, retrieve original RNG seeds and current state */
  fp = fopen(filename, "r");
  if (!fp) {
    fprintf(stderr, "There is no RNG statefile in this directory!\n");
    fflush(stderr);
    return(0);
  }

  /* read statefile */
  fscanf(fp, RNGfileLINE0, &RNGalg);
  setRNGparams(rng,RNGalg); /* set RNGrangem1, RNGstatesize */
  switch (RNGalg) {
    case 0:	fscanf(fp, RNGfileLINE0_0);
		break;
    case 1:	fscanf(fp, RNGfileLINE0_1);
		break;
    case 2:	fscanf(fp, RNGfileLINE0_2);
		break;
    case 3:	fscanf(fp, RNGfileLINE0_3);
		break;
    case 4:	fscanf(fp, RNGfileLINE0_4);
		break;
    case 5:	fscanf(fp, RNGfileLINE0_5);
		break;
    default:	return(0);
  }
  fscanf(fp, RNGfileLINE1, &RNGseed1, &RNGseed2);
  fscanf(fp, RNGfileLINE2, &RNGcount2, &RNGcount1);
  fscanf(fp, RNGfileLINE3);
  if (RNGalg == 4) { /* read as shorts, for code portability */
    sarray = (unsigned short *) newstate;
    fscanf(fp, RNGfileLINEn_4, &sarray[0], &sarray[1], &sarray[2]);
    sarray[3] = 0;
  } else for (i=0; i<RNGstatesize; i++) { /* all other RNG algs use long ints */
    fscanf(fp, "%lx", &newstate[i]);
    if (i == RNGstatesize-1) fscanf(fp,"\n");
  }
  fscanf(fp, RNGfileLINE12, &RNGnextval);
  fclose(fp);

  errflag = 0; /* no errors detected yet */

  /* If reconstruction will be rapid, do it as an error check. */ 
  if (RNGcount1 < 1000 && RNGcount2 == 0) {
    init_rng(rng, RNGalg, RNGseed1, RNGseed2, RNGcount1, RNGcount2);
    /* see if we got to the same state */
    for (i=0; i<RNGstatesize; i++) {
      if (newstate[i] != RNGstate[i]) {
        errflag = 1;
      }
    }
  } else { /* just copy in the state, and update mru_rng */
    /* first, some special hacks for alg 1 */
    if (RNGalg == 1) {
      /* tell random() we've got a 31-word statefile */
      RNGstate[0] = MRNGSTATE0;
      setstate(RNGstate);
      /* and modify random()'s hidden state to correspond to the RNGcount */
      m = RNGcount1%31 + RNGcount2*(BILLION%31);
      for (i=0 ; i<(m%31); i++) {
        r = random();
      }
    }
    /* copy state array */
    for (i=0; i<RNGstatesize; i++) {
      RNGstate[i] = newstate[i];
    }
    mru_rng = rng; /* remember this rng, for use by mrandom() and frandom() */
  }

  /* see if RNG state looks ok */
  errflag = errflag || !checkrandom(rng);

  /* Check nextval() operation, and verify RNGnextval */
  if (nextval(rng) != nextval(rng) || RNGnextval != nextval(rng)) {
    errflag = 1;
  }

  if (errflag) {
    fprintf(stderr,
	"Warning: RNG statefile is inconsistent.  Did you edit it?\n");
    fprintf(stderr,
	"If not, check your program to make sure you:\n");
    fprintf(stderr,
	"   1. use frandom() or mrandom(m), not random();\n");
    fprintf(stderr, "   2. use init_rng(alg, seed1, seed2, count1, count2),");
    fprintf(stderr, " not srandom(seed);\n");
    fprintf(stderr,
	"   3. use restart_rng(filename), not setstate(state); and\n");
    fprintf(stderr,
	"   4. don't overwrite the private storage of mrandom.\n");
    fflush(stderr);
  }
  return(!errflag);
} /* end restart_rng */

/* Generate a length-n vector v[] of uniformly-distributed variates, where
 * 0.0 <= v[i] < 1.0 for i=0,...n-1.  Return the value of v[0]. 
 * 
 * Special-case parameter values: if rng==0, use the most recently
 * initialized or restarted RNG; if n==0, return one random variate
 * and don't write into v[].
 */
extern double frandomrv (rng,n,v)
RNGdata *rng;
long n;
double v[];
{
  long i,j; /* temps */
  long x[64]; /* a place to store a block of random long ints */
  double vdef[1]; /* default result vector */ 

  /* handle special-case parameter values */
  setrng(rng);
  if (n==0) {
    xrandomrv(rng,1,x);
    return ((double)x[0] / RNGrange);
  }

  /* get uniform variates in blocks of 64 */
  for (i=0;i+64<n;i+=64) {
    xrandomrv(rng,64,x);
    for (j=0;j<64;j++) {
      v[i+j] = ((double)x[j])/ RNGrange;
    }
  }

  /* get the last (partial) block, if any */
  if (i < n) {
    xrandomrv(rng,n-i,x);
    for (j=0;i+j<n;j++) {
      v[i+j] = ((double)x[j])/ RNGrange;
    }
  }

  return( v[0] );

} /* end frandomrv */

/* Generate a length-n vector v of random longs, uniformly distributed
 * in the range 0..m-1, using the indicated rng.  Return a copy of the
 * first random variate, v[0].
 *
 * Special-case parameter values: if rng==0, use the RNG that was
 * the most-recently initialized or restarted; if n==0, return one
 * random variate and don't write into v[]. 
 *
 * Our code does not have a deterministic bias for any m, unlike the
 * typical "good" code
 *		(int) floor( frandom() * (double) m )
 * or the commonly-used, but hazardous (because it exposes the flaws
 * in many RNGs) code 
 *		random()%m
 * We remove the bias by making multiple calls (on rare occasions)
 * to the underlying generator.  The expected number of RNG calls
 * is upper-bounded by n/(1 - (RNGrange%m)/RNGrange) < 2n.
 * 
 * The program is aborted, with an error message, if m == 0 or if
 * m exceeds the range of the RNG.  Note: m is treated as an unsigned long,
 * so mrandomr(rng,1<<31) returns a full-range (31-bit) integer if the
 * currently-selected RNG has a 31-bit range.
 *
 * The program will also abort, again with an error message to stderr,
 * if the generator is behaving so non-randomly that our multiple-call
 * bias-removal algorithm makes an excessive number of calls to the
 * underlying generator.
 */
extern long mrandomrv (rng,m,n,v)
RNGdata *rng;
long m,n,v[];
{
  long i,yield; /* temps */
  long ncalls; /* counts number of RNG calls during this mrandomrv call */
  long vdef[1]; /* default value for v */
  char rngdesc[RNGIDSTRLEN]; /* for a describe_rng() call */

  /* we can avoid some calcs&tests if m and rng range are unchanged */ 
  static long lastm=0,lastrangem1=0; /* m and RNGrangem1 on last call */
  static long maxv; /* largest "useful" RNG output for this m */
  static double ifreq; /* multiplicity of RNG -> mrandomr() mapping */
  double dm; /* floated value of m */

  /* handle special-case parameters */
  setrng(rng);
  if (n==0) {
    n = 1;
    v = vdef;
  }

  /* see if m or RNGrange have changed recently */
  if (m != lastm || RNGrangem1 != lastrangem1) {
    /* yes, save current m and RNGrangem1 ... */
    lastm = m;
    lastrangem1 = RNGrangem1;
    /* ... and recalculate ifreq and maxv */
    if ( m < 1 || (m-1) > RNGrangem1) { /* m in range? */ 
      /* watch special case: range may be MAXLONG+1, so m could be < 0 */
      if (RNGrangem1 != MAXLONG || m != ~MAXLONG) {
        fprintf(stderr,
	   "Error: mrandom() argument value, %ld, is out of range.\n", m);
        fprintf(stderr,
	   "The range of this RNG is %.0f.\n", RNGrange);
        fflush(stderr);
        exit(1);
      }
    }
  
    /* Compute ifreq = multiplicity of RNG -> mrandom(m) mapping,
     * and maxv = the largest "useful" RNG output for this m.
     *
     * You might want to rewrite this code if you don't have FP hardware.
     */
    if (m != ~MAXLONG) {
      dm = (double) m;
    } else {
      dm = 1.0 + (double)MAXLONG;
    }
    ifreq = (long) floor(RNGrange/dm);
    maxv = RNGrangem1 - irint( RNGrange - ifreq*dm );
  }

  /* the expected # of calls to underlying RNG is n/(1-xdiscard/range) < 2n */

  ncalls = 0; /* number of RNG calls so far */
  for (yield=0; yield<n;  ) {

    /* fill (or re-fill) v[] */
    xrandomrv(rng,n-yield,&v[yield]);

    /* make sure ncalls doesn't get ridiculously large */
    ncalls += n-yield;
    if (ncalls > 3*n+300) {
      /* For yield ==n, mean(ncalls) < 2*n; std dev < sqrt(2*n);
       * If ncalls > 3*n + 300, we are dozens of stddevs away from mean!
       */
      fprintf(stderr, "Trouble in mrandomrv, m = %ld\n",m);
      fprintf(stderr, "Aborting program: this RNG is apparently stuck!\n");
      fprintf(stderr, describe_rng(rng,rngdesc));
      exit(1);
    }

    /* find first out-of-range v[], if any */
    for (  ; yield<n; yield++) {
      if (v[yield] > maxv) {
        break;
      }
    }

    /* move in-range values to front of v[] */
    for (i=yield+1; i<n; i++) {
      if (v[i] <= maxv) {
        v[yield] = v[i];
        yield++;
      }
    }
  }

  /* map v[] values, with ifreq:1 multiplicity, into the range 0..m-1 */
  for (i=0; i<n; i++) {
    v[i] = (long) ((double)v[i] / ifreq);
    /* Note: some C compilers might do (long) conversion by rounding.
     * If so, you'll have to write something like
     *    v[i] = (long) floor ((double)v[i] / ifreq);
     * I didn't write it this way because a floor() call takes three
     * microseconds (!) on my Sun4.
     */
  }

  return(v[0]);

} /* end mrandomrv */

/* Save the RNG state to a statefile.
 * Check to be sure the RNG can be restarted by calling restart_rng().
 * Return 0 if a problem is detected, printing an error message on stderr.
 * Otherwise return 1.
 */
extern int save_rng(rng,filename)
RNGdata *rng;
char *filename;
{
  FILE *fp;
  long i;
  unsigned short *sarray;

  if (!rng) { /* avoid following null pointers */
    fprintf(stderr,
	 "Bad argument (rng==0) to save_rng.\n");
    fflush(stderr);
    return(0);
  }

  /* write the statefile */
  fp = fopen(filename, "w");
  if (!fp) {
    fprintf(stderr,
	 "Trouble opening RNG statefile %s for writing.\n", filename);
    fflush(stderr);
    return(0);
  }
  fprintf(fp, RNGfileLINE0, RNGalg);
  setRNGparams(rng,RNGalg); /* (only called for its error diagnostic) */
  switch (RNGalg) {
    case 0:	fprintf(fp, RNGfileLINE0_0);
		break;
    case 1:	fprintf(fp, RNGfileLINE0_1);
		break;
    case 2:	fprintf(fp, RNGfileLINE0_2);
		break;
    case 3:	fprintf(fp, RNGfileLINE0_3);
		break;
    case 4:	fprintf(fp, RNGfileLINE0_4);
		break;
    case 5:	fprintf(fp, RNGfileLINE0_5);
		break;
    default:	return(0);
  }
  fprintf(fp, RNGfileLINE1, RNGseed1, RNGseed2);
  fprintf(fp, RNGfileLINE2, RNGcount2, RNGcount1);
  fprintf(fp, RNGfileLINE3);
  if (RNGalg == 4) { /* write as shorts, for code portability */
    sarray = (unsigned short *) RNGstate;
    fprintf(fp, RNGfileLINEn_4, sarray[0], sarray[1], sarray[2]);
  } else {
    for (i=0; i+3<RNGstatesize; i+=4) {
      fprintf(fp, RNGfileLINEn,
	    RNGstate[i], RNGstate[i+1], RNGstate[i+2], RNGstate[i+3]);
    }
    for (   ; i<RNGstatesize; i++) {
      fprintf(fp, RNGfileLINEnn, RNGstate[i]);
      if (i == RNGstatesize-1) fprintf(fp,"\n");
    }
  }
  fprintf(fp, RNGfileLINE12, nextval(rng));
  fclose(fp);

  /* Return after checking that state was saved correctly. */
  return( restart_rng(rng,filename) );

} /* end save_rng */

/* end mrandom.c */
