#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_zeta		WITH_2_ARGS(
	mp_int,		n,
	mp_float,	x
)
/*
Returns x = Riemann zeta(n) for n > 1 (the sum from i = 1 to infinity of i^-n). 
Uses the Euler-Maclaurin series unless n = 2, 3, 4, 6, or 8, so the
space required is O(t^2).  (More accurately, space = nl * t/2 + O(t) words,
where nl is the number of terms used in the asymptotic expansion,
nl = O(t).)  The time taken is O(t^3).
*/
{

    /* BUG: zeta(5) gives different values for different b and t */

    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_acc_float	temp1, temp2;

    DEBUG_BEGIN(DEBUG_ZETA);
    DEBUG_PRINTF_1("+zeta {\n");
    DEBUG_PRINTF_2("n = %d\n", n);

    if (n <= 1)
	mp_error("mp_zeta: n is less than or equal to 1");

    b = mp_b(xp);
    t = mp_t(xp);

    round = MP_TRUNC;

    new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);

    mp_acc_float_alloc_2(b, new_t, temp1, temp2);
    mp_change_up();

    if (n <= 8 && n % 2 == 0)
    {
	/*
	Here zeta(n) == pi^n / known constant.
	*/

	static mp_int		data[] = { 6, 90, 945, 9450 };


	mp_pi(temp1);
	mp_abs_int_power(temp1, n, temp1);

	mp_div_int_eq(temp1, data[n / 2 - 1]);
    }
    else
    {
	/*
	See if n is very large.  We can return 1 to required precision if
	2^n >= b^(new_t - 1).
	*/

	if (n < mp_times_log2_b(new_t - 1, b))
	{
	    /*
	    If n is 3, use Gosper's series for zeta(3).  This is faster and uses
	    less space than the Euler-Maclaurin series.  Series for other odd
	    n are given by H. Riesel in BIT 13 (1973), 97-113.
	    */

	    mp_ptr_type		temp2_ptr = mp_acc_float_ptr(temp2);
	    mp_int		i;


#define	fix_pointers()		if (mp_has_changed()) \
				    temp2_ptr = mp_acc_float_ptr(temp2);

	    if (n == 3)
	    {
		mp_q_to_mp(1, 4, temp1);
		mp_q_to_mp(5, 4, temp2);

		i = 1;

		fix_pointers();

		do
		{

		    mp_length	mul_t = new_t + mp_expt(temp2_ptr);

		    if (mul_t < 2)
			mul_t = 2;
		    else if (mul_t > new_t)
			mul_t = new_t;

		    mp_t(temp2_ptr) = mul_t;

		    mp_mul_q_eq(temp2, -i, 4 * i + 2);
		    mp_mul_double_q(temp2, i, i, i + 1, i + 1);


		    /*
		    Fix t of temp2 for addition.
		    */

		    fix_pointers();
		    mp_t(temp2_ptr) = new_t;

		    mp_add_eq(temp1, temp2);

		    i++;
		    fix_pointers();
		} while (!mp_is_zero(temp2_ptr) && mp_expt(temp2_ptr) > -new_t);

		/*
		Now temp1 has zeta(3) - 1.
		*/
	    }

	    else
	    {

		/*
		Here we use the Euler-Maclaurin series.
		*/

		mp_int		q, kt, nt, nl, nm;


		mp_set_sign(mp_acc_float_ptr(temp1), 0);

		q = 1;
		kt = new_t;
		nt = n - 2;

		nl = -1;

		do
		{
		    /*
		    Start of loop to estimate nl = # of terms needed in 
		    asymptotic expansion, and nm = # of terms needed in
		    finite sum.
		    */

		    nl++;
		    nm = 5 * nl + 2;

		    for (i = 0; i < 2; i++)
		    {
			nt++;

			while (q >= MAX_INT / nt)
			{
			    q /= b;
			    q++;
			    kt++;
			}

			q *= nt;
		    }

		} while (kt >= mp_change_base(b, nm, n - 1) +
			    mp_change_base(b, 6 * nm + nm / 4, 2 * nl + 2));


		/*
		If nl <= 0, there is no need to compute any bernoulli numbers.
		*/

		if (nl > 0)
		{
		    mp_acc_float	temp3;
		    mp_float_array	bern = mp_float_array_alloc(nl,
								    b, new_t);

		    mp_acc_float_alloc(b, new_t, temp3);

		    mp_bern(bern, -nl);

		    mp_q_to_mp(n, 2 * nm, temp2);
		    mp_div_int_eq(temp2, nm);


		    /*
		    Sum Euler-Maclaurin asymptotic series first.
		    */

		    for (i = 0; i < nl; i++)
		    {
			mp_copy(mp_array_element(bern, i), temp3);

			mp_mul_eq(temp3, temp2);
			mp_add_eq(temp1, temp3);

			mp_mul_double_q(temp2, n + 2 * i + 1,
					 n + 2 * i + 2, 2 * i + 3, 2 * i + 4);

			mp_mul_double_q(temp2, 1, 1, nm, nm);
		    }

		    /*
		    Add integral approximation and multiply by nm^(1 - n).
		    */

		    mp_add_q_eq(temp1, 1, n - 1);
		    mp_scale(temp1, nm, 1 - n);

		    mp_acc_float_delete(temp3);
		    mp_delete_float_array(bern);

		    xp = mp_ptr(x);
		    temp2_ptr = mp_acc_float_ptr(temp2);
		}

		/*
		Add finite sum in forward direction so we can reduce t more
		easily than if backward direction were used.
		*/

		mp_set_sign(temp2_ptr, 0);
		mp_set_digits_zero(mp_digit_ptr(temp2_ptr, 0), new_t);

		for (i = 2; i <= nm; i++)
		{
		    /*
		    Reduce t for i^-n computation if possible.
		    */

		    mp_length	t = new_t + mp_expt(temp2_ptr);

		    if (t < 2)
			t = 2;

		    mp_t(temp2_ptr) = t;

		    /*
		    Compute i^-n by mp_scale().
		    */

		    mp_int_to_mp(1, temp2);
		    mp_scale(temp2, i, -n);

		    /*
		    Halve last term in finite sum.
		    */

		    if (i == nm)
			mp_div_int_eq(temp2, 2);

		    
		    /*
		    Restore t of temp2 for addition.
		    */

		    fix_pointers();
		    mp_t(temp2_ptr) = new_t;

		    if (mp_is_zero(temp2_ptr))
			break;

		    mp_add_eq(temp1, temp2);

		    fix_pointers();
		}
	    }
	}

	/*
	Final addition.
	*/

	mp_add_int_eq(temp1, 1);
    }

    round = save_round;
    mp_move_round(temp1, x);

    xp = mp_ptr(x);

    if (mp_cmp_q(x, 1, 2) < 0 || mp_cmp_int(x, 2) >= 0)
	mp_bug("mp_zeta: error occurred, result incorrect");
    
    /*
    Rounding down may have given result less than one.
    */

    if (mp_cmp_int(x, 1) < 0)
	mp_int_to_mp(1, x);

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_change_down();

    DEBUG_1("-} x = ", xp);
    DEBUG_END();

    return x;
}
