/******************************************************************************
  order_units_fundamental_calc.c
******************************************************************************/

#include "kant.h"

t_void
order_units_fundamental_calc WITH_3_ARGS (
	order,		ord,
	t_real,		reg_lbound,
	integer_small,	index_ubound
)
/******************************************************************************
 
Description:	Computes a maximal system of fundamental units
		using a density theorem of Frobenius
		respectivly N. Cebotarev.
		    
Calling sequence:
 
	order_units_fundamental_calc(ord,reg_lbound,index_ubound);

	ord		: order		= t_handle of order
	reg_lbound	: t_real	= lower regulator bound. The user
					  does not need to specify reg_lbound
					  (reg_lbound = MEM_NH).
	index_ubound	: integer_small	= upper index bound. The user does
					  does not need to specify index_ubound
					  (index_ubound = MEM_NH).

	If both of reg_lbound and index_ubound are not specified
	the program computes a lower regulator bound to get
	an upper index bound.

History:

	93-03-01 KW	modular arithmetic + order_mult_assure
	93-02-26 KW	new parameter upper index bound
			adjust real precision
	92-12-09 KW	relations instead of MLLL
	92-12-04 KW	new p'th power test
	92-10-06 KW	written
 
******************************************************************************/
{
	block_declarations;

	anf_elt		alpha,beta,gamma,delta,omega,kasimir;
	anf_ideal	pricebo;
	dyn_arr_handle	primes;
	integer_big	tempa,tempb,tempc,q,s;
	integer_small	bv,i,j,k,m,p,pp,np,r;
	matrix		matum,mutam;
	matrix		transR,transZ;
	order		ordmax;
	t_handle	R,Z;
	t_real		tempu,tempv;
	vector		cnits,mnits,onits,units,vnits;

	if (anf_print_level > 0) printf("Computing a system of fundamental units...\n");

	if (order_units_are_fund(ord)) goto FINIS;

	order_mult_assure(ord);
	order_reals_assure(ord);

	R = order_reals(ord);
	r = order_r(ord);
	Z = structure_z;

	if (reg_lbound || index_ubound)
	{
		order_torsion_subgroup_calc(ord);
		if (reg_lbound) reg_lbound = real_incref(reg_lbound);
	}
	else
	{
		reg_lbound = order_reg_lbound(ord,0,MEM_NH);
                order_relation_eval_torsion_subgroup(ord);
	}
	order_units_indep_calc(ord);

	if (anf_print_level > 0)
	{
		printf("regulator of maximal system is ");
		real_write(R,order_reg(ord),20); printf(".\n");
	}

/*
**	Compute an upper bound s for the index.
*/
	if (reg_lbound)
	{
		tempv = real_divide(R,order_reg(ord),reg_lbound);
		s = conv_real_to_int_floor(R,tempv);
		if (!integer_is_single(s))
			error_internal("order_units_fundamental: upper bound for index is an integer_big.");
		if (index_ubound) s = integer_min(s,index_ubound);
		real_delete(&reg_lbound);
		real_delete(&tempv);
	} else s = index_ubound;
	if (s==1) goto FINIS;

/*
**	Compute all primes <= s.
*/
	tempa = integer_add(s,1);
	primes = integer_read_primes(tempa);
	integer_delref(tempa);
	pp = dyn_arr_curr_length(primes);

/*
**	We need the maximal overorder of ord.
*/
	ordmax = (order_is_maximal(ord)) ? order_incref(ord) : order_maximal(ord);

/*
**	Enlarge the unit group.
*/
	units = vec_new(r+1);
	onits = vec_new(r+1);
	cnits = vec_new(r+1);
	vec_elt(onits,0) = anf_elt_incref(order_torsion_unit(ord));
	vec_elt(units,0) = anf_elt_incref(order_torsion_unit(ord));
	vec_elt(cnits,0) = anf_elt_con(ord,order_torsion_unit(ord));

	while (pp)
	{
		if (anf_print_level > 0) cay_print("New upper index bound: %d (%d primes below).\n",s,pp);
/*		order_units_lll_reduce(ord);
*/

		for (i=1;i<=pp;i++)
		{
			p = dyn_arr_element(primes,i-1);
			if (anf_print_level > 0) cay_print("Check p = %d\n",p);
                        for (j=1;j<=r;j++)
			{
				vec_elt(units,j) = anf_elt_incref(order_unit(ord,j));
				vec_elt(onits,j) = anf_elt_incref(order_unit(ord,j));
				vec_elt(cnits,j) = anf_elt_con(ord,order_unit(ord,j));
    			}
			matum = mat_ring_create_id(Z,r+1);
/*
**			The torsion unit of ord is a p'th power,
**			iff gcd(order_torsion_rank,p) = 1.
*/
			j = (order_torsion_rank(ord) % p) ? 1 : 0;
			for (   ;j<=r;j++)
			{
				if (j)
				{
					mnits = mat_ring_col_to_vector(Z,matum,j+1);
					kasimir = anf_elt_power_product(ord,cnits,mnits);
					vec_delete(Z,&mnits);
					while ((bv = anf_elt_unit_is_mth_power(ord,kasimir,p,&omega)) == -1)
					{
						anf_elt_delete(ord,&kasimir);
/*
**						Delete old stuff...
*/
						for(k=0;k<=r;k++) anf_elt_delete(ord,&vec_elt(cnits,k));
						
/*
**						New precision
*/
						np = order_real_prec(ord);
						np <<= 1;
						if (anf_print_level > 0) printf("New precision is %d.\n",np);
						order_reals_delete(ord);
						order_reals_set(ord,np);
						order_reals_create(ord);
/*
**						Recalc...
*/
						dyn_arr_element(cnits,0) = anf_elt_con(ord,order_torsion_unit(ord));
						for (k=1;k<=r;k++) vec_entry(cnits,k) = anf_elt_con(ord,order_unit(ord,k));
						mnits = mat_ring_col_to_vector(Z,matum,j+1);
						kasimir = anf_elt_power_product(ord,cnits,mnits);
						vec_delete(Z,&mnits);
					}
					anf_elt_delete(ord,&kasimir);
					if (bv) goto EVAL;
				}
				if (j == r) continue;
				pricebo = anf_ideal_cebotarev(ord,ordmax,p,vec_elt(units,j),&q);
				anf_ideal_norm(ord,pricebo,&tempa,&tempb);
				tempb = integer_subtract(tempa,1);
				tempc = integer_div(tempb,p);
				for (k=j+1;k<=r;k++)
				{
                    			delta = m = -1;
					do
                    			{
						m++;
						anf_elt_delete(ord,&delta);
						alpha = anf_elt_power_mod(ord,q,vec_elt(units,j),m);
       	            				beta  = anf_mult_mod(ord,q,alpha,vec_elt(units,k));
               	    				gamma = anf_elt_power_mod(ord,q,beta,tempc);
						delta = anf_elt_subtract(ord,gamma,1);
						anf_elt_delete(ord,&alpha);
						anf_elt_delete(ord,&beta);
						anf_elt_delete(ord,&gamma);
                    			}
					while(!anf_elt_in_ideal(ord,delta,pricebo));
					anf_elt_delete(ord,&delta);
					anf_elt_delete(ord,&vec_elt(units,k));
               				mat_ring_col_add(Z,matum,j+1,k+1,m,1,r+1);
					mutam = mat_z_mod(Z,matum,p);
					mat_delref(Z,&matum);
					matum = mutam;
					mnits = mat_ring_col_to_vector(Z,matum,k+1);
					vec_elt(units,k) = anf_elt_power_product(ord,onits,mnits);
					vec_delete(Z,&mnits);
				}
				anf_ideal_delete(ord,&pricebo);
				integer_delref(q);
				integer_delref(tempa);
				integer_delref(tempb);
				integer_delref(tempc);
			}
			for (j=1;j<=r;j++)
			{
				anf_elt_delete(ord,&vec_elt(units,j));
				anf_elt_delete(ord,&vec_elt(cnits,j));
			}
			mat_delref(Z,&matum);
		}
		pp = 0;
		continue;

/*
**		Enlarge sublattice using relations
*/

EVAL:		if (anf_print_level > 0) printf("\nEnlarging sublattice...\n");

		mnits = mat_ring_col_to_vector(Z,matum,j+1);
		vec_entry(mnits,1) = -p;
		transZ = lat_sublattice_enlarge(mnits);
		vec_delete(Z,&mnits);

		vnits = vec_new(r+1);
		for (i=1;i<=r;i++)
		{
			vec_entry(vnits,i+1) = anf_elt_incref(order_unit(ord,i));
		}
		vec_entry(vnits,1) = omega;
		
		for (i=1;i<=r;i++)
		{
			anf_elt_delete(ord,&order_unit(ord,i));
			mnits = mat_ring_col_to_vector(Z,transZ,i);
			order_unit(ord,i) = anf_elt_power_product(ord,vnits,mnits);
			vec_delete(Z,&mnits);
		}
		mat_delref(Z,&transZ);
		vec_delete(ord,&vnits);

		for (j=1;j<=r;j++)
		{
			anf_elt_delete(ord,&vec_elt(units,j));
			anf_elt_delete(ord,&vec_elt(cnits,j));
		}
		mat_delref(Z,&matum);

		s = s/p;
		while ((pp) && (dyn_arr_element(primes,pp-1) > s)) pp--;
/**/		if (order_reg_known(ord)) real_delete(&order_reg(ord));
/**/		order_write(ord);
	}
	vec_delete(ord,&units);
	vec_delete(ord,&cnits);
	dyn_arr_delete(&primes);
	order_delete(&ordmax);

/*
**	Delete old data...
*/
	if (order_units_logs_known(ord))
	{         
		mat_delref(R, &order_units_logs(ord));
		lat_enum_delete(order_units_log_lat(ord),       
					&order_units_log_lat_env(ord));
		lat_delete(&order_units_log_lat(ord));
	}
	if (order_reg_known(ord))
		real_delete(&order_reg(ord));

FINIS:	order_set_units_are_fund(ord);	
	order_reg_assure(ord);

	return;
}



