// This may look like C code, but it is really -*- C++ -*-
// if cells don't have unit area
#ifndef UNIT_CELLS

#include "representation.h"
#include "cell.h"

extern Cell* cell[];
cellList*
Representation::neighborList(Int c, int& perimeter)
     // Returns a list of neighboring cells of the cell c 
     // There are two contact lengths that can be counted, we count the
     // exterior one
{
  Point direction = randomDirection();
  Point neighborPoint = findBoundary( c, direction);
  // neighborPoint is a point on the boundary not belonging to cell c
  
  // Find a point on the boundary belonging to cell c
  Point p, cellPoint;
  
  PointList search; search.addNeighbors( neighborPoint);
  do {
    p = search.delget();
    if (getCell(p) == c) cellPoint = p;
  } while  (!search.emptyList() && getCell(p) != c);
  
  Int neighborCell = getCell( neighborPoint);
  PointList* boundary = new PointList( neighborPoint);
  cellList* cellNeighbors = new cellList();
  int contactLength = 1;
  perimeter = 0;
  FPoint directionToNeighbor = cellPoint.relDirectionFloat( neighborPoint);
  Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
  do {
    // Step through cellPoint and neighborPoint 
    p = cellPoint.nextPointInRotation( neighborPoint);
    if ( getCell( p) == c) cellPoint = p;
    else {
      neighborPoint = p;
      if ( getCell( p) != neighborCell) {
	// new cell neighbor found
	cellNeighbors->insert( cell[neighborCell], boundary, contactLength,
			      directionToNeighbor);
	perimeter += contactLength;
	neighborCell= getCell( neighborPoint);
	boundary = new PointList( neighborPoint);
	contactLength = 0;
	directionToNeighbor.zeroPoint();
      }
      else boundary->insert( p); // exterior boundary
      contactLength ++;
    }
    directionToNeighbor += cellPoint.relDirectionFloat( neighborPoint);
  } while ( (startCellPoint != cellPoint) ||
	   (startNeighborPoint != neighborPoint));
  cellNeighbors->insert( cell[neighborCell], boundary, contactLength - 1,
			directionToNeighbor);
  perimeter += contactLength - 1;
  return cellNeighbors;
}

int Representation::getArea(Int c)
{ // Compute and return area of cell c, just totals points over the entire
  // representation 
  int area=0;
  for (int y = 0; y < height; y++)
    for(int x = 0; x < width; x++)
      if ( *(representation + y * width + x) == c) area++;
  return( area );
}

void 
Representation::swapCells( Int c1, Int c2, FPoint direction)
     // swap the locations of the adjoining cells c1 and c2
{
  const float WeakDistance = 0.0;
//  cout << " Swap try " <<c1<< " and " << c2 << "--" <<
//    direction.distanceSquared(Point(0,0)) << '\n';
//  swapAttempt++;
  if (direction != Point(0,0) && direction.distanceSquared(Point(0,0)) < 
      WeakDistance) return; // very weak direction
  
  if (cell[ c2]->dummy) {
	// do not swap with dummy cells
    return; 
  }

  if (direction == Point(0,0)) {
    enum { LARGE_INT = 100};
    Point direction = ( location[ c2] - location[ c1]) * LARGE_INT;
  }  

  if (direction == Point(0,0)) direction = randomDirection();
  Point neighborPoint = findBoundary( c1, direction);
  // neighborPoint is a point on the boundary not belonging to cell c1
  
//  cout << neighborPoint << c1 << " " << c2 << " in swap \n";
  if ( getCell( neighborPoint) != c2) {
    // check if adjoining
    // Find a point on the boundary belonging to cell c1
//    cerr << "bad choice of direction in swap " << c1 << direction << c2 <<
//      '\n'; // TODO look it up, why should this happen 4/15/92
    Point p, cellPoint;
    
    PointList search; search.addNeighbors( neighborPoint);
    do {
      p = search.delget();
    } while  (!search.emptyList() && getCell(p) != c1);
    cellPoint = p;
    
    Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
    do {
      // Step through cellPoint and neighborPoint 
      p = cellPoint.nextPointInRotation( neighborPoint);
      if ( getCell( p) == c1) cellPoint = p;
      else neighborPoint = p;
    } while (( (startCellPoint != cellPoint) ||
	      (startNeighborPoint != neighborPoint)) && 
	     getCell( neighborPoint) != c2);
    if ( getCell( neighborPoint) != c2) {
      cerr << c1 << ", " << c2 ;
      error(" cells not adjoining in swap ");
      return; // failure, but we don't want to try again
    }
  }
  int x,y;
  for ( y = 0; y < height; y++)
    for( x = 0; x < width; x++)
      if ( *(representation + y * width + x) == c1)
	*(representation + y * width + x) = -c1;
  for ( y = 0; y < height; y++)
    for( x = 0; x < width; x++)
      if ( *(representation + y * width + x) == -c1)
	*(representation + y * width + x) = c2;
      else if ( *(representation + y * width + x) == c2)
	*(representation + y * width + x) = c1;
  Point loc = location[ c1];
  location[ c1] = location[ c2];
  location[ c2] = loc;

  transferAreaBetweenCells( c1, c2, (getArea( c1) - getArea(c2))/2);
  setLocationToCentreofCell( c1);
  setLocationToCentreofCell( c2);
}

int 
Representation::pointEnergy( Point p, Int c) const
{ // the adhesive energy at point p if occupied by a point of cell c
  int energy = 0;
  PointList search; 
  search.addNeighbors( p);
  do { 
    Point q = search.delget();
    if ( getCell( q) == c) energy += 1;
  } while  (!search.emptyList());

  return energy;
}

void
Representation::configure( Int c)
{ //move towards higher adhesive energy
  bool swapped = FALSE; //keeps track if any points on the boundary were
			//swapped 
  int areaDiscrepancy[ MAX_NUMBER_CELLS ];
  for (int i=0;i<= Cell::numberDefined; i++) areaDiscrepancy[i]=0;

  Point neighborPoint = findBoundary( c, randomDirection());
  // neighborPoint is a point on the boundary not belonging to cell c
  
  Point p, cellPoint;
  PointList search; search.addNeighbors( neighborPoint);
  do { // Find a cell neighbor
    p = search.delget();
  } while  (!search.emptyList() && getCell(p) != c);
  cellPoint = p;
  
  Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
  PointList cellBoundary[ NumberOfNeighbors], neighborBoundary[ NumberOfNeighbors];
  // lists of points on the boundary classified according to their energy or
  // neighbors. cellBoundary[i] contains all the point on the cell boundary
  // with i cell neighbors. neighborBoundary[i] contains all the point
  // neighborly to the cell boundary (external) with i cell neighbors. 
  // Thus cellBoundary[1] and neighborBoundary[5] would be the most probable
  // swap, the cellBoundary[0] list is not used.
  do {
    // Step through cellPoint and neighborPoint 
    p = cellPoint.nextPointInRotation( neighborPoint);
    if ( getCell( p) == c) {
      cellPoint = p;
      cellBoundary[ pointEnergy(p, c)].insert( p);
    }
    else {
      neighborPoint = p;
      neighborBoundary[ pointEnergy(p, c)].insert( p);
    }
  } while (( (startCellPoint != cellPoint) ||
	    (startNeighborPoint != neighborPoint)));

  initializeTags();
  // try to find better locations for exterior points
  int cellList = 1, neighborList = 5;
  while (cellList < neighborList) {
    while ( !cellBoundary[ cellList].emptyList() && ! neighborBoundary[ neighborList].emptyList()) {
      // find first eligible cellPoint
      do 
	cellPoint = cellBoundary[ cellList].delget();
      while ( ! ( cellBoundary[ cellList].emptyList() || 
		  ( *(tag + cellPoint.arrayindex(width)) == FALSE && 
		    canStealPoint( cellPoint))));

      if (DEBUG) cerr << "Candidate Cell Point " << cellPoint << "\n";
      do
	neighborPoint = neighborBoundary[ neighborList].delget();
      while (!( neighborBoundary[ neighborList].emptyList() ||
		(*(tag + neighborPoint.arrayindex(width)) == FALSE && 
		 canStealPoint( neighborPoint))));
      if (DEBUG) cerr << "Candidate Neighbor Point " << neighborPoint << "\n";

      if ( neighborList > cellList + cellPoint.ifNeighbor( neighborPoint)) {
	// succesful. If the two points are neighbors then the difference
	// must be more
	if (DEBUG) cerr << "Good energy " << "\n";
	Int neighborCell = getCell( neighborPoint);
	if ( canStealPoint( neighborPoint)) {
          // neighborPoint can be changed
	  putCell( neighborPoint, c);
	  // we do not want to form bridges, so check if this does
	  if (canStealPoint( neighborPoint)) {
	    // Find a cell which can be given cellPoint
	    search.deleteList(); 
	    search.addNeighbors( cellPoint);
	    int maxEnergy = -99;
	    Int bestCell;
	    do { // Find a non-cell neighbor
	      p = search.delget(); 
	      // Find the non-cell neighbor which would do best at this spot
	      if ( getCell(p) != c) {
		int e = pointEnergy( cellPoint, getCell(p));
		if ( maxEnergy < e) { 
		  maxEnergy = e; 
		  bestCell = getCell( p);
		}
	      }
	    } while  (!search.emptyList());
	    
	    if (canStealPoint( cellPoint)) {
	       //does not form a bridge with other neighboring points
		putCell( cellPoint, bestCell);
		if (canStealPoint( cellPoint)) {
		  swapped = TRUE;
		  tagPointAndNeighbors( cellPoint);
		  tagPointAndNeighbors( neighborPoint);
		  if (neighborCell != bestCell) {
		    areaDiscrepancy[ neighborCell]++; // has to be incremented
		    areaDiscrepancy[ bestCell]--; // has to be decremented
//		    incrementCellArea( neighborCell, Point(0,0));
//		    decrementCellArea(  bestCell, Point(0,0));
		  }
		}
		else {
		  putCell( cellPoint, c);
		}
	    }
	  }
	  else {
	    putCell( neighborPoint, neighborCell);
	  }
	}
      }
      else { // if no energy gain
	cellBoundary[ cellList].deleteList();
	neighborBoundary[ neighborList].deleteList();
      }
    }
    if (cellBoundary[ cellList].emptyList()) cellList += 1; 
    if (neighborBoundary[ neighborList].emptyList()) neighborList -=1;
  }
  for (i=2;i<= Cell::numberDefined; i++) 
    if (areaDiscrepancy[  i] > 0)
      for (int a=1;a<= areaDiscrepancy[  i]; a++)
	incrementCellArea( a, Point(0,0));
    else
      for (a= 1;a<= -areaDiscrepancy[  i]; a++)
	decrementCellArea( a, Point(0,0));
  if (swapped) {
    configure( c);
  }
}

void Representation::growCell(Int c, int areaInc, FPoint direction )
{
  if (areaInc < 0) 
    shrinkCell( c, -areaInc, direction); 
  else
    for (int ar = 1; ar <= areaInc; ar++) 
      incrementCellArea( c, direction.closestPoint()); 
}

bool
Representation::central( Point p, Int c) const
// if point p is inside the cell, has no other neighbor
{
  if (pointEnergy(p,c) == 6) return TRUE;
  else return FALSE;
}

bool
Representation::bridge( Point p) const
// Check if point p is the only connection between two parts of the cell
// at point p
{
  enum {BRIDGE=TRUE, NOT_BRIDGE= FALSE};
  Int c = getCell( p);
  PointList neighList; 
  neighList.addNeighbors( p);
  Point scan, c1, c2, n1; // scan goes thru the list, 
                              // c1,c2 always belongs to type c, 
                              // n1 is not of type c
  scan = neighList.delget();
  // All these neighbors should from one connected component without p
  // belonging to the cell

  // Find the first cell neighbor
  while  (!neighList.emptyList() && getCell( scan) != c)
    scan = neighList.delget();
  c1 = scan;

  if (neighList.emptyList()) 
    if (getCell( c1) != c) return BRIDGE; // Single point cell
    else return NOT_BRIDGE; // one neighboring point belongs to the cell

  scan = neighList.delget();
  while ( getCell( scan) == c && !neighList.emptyList()) 
    scan = neighList.delget();
  n1 = scan;
  // this advances past the first cell component which began at c1
  
  if ( neighList.emptyList())
    return NOT_BRIDGE;
  // just one path out of p

  while  (!neighList.emptyList() && getCell( scan) != c)
    scan = neighList.delget(); 
  c2 = scan; // start of possibly the second cell component
  // unless it joins up with the first
  
  if (neighList.emptyList())
    if ( (getCell( c2) != c) || c1.ifNeighbor( c2)) 
      return NOT_BRIDGE; // one path out of p
    else return BRIDGE;
  
  // Found the next cell point in c2
  scan = neighList.delget();
  while ( getCell( scan) == c && !neighList.emptyList()) 
    scan = neighList.delget();
  // this advances past the second cell component 

  if ((getCell( scan) == c) && c1.ifNeighbor( scan))
    return NOT_BRIDGE;
  else 
    return BRIDGE;
}

void
Representation::findExtremePoints( Int c, Point direction, Point& lowPoint,
				  Point& highPoint) 
{
  // compute the inner cell boundary, for each point on the boundary compute
  // the product of the point and the perpendicular to the direction, keep
  // track of the low and high these two are the support points
  //  cout << direction << " = direction \n";
  
//  cerr << " location " << getLocation( c);
//  dumpPartOfImage( getLocation(c));
  Point neighborPoint = findBoundary( c, getLocation(c)+direction);
  // neighborPoint is a point on the boundary not belonging to cell c
  
  // Find a point on the boundary belonging to cell c
  Point p, cellPoint;
  
  PointList search; search.addNeighbors( neighborPoint);
  do {
    p = search.delget();
    if (getCell(p) == c) cellPoint = p;
  } while  (!search.emptyList() && getCell(p) != c);
  
  Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
  Point perpDirection = direction.perpendicular();
  int low = cellPoint.dotProduct( perpDirection);
  int high = low;
  PointList highList( cellPoint), lowList( cellPoint);
  do {
    // Step through cellPoint and neighborPoint 
    p = cellPoint.nextPointInRotation( neighborPoint);
    if ( getCell( p) == c) {
      cellPoint = p;
      int product = p.dotProduct( perpDirection);
      if (product > high) { 
	high = product;
	highList = PointList( p);
      }
      else 
	if (product == high) highList.insert( p);
      if (product < low) {
	low = product;
	lowList = PointList( p);
      }
      else 
	if (product == low) lowList.insert( p);
    }
    else
      {
	neighborPoint = p;
      }
  } while ( (startCellPoint != cellPoint) ||
	   (startNeighborPoint != neighborPoint));
  lowPoint = lowList.middle();
  highPoint = highList.middle();
// Make sure the two aren't the same, unless there is just one point
  while (lowPoint == highPoint && !lowList.emptyList()) 
    lowPoint =  lowList.delget();
  while (lowPoint == highPoint && !highList.emptyList()) 
    highPoint =  highList.delget();
  if (lowList.emptyList() && highList.emptyList() && lowPoint == highPoint){
    lowList.display(); 
    cerr <<"\n";
    highList.display(); 
    error("in find extreme points"); 
  }
}

Point Representation::divideCell(Int c, divideOptions dO, int childNumber,
				Point previousDivision)
{
  // Find the direction of the divide (say horizontal = (1,0))
  // FInd the supporting tangents to the cell parallel to this direction
  // divide the cell into two by growing these two extreme points until the
  // two daughter cells swallow the whole parent
  bool errorFlagged = FALSE;
  while (!cell[c]->dummy && getArea(c) != cell[c]->area) {
      cerr << "Cell number " << c << ": Actual area = " << getArea(c) << 
	", it should have been = " <<  cell[ c]->area <<'\n'; 
      error("Area discrepancy in cell while dividing.");
      errorFlagged = TRUE;
      growCell(c, cell[c]->area - getArea(c), randomDirection());
      configure(c);
    }
  if ( getArea(c) <=1) {
    cout << "Unable to divide cell " << c << ". Its area is <= 1.\n";
    return Point(0,0);
  }

  Point direction;
  switch (dO)
    {
    case HORIZONTAL:
      direction = Point( 1,0);
      break;
    case VERTICAL:
      direction = Point( 0,1);
      break;
    case PERPENDICULAR:
      direction = previousDivision.perpendicular();
      break;
    case SHORTEST:
    case ANY:
      direction = randomDirection();
      break;
    }
  Point low, high;
  findExtremePoints( c, direction, low, high);
  
  // Implementing a simple bfs on neighbors, tagging all visited points
  PointList highSearch( high);
  PointList lowSearch( low);
  Point p;
  while ( ! highSearch.emptyList() || ! lowSearch.emptyList())
    {
      while ( ! highSearch.emptyList() && 
	     getCell( p = highSearch.delget()) != c)
	; // empty while body
      if ( getCell( p) == c) {
	putCell( p, childNumber);
	highSearch.addNeighbors( p);
      }
      // else if highSearch empty then maybe disconnected cell
      while ( ! lowSearch.emptyList() && 
	     getCell( p = lowSearch.delget()) != c)
	;// empty while body
      if ( getCell( p) == c) {
	putCell( p, -childNumber);
	lowSearch.addNeighborsInReverse( p);
      }
      // else if lowSearch empty then disconnected cell
    }
  
  for (int y = 0; y < height; y++)
    for(int x = 0; x < width; x++)
      if ( *(representation + y * width + x) == -childNumber) 
	*(representation + y * width + x) = c;
  // replace -childNumber by c
  int gAc= getArea( c);
  int gAcN=getArea( childNumber);
  if ( gAc + gAcN  != cell[ c]->area && !cell[c]->dummy && !errorFlagged)
    {
      cerr << "Cell number " << c << ": Actual area = " << gAc+gAcN << 
	", it should have been = " <<  cell[ c]->area <<'\n'; 
      error("Area discrepancy in cell while dividing.");
    }
  if ( abs( gAcN -gAc) > 1) 
    // if not good division almost equal areas
   {
      setLocationToCentreofCell( c);
      setLocationToCentreofCell( childNumber);
      transferAreaBetweenCells( c, childNumber, (gAc - gAcN)/2);
    }
  setLocationToCentreofCell( c);
  setLocationToCentreofCell( childNumber);
//  configure(c); //EEEE temporary insertion on 7/13/93
//  configure(childNumber); //EEEE temporary insertion on 7/13/93
  return direction;
}

void Representation::shrinkCell(Int c, int areaDec, FPoint direction)
{
  if (areaDec < 0) 
    growCell( c, -areaDec, direction); 
  else
    for (int ar = 1; ar <= areaDec; ar++) 
      decrementCellArea( c, direction.closestPoint());
}

void
Representation::transferAreaBetweenCells( Int c1, Int c2, int area)
     // move area from c1 to c2
{
  
  if (area == 0) return;
  if (area < 0) { transferAreaBetweenCells( c2, c1, -area); return;}
  
  int areaDiff = area;
  Point direction = randomDirection(); // choose a direction
  Point neighborPoint = findBoundary( c1, direction);
  // neighborPoint is a point on the boundary not belonging to cell c1
  
  // Find a point on the boundary belonging to cell c1
  Point p, cellPoint;
  
  PointList search; search.addNeighbors( neighborPoint);
  do {
    p = search.delget();
  } while  (!search.emptyList() && getCell(p) != c1);
  cellPoint = p;
  Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
  PointList transferList;
  do {
    // Step through cellPoint and neighborPoint 
    if ( getCell( neighborPoint) == c2) {
      // try and transfer the cellPoint
      if ( getCell( cellPoint) == c1 && canStealPoint( cellPoint)) { 
	// Put all those points in a list which will be transferred
	if ( transferList.insertNonDuplicate( cellPoint)) area--;
      }
    }
    p = cellPoint.nextPointInRotation( neighborPoint);
    if ( getCell( p) == c1) cellPoint = p;
    else neighborPoint = p;
  } while (( (startCellPoint != cellPoint) ||
	    (startNeighborPoint != neighborPoint)) && area > 0);
  while (!transferList.emptyList()) {
    putCell( transferList.delget(), c2);
  }

  if (area > 0 && areaDiff > area) { 
    // if some progress was made, try again new boundaries now
    transferAreaBetweenCells( c1, c2, area); 
  }
  else if (area == areaDiff) {
    shrinkCell( c1, area, Point(0,0));
    growCell( c2, area, Point(0,0));
  }
}

bool
Representation::canStealPoint( Point p) const
 // if the point p can be stolen from its present cell without breaking it up
{
  // can't disconnect anything except environment 
  // dummy cells may not keep their area consistent but any cell which
  // executes a program must be connected
  if ( cell[ getCell( p)]->tissueType != 0) 
    if ( bridge(p) || central(p , getCell(p)))
      return FALSE; 
  return TRUE;
}

void
Representation::connect( Int c)
// ensure that cell c is connected, all its point are accessible from centre
// and the ones that aren\'t are reverted to environment
{
  Point p = getLocation( c);
  int area = 0;
  // Implementing a simple bfs on neighbors, tagging all visited points
    
  initializeTags();
  PointList search(p);
  *(tag + p.arrayindex(width)) = TRUE; // This point visited
  // The only points ever added to search or tagged belong to the cell
  while (!search.emptyList())
    {
      p = search.delget();
      area++;
      PointList temp; temp.addNeighbors(p);
      // For each of the points in the temp list if they have not been
      // tagged and they belong to the cell, add them to the search list
      while (!temp.emptyList())
	{
	  p = temp.delget();
	  if ( *(tag + p.arrayindex(width)) == FALSE // not visited yet
	      && getCell( p) == c) {
	    *(tag + p.arrayindex(width)) = TRUE; // This point visited
	    search.insert(p);
	  }
	}
    }
  if (area != getArea( c)) {
    cerr << "Cell number " << c << " found disconnected. Only " << area << 
      " points found of possible " << getArea(c) << ". Attempting to fix.";
    error("");
    for (int y = 0; y < height; y++)
      for(int x = 0; x < width; x++)
	if (( *(representation + y * width + x) == c) && !(tag+y*width+x)) {
	  // found a point not connected to the mainstream
	  putCell( Point(x,y), 0);
	  cerr << "Disconnected piece of cell " << c << " found.\n";
	  area--;
	}
  }
  if (cell[ c]->area != getArea( c)) {
    growCell( c, cell[c]->area - getArea(c), randomDirection() );
    connect( c);
  }
}

void
Representation::incrementCellArea(Int c, Point direction)
     //  Increments area of cell by 1 by stealing a point from the cell in
     //  direction. direction should be either (0,0) indicating random or one
     //  of the 6 possible
{
  if (cell[ c]->dummy) //area flexible
    return;
  Point incDirection;
  if (direction == Point(0,0)) incDirection = random6Direction();
  else incDirection = direction;
  Point neighborPoint = findBoundary( c,  getLocation(c)+incDirection);
  PointList search;
  search.addNeighbors( neighborPoint);
  Point p;
  do {
    p = search.delget();
  } while  (!search.emptyList() && getCell(p) != c);
  Point cellPoint = p;
  
  PointList boundaryList;
  Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
  do {
    // Step through cellPoint and neighborPoint 
    p = cellPoint.nextPointInRotation( neighborPoint);
    if ( getCell( p) == c) cellPoint = p;
    else {
      neighborPoint = p;
      if (direction == Point(0,0)) boundaryList.insert( p);
      else if ( getCell( p - incDirection) == c) boundaryList.insert( p);
      // possible point of growth if in proper direction
    }
  } while ( (startCellPoint != cellPoint) ||
	   (startNeighborPoint != neighborPoint));
  int boundarySize = boundaryList.length();
  // try to replace the rth item in the list
  while ( !boundaryList.emptyList()) {
    int r = randBetween(1, boundarySize--);
    for (int i = 1; i < r; i++) {
      p = boundaryList.delget();
      boundaryList.insert( p);
      // delete from front and insert in back
    }
    p = boundaryList.delget();
    if ( canStealPoint( p)) {
      Int newCell = getCell( p);
      putCell( p, c);
      if ( canStealPoint( p)) { // This ensures that the cell does not have
				// holes in it
	if (direction != Point(0,0)) incrementCellArea( newCell, incDirection);
	else { // propogate steal in same direction
	  PointList search;
	  search.addNeighbors( p);
	  Point q;
	  do {
	    q = search.delget();
	  } while  (!search.emptyList() && getCell(q) != c);
	  incrementCellArea( newCell, p-q);
	}
	return;
      }
      else putCell( p, newCell);
    }
  }
  if (direction != Point(0,0)) { // try any direction
    incrementCellArea( c, Point(0,0));
    return;
  }
  // Not possible to increase area
  cerr << "Cell number " << c <<':';
  error(" failure in incrementing cell area.");
}

void
Representation::decrementCellArea(Int c, Point direction)
     //  Decrements area of cell by 1 by donating a point to the cell in
     //  direction. direction should be either (0,0) indicating random or one
     //  of the 6 possible
{
  if (cell[ c]->dummy) //area flexible
    return;
  Point decDirection = (direction == FPoint(0,0))? random6Direction():direction; 
  Point neighborPoint = findBoundary( c,  getLocation(c) + decDirection);
  PointList search;
  search.addNeighbors( neighborPoint);
  Point p;
  do {
    p = search.delget();
  } while  (!search.emptyList() && getCell(p) != c);
  Point cellPoint = p;
  
  PointList boundaryList;
  Point startCellPoint = cellPoint, startNeighborPoint = neighborPoint;
  do {
    // Step through cellPoint and neighborPoint 
    p = cellPoint.nextPointInRotation( neighborPoint);
    if ( getCell( p) == c) {
      cellPoint = p;
      if (direction == FPoint(0,0)) boundaryList.insert( p);
      else if ( getCell( p - decDirection) != c) boundaryList.insert( p);
      // possible point of shrinkage if in proper direction
    }
    else neighborPoint = p;
  } while ( (startCellPoint != cellPoint) ||
	   (startNeighborPoint != neighborPoint));
  int boundarySize = boundaryList.length();
  // try to replace the rth item in the list
  while ( !boundaryList.emptyList()) {
    int r = randBetween(1, boundarySize--);
    for (int i = 1; i < r; i++) {
      p = boundaryList.delget();
      boundaryList.insert( p);
      // delete from front and insert in back
    }
    p = boundaryList.delget();
    if ( canStealPoint( p)) {
      Int newCell;
      if ((newCell = getCell( p - decDirection)) == c) {// find non cell
							// neighbor
	PointList search;
	search.addNeighbors( p);
	Point q;
	do {
	  q = search.delget();
	} while  (!search.emptyList() && getCell(p) == c);
	newCell = getCell(p);
      }
      putCell( p, newCell);
      if ( canStealPoint( p)) { // This ensures that the neighboring cell does
				// not have holes in it
	decrementCellArea( newCell, decDirection);
	return;
      }
      else putCell( p, c);
    }
  }
  if (direction != Point(0,0)) { // try any direction
    decrementCellArea( c, Point(0,0));
    return;
  }
  // Not possible to decrease area
  cerr << "Cell number " << c <<':';
  error(" failure in decrementing cell area.");
}

#endif not UNIT_CELLS

