/*****************************************************************************
**
**    imd.c                             NQ                        3 Oct 1991
**
**    Copyright 1991.
**    Werner Nickel
**    Mathematics Research Section
**    School of Mathematical Sciences 
**    Australian National University, Canberra, Australia
*/

#include <stdio.h>

/*
**    This module uses the GNU arbitrary precision integer package gmp.
*/
#include "/usr/local/include/gmp.h"

/*
**    Define the data type for large integers and vectors of large integers.
*/
typedef	MP_INT	large;
typedef large	*lvec;

/*
**    Three macros to compare a large integer with 0.
*/
#define	ISZERO( x )  (mpz_cmp_ui( &(x), 0L ) == 0)
#define	ISNEG( x )   (mpz_cmp_ui( &(x), 0L )  < 0)
#define	NOTZERO( x ) (mpz_cmp_ui( &(x), 0L ) != 0)

/*
**    Taking times needs a variable.
*/
long    ImdTime = 0;
long    ImtTime = 0;

/*
**    The variable 'Matrix' contains the pointer to the integer matrix.
**    The variable 'Heads' contains the pointer to an array whose i-th
**    component contains the position of the first non-zero entry in
**    i-th row of Matrix[].
*/
       lvec     *Matrix   = (lvec *)0;
static lvec     *TMatrix  = (lvec *)0;
static long	*Heads    = (long *)0;
static long     Transform = 0;

/*
**    The number of rows and columns in the integer matrix are stored in
**    the following two variables.
*/
static	long     NrRows  = 0;
static	long     NrTRows = 0;
static	long     NrCols  = 0;

/*
**    Some functions to free memory.
*/
static void	FreeVector( v, size )
lvec	v;

{	long	i;

	for( i = 0; i < size; i++ )
	    mpz_clear( &(v[i]) );
	Free( v );
}

void	FreeIMD() {

	long	i;

	for( i = 0; i < NrTRows; i++ )
	    FreeVector( TMatrix[i], NrTRows );
	Free(TMatrix);

	for( i = 0; i < NrRows; i++ )
	    FreeVector( Matrix[i], NrCols );
	Free(Matrix);

	NrRows  = 0;
	NrCols  = 0;
	Matrix  = (lvec *)0;

	NrTRows = 0;
	TMatrix = (lvec *)0;
}

/*
**    Some functions to convert formats.
*/
lvec	VectorShortLarge( v, a, z )
short	*v;
long	a, z;

{	long	i = 0;
	lvec	w;

	w = (lvec)Allocate( (z-a)*sizeof(large) );

	while( a < z ) mpz_init_set_si( &w[i++], v[a++] );

	return w;
}

/*
**    Initialize IMD.
*/
void	InitIMD( nrRows, nrCols )
int	nrRows, nrCols;

{	long	i, j;

        ImdTime = 0;
	if( Matrix == (lvec*)0 ) {
	    Matrix = (lvec*)Allocate( nrRows * sizeof(lvec) );
	    Heads  = (long*)Allocate( nrRows * sizeof(long) );
	    NrRows = nrRows;
	    NrCols = nrCols;
	}

	TMatrix = (lvec *)Allocate( NrCols * sizeof(lvec) );
	for( i = 0; i < NrCols; i++ ) {
	    TMatrix[i] = (lvec)Allocate( NrCols * sizeof(large) );
	    for( j = 0; j < NrCols; j++ )
		mpz_init( &TMatrix[i][j] );
	    mpz_init_set_ui( &TMatrix[i][i], 1 );
	}
	NrTRows  = NrCols;
}

/*
**    Print the contents of Matrix[].
*/
void	PrintGapVector( v, size )
lvec	v;

{	int	i;

	printf( "[ " );
	mpz_out_str( stdout, 10, &v[0] );
	for( i = 1; i < size; i++ ) {
	    printf( ", " );
	    mpz_out_str( stdout, 10, &v[i] );
	}
	printf( " ]" );
}

void	PrintGapMatrix() {

	long	i, j;

	printf( "D := [ " );
	if( NrRows > 0 ) PrintGapVector( Matrix[0], NrCols );
	for( i = 1; i < NrRows; i++ ) {
	    printf( ",\n       " );
	    PrintGapVector( Matrix[i], NrCols );
	}
	printf( " ];\n" );
}

void	PrintGapTMatrix() {

	long	i, j;

	printf( "T := [ " );
	if( NrTRows > 0 ) PrintGapVector( TMatrix[0], NrTRows );
	for( i = 1; i < NrTRows; i++ ) {
	    printf( ",\n       " );
	    PrintGapVector( TMatrix[i], NrTRows );
	}
	printf( " ];\n" );
}

void	PrintVector( v, size )
lvec	v;

{	int	i;

	mpz_out_str( stdout, 10, &v[0] );
	for( i = 1; i < size; i++ ) {
	    printf( " " );
	    mpz_out_str( stdout, 10, &v[i] );
	}
}

void	PrintMatrix() {

	long	i, j;

	printf( "\n%d\n", NrCols );
	for( i = 0; i < NrRows; i++ ) {
	    PrintVector( Matrix[i], NrCols );
	    printf( "\n" );
	}
}

void	PrintTMatrix() {

	long	i, j;

	printf( "Transforming matrix\n" );
	printf( "\n%d\n", NrTRows );
	for( i = 0; i < NrTRows; i++ ) {
	    for( j = 0; j < NrTRows; j++ ) {
		printf( " " );
		mpz_out_str( stdout, 10, &TMatrix[i][j] );
	    }
	    printf( "\n" );
	}
}

static void	Transpose() {

	long	i, j;
	lvec	*newMatrix;

	newMatrix = (lvec *)Allocate( NrCols*sizeof(lvec) );
	for( i = 0; i < NrCols; i++ ) {
	    newMatrix[i] = (lvec)Allocate( NrRows*sizeof(large) );
	    for( j = 0; j < NrRows; j++ )
		newMatrix[i][j] = Matrix[j][i];
	}
	for( i = 0; i < NrRows; i++ ) Free(Matrix[i]);
	Free( Matrix );

	Matrix = newMatrix;

	Heads = (long *)ReAllocate( Heads, NrCols*sizeof(long) );

	i = NrCols;
	NrCols = NrRows;
	NrRows = i;
}

static long	NotDiagonal() {

	long	i, j;

	if( NrCols != NrRows ) return 1;

	for( i = 0; i < NrRows; i++ )
	    for( j = 0; j < NrCols; j++ )
		if( i != j && NOTZERO( Matrix[i][j] ) )
		    return 1;

	return 0;
}

/*
**    Miscellaneous vector operations.
*/
static void	VectorScalarSub( v, w, s, a, z )
lvec	v, w;
large	*s;
long	a, z;

{	large	t;

	mpz_init( &t );
	/* Calculate v - s * w. */
	while( a < z ) { 
	    mpz_mul( &t, &w[a], s );
	    mpz_sub( &v[a], &v[a], &t );
	    a++;
	}
	mpz_clear( &t );
}

static void	VectorScalarAdd( v, w, s, a, z )
lvec	v, w;
large	*s;
long	a, z;

{	large	t;

	mpz_init( &t );
	/* Calculate v + s * w. */
	while( a < z ) { 
	    mpz_mul( &t, &w[a], s );
	    mpz_add( &v[a], &v[a], &t );
	    a++;
	}
	mpz_clear( &t );
}

static lvec	VectorScalarProd( v, s, a, size )
lvec	v;
large	*s;
long	a, size;

{	lvec	w;
	long	b = 0;

	w = (lvec)Allocate( size * sizeof(large) );
	while( b < size ) { mpz_init( &w[b] ); b++; }
	while( a < size ) { mpz_mul( &w[a], s, &v[a] ); a++; }

	return w;
}

static void	VectorScalarMult( v, s, a, z )
lvec	v;
large	*s;
long	a, z;

{	while( a < z ) { mpz_mul( &v[a], s, &v[a] ); a++; } }

static void	VectorNegate( v, a, z )
lvec    v;
long	a;

{       while( a < z ) { mpz_neg( &v[a], &v[a] ); a++; }    }

static void	VectorAdd( v, w, a, z )
lvec	v, w;
long	a, z;

{	while( a < z ) { mpz_add( &v[a], &v[a], &w[a] ); a++; } }

static void	VectorSub( v, w, a, z )
lvec	v, w;
long	a, z;

{	while( a < z ) { mpz_sub( &v[a], &v[a], &w[a] ); a++; } }

static void	Add( j, i )
long	j, i;

{	VectorAdd( Matrix[j],  Matrix[i],  Heads[i], NrCols );
	if( Transform ) VectorSub( TMatrix[i], TMatrix[j], 0, NrTRows );
}

static void	ScalarSub( j, i )
long    j, i;

{       large	s;
	long	h;

	h = Heads[i];
	if( NOTZERO( Matrix[i][h] ) ) {
	    mpz_init(&s);
	    mpz_div( &s, &Matrix[j][h], &Matrix[i][h] );
	    if( NOTZERO( s ) ) {
		VectorScalarSub( Matrix[j],  Matrix[i],  &s, h, NrCols );
		if( Transform )
		    VectorScalarAdd( TMatrix[i], TMatrix[j], &s, 0, NrTRows );
	    }
	    mpz_clear(&s);
	}
}

static void	Negate( i )
long	i;

{	VectorNegate( Matrix[i], Heads[i], NrCols );
	if( Transform ) VectorNegate( TMatrix[i], 0, NrTRows );
}

static void	ReduceOffDiagonal( n ) {

	long	i, j;

	/* Reduce the entries in the head column of row i. */
	for( i = 0; i < n; i++ ) {
	    if( ISNEG( Matrix[i][Heads[i]] ) ) Negate( i );
	    for( j = 0; j < i; j++ ) {
	    	ScalarSub( j, i );
		if( ISNEG( Matrix[j][Heads[i]] ) ) Add( j, i );
	    }
	}
}

static void	Bobbing( k, i, h )
long	k, i, h;

{	large	g, s, t, sg, tg;
	lvec	u, v, w;

	mpz_init( &g );
	mpz_init( &s );
	mpz_init( &t );
	mpz_init( &sg );
	mpz_init( &tg );

	mpz_gcdext( &g, &s, &t, &Matrix[k][h], &Matrix[i][h] );
	mpz_div( &sg, &Matrix[k][h], &g );
	mpz_div( &tg, &Matrix[i][h], &g );

	/*
	**   [         s               t       ]     [  Matrix[k]  ]    (k)
	**   [                                 ]  *  [             ]
	**   [ -Matrix[i][h]/g  Matrix[k][h]/g ]     [  Matrix[i]  ]    (i)
	*/
	u = Matrix[k];
	v = VectorScalarProd( Matrix[k], &s, h, NrCols );
	w = VectorScalarProd( Matrix[i], &t, h, NrCols );
	VectorAdd( v, w, h, NrCols );
	Matrix[k] = v;
	FreeVector( w, NrCols );
	VectorScalarMult( u, &tg, h, NrCols );
	VectorScalarMult( Matrix[i], &sg, h, NrCols );
	VectorSub( Matrix[i], u, h, NrCols );
	FreeVector( u, NrCols );

	if( Transform ) {
	    /*            (k)          (i)               (k)       (i)
	    **                                   [ Matrix[k][h]/g  -t ]  (k)
	    **   [ TMatrix[k]   TMatrix[i] ]  *  [                    ]
	    **                                   [ Matrix[i][h]/g   s ]  (i)
	    */
	    u = TMatrix[k];
	    v = VectorScalarProd( TMatrix[k], &sg, 0, NrTRows );
	    w = VectorScalarProd( TMatrix[i], &tg, 0, NrTRows );
	    VectorAdd( v, w, 0, NrTRows );
	    TMatrix[k] = v;
	    FreeVector( w, NrTRows );
	    VectorScalarMult( u, &t, 0, NrTRows );
	    VectorScalarMult( TMatrix[i], &s, 0, NrTRows );
	    VectorSub( TMatrix[i], u, 0, NrTRows );
	    FreeVector( u, NrTRows );
	}

	mpz_clear( &g );
	mpz_clear( &s );
	mpz_clear( &t );
	mpz_clear( &sg );
	mpz_clear( &tg );
}

/* Insert() inserts row i of Matrix before row j. */
Insert( i, j, h )
long	i, j, h;

{	lvec	v;
	long	k;

	v = Matrix[i];
	for( k = i; k > j; k-- ) {
	    Matrix[k]  = Matrix[k-1];
	    Heads[k]   = Heads[k-1];
	}
	Matrix[j] = v;
	Heads[j] = h;
	if( Transform ) {
	    v = TMatrix[j];
	    for( k = j; k < i; k++ )
		TMatrix[k]  = TMatrix[k+1];
	    TMatrix[i] = v;
	}
}
	    
/*    Swap() interchanges the i-th row and the last row of Matrix[].    */
static	void	Swap( i )
long	i;

{	lvec	v;

	FreeVector( Matrix[i], NrCols );
	NrRows--;
	Matrix[i] = Matrix[NrRows];
	if( Transform ) {
	    v = TMatrix[i];
	    TMatrix[i] = TMatrix[NrRows];
	    TMatrix[NrRows] = v;
	}
}

static void	RowReduce() {

	long	h, i, j, k;
	lvec	v;

	i = 0;
	while( i < NrRows ) {
	    /* first find the head of the i-th row. */
	    h = 0; while( h < NrCols && ISZERO( Matrix[i][h] ) ) h++;
	    if( h < NrCols )
	        /* Reduce the i-th row against the rows above. */
		for( j = 0; j < i && Heads[j] <= h; j++ )
		    if( Heads[j] == h ) {
			Bobbing( j, i, h );
			while( h < NrCols && ISZERO(Matrix[i][h]) ) h++;
			if( h >= NrCols ) break;
		    }
	    if( h < NrCols ) {
		Insert( i, j, h );
		i++;
	    }
	    else {
		Swap( i );
	    }
	    ReduceOffDiagonal(i);
	}
}

void	RunImd() {

        ImdTime -= runTime();
	while( NotDiagonal() ) {
	    Transform = 0;
	    RowReduce();

	    Transpose();
	    Transform = 1;
	    RowReduce();
	    Transpose();
	}
        ImdTime += runTime();
}

void	Imd( nrRows, nrCols, M )
long	nrRows, nrCols;
lvec	*M;

{	long	i;

	Matrix   = M;
	Heads    = (long*)Allocate( nrRows * sizeof(long) );
	NrRows   = nrRows;
	NrCols   = nrCols;

	InitIMD( nrRows, nrCols );

	RunImd();
}

void	InitIMT( nrCols ) {

	if( Matrix != (lvec *)0 ) {
	    printf( "IMT already initialized\n" );
	    exit( 666 );
	}

        ImtTime = 0;
	Transform = 0;
	Matrix = (lvec*)Allocate( 200 * sizeof(lvec) );
	Heads  = (long*)Allocate( 200 * sizeof(long) );
	NrCols = nrCols;
}

static void	EnlargeIMT() {

	if( Matrix == (lvec *)0 ) {
	    printf( "IMT not initialized\n" );
	    exit( 666 );
	}
	else if( NrRows % 200 == 0 ) {
	    Matrix = (lvec*)ReAllocate( Matrix, (NrRows+200)*sizeof(lvec) );
	    Heads  = (long*)ReAllocate( Heads,  (NrRows+200)*sizeof(long) );
        }
}

/*
**    AppendRow() reduces the vector v against the vectors in Matrix[].
*/
void	AppendRow( v )
lvec	v;

{	long	j, h;
	lvec	w;

        ImtTime -= runTime();
	h = 0;
	while( h < NrCols && ISZERO( v[h] ) ) h++;
	if( h >= NrCols ) { FreeVector( v, NrCols ); return; }

/*	PrintVector( v, NrCols ); printf( "\n" );*/

	EnlargeIMT();
	Matrix[NrRows] = v;
	NrRows++;
	/* Reduce the last row against the rows above. */
	for( j = 0; j < NrRows-1 && Heads[j] <= h; j++ )
	    if( Heads[j] == h ) {
		Bobbing( j, NrRows-1, h );
		while( h < NrCols && ISZERO(Matrix[NrRows-1][h]) ) h++;
		if( h >= NrCols ) break;
	    }
	if( h < NrCols )
	    Insert( NrRows-1, j, h );
	else {
	    NrRows--;
	    FreeVector( Matrix[NrRows], NrCols );
	}
	ReduceOffDiagonal( NrRows );
        ImtTime += runTime();
}

int	ElemDivisors( v )
lvec	v;

{	int	i, n;

	for( n = 0, i = 0; i < NrCols; i++ )
	    if( mpz_cmp_si( &(Matrix[i][i]), 1 ) )
		v[n++] = Matrix[i][i];

	return n;
}
