/*
 * viewing.c
 *
 * Copyright (C) 1989, 1991, Craig E. Kolb, Rod G. Bogart
 * All rights reserved.
 *
 * This software may be freely copied, modified, and redistributed
 * provided that this copyright notice is preserved on all copies.
 *
 * You may not distribute this software, in whole or in part, as part of
 * any commercial product without the express consent of the authors.
 *
 * There is no warranty or other guarantee of fitness of this software
 * for any purpose.  It is provided solely "as is".
 *
 * $Id: viewing.c,v 4.0 91/07/17 14:48:18 kolb Exp Locker: kolb $
 *
 * $Log:	viewing.c,v $
 * Revision 4.0  91/07/17  14:48:18  kolb
 * Initial version.
 *
 */
#include "rayshade.h"
#include "viewing.h"
#include "sampling.h"
#include "options.h"
#include "defaults.h"
#include "picture.h"
#include "stats.h"
 
RSCamera Camera;
RSScreen Screen;
 
void SampleScreen(), SampleScreenFiltered();
 
void
RSViewing()
{
	Float magnitude;
 
	VecSub(Camera.lookp, Camera.pos, &Camera.dir);
	Screen.firstray = Camera.dir;
 
	Camera.lookdist = VecNormalize(&Camera.dir);
	if (VecNormCross(&Camera.dir, &Camera.up, &Screen.scrni) == 0.)
		RLerror(RL_PANIC,
			"The view and up directions are identical?\n");
	(void)VecNormCross(&Screen.scrni, &Camera.dir, &Screen.scrnj);
 
	/*
	 * Add stereo separation if desired.
	 */
	if (Options.stereo) {
		if (Options.stereo == LEFT)
			magnitude = -.5 * Options.eyesep;
		else
			magnitude =  .5 * Options.eyesep;
		Camera.pos.x += magnitude * Screen.scrni.x;
		Camera.pos.y += magnitude * Screen.scrni.y;
		Camera.pos.z += magnitude * Screen.scrni.z;
		VecSub(Camera.lookp, Camera.pos, &Screen.firstray);
		Camera.dir = Screen.firstray;
		Camera.lookdist = VecNormalize(&Camera.dir);
		(void)VecNormCross(&Camera.dir, &Camera.up, &Screen.scrni);
		(void)VecNormCross(&Screen.scrni, &Camera.dir, &Screen.scrnj);
	}
 
	magnitude = 2.*Camera.lookdist * tan(deg2rad(0.5*Camera.hfov)) /
				Screen.xres;
 
	VecScale(magnitude, Screen.scrni, &Screen.scrnx);
	magnitude = 2.*Camera.lookdist * tan(deg2rad(0.5*Camera.vfov)) /
				Screen.yres;
	VecScale(magnitude, Screen.scrnj, &Screen.scrny);
 
	Screen.firstray.x -= 0.5*Screen.yres*Screen.scrny.x +
			     0.5*Screen.xres*Screen.scrnx.x;
	Screen.firstray.y -= 0.5*Screen.yres*Screen.scrny.y +
			     0.5*Screen.xres*Screen.scrnx.y;
	Screen.firstray.z -= 0.5*Screen.yres*Screen.scrny.z +
			     0.5*Screen.xres*Screen.scrnx.z;
 
	if (Camera.focaldist == UNSET)
		Camera.focaldist = Camera.lookdist;
}
 
/*
 * Adjust the initial ray to account for an aperture and a focal
 * distance.  The ray argument is assumed to be an initial ray, and
 * always reset to the eye point.  It is assumed to be unit length.
 */
void
focus_blur_ray(ray)
Ray *ray;
{
	Vector circle_point, aperture_inc;
	extern void UnitCirclePoint();
	/*
	 * Find a point on a unit circle and scale by aperture size.
	 * This simulates rays passing thru different parts of the aperture.
	 * Treat the point as a vector and rotate it so the circle lies
	 * in the plane of the screen.  Add the aperture increment to the
	 * starting position of the ray.  Stretch the ray to be focaldist
	 * in length.  Subtract the aperture increment from the end of the
	 * long ray.  This insures that the ray heads toward a point at
	 * the specified focus distance, so that point will be in focus.
	 * Normalize the ray, and that's it.  Really.
	 */
	UnitCirclePoint(&circle_point, ray->sample);
	VecComb(Camera.aperture * circle_point.x, Screen.scrni,
		    Camera.aperture * circle_point.y, Screen.scrnj,
		    &aperture_inc);
	VecAdd(aperture_inc, Camera.pos, &(ray->pos));
	VecScale(Camera.focaldist, ray->dir, &(ray->dir));
	VecSub(ray->dir, aperture_inc, &(ray->dir));
	(void)VecNormalize(&ray->dir);
}
 
void
ViewingSetup()
{
	if (Options.stereo && Options.eyesep == UNSET)
		RLerror(RL_PANIC,
			"No eye separation specified for stereo rendering.\n");
	/*
	 * Because we want the user to be able to override the input file
	 * through the command line, we have to initialize some variables to
	 * bogus values so that when the file is being parsed, it is
	 * possible to tell if a given variable has been set on the
	 * command line.
	 *
	 * If such variables are not set to legal values on the command
	 * line or in the input file, we must do it here.
	 */
	if (Screen.xres == UNSET)
		Screen.xres = XRESOLUTION;
	if (Screen.yres == UNSET)
		Screen.yres = YRESOLUTION;
 
#ifdef URT
	/*
	 * If using the URT, we can use the RLE file header to
	 * determine window size.  Screen size (Screen.xres, Screen.yres)
	 * is determined as usual (from command line or input file).
	 */
	if (Options.appending) {
		/*
		 * Read image header to determine window size.
		 */
		PictureSetWindow();
	} else {
#endif
	if (Options.xmin < 0.) Options.xmin = 0.;
	if (Options.ymin < 0.) Options.ymin = 0.;
	if (Options.xmax > 1.) Options.xmax = 1.;
	if (Options.xmax > 1.) Options.xmax = 1.;
 
	Screen.minx = (int)(Options.xmin * Screen.xres);
	Screen.maxx = (int)(Options.xmax * Screen.xres -1);
	Screen.miny = (int)(Options.ymin * Screen.yres);
	Screen.maxy = (int)(Options.ymax * Screen.yres -1);
#ifdef URT
	}
#endif
 
	Screen.xsize = Screen.maxx - Screen.minx + 1;
	Screen.ysize = Screen.maxy - Screen.miny + 1;
 
	/*
	 * Sanity check.
	 */
	if (Screen.minx < 0 || Screen.miny < 0 ||
	    Screen.maxx >= Screen.xres || Screen.maxy >= Screen.yres){
	/*	fprintf(stderr,"minx %d miny %d maxx %d maxy %d xres %d yres %d\n",Screen.minx,Screen.miny,Screen.maxx,Screen.maxy,Screen.xres,Screen.yres);*/
		RLerror(RL_PANIC, "Invalid window specification.\n");
 	}
 
	/*
	 * If not defined in the input file, calculate VFOV
	 * by hand.  This assumes that pixels are square, which is
	 * probably a bad idea.  ("aspect" option?)
	 */
	if (Camera.vfov == UNSET)
		Camera.vfov = Camera.hfov * Screen.yres / Screen.xres;
}
 
void
SampleScreenFiltered(x, y, u, v, ray, color, sample)
Float x, y;
Ray *ray;
Pixel *color;
int sample, u, v;
{
	SampleScreen(x, y, ray, color, sample);
	color->r *= Sampling.filter[u][v];
	color->g *= Sampling.filter[u][v];
	color->b *= Sampling.filter[u][v];
	color->alpha *= Sampling.filter[u][v];
}	
 
void
SampleScreen(x, y, ray, color, sample)
Float x, y;		/* Screen position to sample */
Ray *ray;		/* ray, with origin and medium properly set */
Pixel *color;		/* resulting color */
int sample;		/* sample number */
{
	Float dist;
	HitList hitlist;
	Color ctmp, fullintens;
	extern void focus_blur_ray(), ShadeRay();
 
	/*
	 * Calculate ray direction.
	 */
	Stats.EyeRays++;
	ray->dir.x = Screen.firstray.x + x*Screen.scrnx.x + y*Screen.scrny.x;
	ray->dir.y = Screen.firstray.y + x*Screen.scrnx.y + y*Screen.scrny.y;
	ray->dir.z = Screen.firstray.z + x*Screen.scrnx.z + y*Screen.scrny.z;
 
	(void)VecNormalize(&ray->dir);
 
	ray->sample = sample;
 
	if (Camera.aperture > 0.0) {
		/*
		 * If the aperture is open, adjust the initial ray
		 * to account for depth of field.
		 */
		focus_blur_ray(ray);
	}
 
	/*
	 * Do the actual ray trace.
	 */
	fullintens.r = fullintens.g = fullintens.b = 1.;
	dist = FAR_AWAY;
	hitlist.nodes = 0;
	(void)TraceRay(ray, &hitlist, EPSILON, &dist);
	ShadeRay(&hitlist, ray, dist, &Screen.background, &ctmp, &fullintens);
	color->r = ctmp.r;
	color->g = ctmp.g;
	color->b = ctmp.b;
	if (hitlist.nodes != 0) {
		color->alpha = 1.;
	} else {
		color->alpha = 0.;
	}
}
