/* m3dminmax.c -- 3D morphology minmax operators */

/* contains: BinSetMinMax(), GraySetMinMax(), GrayFctMinMax() */

/* dependencies: none */

/* morph3d version 4.0   1 June 1993               */
/* 3D image morphology program                     */ 
/*                                                 */ 
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */

/* minmax operators -- finds the minimum of the set {m1,...,mk}, where */
/* k == sZ (the depth or Z-dim of the SE) and each mj is the maximum   */
/* over the xy neighborhood in the jth plane of the SE */


#include "morph3d_sub.h"



/* Morphological 3D MinMax of a gray-level image with a binary SE.          */
/* Binary is interpreted as zero and positive (not zero). Images are        */
/* unsigned. A negative SE element is treated as a DON'T CARE. The MinMax   */
/* is perfomed over the positive (>0) support of of the SE.                 */

void BinSetMinMax( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, unused ) 
   byte **In, *Out; /* input and output images */
   int X;           /* image horizontal dimension */
   int tX,tY;       /* transform scan horiz., vert. dimensions */
   int *stelem;     /* SE in row-major order */
   int sX,sY,sZ;    /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int unused;      /* to conform with other routines' param list */
   {
   byte *I,*O;      /* image pointers */
   int i,j,k;       /* transform indices */
   int x,y;         /* image indices */
   int p;           /* pixel value */
   int r,t;         /* transform result */
   int *s;          /* pointer to SE  */

   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         s = stelem;
         t = WHITE;   
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            r = BLACK;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s > 0)  
                     {
                     p = *(I + j*X + i);  /* get pixel   */
                     r = MAX(r,p);
                     if ( r == WHITE ) break; /* out of col. scan */
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               if ( r == WHITE ) break;    /* out of SE row scan */
               }                       /* end of SE row scan */
            t = MIN(t,r);
            if ( t == BLACK ) break;       /* out of SE z-dim scan */
            }                          /* end of SE z-dim scan */
         *(O++) = (byte)t; /* output result */
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of BinSetMinMax */


 

/* Morphological set MinMax of a gray-level image with a gray-level SE.    */
/* Images are unsigned. A negative SE element is treated as a DON'T CARE.  */
/* The MinMax is perfomed as a function-set operation over the nonnegative */
/* (>= 0) support of of the SE. */

void GraySetMinMax( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, unused ) 
   byte **In, *Out; /* input and output images */
   int X;           /* image horizontal dimension */
   int tX,tY;       /* transform scan horiz., vert. dimensions */
   int *stelem;     /* SE in row-major order */
   int sX,sY,sZ;    /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int unused;      /* to conform with other routines' param list */
   {
   byte *I,*O;      /* image pointers */
   int i,j,k;       /* transform indices */
   int x,y;         /* image indices */
   int p;           /* pixel value */
   int r,t;         /* transform result */
   int *s;          /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         s = stelem;     
         t = WHITE;
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            r = BLACK;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s >= 0)  
                     {
                     p = *(I + j*X + i);  /* get pixel   */
                     r = MAX(r,p);
                     if ( r == WHITE ) break; /* out of col. scan */
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               if ( r == WHITE ) break;    /* out of SE row scan */
               }                       /* end of SE row scan */
            t = MIN(t,r);
            if ( t == BLACK ) break;       /* out of SE z-dim scan */
            }                          /* end of SE z-dim scan */
        *(O++) = (byte)t; /* output result */
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of GraySetMinMax */




/* Morphological function MinMax of a gray-level image with a gray-level SE. */
/* Images are unsigned. A negative SE element is treated as a DON'T CARE.    */
/* The MinMax is perfomed as a function operation over the nonnegative       */
/* (>= 0) support of of the SE. */

void GrayFctMinMax( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, unused ) 
   word **In, *Out; /* input and output images */
   int X;           /* image horizontal dimension */
   int tX,tY;       /* transform scan horiz., vert. dimensions */
   int *stelem;     /* SE in row-major order */
   int sX,sY,sZ;    /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz;    /* SE origin offset from upper LH corner. */
   {
   word *I,*O;      /* image pointers */
   int i,j,k;       /* transform indices */
   int x,y;         /* image indices */
   int q,r,t;       /* transform result */
   int *s;          /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         s = stelem;     
         t = WHITE;
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            r = BLACK;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s >= 0)  
                     {
                     q = *(I + j*X + i) + *s; /* pixel plus SE value */
                     r = MAX(q,r);
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               }                       /* end of SE row scan */
            t = MIN(t,r);
            }                          /* end of SE z-dim scan */
         *(O++) = (word)t; /* output result */
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of GrayFctMinMax */
