/*
 * rdev.c -- readdev() function for lsof library
 */


/*
 * Copyright 1997 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */


#include "../machine.h"

#if	defined(USE_LIB_READDEV)

# if	!defined(lint)
static char copyright[] =
"@(#) Copyright 1997 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: rdev.c,v 1.5 97/07/09 08:31:11 abe Exp $";
# endif	/* !defined(lint) */

#include "../lsof.h"


_PROTOTYPE(static int rmdupdev,(struct l_dev ***dp, int n, char *nm));
_PROTOTYPE(static void stkdir,(char ***d, int *n, int *x, char *p));


/*
 * To use this source file:
 *
 * 1. Define DIRTYPE as:
 *
 *	  #define DIRTYPE direct
 *    or  #define DIRTYPE dirent
 *
 * 2. Define HASDNAMLEN if struct DIRTYPE has a d_namlen element, giving
 *    the length of d_name.
 *
 * 3. Define the RDEV_EXPDEV macro to apply special handling to device
 *    numbers, as required.  For example, for EP/IX 2.1.1:
 *
 *	#define RDEV_EXPDEV(n)	expdev(n)
 *
 *    to use the expdev() function to expand device numbers.  If
 *    no RDEV_EXPDEV macro is defined, it defaults to:
 *
 *	#define RDEV_EXPDEV(n)	(n)
 *
 * 4. Define HASBLKDEV to request that information on S_IFBLK devices be
 *    recorded in BDevtp[].
 *
 * 5. Define RDEV_STATFN to be a stat function other than stat() or lstat()
 *    -- e.g.,
 *
 *	#define	RDEV_STATFN	private_stat
 */


#if	!defined(RDEV_EXPDEV)
#define	RDEV_EXPDEV(n)		(n)
#endif	/* !defined(RDEV_EXPDEV) */

#if	!defined(RDEV_STATFN)
# if	defined(USE_STAT)
#define	RDEV_STATFN	stat
# else	/* !defined(USE_STAT) */
#define	RDEV_STATFN	lstat
# endif	/* defined(USE_STAT) */
#endif	/* !defined(RDEV_STATFN) */


/*
 * readdev() - read device names, modes and types
 */

void
readdev(skip)
	int skip;			/* skip device cache read if 1 */
{
#if	defined(HASDCACHE)
	int dcrd;
#endif	/* defined(HASDCACHE) */

	DIR *dfp;
	int dn = 0;
	struct DIRTYPE *dp;
	char **dstk = NULL;
	int dx = 0;
	int i = 0;

#if	defined(HASBLKDEV)
	int j = 0;
#endif	/* defined(HASBLKDEV) */

#if	!defined(HASDNAMLEN)
	STRNCPY_L dnamlen;
#endif	/* !defined(HASDNAMLEN) */

	MALLOC_S nl;
	char path[MAXNAMLEN+1];
	int pl;
	struct stat sb;

	if (Sdev)
	    return;

#if	defined(HASDCACHE)
/*
 * Read device cache, as directed.
 */
	if (!skip) {
	    if (DCstate == 2 || DCstate == 3) {
		if ((dcrd = read_dcache()) == 0)
		    return;
	    }
	} else
	    dcrd = 1;
#endif	/* defined(HASDCACHE) */

	(void) stkdir(&dstk, &dn, &dx, "/dev");
/*
 * Unstack the next /dev or /dev/<subdirectory> directory.
 */
	while (--dx >= 0) {
	    (void) strcpy(path, dstk[dx]);
	    if ((dfp = opendir(path)) == NULL) {

#if	defined(WARNDEVACCESS)
		if (!Fwarn)
		    (void) fprintf(stderr, "%s: WARNING: can't open %s\n",
			Pn, path);
#endif	/* defined(WARNDEVACCESS) */

		continue;
	    }
	    (void) strcat(path, "/");
	    pl = strlen(path);
	    (void) free((FREE_P *)dstk[dx]);
	    dstk[dx] = NULL;
	/*
	 * Scan the directory.
	 */
	    for (dp = readdir(dfp); dp; dp = readdir(dfp)) {
		if (dp->d_ino == 0 || dp->d_name[0] == '.')
				continue;
	    /*
	     * Form the full path name and get its status.
	     */

#if	defined(HASDNAMLEN)
		if ((nl = pl + dp->d_namlen) >= sizeof(path))
#else	/* !defined(HASDNAMLEN) */
		dnamlen = strlen(dp->d_name);
		if ((nl = pl + dnamlen) >= sizeof(path))
#endif	/* defined(HASDNAMLEN) */

		{
		    (void) fprintf(stderr,
			"%s: /dev entry name too long: %s\n", Pn, dp->d_name);
			Exit(1);
		}
		(void) strncpy(&path[pl], dp->d_name,

#if	defined(HASDNAMLEN)
		    (STRNCPY_L)dp->d_namlen
#else	/* !defined(HASDNAMLEN) */
		    dnamlen
#endif	/* !defined(HASDNAMLEN) */

		);

		path[nl++] = '\0';
		if (RDEV_STATFN(path, &sb) != 0) {
		    if (errno == ENOENT)	/* a sym link to nowhere? */
			continue;

#if	defined(WARNDEVACCESS)
		    if (!Fwarn)
			(void) fprintf(stderr, "%s: can't stat %s: %s\n",
			    Pn, path, strerror(errno));
#endif	/* defined(WARNDEVACCESS) */

		    continue;
		}
	    /*
	     * If it's a subdirectory, stack its name for later
	     * processing.
	     */
		if ((sb.st_mode & S_IFMT) == S_IFDIR) {
		    (void) stkdir(&dstk, &dn, &dx, path);
		    continue;
		}
		if ((sb.st_mode & S_IFMT) == S_IFCHR) {

		/*
		 * Save character device information in Devtp[].
		 */
		    if (i >= Ndev) {
			Ndev += DEVINCR;
			if (Devtp == NULL)
			    Devtp = (struct l_dev *)malloc(
				    (MALLOC_S)(sizeof(struct l_dev)*Ndev));
			else
			    Devtp = (struct l_dev *)realloc((MALLOC_P *)Devtp,
				    (MALLOC_S)(sizeof(struct l_dev)*Ndev));
			if (Devtp == NULL) {
			    (void) fprintf(stderr,
				"%s: no space for character device\n", Pn);
			    Exit(1);
			}
		    }
		    Devtp[i].rdev = RDEV_EXPDEV(sb.st_rdev);
		    Devtp[i].inode = sb.st_ino;
		    if ((Devtp[i].name = (char *)malloc(nl)) == NULL) {
			(void) fprintf(stderr, "%s: no space for /dev/%s\n",
			    Pn, dp->d_name);
			Exit(1);
		    }
		    (void) strcpy(Devtp[i].name, path);
		    Devtp[i].v = 0;
		    i++;
		}

#if	defined(HASBLKDEV)
		if ((sb.st_mode & S_IFMT) == S_IFBLK) {

		/*
		 * Save block device information in BDevtp[].
		 */
		    if (j >= BNdev) {
			BNdev += DEVINCR;
			if (BDevtp == NULL)
			    BDevtp = (struct l_dev *)malloc(
				     (MALLOC_S)(sizeof(struct l_dev)*BNdev));
			else
			    BDevtp = (struct l_dev *)realloc((MALLOC_P *)BDevtp,
				     (MALLOC_S)(sizeof(struct l_dev)*BNdev));
			if (BDevtp == NULL) {
			    (void) fprintf(stderr,
				"%s: no space for block device\n", Pn);
			    Exit(1);
			}
		    }
		    BDevtp[j].rdev = RDEV_EXPDEV(sb.st_rdev);
		    BDevtp[j].inode = sb.st_ino;
		    if ((BDevtp[j].name = (char *)malloc(nl)) == NULL) {
			(void) fprintf(stderr, "%s: no space for /dev/%s\n",
			    Pn, dp->d_name);
			Exit(1);
		    }
		    (void) strcpy(BDevtp[j].name, path);
		    BDevtp[j].v = 0;
		    j++;
		}
#endif	/* defined(HASBLKDEV) */

	    }
	    (void) closedir(dfp);
	}
/*
 * Free any directory stack space.
 */
	if (dstk != NULL)
	    (void) free((FREE_P *)dstk);
/*
 * Reduce the BDevtp[] (optional) and Devtp[] tables to their minimum
 * sizes; allocate and build sort pointer lists; and sort the tables by
 * device number.
 */

#if	defined(HASBLKDEV)
	if (BNdev) {
	    if (BNdev > j) {
		BNdev = j;
		BDevtp = (struct l_dev *)realloc((MALLOC_P *)BDevtp,
			 (MALLOC_S)(sizeof(struct l_dev) * BNdev));
	    }
	    if ((BSdev = (struct l_dev **)malloc(
		(MALLOC_S)(sizeof(struct l_dev *) * BNdev)))
	    == (struct l_dev **)NULL) {
		(void) fprintf(stderr,
		    "%s: no space for block device pointers\n", Pn);
		Exit(1);
	    }
	    for (j = 0; j < BNdev; j++) {
		BSdev[j] = &BDevtp[j];
	    }
	    (void) qsort((QSORT_P *)BSdev, (size_t)BNdev,
		(size_t)sizeof(struct l_dev *), compdev);
	    BNdev = rmdupdev(&BSdev, BNdev, "block");
	} else {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: no block devices found\n", Pn);
	}
#endif	/* defined(HASBLKDEV) */

	if (Ndev) {
	    if (Ndev > i) {
		Ndev = i;
		Devtp = (struct l_dev *)realloc((MALLOC_P *)Devtp,
			(MALLOC_S)(sizeof(struct l_dev) * Ndev));
	    }
	    if ((Sdev = (struct l_dev **)malloc(
		(MALLOC_S)(sizeof(struct l_dev *) * Ndev)))
	    == (struct l_dev **)NULL) {
		(void) fprintf(stderr,
		    "%s: no space for character device pointers\n", Pn);
		Exit(1);
	    }
	    for (i = 0; i < Ndev; i++) {
		Sdev[i] = &Devtp[i];
	    }
	    (void) qsort((QSORT_P *)Sdev, (size_t)Ndev,
		(size_t)sizeof(struct l_dev *), compdev);
	    Ndev = rmdupdev(&Sdev, Ndev, "char");
	} else {
	    (void) fprintf(stderr, "%s: no character devices found\n", Pn);
	    Exit(1);
	}

#if	defined(HASDCACHE)
/*
 * Write device cache file, as required.
 */
	if (DCstate == 1 || (DCstate == 3 && dcrd))
		write_dcache();
#endif	/* defined(HASDCACHE) */

}


#if	defined(HASDCACHE)
/*
 * rereaddev() - reread device names, modes and types
 */

void
rereaddev()
{
	(void) clr_devtab();

# if	defined(DCACHE_CLR)
	(void) DCACHE_CLR();
# endif	/* defined(DCACHE_CLR) */

	readdev(1);
	DCunsafe = 0;
}
#endif	/* defined(HASDCACHE) */


/*
 * rmdupdev() - remove duplicate (major/minor/inode) devices
 */

static int
rmdupdev(dp, n, nm)
	struct l_dev ***dp;	/* device table pointers address */
	int n;			/* number of pointers */
	char *nm;		/* device table name for error message */
{
	int i, j, k;
	struct l_dev **p;

	for (i = j = 0, p = *dp; i < n ;) {
	    for (k = i + 1; k < n; k++) {
		if (p[i]->rdev != p[k]->rdev || p[i]->inode != p[k]->inode)
		    break;
	    }
	    if (i != j)
		p[j] = p[i];
	    j++;
	    i = k;
	}
	if (n == j)
	    return(n);
	if (!(*dp = (struct l_dev **)realloc((MALLOC_P *)*dp,
	    (MALLOC_S)(j * sizeof(struct l_dev *)))))
	{
	    (void) fprintf(stderr, "%s: can't realloc %s device pointers\n",
		Pn, nm);
	    Exit(1);
	}
	return(j);
}



/*
 * stkdir() - stack directory name
 */

static void
stkdir(d, n, x, p)
	char ***d;		/* array of directory pointers */
	int *n;			/* number of pointers */
	int *x;			/* current index */
	char *p;		/* directory path */
{
	if (*d == NULL) {

	/*
	 * Allocate first entry.
	 */
	    if ((*d = (char **)malloc(sizeof(char *))) == NULL) {

stkdir_nospace:

		(void) fprintf(stderr,
		    "%s: no space for directory stack at %s\n", Pn, p);
		Exit(1);
	    }
	    *n = 1;
	    *x = 0;
	} else if (*x >= *n) {

	/*
	 * Allocate additional space as required.
	 */
	    *n += 1;
	    if ((*d = (char **)realloc((MALLOC_P *)*d,
		          (MALLOC_S)(*n * sizeof(char *))))
	    == NULL)
		goto stkdir_nospace;
	}
/*
 * Allocate space for the name, copy it there and put its pointer on the stack.
 */
	if (((*d)[*x] = (char *)malloc((MALLOC_S)(strlen(p) + 1))) == NULL) {
	    (void) fprintf(stderr, "%s: no space for %s\n", Pn, p);
	    Exit(1);
	}
	(void) strcpy((*d)[*x], p);
	*x += 1;
}


#if	defined(HASDCACHE)
/*
 * vfy_dev() - verify a device table entry (usually when DCunsafe == 1)
 *
 * Note: rereads entire device table when an entry can't be verified.
 */

int
vfy_dev(dp)
	struct l_dev *dp;		/* device table pointer */
{
	struct stat sb;

	if (!DCunsafe || dp->v)
	    return(1);
	if (RDEV_STATFN(dp->name, &sb) != 0
	||  dp->rdev != RDEV_EXPDEV(sb.st_rdev)
	||  dp->inode != sb.st_ino) {
	   (void) rereaddev();
	   return(0);
	}
	dp->v = 1;
	return(1);
}
# endif	/* defined(HASDCACHE) */
#else	/* !defined(USE_LIB_READDEV) */
static char d1[] = "d"; static char *d2 = d1;
#endif	/* defined(USE_LIB_READDEV) */
