/**** Program to display fixed data base of net land.  ****/

/*	Copyright (C) 1989, 1990, 1991 Massachusetts Institute of Technology
 *		Right to copy granted under the terms described
 *		in the file Copyright accompanying this distribution.
 */


/* This program displays some network connectivity.  The data presently
 * is more or less fixed by values in certain data files.  Eventually I
 * hope to be able to do this more dynamically.
 */

#include "map.h"

#ifdef EDIT_HISTORY

Nbr Date	Author		Description
--- ----	------		-----------
 3  20-Feb-89	M. A. Patton	First release version.
 3A  4-Mar-89	M. A. Patton	Added option aliases for compatability.  Made
				some display related bits indirect there.
 4  17-Mar-89	M. A. Patton	Fixed up greeting stuff to aid variable version
				numbers.  Made using Chaos protocols optional.
 4A 20-Mar-89	M. A. Patton	Fixed byte ordering in same_inet & WriteAddress
 4B 23-Mar-89	M. A. Patton	Keep addresses from Geometry seperate from
				those from Hosts, use ones from Geometry but
				default to Hosts if not given.
 4C 23-Mar-89	M. A. Patton	Added "Version" and "Load Geometry <file>"
				commands.
 4D 23-Mar-89	M. A. Patton	Cosmetics from report by Chris Tengi.
 4E 27-Mar-89	M. A. Patton	General cleanup of FlushGeometryInfo().
 4F  6-Apr-89	M. A. Patton	Added debug and undebug commands and switches.
 4G 29-Jun-89	M. A. Patton	Use new find_file routine for data files.
				Added "debug files" and made messages in Parse
				routines conditional on it.  Added messages at
				start and end of parsing for progress report.
 4H  8-Jul-89	M. A. Patton	Made file lookup outside ParseGeometryFile so
				that file not found could be soft from "Load
				Geometry" command.  geo_path and host_path
				share common table.  Added sleep on fatal error
				so it stays on display to read.  Improve action
				of "Help" key when typein exists.  Added the
				command field to FlushGeometryInfo.  Added code
				to respond to "debug state".
 4I 15-Jul-89	M. A. Patton	Corrected ParseNet to not do Place if no
				display, now runs in ttymode.  Added -ttymode
				option to force no display.
 4J 12-Sep-89	M. A. Patton	Added printing of connectivity data when debug
				flag is on.
 4K 15-Sep-89	M. A. Patton	Made Networks without a name print better, both
				in the above and when queried.
 4L 19-Sep-89	M. A. Patton	String printed in 4J had mismatched parens.
 4M 24-Jan-90	M. A. Patton	InitICMP now fails soft, handle it.  Added take
				and echo commands.  Startup uses the new take.
				Added supplemental host name info in geo file.
 4N 10-Feb-90	M. A. Patton	Added various casting &c. to make call to
				inet_ntoa portable (Sun-4).  Added defaults for
				initial take file not found.
 4O  6-Mar-90	M. A. Patton	Added comment in cmd_quit for lint (and strict
				compilers) to indicate that exit does not
				return.  Removed extra wordiness from message
				for connectivity debugging (it was too busy).
 4P 13-Mar-90	Karl Owen	DG/UX returns a struct from inet_addr...
 5  12-Jun-90	M. A. Patton	Added SNMP (just test for now).  Reworked
				main() for conciseness [added STATE() macro.]
				Moved selectIPAddr here and split for better
				usefulness.
 5A 14-Jun-90	M. A. Patton	Renamed findHostAddr since it specifically
				finds IP addresses.  Moved "command processing"
				part of SNMP stuff back here.  Added args for
				all command functions.
 5B 15-Jun-90	M. A. Patton	Reorganized host description somewhat.
				Including "summarizing" namelist...
				Various cleanup.  Redid commands:
				Hostat->Status, Select->Host.
				Added variables, only one now: snmp_auto.
				Loading files never closed...
 5C 21-Jun-90	M. A. Patton	Added SNMP Neighbors command.  Printing an
				unplaced host correlates networks if the flag
				for connectivity debugging is on.  This makes
				it not print as "[network not displayed]" if
				in fact it is.
 5D 27-Jun-90	M. A. Patton	Spelled out Tektronix in "write TEK" command.
 5E  2-Jul-90	M. A. Patton	Added more SNMP conditionalization...
 5F  8-Aug-90	M. A. Patton	Added "set hardcopy ps/tek" and removed "write
				tek" command.  Added cd command (useful as
				default output directory).  Also removed a
				case that couldn't happen in DescribeMachine.
 5G 15-Aug-90	M. A. Patton	Removed "testing" comment from SNMP command.
				Don't print geometry if not set.  Added space
				to help listings for longer commands.
 5H 15-Aug-90	Edwin Kremer	HP-UX 7.0 prefers "setresuid()" instead of
				"setuid()" -- setuid() is likely to disappear
				in future releases.
 5I 25-Aug-90	M. A. Patton	ParseHost now sets selected_host so errors
				can specify which host...
 5J  1-Apr-91	Mike Patton	Added variable "frame" which is used by the
				PostScript for whether to draw the frame,
				could be used by others, but not put in.
				Improved list of variables (format better) and
				help for set (prints legal values).
				Added support for point-to-point nets.
				[This uses a kludege and displays twice]
 5K  5-Apr-91	Mike Patton	Fixed newline bug in "set ?" output.
				By adding a routine to do all placement
				calculations in the right order before the
				first full Redisplay it's a little faster.
 5L  3-May-91	Mike Patton	Fixed bug, "write geometry" didn't handle the
				new point-to-point nets.  Also made geometry not
				show addresses if unneeded.

#endif
/**** Documentation ****/


#ifdef DOCUMENTATION


Mouse buttons:
    Left:	Describe object pointed to.
		Shift->Verbose form of description.
			On net:		List hosts on that net as well.
			On Host:	Try and ping that host.
		Control->Print Geometry information.
    Middle:	Unmodified: Place selected host here.
		Shift: Delete object pointed to.
    Right:	Reserved for menu interface?

The following commands are also accepted:
    CD			Change working directory.
    Check		Check a host (by name).
    Debug <f>		Turn on debugging flag(s).
    Echo <text>		Echo the text to the "user".
    Help (or ?)		Print internal documentation.
    Host		Select a host (by name).
    List Debug		List the current debug flags set.
    List Protocols	Print a list of all protocols declared by any host.
    List Variables	List all variables and current values.
    Load Geometry	Load a new Geometry file (replacing the old).
    Load Hosts		Load host data from a file (adding to existing data).
    Ping		Send ICMP Echo, report result.
    Quit		Exit from the program.
    Set <var> <val>	Set an operational variable.
    Show All		Show all hosts/nets.
    Show Chaos		Show only ChaosNet hosts/nets.
    Show IP		Show only InterNet hosts/nets.
    SNMP Interface	SNMP query for interface table stats
    SNMP Neighbors	Show neighbors found by traversal of routing table
    SNMP Route		SNMP query to dump routing table
    SNMP System		SNMP query for system info
    Status		Send Chaos STATUS, report result.
    Take <file>		Read commands from file (may be nested).
    UnDebug <f>		Turn off debugging flag(s).
    Version		Print version of running program.
    Write Geometry	Write out geometry database to file.
    Write Hardcopy	Write out file for hardcopy version

See output from help command to get up-to-date list of commands.

#endif
/****	Compiler set up and configuration conditionals.	****/


#include "cmd.h"
#include "icmp.h"
#include "debug.h"
#include <errno.h>


#define STATE(str)	if (DBG_STATE_P) printf(str)

#define DISCARD(p)	{ caddr_t tmp=(caddr_t)p;p=p->next;free(tmp); }

/* Converting a string address to a long value may be system dependant? */
#ifdef DGUX
#define inet_s2l(cp)	(inet_addr(cp).s_addr)
#else
#define inet_s2l(cp)	(inet_addr(cp))
#endif DGUX



#define USAGE "\
usage: %s [host:display] [=<geom>] [-bw #] [-bd <color>] [-bg <color>]\n\
		[-fg <color>] [-hl <color>] [-font <font>] [-W] [-ttymode]\n\
		[-G <GeometryFile>] [-H <HosttableFile>] [<CommandFile>]\n\
"


char *greeting = "MIT LCS interactive network map  	       ";
extern char *version;
char *copyright = "\n\n\
     Copyright (C) 1989, 1990, 1991 Massachusetts Institute of Technology\n\
            This program comes with ABSOLUTELY NO WARRANTY.\n\n";
/****	Random Data	****/


/** Where the data files live: **/
char *dflt_path[] = {
    "",
    "nets/",			/* Really need ~/ here but no globbing yet */
    "/usr/lib/nets/",
    NULL };


/* geometry data */
char *geo_file = "Geometry";
#define geo_path dflt_path
char *geo_ext[] = {
    "",
    ".Geometry",
    NULL };

/* Input for "take" command */
char *take_file = "map.cf";
#define take_path dflt_path
char *take_ext[] = {
    "",
    ".map",
    NULL };

/* master host table */
char *host_file = "Hosts";
#define host_path dflt_path
char *host_ext[] = {
    "",
    ".Host",
    NULL };



/* output files */
/* See later for def of dflt_Hardcopy_output */
static char *dflt_Geometry_output = "map.Geometry";
/****	Main database pointers	****/


Protocol *all_protocols = NULL; /* Union of all protocols spoken by any host */
Media *media_base = NULL;
Network *net_base = NULL;
Machine *machine_base = NULL;
Host *host_base = NULL;
Host *FindHost();
Title *title_base = NULL;




/** Globals to hold original values of argc and argv. **/
int Argc;
char **Argv;



/** Random global flags (controlled by user) **/
int show_chaos = TRUE;
int show_IP = TRUE;
int use_display = TRUE;


int ICMP_ok = FALSE;

extern	int errno;



Host *selected_host = NULL;
/**** Main procedure ****/


main(argc,argv)
int argc;
char **argv;
{   Argc = argc;
    Argv = argv;
    if (Argc>1&&streq(Argv[1],"-D"))
	debug_set(Argv[2]);
    STATE("Starting main procedure.\n");
    {	char *cp;

	cp = InitICMP();
#if defined(hp9000s300)
	setresuid(-1, getuid(), -1);
#else
	setuid(getuid());
#endif
	ICMP_ok = (cp == NULL);
	if (!ICMP_ok)
	{   fprintf(stderr,"ICMP protocol could not initialize.\n\
Reason given was: %s\n\
Continuing, will be unable to test IP host up/down.\n", cp);
	}
    }
    STATE("Getting defaults.\n");
    GetDefaults();
    STATE("Parsing command line.\n");
    if (!ParseArgs())
    {   fprintf(stderr, USAGE, Argv[0]);
	exit(1);
    }
    if (use_display)
    {	STATE("Initializing display.\n");
	use_display = ( InitDisplay()!=NULL );
	if (use_display) STATE("Are using display.\n");
	else STATE("Are not using display.\n");
    }
    Message(greeting);  Message(version);  Message(copyright);
    {	FILE *fd=find_file(take_file, take_path, take_ext, NULL);
	if ( fd != NULL )
	    Take_File(fd);
	else
	{   fprintf(stderr,
		"Can't find file '%s' to initialize from, loading defaults.\n",
				take_file);
	    cmd_ld_host(NULL, NULL);
	    cmd_ld_geo(NULL, NULL);
	}
    }
    Message("\nType \"Help\" for command list and documentation.\n");
#ifdef SNMPDK
    STATE("Initializing SNMP.\n");
    snmpInit();
#endif
    STATE("Off we go.\n");
    if (use_display)
	Display_BackgroundLoop();
    else
	Term_BackgroundLoop();
}
/****	Random "top level" helper functions.	****/



error(cp)
char *cp;
{   fprintf(stderr,"%s: FATAL ERROR: %s.\n",Argv[0],cp);
    /* Since this is often started direct from an xterm which exits when
     * we do, put in a pause for the message to get read.
     */
    if (use_display) sleep(60);
    exit(1);
}



Message(cp)
char *cp;
{   if (use_display)
	Display_Message(cp);
    else
	fputs(cp, stdout);
}



/**	Background Loop for non-display interaction	**/


Term_BackgroundLoop()
{   char buffer[80];

    /* This should check for terminal first.  */
    /*  In fact it should probably just use the following... */
    printf("%s: ",Argv[0]);
    /* This should be using GetLine, and defaulting on blank line, and ... */
    /*     But all of that should wait for the improved GetLine		   */
    while (gets(buffer))
    {	if (buffer[0]!='\0') HandleInput(buffer);
	printf("%s: ",Argv[0]);
    }
}



/**	Process commands from a file	**/
Take_File(fd)
FILE *fd;
{   char *cp;

    /* This should check for terminal first and prompt if so.  */
    while (cp=GetLine(fd))
	HandleInput(cp);
}
/**** Command line processing ****/




ParseArgs()
{   register int i;

    for (i=1; i<Argc; i++)
    {	if (Argv[i][0] == '=')
	{   geometry = Argv[i];
	    continue;
	}
	if (index(Argv[i],':')!=NULL)
	{   display_name = Argv[i];
	    use_display = TRUE;
	    continue;
	}
	S_ARG("-background",back_color);
	S_ARG("-bd",border_color);
	S_ARG("-bg",back_color);
	S_ARG("-bordercolor",border_color);
	I_ARG("-borderwidth",border_width);
	I_ARG("-bw",border_width);
	if (streq(Argv[i],"-D") || streq(Argv[i],"-DEBUG"))
	{   if (++i>=Argc) return (FALSE);
	    debug_set(Argv[i]);
	    continue;
	}
	if (streq(Argv[i],"-display"))
	{   if (++i>=Argc) return (FALSE);
	    display_name = Argv[i];
	    use_display = TRUE;
	    continue;
	}
	S_ARG("-fg",fore_color);
	S_ARG("-font",font_name);
	S_ARG("-foreground",fore_color);
	S_ARG("-geometry",geometry);
	S_ARG("-G",geo_file);
	S_ARG("-H",host_file);
	S_ARG("-hl",highlight_color);
	if (streq(Argv[i],"-ttymode"))
	{   use_display = FALSE;
	    continue;
	}
	if (streq(Argv[i],"-U") || streq(Argv[i],"-UNDEBUG"))
	{   if (++i>=Argc) return (FALSE);
	    debug_unset(Argv[i]);
	    continue;
	}
	B_ARG("-W",use_display);
	if (*Argv[i] == '-') return(FALSE);
	take_file = Argv[i];
    }
    return (TRUE);
}
/****	Main command processing and help table	****/


/* Forward references to all implementing functions... */
extern int cmd_show(), cmd_cwd(), cmd_set(), cmd_ping(), cmd_status();
extern int cmd_quit(), cmd_list(), cmd_load(), cmd_debug(), cmd_undebug();
extern int cmd_host(), cmd_check(), cmd_echo(), cmd_take(), cmd_help();
extern int cmd_write(),  cmd_version();
extern int cmd_snmp(), cmd_snmp_help(), cmd_for_SNMP();
extern int cmd_print(), cmd_wr_geo(), cmd_write_help();
extern int cmd_chaos(), cmd_IP(), cmd_all(), cmd_show_help();
extern int cmd_proto();
extern int cmd_list_vars(), cmd_list_help(), cmd_list_debug();
extern int cmd_ld_geo(), cmd_ld_host(), cmd_load_help();

char *(DisplayHelpMessage[]) = {
    "Point at something and click left to get data about it.  Using the shift",
    "	key prints additional information, and the control key adds the",
    "	placement information.",
    "The middle mouse button is for modification.  Normally specifies a new",
    "	placement of selected host.  Shift means delete the object.",
    "The right mouse button is reserved for future menu interface.",
    "",
    "The following commands are also accepted:",
    NULL
    };
char *(HelpMessage[]) = {
    "List of commands:",
    NULL
    };
CmdTable cmd_table[] = {
    { "?",	cmd_help, NULL,
	  "Print this information." },
    { "CD",  cmd_cwd, NULL,
	  "Change default directory." },
    { "Check",  cmd_check, NULL,
	  "Check a host (by name)." },
    { "Debug",  cmd_debug, NULL,
	  "Turn on some debug flags." },
    { "Echo",	cmd_echo, NULL,
	  "Type out a string to the user." },
    { "Help",	cmd_help, NULL,
	  "Print internal documentation." },
    { "Host",	cmd_host, NULL,
	  "Select a host (by name)." },
    { "List",	cmd_list, NULL,
	  "List general information.  Use 'List ?' for sub-commands." },
    { "Load",	cmd_load, NULL,
	  "Load information.  Use 'Load ?' for sub-commands." },
    { "Ping",	cmd_ping, NULL,
	  "Send ICMP Echo, report result." },
    { "Quit",	cmd_quit, NULL,
	  "Exit from the program." },
    { "Set",	cmd_set, NULL,
	  "Set an operational variable.  'Set ?' for list." },
    { "Show",	cmd_show, NULL,
	  "Control what is displayed.  'Show ?' for list." },
#ifdef SNMPDK
    { "SNMP",	cmd_snmp, NULL,
	  "SNMP queries for various things.  'SNMP ?' for list." },
#endif	/* SNMPDK */
#ifdef USE_CHAOS
    { "Status",	cmd_status, NULL,
	  "Send Chaos STATUS, report result." },
#endif	/* USE_CHAOS */
    { "Take",	cmd_take, NULL,
	  "Read commands from a file." },
    { "UnDebug",  cmd_undebug, NULL,
	  "Turn off some debug flags." },
    { "Version", cmd_version, NULL,
	  "Print out information on current version of program." },
    { "Write",  cmd_write, NULL,
	  "Write out data.  Use 'Write ?' for list." },
    { NULL, NULL, NULL, NULL }
};
/****	Sub-command processing and help tables	****/




char *(WriteHelpMessage[]) = {
    "The Write command is used to write out various data to files.",
    "The sub-commands may be followed by a file name to use, or the default",
    "of 'map.<type>' will be used.",
    "",
    "The following sub-commands are used:",
    NULL
    };
CmdTable write_cmd_table[] = {
    { "?",	  cmd_write_help, NULL,
	  "List of Write sub-commands." },
    { "Geometry", cmd_wr_geo, NULL,
	  "Write out Geometry file of current layout." },
#ifdef HAVE_HARDCOPY
    { "Hardcopy", cmd_print, NULL,
	  "Write out file for hardcopy." },
#endif
    { NULL, NULL, NULL, NULL }
};


char *(ShowHelpMessage[]) = {
    "The Show command is used to control what information is displayed.",
    "",
    "The following sub-commands are used:",
    NULL
    };
CmdTable show_cmd_table[] = {
    { "?",	cmd_show_help, NULL,
	  "List of Show sub-commands." },
    { "All",	cmd_all, NULL,
	  "Show all hosts/nets." },
    { "Chaos",	cmd_chaos, NULL,
	  "Show only ChaosNet hosts/nets." },
    { "IP",	cmd_IP, NULL,
	  "Show only InterNet hosts/nets." },
    { NULL, NULL, NULL, NULL }
};


char *(ListHelpMessage[]) = {
    "The List command is used to list general information.",
    "",
    "The following sub-commands are used:",
    NULL
    };
extern int cmd_proto(), cmd_list_help(), cmd_list_debug();
CmdTable list_cmd_table[] = {
    { "?",	cmd_list_help, NULL,
	  "List of List sub-commands." },
    { "Debug",	cmd_list_debug, NULL,
	  "List of what debugging flags are currently set." },
    { "Protocols",	cmd_proto, NULL,
	  "List of protocols declared by any host." },
    { "Variables",	cmd_list_vars, NULL,
	  "List of variables and their current values." },
    { NULL, NULL, NULL, NULL }
};


char *(LoadHelpMessage[]) = {
    "The Load command is used to load information from a file.",
    "",
    "The following sub-commands are used:",
    NULL
    };
CmdTable load_cmd_table[] = {
    { "?",	cmd_load_help, NULL,
	  "List of Load sub-commands." },
    { "Geometry",	cmd_ld_geo, NULL,
	  "Load new display geometry file." },
    { "Hosts",	cmd_ld_host, NULL,
	  "Load host data file." },
    { NULL, NULL, NULL, NULL }
};





HandleInput(cmd)
register char *cmd;
{   STATE("Processing command \"");STATE(cmd);STATE("\".\n");
    if (!DoCommand(cmd,cmd_table))
	fprintf(stderr,"\
Invalid command \"%s\", ignored.\n\
Use \"Help\" for more information.\n",	cmd);
}


cmd_write(arg, cmd)
caddr_t arg;
register char *cmd;
{   arg;
    if (!DoCommand(cmd,write_cmd_table))
	fprintf(stderr,"\
Invalid write sub-command \"%s\", ignored.\n\
Use \"write ?\" for more information.\n",	cmd);
}



cmd_show(arg, cmd)
caddr_t arg;
register char *cmd;
{   arg;
    if (!DoCommand(cmd,show_cmd_table))
	fprintf(stderr,"\
Invalid show sub-command \"%s\", ignored.\n\
Use \"show ?\" for more information.\n",	cmd);
}



cmd_list(arg, cmd)
caddr_t arg;
register char *cmd;
{   arg;
    if (!DoCommand(cmd,list_cmd_table))
	fprintf(stderr,"\
Invalid list sub-command \"%s\", ignored.\n\
Use \"list ?\" for more information.\n",	cmd);
}



cmd_load(arg, cmd)
caddr_t arg;
register char *cmd;
{   arg;
    if (!DoCommand(cmd,load_cmd_table))
	fprintf(stderr,"\
Invalid load sub-command \"%s\", ignored.\n\
Use \"load ?\" for more information.\n",	cmd);
}
/****	Generalized host selection/deselection used by many commands.	****/


SelectHostByName(cp)
register char *cp;
{   DeselectHost();
    selected_host = FindHost(cp);
    if (selected_host == NULL) return;
    if ( (selected_host->preferred_name==NULL) ||
	 (!streq(selected_host->preferred_name,cp)) )
	selected_host->preferred_name = new_string(cp);
    /* !!!! The above line provides a slow memory leak since it overwrites !!!!
     * !!!! a pointer to allocated data which may be the last one. !!!!
     */
    if (use_display && selected_host->machine != NULL)
	DisplayHost(selected_host);
}


DeselectHost()
{   if (selected_host != NULL)
    {   Host *hp = selected_host;
	selected_host = NULL;
	if (use_display && hp->machine != NULL)
	    DisplayHost(hp);
    }
}


SelectHost(hp)
register Host *hp;
{   DeselectHost();
    if (hp==NULL) return;
    selected_host = hp;
    if (hp->preferred_name==NULL) hp->preferred_name = hp->names->name;
    if (use_display && (hp->machine != NULL))
	DisplayHost(hp);
}


SelectHostMaybe(host)
char	*host;
{   char	*trim();
    int		addr;

    if (host != NULL)
    {	host = trim(host);
	if (*host == '\0') host = NULL;
    }
    if (host == NULL)
    {	if (selected_host == NULL)
	{   fprintf(stderr,"No host specified.\n");
	    return(0);
	}
	host = selected_host->preferred_name;
    }
    else
    {	SelectHostByName (host);
    }
}


int
findIPAddr(hp)
Host *hp;
{   AddressList *ap;
    MAPCAR(ap,hp->addresses)
	if (ap->address.class == AF_INET)
	    break;
    if (ap==NULL)
    {   fprintf(stderr, "Host %s doesn't have an IP address!\n",
		hp->preferred_name);
	return (0);
    }
    return (ap->address.value);
}
/****	Help commands	****/



cmd_help(arg, rest)
caddr_t arg;
{   arg; rest;
    PrintHelp(use_display?DisplayHelpMessage:HelpMessage,cmd_table);
    return (TRUE);
}

cmd_write_help(arg, rest)
caddr_t arg;
{   arg; rest;
    PrintHelp(WriteHelpMessage,write_cmd_table);
    return (TRUE);
}

cmd_show_help(arg, rest)
caddr_t arg;
{   arg; rest;
    PrintHelp(ShowHelpMessage,show_cmd_table);
    return (TRUE);
}

cmd_list_help(arg, rest)
caddr_t arg;
{   arg; rest;
    PrintHelp(ListHelpMessage,list_cmd_table);
    return (TRUE);
}

cmd_load_help(arg, rest)
caddr_t arg;
{   arg; rest;
    PrintHelp(LoadHelpMessage,load_cmd_table);
    return (TRUE);
}



/* Sort of a hack to help a little.  Eventually needs real partial parser. */
HelpOn(cp)
char *cp;
{   int i=strlen(cp);
    char *buf=(char *)malloc(i+3);
    char *t1=cp, *t2=buf; 

    while (*t1!='\0')
	*t2++ = *t1++;
    *t2++ = ' ';
    *t2++ = '?';
    *t2++ = '\0';
    if (!DoCommand(buf,cmd_table))
	fprintf(stderr, "The input \"%s\" is not a valid command.\n",cp);
    return (TRUE);
}



static
PrintHelp(tp,p)
register char **tp;
register CmdTable *p;
{   printf("----------------------------------\n");
    for(;*tp!=NULL;tp++)
	printf("%s\n",*tp);
    for(;p->cmd!=NULL;p++)
	printf("   %-13s%s\n",p->cmd,p->hlpmsg);
}
/****	Simple commands	****/


int
cmd_chaos(arg, rest)
caddr_t arg;
{   arg; rest;
    show_chaos = TRUE;
    show_IP = FALSE;
    if (use_display) ForceRedisplay();
    return (TRUE);
}

int
cmd_IP(arg, rest)
caddr_t arg;
{   arg; rest;
    show_chaos = FALSE;
    show_IP = TRUE;
    if (use_display) ForceRedisplay();
    return (TRUE);
}


int
cmd_all(arg, rest)
caddr_t arg;
{   arg; rest;
    show_chaos = TRUE;
    show_IP = TRUE;
    if (use_display) ForceRedisplay();
    return (TRUE);
}


int
cmd_version(arg, rest)
caddr_t arg;
{   arg; rest;
    Message("\n\n");
    Message(greeting);
    Message(version);
    Message("\n\n");
    return (TRUE);
}


int
cmd_cwd(arg, cp)
caddr_t arg;
char *cp;
{   extern int sys_nerr;
    extern char *sys_errlist[];

    arg;
    if (cp == NULL)
    {	Message("CD command must have argument.\n");
	return(FALSE);
    }
    if (chdir(trim(cp)) != 0)
    {	Message("CD command failed, reason given was ");
	if (errno < sys_nerr)
	    Message(sys_errlist[errno]);
	else
	    Message("not a valid value.");
	Message("\n");
	return(FALSE);
    }
    return(TRUE);
}



int
cmd_echo(arg, cp)
caddr_t arg;
char *cp;
{   arg;
    if (cp!=NULL)
    {	while (isspace(*cp)&&*cp!='\0')
	    cp++;
	Message(cp);
    }
    Message("\n");
    return (TRUE);
}


int
cmd_quit(arg, rest)
caddr_t arg;
{   arg; rest;
    STATE("Exiting normally.\n");
    exit(0);
    /*NOTREACHED*/
}


int
cmd_take(arg, cp)
caddr_t arg;
char *cp;
{   char *trim(), *name;
    FILE *fd;

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
	cp = take_file;
    if ( (fd=find_file(cp, take_path, take_ext, &name)) == NULL)
    {	fprintf(stderr,"Can't find file '%s' to take, ignored.\n",cp);
	return(FALSE);
    }
    if (DBG_STATE_P)
	fprintf(stderr,"Reading commands from %s ...\n",name);
    Take_File(fd);
    if (DBG_STATE_P)
	fprintf(stderr,"       ... %s read.\n",name);
    free(name);
    return (TRUE);
    /*NOTREACHED*/
}


int
cmd_proto(arg, rest)
caddr_t arg;
{   arg; rest;
    ListProtocols(all_protocols,0);
    return (TRUE);
}
/****	SNMP sub-commands	****/

#ifdef SNMPDK



char *(SnmpHelpMessage[]) = {
    "The SNMP command is used to send SNMP queries to the selected host.",
    "",
    "The following sub-commands are used:",
    NULL
    };
extern int SNMPipAddr(), SNMPintfc(), SNMProute(), SNMPneigh(), SNMPsysinf();
CmdTable snmp_cmd_table[] = {
    { "?",	cmd_snmp_help, NULL,
	  "List of SNMP sub-commands." },
    { "Addresses",	cmd_for_SNMP, (char *)SNMPipAddr,
	  "SNMP query for address list." },
    { "Interfaces",	cmd_for_SNMP, (char *)SNMPintfc,
	  "SNMP query for interface table stats." },
    { "Neighbors",	cmd_for_SNMP, (char *)SNMPneigh,
	  "List all neighbors found in routing table." },
    { "Routing",	cmd_for_SNMP, (char *)SNMProute,
	  "SNMP query for routing table." },
    { "System",		cmd_for_SNMP, (char *)SNMPsysinf,
	  "SNMP query for sytem info." },
    { NULL, NULL, NULL, NULL }
};


cmd_snmp(arg, cmd)
caddr_t arg;
register char *cmd;
{   arg;
    if (!DoCommand(cmd,snmp_cmd_table))
	fprintf(stderr,"\
Invalid snmp sub-command \"%s\", ignored.\n\
Use \"snmp ?\" for more information.\n",	cmd);
}



cmd_snmp_help(arg, rest)
caddr_t arg;
{   arg; rest;
    PrintHelp(SnmpHelpMessage,snmp_cmd_table);
    return (TRUE);
}



cmd_for_SNMP(rtn, host)
int (*rtn)();
char	*host;
{   SelectHostMaybe(host);
    if ( selected_host == NULL )
    {	if (host)
	    fprintf(stderr, "Don't know host %s.\n", host);
	else
	    fprintf(stderr, "Must specify host.\n");
	return(FALSE);
    }
    (*rtn)(selected_host);
}


#endif
/****	Debugging on/off	****/
cmd_debug(arg, cp)
caddr_t arg;
char *cp;
{   char *trim();

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') return(FALSE);
    }
    if (cp==NULL) return(FALSE);
    if (!debug_set(cp))
    	fprintf(stderr,"Don't recognize debug option %s.\n",cp);
    return (TRUE);
}

cmd_undebug(arg, cp)
caddr_t arg;
char *cp;
{   char *trim();

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (!debug_unset(cp))
    	fprintf(stderr,"Don't recognize debug option %s.\n",cp);
    return (TRUE);
}


cmd_list_debug(arg, rest)
caddr_t arg;
{   arg; rest;
    debug_list();
    return (TRUE);
}
/****	Variables	****/

static char *VarHelpMessage[] = {
    "The set command is used to change operational variables.",
    "To see what values the variables have now use \"list var\".",
    "",
    "List of variables, what they are for and legal values:",
    "",
    NULL,
};


#ifdef SNMPDK
/* Variable:	SNMP	When to do snmp queries	*/
static char *val_snmp[] = { "Never", "Auto", "Always", NULL };
enum { Never, Auto, Always } snmp_auto = Auto;
#endif /* defined(SNMPDK) */


#ifdef HAVE_HARDCOPY
#if (defined(HARDCOPY_PS) + defined(HARDCOPY_TEK)) > 1
/* Variable:	Hardcopy	What type of hardcopy to write	*/
static char *val_hardcopy[] = {
#ifdef HARDCOPY_PS
    "PS",
#endif
#ifdef HARDCOPY_TEK
    "TEK",
#endif
    NULL };
enum {
#ifdef HARDCOPY_PS
    PS,
#endif
#ifdef HARDCOPY_TEK
    TEK
#endif
    } hardcopy_type = PS;
#else	/* (i.e. if only one kind of hardcopy...) */
#define hardcopy_type 0
#endif
extern int Hardcopy(), HardcopyTEK();
static int (*hardcopy_func[])() = {
#ifdef HARDCOPY_PS
    Hardcopy,
#endif
#ifdef HARDCOPY_TEK
    HardcopyTEK
#endif
    };
static char *dflt_Hardcopy_output[] = {
#ifdef HARDCOPY_PS
    "map.PS",
#endif
#ifdef HARDCOPY_TEK
    "map.TEK"
#endif
    };
#endif


#ifdef HARDCOPY_PS
/* Variable:	Frame	Draw frame around diagram	*/
static char *val_frame[] = { "Noshow", "Show", NULL };
enum { Noshow, Show } ps_frame = Noshow;
#endif /* defined(HARDCOPY_PS) */


static struct _vars_type_ {
	char *name;
	int *valp;
	char **valtbl;
	char *doc;
    } vars[] = {
#ifdef HARDCOPY_PS
	{ "Frame", (int *)&ps_frame, val_frame,
	      "Draw frame around diagram" },
#endif /* defined(HARDCOPY_PS) */
#if (defined(HARDCOPY_PS) + defined(HARDCOPY_TEK)) > 1
	{ "Hardcopy", (int *)&hardcopy_type, val_hardcopy,
	      "What type of hardcopy to write" },
#endif
#ifdef SNMPDK
	{ "SNMP", (int *)&snmp_auto, val_snmp, "When to do snmp queries" },
#endif
	{ NULL, NULL, NULL, NULL }
    };

cmd_set(arg, cp)
caddr_t arg;
char *cp;
{   char *trim();
    char *tp;
    struct _vars_type_ *p;
    struct _vars_type_ *result = NULL;
    char **vp;
    int val = -1;

    arg;
    if (cp==NULL) return(FALSE);
    cp = trim(cp);
    if (*cp=='\0') return(FALSE);
    if (streq(cp,"?"))
    {	printf("----------------------------------\n");
	for(vp=VarHelpMessage;*vp!=NULL;vp++)
	    printf("%s\n",*vp);
	for(p=vars;p->name!=NULL;p++)
	{   printf("   %-10s%s\n%25s",p->name,p->doc,"Values");
	    tp = ": ";
	    for (vp=p->valtbl;*vp!=NULL;vp++)
	    {	printf("%s%s",tp,*vp);
		tp = ", ";
	    }
	    printf("\n");
	}
	return(TRUE);
    }
    for (tp=cp;*tp!='\0'&&!isspace(*tp);tp++)
	;
    *tp++ = '\0';
    cp = trim(cp);
    tp = trim(tp);
    for(p=vars;p->name!=NULL;p++)
    {	char *tp1=p->name;
	char *tp2=cp;
	while (*tp2!='\0'&&upcase(*tp1)==upcase(*tp2))
	{   tp1++;
	    tp2++;
	}
	if (*tp2=='\0')
	{   if (result!=NULL)
	    {	/* duplicate */
		fprintf(stderr,"Ambiguous variable %s.\n",cp);
		return (FALSE);
	    }
	    result = p;
	}
    }
    if (result == NULL)
    {	fprintf(stderr,"Don't recognize variable %s.\n",cp);
	return (FALSE);
    }
    for (vp=result->valtbl;*vp!=NULL;vp++)
    {	char *tp1=(*vp);
	char *tp2=tp;
	while (*tp2!='\0'&&upcase(*tp1)==upcase(*tp2))
	{   tp1++;
	    tp2++;
	}
	if (*tp2=='\0')
	{   if (val!=-1)
	    {	/* duplicate */
		fprintf(stderr,"Ambiguous value %s.\n",tp);
		return (FALSE);
	    }
	    val = vp - result->valtbl;
	}
    }
    if (val == -1)
    {	fprintf(stderr,"Don't recognize value %s.\n",tp);
	return (FALSE);
    }
    *(result->valp) = val;
}

cmd_list_vars(arg, rest)
caddr_t arg;
{   struct _vars_type_ *p;
    char **vp;
    static char *fmt_string = "%10s %-10s\n";

    arg; rest;
    printf("----------------------------------\n");
    printf("List of variables and their present values:\n\n");
    printf(fmt_string, "Name", "Current value");
    printf(fmt_string, "--------", "------------");
    for(p=vars;p->name!=NULL;p++)
    {	int i = *(p->valp);
	for (vp=p->valtbl; i>0; vp++)
	    i--;
	printf(fmt_string, p->name, *vp);
    }
    return (TRUE);
}
/****	Command to print out a hardcopy.	****/


#ifdef HAVE_HARDCOPY
int
cmd_print(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim();

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
	cp = dflt_Hardcopy_output[(int)hardcopy_type];
    (*hardcopy_func[(int)hardcopy_type])(cp);
    return (TRUE);
}
#endif
/****	Commands to read and write geometry data	****/


int
cmd_wr_geo(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim();

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
	cp = dflt_Geometry_output;
    WriteGeometry(cp);
    return (TRUE);
}


int
cmd_ld_geo(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim(), *name;
    FILE *fd;

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
	cp = geo_file;
    if ( (fd=find_file(cp,geo_path,geo_ext,&name)) == NULL)
    {	fprintf(stderr,"No geometry file for '%s', command ignored.\n",cp);
	return(FALSE);
    }
    if (use_display) ForceRedisplay();
    FlushGeometryInfo();
    if (use_display)
    {	ForceDisplay();
	printf("Loading geometry data from %s ...\n",name);
	ParseGeometryFile(fd);
	fclose(fd);
	printf("       ... %s loaded.\n",name);
	free(name);
	CalculatePlacement();
    }
    return (TRUE);
}
/****	Commands to read and write host data	****/


int
cmd_ld_host(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim(), *name;
    FILE *fd;

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
	cp = host_file;
    if ( (fd=find_file(cp,host_path,host_ext,&name)) == NULL)
    {	fprintf(stderr,"No Hosts file for '%s', command ignored.\n",cp);
	return(FALSE);
    }
    if (use_display) ForceDisplay();
    printf("Loading host data from %s ...\n",name);
    ParseHostFile(fd);
    fclose(fd);
    printf("       ... %s loaded.\n",name);
    free(name);
    if (use_display) ForceRedisplay();
    return (TRUE);
}
/****	Host selection	****/


int
cmd_host(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim();

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
    {	fprintf(stderr,"No host to select?\n");
	return (TRUE);
    }
    SelectHostByName(cp);
    if (selected_host == NULL)
	fprintf(stderr,"Unknown host \"%s\".\n",cp);
    else
    {	printf("----------------------------------\n");
	DescribeHost(selected_host,0);
    }
    return (TRUE);
}



int
cmd_check(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim();

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp!=NULL)
	SelectHostByName(cp);
    if (selected_host == NULL)
    {	if (cp==NULL)
	    fprintf(stderr,"No selected host to check.\n");
	else
	    fprintf(stderr,"Unknown host \"%s\".\n",cp);
    }
    else
    {	printf("----------------------------------\n");
	DescribeHost(selected_host,-1);
    }
    return (TRUE);
}
/****	IP status command	****/


/* This is a dummy routine for now, eventually it will call on the domain
 * code, but that isn't in yet.  This means you can't ping anyone not in
 * the table by name. */
int IP_Addr(cp) char *cp;
{   if (isdigit(*cp))
	return (inet_s2l(cp));
    return(0);
}


int
cmd_ping(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim();
    int addr;

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
    {   if (selected_host == NULL)
	{   fprintf(stderr,"No host to ping.\n");
	    return (FALSE);
	}
	cp = selected_host->preferred_name;
    }
    else
	SelectHostByName(cp);
    if (selected_host == NULL)
    {   addr = IP_Addr(cp);
	if (addr == 0)
	{   printf("Can't resolve name %s.\n",cp);
	    return (FALSE);
	}
    }
    else
    {   AddressList *ap;
	MAPCAR(ap,selected_host->addresses)
	    if (ap->address.class == AF_INET)
		break;
	if (ap==NULL)
	{   printf("Host %s doesn't have an IP address!\n",cp);
	    return (FALSE);
	}
	addr = ap->address.value;
    }
    if (use_display) ForceDisplay();
    if (ICMP_ok)
	if (IP_Poll(addr))
	    printf("%s is responding to ICMP/Echo.\n", cp);
	else
	    printf("%s not responding to ICMP/Echo.\n", cp);
    else
	printf("Cannot ping hosts, ICMP protocol not initialized.\n");
    return (TRUE);
}
/****	Chaos status command	****/


#ifdef USE_CHAOS


int
cmd_status(arg, cp)
caddr_t arg;
register char *cp;
{   char *trim();
    register int addr;

    arg;
    if (cp!=NULL)
    {	cp = trim(cp);
	if (*cp=='\0') cp = NULL;
    }
    if (cp==NULL)
    {	if (selected_host == NULL)
	{   fprintf(stderr,"No host to test.\n");
	    return (TRUE);
	}
	cp = selected_host->preferred_name;
    }
    else
	SelectHostByName(cp);
    if (selected_host == NULL)
    {	if ((addr = chaos_addr(cp, 0)) == 0)
	{   printf("%s unknown hostname on Chaos net.\n",cp);
	    return (FALSE);
	}
    }
    else
    {   AddressList *ap;
	MAPCAR(ap,selected_host->addresses)
	    if (ap->address.class == AF_CHAOS)
		break;
	if (ap==NULL)
	{   printf("Host %s doesn't have a CHAOS address!\n",cp);
	    return (FALSE);
	}
	addr = ap->address.value;
    }
    if (use_display) ForceDisplay();
    if (ChaosHostat(addr))
	printf("%s is responding to Chaos STATUS.\n",cp);
    else
	printf("%s not responding to Chaos STATUS.\n",cp);
    return (TRUE);
}

#endif
/**** Detailed output routines, Hosts ****/


DescribeMachine(p,modifier)
register Machine *p;
register int modifier;
{   Host *hp = FindHost(p->name);

    printf("Machine %s",p->name);
    if (hp != p->host)
	printf(" (Full name %s)",p->host->names->name);
    printf(":   ");
    DescribeHost(p->host,modifier);
}



static
DescribeHost(hp,modifier)
register Host *hp;
register int modifier;
{   register char *cp;
#ifdef SNMPDK
    int do_snmp = FALSE;

    if (VERBOSE_P(modifier))
    {	if (snmp_auto == Auto)
	{   Protocol *p = NULL;
	    MAPCAR(p,hp->protocols)
		if (match(p->name,"IP"))
		{   MAPCAR(p,p->child)
			if (match(p->name,"UDP"))
			{   MAPCAR(p,p->child)
				if (match(p->name,"SNMP"))
				    break;
			    break;
			}
		    break;
		}
	    do_snmp = (p!=NULL);
	}
	else if (snmp_auto == Always)
	    do_snmp = TRUE;
    }
#endif
    if ((cp=hp->preferred_name) == NULL)
	cp = hp->names->name;
    printf("%s ", cp);
    printf("is registered as a %s running %s.\n", hp->hardware, hp->opsys);
#ifdef SNMPDK
    if (do_snmp)
    {	printf("It reports itself as:  ");
	SNMPsysinf(hp);
    }
#endif
    PrintNameListExclusive(hp->names,cp);
    PrintAddresses(hp->addresses,VERBOSE_P(modifier));
    if (GEOMETRY_P(modifier) && (hp->geo_flag&AlreadyPlaced) )
    {	printf("Specified geometry (flag=%x(hex)) pos=(%d,%d), size=(%d,%d).\n",
	       hp->geo_flag, hp->x_pos, hp->y_pos, hp->width, hp->height);
	printf("Screen position: Upper Left @ (%d,%d), Lower right @ (%d,%d).\n",
	       hp->x1, hp->y1, hp->x2, hp->y2);
    }
    if (hp->protocols == NULL)
	printf("No advertised protocols.\n");
    else
    {	printf("Advertised protocols:\n\t");
	ListProtocols(hp->protocols,8);
    }
}


PrintNameListExclusive(np,cp)
register NameList *np;
char *cp;
{   register NameList *tp, *rp = NULL;
    register int first = TRUE;

    for (;np!=NULL;np=np->next)
    {	if (match(np->name,cp)) continue;
	if (rp==NULL)
	{   rp = (NameList *) malloc(sizeof(NameList));
	    rp ->next = NULL;
	    rp->name = np->name;
	    continue;
	}
	MAPCAR (tp,rp)
	{   if (MatchHost(tp->name,np->name))
	    {	tp->name = np->name;
		break;
	    }
	    if (MatchHost(np->name,tp->name))
		break;
	    if (tp->next==NULL)
	    {	tp->next = (NameList *) malloc(sizeof(NameList));
		tp = tp->next;
		tp->next = NULL;
		tp->name = np->name;
		break;
	    }
	}
	if (tp == NULL)
	    fprintf(stderr,
		"Internal error in name printing, THIS SHOULDN'T HAPPEN!!\n");
    }
    while (rp!=NULL)
    {	printf("%-16s%s\n",first?"Known aliases:":"",rp->name);
	first = FALSE;
	DISCARD(rp);
    }
}
/**** Detailed output routines, Networks ****/



DescribeNetwork(np,modifier)
register Network *np;
int modifier;				/* Verbosity */
{   register Machine *gp;
    register int firstline = 1;
    register AddressList *ap;

    if (np->name==NULL  || *(np->name)=='\0' )
	printf("Unnamed network");
    else
	printf("Network %s", np->name);
    printf(", Media=%s.\n", np->media->name);
    if (VERBOSE_P(modifier) || (np->nbr_hosts == -1))
    {	if (use_display) ForceDisplay();
	CountHosts(np,VERBOSE_P(modifier));
    }
    printf("%d Hosts registered.\n",np->nbr_hosts);
    PrintAddresses(np->addresses,FALSE);
    MAPCAR (gp,machine_base)
    {	register int firstaddr = TRUE;
	MAPCAR (ap,gp->host->addresses)
	{   if (ap->net_p == np)
	    {	if (firstaddr)
		{   printf("%-20s%-16s",firstline?"Displayed machines:":"",gp->name);
		    firstline = 0;
		}
		else
		    printf(", ");
		firstaddr = FALSE;
		WriteAddress(stdout,ap->address,TRUE);
	    }
	}
	if (!firstaddr)
	    printf("\n");
    }
    if (GEOMETRY_P(modifier))
    {	printf("Specified geometry (flag=%x(hex)) pos=(%d,%d), size=(%d,%d).\n",
	       np->geo_flag, np->x_pos, np->y_pos, np->width, np->height);
	printf("Screen position: Upper Left @ (%d,%d), Lower right @ (%d,%d).\n",
	       np->x1, np->y1, np->x2, np->y2);
    }
}
/**** Output helper routines ****/


#define CANNOT_TEST	2

PrintAddresses(ap,testp)
register AddressList *ap;
register int testp;
{   register int first = TRUE;
    extern Network *FindNetwork();
    
    MAPCAR (ap,ap)
    {	printf("%-12s",first?"Addresses:":"");
	first = FALSE;
	WriteAddress(stdout,ap->address,TRUE);
	if (DBG_CONNECTIVITY_P)
	{   if (ap->net_p == NULL) ap->net_p = FindNetwork(ap->address);
	    if (ap->net_p)
	    {	AddressList *tp;
		char c='(';
		if (ap->net_p->name==NULL  || *(ap->net_p->name)=='\0' )
		    printf(" [Unnamed ");
		else
		    printf(" [%s ",ap->net_p->name);
		MAPCAR (tp,ap->net_p->addresses)
		{   printf("%c",c);
		    WriteAddress(stdout,tp->address,FALSE);
		    c=',';
		}
		if (c=='(') printf("(No addresses");
		printf(")]");
	    }
	    else
		printf(" [network not displayed]");
	}
	if (testp)
	{   register int responding;
	    fflush(stdout);
	    if (use_display) ForceDisplay();
	    switch(ap->address.class)
	    {
	    case AF_INET:
		responding = ICMP_ok ?
				IP_Poll(ap->address.value) : CANNOT_TEST;
		break;
#ifdef USE_CHAOS
	    case AF_CHAOS:
		responding = ChaosHostat(ap->address.value);
		break;
#endif
	    default:
		responding = CANNOT_TEST;
	    }
	    switch (responding)
	    {
	    case FALSE:
		printf(" not responding.");
		break;
	    case TRUE:
		printf(" responding.");
		break;
	    case CANNOT_TEST:
		printf(" cannot be tested (protocol not installed).");
	    }
	}
	printf("\n");
    }
}



WriteAddress(fd,addr,verbose)
register FILE *fd;
NetAddress addr;
register int verbose;
{   switch (addr.class)
    {
    case AF_INET:
	fputs(inet_ntoa(*((struct in_addr *)&(addr.value))),fd);
	break;

    case AF_CHAOS:
	fprintf(fd,"Chaos %o",addr.value);
	if (verbose)
	    fprintf(fd," (%o/%o)",addr.value>>8,addr.value&0xFF);
	break;
    }
}
/****	Print a formatted listing of a protocol structure.	****/

ListProtocols(p,indent)
register Protocol *p;
register int indent;
{   register int i;
    register int printed = FALSE; /* Whether any items have been printed */

    while (p!=NULL)
    {	if (p->child == NULL)
	{   /* Childless protocols just list on same line */
	    if (printed) printf(", ");
	    printed = TRUE;
	    printf("%s", p->name);
	    p = p->next;
	    continue;
	}
	/* This node does have children! */
	if (printed)
	{   printf("\n");
	    for (i=0;i<indent;i++) printf(" ");
	}
	printf("%s: ", p->name);
	ListProtocols(p->child,indent+strlen(p->name)+2);
	p = p->next;
	if (p!=NULL)
	    for (i=0;i<indent;i++) printf(" ");
	printed = FALSE;
    }
    if (printed) printf("\n");
}
/**** Database lookup routines ****/



Media *
FindMedia(cp)
register char *cp;
{   register Media *p;

    MAPCAR (p,media_base)
	if (match(p->name,cp))
	    return (p);
    p = (Media *)malloc(sizeof(Media));
    if (p==NULL)
	error("Allocation error for media");
    p->pixel = use_display ? InterpretColor(cp,foreground) : 1;
    p->name = new_string(cp);
    if (p->name == NULL)
	error("Allocation error for media name");

    p->next = media_base;
    media_base = p;

    return (p);
}




Host *
FindHost(cp)
register char *cp;
{   register Host *hp;
    register NameList *np;

    MAPCAR (hp,host_base)
	MAPCAR (np,hp->names)
	    if (MatchHost(cp,np->name))
		return (hp);
    return (NULL);
}
/**** Database lookup routines, continued ****/



Network *
FindNetwork(addr)
NetAddress addr;
{   register Network *p;
    MAPCAR (p,net_base)
	if (AddrInList_p(addr,p->addresses))
	    return(p);
    return (NULL);
}


/* True if the address is in the address list */
AddrInList_p(addr,p)
NetAddress addr;
AddressList *p;
{   register AddressList *ap;

    MAPCAR (ap,p)
    {   if (ap->address.class != addr.class) continue;
	switch (addr.class)
	{
	case AF_INET:
	    if (same_inet(ap->address.value,addr.value))
		return(TRUE);
	    break;
	    
	case AF_CHAOS:
	    if (ap->address.value == (addr.value&0xFF00))
		return (TRUE);
	    break;
	    
	default:
	    error("Bad addr.class");
	}
    }
    return(FALSE);
}


/* a1 is net, a2 is host */
same_inet(a1,a2)
register long a1, a2;
{   register long byt = 0xFF;
    register long msk = 0x00;

    /*  This is a bit of a hack for now, it makes up a mask to be whatever
	bytes exist in the network address part.  Eventually networks should
	specify masks (and a geometry file should give a default)!!!! */
    while (byt != 0)
    {	if ( (a1&byt) != 0) msk |= byt;
	byt <<= 8;
    }
    return (a1 == (a2&msk));
}
/**** Database load ****/



ParseGeometryFile(fd)
FILE *fd;
{   char *bp, *cp;

    while (bp=GetLine(fd))
    {	if ((cp=index(bp,':')) == NULL)
	{   fprintf(stderr,"No colon on line in geometry file:\n%s\n",bp);
	    continue;
	}
	*cp++ = '\0';
	bp = trim(bp);
	if (match("Net",bp))
	    ParseNet(cp);
	else if (match("Machine",bp))
	    ParseMachine(cp);
	else if (match("Title",bp))
	    ParseTitle(cp);
	else
	    fprintf(stderr,"Unknown entry type %s in geometry file.\n",bp);
    }
}


ParseHostFile(fd)
FILE *fd;
{   register char *bp;
    register char *cp;
    extern char *trim();
    
    while (bp=GetLine(fd))
    {	if ((cp=index(bp,':')) == NULL)
	{   fprintf(stderr,"No colon on line in host file:\n%s\n",bp);
	    continue;
	}
	*cp++ = '\0';
	bp = trim(bp);
	if (match("HOST",bp) || match("GATEWAY",bp))
	    ParseHost(cp);
	else if (match("NET",bp))
	    /* Parse a NIC NET entry!! */ ;
	else
	    fprintf(stderr,"Unknown entry type %s in host file.\n",bp);
    }
}
/****	Database output	****/



WriteGeometry(fname)
register char *fname;
{   register FILE *fd;
    register Network *np;
    register Machine *gp;
    register Title *tp;

    if ((fd=fopen(fname,"w"))==NULL)
    {	fprintf(stderr,"Couldn't open Geometry output file.\n");
	return;
    }
    fprintf(fd,";Net Drawing Geometry file\n\n");
    MAPCAR (tp,title_base)
    {	fprintf(fd,"Title: %s : ",tp->text);
	fprintgeo(fd,tp->geo_flag,tp->width,tp->height,tp->x_pos,tp->y_pos);
	fprintf(fd," :\n");
    }
    MAPCAR (np,net_base)
    {	fprintf(fd,"Net: %s : ",np->name);
	if (np->geo_flag & P_P_NET)
	    fprintf(fd,"|%d",np->width);
	else
	    fprintgeo(fd,np->geo_flag,np->width,np->height,np->x_pos,np->y_pos);
	fprintf(fd," : %s : ",np->media->name);
	WriteAddresses(fd,np->addresses);
	fprintf(fd," :\n");
    }
    MAPCAR (gp,machine_base)
    {	Host *hp = FindHost(gp->name);

	if (hp == gp->host)
	    fprintf(fd,"Machine: %s : ", gp->name);
	else
	{   hp = gp->host;
	    fprintf(fd,"Machine: %s {%s} : ", gp->name, hp->names->name);
	}
	fprintgeo(fd,hp->geo_flag,hp->width,hp->height,hp->x_pos,hp->y_pos);
	fprintf(fd," : ");
	if (hp->addresses != hp->table_addresses)
	    WriteAddresses(fd,hp->addresses);
	fprintf(fd," :\n");
    }
    fclose(fd);
}



WriteAddresses(fd,ap)
register FILE *fd;
register AddressList *ap;
{   register int first = 1;

    MAPCAR (ap,ap)
    {	fprintf(fd,"%s",first?"":",");
	first = 0;
	WriteAddress(fd,ap->address,0);
    }
}
/**** Input parser helping routines ****/


long
Octal2Long(cp)
register char *cp;
{   register long result = 0;

    while (*cp)
    {	result *= 8;
	result += *cp++ - '0';
    }
    return (result);
}



NetAddress
net_address(cp)
register char *cp;
{   NetAddress result;
    register char *tp;

    if (isdigit(*cp))
    {	result.class = AF_INET;
	result.value = inet_s2l(cp);
	return (result);
    }
    for (tp=cp; !isspace(*tp); tp++ )
	;
    *tp = '\0';
    while (isspace(*++tp))
	;
    if (match("CHAOS",cp))
    {   result.class = AF_CHAOS;
	result.value = Octal2Long(tp);
	return (result);
    }
    fprintf(stderr,"Can't parse network address %s %s.\n",cp,tp);
    error("Unknown Network");
}
/**** Main input parsing ****/



AddressList *
ParseAddresses(cp1)
register char *cp1;
{   register char *cp2;
    register AddressList *result;
    extern char *trim();

    cp2=cp1;
    if ((cp1=index(cp1,',')) != NULL)
	*cp1++ = '\0';
    cp2 = trim(cp2);
    if ( (cp2==NULL) || (*cp2=='\0')) return (NULL);
    result = (AddressList *)malloc(sizeof(AddressList));
    if (result == NULL)
	error("Allocation error for address");
    result->address = net_address(cp2);
    result->net_p = NULL;
    if (cp1 == NULL)
	result->next = NULL;
    else
	result->next = ParseAddresses(cp1);
    return (result);
}
/**** ???? ****/


ParseNet(cp1)
register char *cp1;
{   char *cp2;
    register Network *p;
    extern char *trim();

    p = (Network *)malloc(sizeof(Network));
    if (p == NULL)
	error("Allocation error for network");
    p->next = NULL;
    cp2 = cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,"Illegally formatted NET line in %s:\n%s\n",
							host_file,cp2);
	free(p);
	return;
    }
    *cp1++ = '\0';
    p->name = new_string(trim(cp2));
    if (p->name == NULL)
	error("Allocation error for network name");
    cp2 = cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,"Illegally formatted NET line for net %s in %s\n",
							p->name,host_file);
	free(p->name);
	free(p);
	return;
    }
    *cp1++ = '\0';
    cp2 = trim(cp2);
    switch (*cp2) {
    default:	p->geo_flag = XParseGeometry(cp2,
					     &p->x_pos, &p->y_pos,
					     &p->width, &p->height);
		break;

    case '|':	p->width = atoi(cp2+1);
		p->geo_flag = P_P_NET;
		break;
    }
    cp2=cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,"Illegally formatted NET line for net %s in %s\n",
							p->name,host_file);
	free(p->name);
	free(p);
	return;
    }
    *cp1++ = '\0';
    p->media = FindMedia(trim(cp2));
    cp2 = cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,"Illegally formatted NET line for net %s in %s\n",
							p->name,host_file);
	free(p->name);
	free(p);
	return;
    }
    *cp1++ = '\0';
    p->addresses = ParseAddresses(cp2);
    p->nbr_hosts = -1;
    /* Now link in */
    if (net_base==NULL)
    {	net_base = p;
    }
    else
    {	register Network *tp;

	tp = net_base;
	while (tp->next!=NULL)
	    tp = tp->next;
	tp->next = p;
    }
}
/**** ???? ****/


static char dummy_sw[] = "Unspecified software";
static char dummy_hw[] = "Unknown hardware";


ParseMachine(cp1)
register char *cp1;
{   char *cp2, *cp3;
    register Machine *p;
    register Host *hp;
    AddressList *ap;
    extern char *trim();
    extern NameList *ParseNames();
    int geo_f, geo_x, geo_y, geo_w, geo_h;

    p = (Machine *)malloc(sizeof(Machine));
    if (p == NULL)
	error("Allocation error for machine");
    p->next = NULL;
    cp2 = cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,"Illegally formatted MACHINE line in %s:\n%s\n",
							host_file,cp2);
	free(p);
	return;
    }
    *cp1++ = '\0';
    if ((cp3=index(cp2,'{')) != NULL)
	*cp3++ = '\0';
    p->name = new_string(trim(cp2));
    if (p->name == NULL)
	error("Allocation error for machine name");
    if (cp3 != NULL)
    {	if ((cp2=index(cp3,'}')) == NULL)
	{   fprintf(stderr,
		    "Bad host name structure (unmatched '{'), host %s.\n",
		    p->name);
	    free(p->name);
	    free(p);
	    return;
	}
	*cp2 = '\0';
	cp3 = new_string(trim(cp3));
	if (cp3 == NULL)
	    error("Allocation error for machine's real name");
    }
    else
	cp3 = p->name;
    cp2=cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,
		"Illegally formatted MACHINE line for machine %s in %s\n",
							p->name,host_file);
	free(p->name);
	free(p);
	return;
    }
    *cp1++ = '\0';
    /* Parse the geometry specified. */
    geo_f = XParseGeometry(trim(cp2),&geo_x,&geo_y,&geo_w,&geo_h);
    cp2=cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,
		"Illegally formatted MACHINE line for machine %s in %s\n",
							p->name,host_file);
	free(p->name);
	free(p);
	return;
    }
    *cp1++ = '\0';
    ap = ParseAddresses(cp2);
    p->host = NULL;
    if (machine_base == NULL)
    {   machine_base = p;
    }
    else
    {	register Machine *tp;

	tp = machine_base;
    	while (tp->next!=NULL)
	    tp = tp->next;
	tp->next = p;
    }
    /* Now compare it against what was in host table */
    if ((hp=FindHost(cp3))==NULL)
    {   if (DBG_FILES_P)
	    fprintf(stderr,
	 "Geometry file lists machine %s, which is not in master host file.\n",
			cp3);
	/* Build a fake host to hold the data */
	hp = (Host *)malloc(sizeof(Host));
	if (hp==NULL) error("Allocation error for forced Host");
	hp->names = ParseNames(cp3);
	hp->addresses = ap;
	hp->table_addresses = NULL; /* i.e. Host file listed no addresses */
	ap = NULL;		/* So it doesn't get freed. */
	hp->machine = NULL;
	hp->hardware = dummy_hw;
	hp->opsys = dummy_sw;
	hp->protocols = NULL;
	hp->next = host_base;
	host_base = hp;
    }
    if (hp->machine != NULL)
    {   fprintf(stderr,
	 "Geometry file lists machine %s, which is already declared.\n\
	Ignoring second entry.\n",
			cp3);
	/* Should combine the data somehow? */
	return;
    }
    p->host = hp;
    hp->machine = p;
    hp->preferred_name = cp3;
    if (ap != NULL) hp->addresses = ap;
    hp->geo_flag = geo_f;
    hp->x_pos = geo_x;
    hp->y_pos = geo_y;
    hp->width = geo_w;
    hp->height = geo_h;
}
/****	Parse a title (random text to be displayed)	****/


ParseTitle(cp1)
register char *cp1;
{   char *cp2;
    register Title *tp;
    extern char *trim();

    tp = (Title *)malloc(sizeof(Title));
    if (tp == NULL)
	error("Allocation error for title");
    tp->next = NULL;
    cp2 = cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,"Illegally formatted TITLE line:\n%s\n",cp2);
	free(tp);
	return;
    }
    *cp1++ = '\0';
    tp->text = new_string(trim(cp2));
    if (tp->text == NULL)
	error("Allocation error for title text");
    cp2=cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,
		"Illegally formatted TITLE line for title %s in %s\n",
							tp->text,host_file);
	free(tp->text);
	free(tp);
	return;
    }
    *cp1++ = '\0';
    /* Parse the geometry specified. */
    tp->geo_flag = XParseGeometry(trim(cp2),
				  &tp->x_pos, &tp->y_pos,
				  &tp->width, &tp->height);
    /* Rest is command to exec if selected */
    cp1 = trim(cp1);
    tp->command =  (*cp1 == '\0')?NULL:new_string(cp1);
    if (title_base == NULL)
    {   title_base = tp;
    }
    else
    {	register Title *tmp;

	tmp = title_base;
    	while (tmp->next!=NULL)
	    tmp = tmp->next;
	tmp->next = tp;
    }
}
/**** ???? ****/



NameList *
ParseNames(cp1)
register char *cp1;
{   register char *cp2;
    register NameList *result = NULL;
    extern char *trim();

    cp2=cp1;
    if ((cp1=index(cp1,',')) != NULL)
	*cp1++ = '\0';
    cp2 = trim(cp2);
    result = (NameList *)malloc(sizeof(NameList));
    if (result == NULL)
	error("Allocation error for name structure");
    result->name = new_string(cp2);
    if (result->name == NULL)
	error("Allocation error for name text");
    if (cp1==NULL)
	result->next = NULL;
    else
	result->next = ParseNames(cp1);
    return (result);
}
/****	Parse the protocols entry from host table.	****/

/* Routine to create, and return one protocol structure. */
Protocol *
MakeProto(nm)
register char *nm;
{   register Protocol *p=(Protocol *)malloc(sizeof(Protocol));

    if (p==NULL)
	error("Allocation error for protocol");
    p->next = p->child = NULL;
    p->name = nm;
    return(p);
}



/* Helper routine for mutual recursive decent of the two data structures */
AddProtocol_internal(pfx,rst,base,all)
register char *pfx;	/* The initial part or NULL. */
register char *rst;	/* The rest, or all */
Protocol **base, **all;
{   register Protocol *t1, *t2;

    if (pfx == NULL)
    {	pfx = rst;
	rst = index(rst,'/');
	if (rst != NULL) *rst++ = '\0';
    }
    pfx = trim(pfx);
    /* pfx = Top protocol, rst = rest of chain or NULL. */
    if (*all == NULL)
	t1 = *all = MakeProto(new_string(pfx));
    else
    {	MAPCAR(t1,*all)
	{   if (match(pfx,t1->name))
		break;
	    if (t1->next == NULL)
	    	t1->next = MakeProto(new_string(pfx));
	}
    }
    pfx = t1->name;
    if (*base == NULL)
	t2 = *base = MakeProto(new_string(pfx));
    else
    {	MAPCAR(t2,*base)
	{   if (match(pfx,t2->name))
		break;
	    if (t2->next == NULL)
	    	t2->next = MakeProto(pfx);
	}
    }
    if (rst != NULL)
	AddProtocol_internal(NULL,rst,&(t2->child),&(t1->child));
}

Protocol *
AddProtocol(pfx,rst,base)
register char *pfx;	/* The initial part or NULL. */
register char *rst;	/* The rest, or all */
Protocol *base;
{   AddProtocol_internal(pfx,rst,&base,&all_protocols);
    return (base);
}
int
match_proto(cp,pat)
register char *cp;
register char *pat;
{   while (upcase(*cp)==*pat)
    {	if (*pat=='\0')
	    break;
	cp++; pat++;
    }
    if (*pat != '\0')
	return (FALSE);
    if ((*cp == '\0') || (*cp=='/'))
	return (TRUE);
    return (FALSE);
}


Protocol *
ParseProtocols(cp1)
register char *cp1;
{   register char *cp2;
    register Protocol *result = NULL;

    while (cp1 != NULL)
    {	cp2 = cp1;
	if ((cp1=index(cp1,',')) != NULL)
	    *cp1++ = '\0';
	cp2 = trim(cp2);
	if (*cp2=='\0') continue;
	/** This kludge is to hack on IP and mung CHAOS-SIMPLE **/
	/**	because they are normally not right in the file. **/
	if (cp2[0]=='C'&&cp2[1]=='H'&&cp2[2]=='A'&&cp2[3]=='O'&&cp2[4]=='S')
	    if (cp2[5]=='-')
	    {	cp2[5]='/';
		result = AddProtocol(NULL,cp2,result);
	    }
	    else if (cp2[5]=='/')
	    {	*--cp2 = 'S';
		cp2[1] = 'T';
		cp2[2] = 'R';
		cp2[3] = 'E';
		cp2[4] = 'A';
		cp2[5] = 'M';
		result = AddProtocol("CHAOS",cp2,result);
	    }
	    else if (cp2[5]=='\0')
		result = AddProtocol(NULL,cp2,result);
	    else
	    {	if (DBG_FILES_P)
		    fprintf(stderr,
	"Host %s: Can't grok CHAOS entry \"%s\" to kludge it, leaving as is.\n",
			    selected_host->names->name, cp2);
		result = AddProtocol(NULL,cp2,result);
	    }
	/* The known protocols under IP */
	else if (match_proto(cp2,"TCP") || match_proto(cp2,"UDP") ||
		 match_proto(cp2,"ICMP") || match_proto(cp2,"EGP") ||
		 match_proto(cp2,"HMP") || match_proto(cp2,"GGP"))
	    result = AddProtocol("IP",cp2,result);
	/* The known top level protocols */
	else if (match_proto(cp2,"MMDF") || match_proto(cp2,"X.25") ||
		 match_proto(cp2,"PUP") || match_proto(cp2,"IP"))
	    result = AddProtocol(NULL,cp2,result);
	else
	{   if (DBG_FILES_P)
		fprintf(stderr,
	"Host %s: Don't know protocol \"%s\".  Defaulting to top level.\n",
			selected_host->names->name, cp2);
	    result = AddProtocol(NULL,cp2,result);
	}
    }
    return(result);
}
/**** ???? ****/



ParseHost(cp1)
register char *cp1;
{   char *cp2;
    register Host *p;
    extern char *trim();

    /* This is built backwards on purpose.  The most frequent operation on
     * this table is these here insertions, and the table is quite long.
     * therefore to speed things up we do it this way since nothing cares.
     * In fact in the final analysis this might want to build some kind of
     * hash table on the names for faster lookup.
     */
    p = (Host *)malloc(sizeof(Host));
    if (p == NULL)
	error("Allocation error for host");
    selected_host = p;
    cp2 = cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,
		"Illegally formatted HOST line in %s:\n%s\n",
						host_file,cp2);
	free(p);
	return;
    }
    *cp1++ = '\0';
    p->table_addresses = p->addresses = ParseAddresses(cp2);
    cp2=cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	register AddressList *tp1, *tp2;
	fprintf(stderr,
		"Illegally formatted HOST line in %s:\n%s\n",
						host_file,cp2);
	for (tp1=p->addresses;tp1!=NULL;)
	{   tp2 = tp1;
	    tp1 = tp1->next;
	    free(tp2);
	}
	free(p);
	return;
    }
    *cp1++ = '\0';
    p->names = ParseNames(cp2);
    p->preferred_name = NULL;
    cp2=cp1;
    cp1 = index(cp1,':');
    if (cp1 == NULL)
    {	fprintf(stderr,
	"Illegally formatted HOST line for %s in %s, hardware, etc. dropped\n",
					p->names->name,host_file);
	p->hardware = NULL;
	p->opsys = NULL;
	p->protocols = NULL;
    }
    else
    {   *cp1++ = '\0';
	p->hardware = new_string(trim(cp2));
	if (p->hardware == NULL)
	    error("Allocation error for hardware type");
	cp2=cp1;
	cp1 = index(cp1,':');
	if (cp1 == NULL)
	{   fprintf(stderr,
	"Illegally formatted HOST line for %s in %s, opsys, etc. dropped\n",
					p->names->name,host_file);
	    p->opsys = NULL;
	    p->protocols = NULL;
	}
	else
	{   *cp1++ = '\0';
	    p->opsys = new_string(trim(cp2));
	    if (p->opsys == NULL)
		error("Allocation error for opsys name");
	    cp2=cp1;
	    cp1 = index(cp1,':');
	    if (cp1 == NULL)
	    {   fprintf(stderr,
	"Illegally formatted HOST line for %s in %s, protocols dropped\n",
					p->names->name,host_file);
		p->protocols = NULL;
	    }
	    else
	    {   *cp1++ = '\0';
		p->protocols = ParseProtocols(cp2);
	    }
	}
    }
    p->machine = NULL;
    p->next = host_base;
    host_base = p;
    selected_host = NULL;
}
/**** Utility subroutines for database interface ****/



/* MatchHost(cp1,cp2)
 *
 * Returns true if the whole first string matches the first characters in
 *	the second string, which then must be followed by the EOS or a dot
 *	seperator.  The comparison is case independant.
 */
MatchHost(cp1,cp2)
register char *cp1, *cp2;
{   for (;*cp1;cp1++,cp2++)
	if (upcase(*cp1)!=upcase(*cp2))
	    return (0);
    return ((*cp2=='\0')||(*cp2=='.'));
}





CountHosts(np,v)
register Network *np;
int v;				/* Verbose? */
{   register Host *hp;
    register AddressList *ap;
    char sep = '\t';

    np->nbr_hosts = 0;
    if (v)
	printf("Registered hosts:");
    MAPCAR (hp,host_base)
    {	MAPCAR (ap,hp->addresses)
	{   if (FindNetwork(ap->address)==np)
	    {	np->nbr_hosts++;
		if (v)
		{   printf("%c",sep);
		    sep = ',';
		    PrintHostName(hp);
		}
		break;
	    }
	}
    }
    if (v)
	printf("\n");
}



PrintHostName(hp)		/* !!!!Should eventually pick "best". */
register Host *hp;
{   if (hp->preferred_name != NULL)
	printf("%s",hp->preferred_name);
    else
	printf("%s",hp->names->name);
}
/**** Routine to dump the geometry data base ****/



FlushGeometryInfo()
{   Network *np;
    Machine *gp;
    Host *hp;
    Title *tp;

    MAPCAR (hp,host_base)
    {	AddressList *ap;
	if (hp->addresses != hp->table_addresses)
	{   ap = hp->addresses;
	    while (ap != NULL) DISCARD(ap);
	    hp->addresses = hp->table_addresses;
	}
	MAPCAR (ap,hp->addresses)
	    ap->net_p = NULL;
	if (hp->machine != NULL)
	{   if (hp->machine->host != hp)
		fprintf(stderr,"Linkage error from host %s during flush.\n",
					hp->preferred_name);
	    else
	    	hp->machine->host = NULL;
	    hp->machine = NULL;
	}
    }
    np = net_base;
    net_base = NULL;
    while (np != NULL)
    {	AddressList *ap=np->addresses;
	while (ap!=NULL) DISCARD(ap);
	free(np->name);
    	DISCARD(np);
    }
    gp = machine_base;
    machine_base = NULL;
    while (gp != NULL)
    {	if (gp->host!=NULL)
	    fprintf(stderr,"Machine %s still has host (during flush).\n",
				gp->name);
	/* Can't free name because it might be held in common! */
	DISCARD(gp);
    }
    tp = title_base;
    title_base = NULL;
    while (tp != NULL)
    {	free(tp->text);
	if (tp->command != NULL)
	    free(tp->command);
	DISCARD(tp);
    }
}
/**** String processing subroutines ****/



/* trim(cp) - remove extraneous characters at ends of string.
 *
 * The actual input string may be modified to place an end-of-string marker
 * if there are trailing blanks, returns pointer to first non-blank.
 */
char *
trim(cp)
register char *cp;
{   register char *rp;
    register char *tp;

    if (cp==NULL) return(NULL);
    while (*cp!='\0')
    {	if (!isspace(*cp)) break;
	cp++;
    }
    if (*cp=='\0') return (cp);
    tp = rp = cp;
    while (*++cp!='\0')
	if (!isspace(*cp))
	    tp = cp;
    *++tp = '\0';
    return (rp);
}




/* match(cp1,cp2)
 *
 * Returns TRUE if the two strings match.  The comparison is case independant.
 * This is a case independant streq.
 */
match(cp1,cp2)
register char *cp1, *cp2;
{   for (;*cp1;cp1++,cp2++)
    {	if (*cp1==*cp2)
	    continue;
	if (islower(*cp1)&&isupper(*cp2))
	    if (*cp1==*cp2+('a'-'A'))
		continue;
	if (isupper(*cp1)&&islower(*cp2))
	    if (*cp1+('a'-'A')==*cp2)
		continue;
	return (0);
    }
    return (*cp2=='\0');
}
