/* ex:set ts=4 sw=4:
 *
 * score.t: you can't tell the players without a scorecard
 *
 * This module provides a single object 'scoreobj' which defines all
 * activity relative to the scoring within Dungeon.  It provides score
 * update notification, player ranking, automatic handling of treasures
 * and treasure repositories.
 *
 * Some code influenced by suggestions and code which was identified as
 * "Copyright (c) 1993 Michael J. Roberts".
 *
 * Non-deterministic (good word) debug code snarfed from Dave Baggett
 *
 * This module is Copyright (c) 1994 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * an backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */
zscoreVersion : versionTag
	id="$Id: score.t_v 1.6 1994/05/03 07:59:52 jeffl Exp jeffl $\n"
	func='scoring'
	author='Jeff Laing with suggestions from Mike Roberts and Dave Baggett'
;

deterministicverb: sysverb
	verb='deterministic' 'norandom' 'norandomize'
	action(a) = { scoreobj.nondeterministic := nil; }
;

// This object hold all information to do with scoring.  it also handles
// the initial "randomization" of the game.  after all, who needs to
// randomize if you are not keeping score...
scoreobj: initialization

	// called by preinit()
	preinit_phase={
		local o;

		// random games by default
		self.nondeterministic := true;		// random games by default

		// reset maximum possible to zero
		global.score := 0;
		global.maxscore := 0;

		// some rooms have points attached
		for (o:=firstobj(room); o<>nil; o:=nextobj(o,room)) {
			if (o.room_value <> nil)
				global.maxscore += o.room_value;
		}

		// treasures have scores attached
		for (o:=firstobj(treasure); o<>nil; o:=nextobj(o,treasure)) {
			if (o.findscore <> nil)
				global.maxscore += o.findscore+o.keepscore;
		}
	}

	// called by init()
	init_phase={
		notify(self,#kick_random_generator,1);
	}

	// this will be called after 1 turn -- this allows the player to
	// turn it off for walkthrough scripts.
	kick_random_generator={
		if (self.nondeterministic) {
			randomize();
		} else {
			"\b*** This will be a deterministic game ***";
		}
	}

    notifyon = true               // announce score increases
    rank = {
        local rk, i;

        "In <<global.turnsofar>> turn(s), your score is <<global.score>>";

		if (global.maxscore=0) {
			".\n";
			return;
		}

		i:=(global.score*20)/global.maxscore;
						rk := 'Beginner';
		if (i >=  1)    rk := 'Amateur Adventurer';
		if (i >=  2)    rk := 'Novice Adventurer';
		if (i >=  4)    rk := 'Junior Adventurer';
		if (i >=  8)    rk := 'Adventurer';
		if (i >= 12)    rk := 'Hacker';
		if (i >= 16)    rk := 'Winner';
		if (i >= 18)    rk := 'Master';
		if (i >= 19)    rk := 'Wizard';
		if (i >= 20)    rk := 'Cheater';

        " out of a possible <<global.maxscore>>. This gives you the rank
        of <<rk>>.\n";
    }

    // This method is called whenever the player's score needs to be
    // adjusted.  If self.notifyon is true, the update will be announced.
    // Its safe to call this routine with 'nil'.
    adjust(amount) = {

        if (amount=nil or amount=0) return; // nothing - forget it then

        if (self.notifyon) {
	        "\b[Your score just went ";
	        if (amount>0) "up by <<amount>>";
	        else          "down by <<-amount>>";
	        " points]\b";
		}

		global.score := global.score + amount;
		setscore( global.score, global.turnsofar );
    }
;

// Treasures give us a score when we find them and again when we stash them
// away in a treasure repository.
class treasure: item        // valuables
	noun='treasure' 'valuable'
	plural='treasures' 'valuables'
    findscore=1
    keepscore=0
    moveInto( obj ) = {

		inherited.moveInto(obj);

        // handle case of moving into player
        if (obj.isIn(Me)) {				// look in knapsack as well!
            if (self.findscore>0) {
                scoreobj.adjust(self.findscore);
                self.findscore:=0;
            }
        }
    }
;

// repositories are a scoring place to put a treasure.  once here, thieves
// should not touch it.
class repository : container   // scoring place to put treasures

    // someone is putting something into the repository.  if its a treasure
    // and it ends up in me, we score it.
    ioPutIn( actor, dobj ) = {
        inherited.ioPutIn( actor, dobj );       // handle it normally
        if (dobj.isIn(self) and isclass(dobj,treasure))
            scoreobj.adjust(dobj.keepscore);
    }

    // someone is taking an object from a repository.  if its a treasure,
    // we knock the score down.
    Grab( obj ) = {
        if (isclass(obj,treasure)) scoreobj.adjust(-obj.keepscore);
        pass Grab;
    }
;

// entry point used by standard adv.t
scoreRank: function
{
	scoreobj.rank;
}

// turn on/off score update notification
notifyVerb: sysverb
	verb = 'notify'
	action( actor ) = {
        scoreobj.notifyon := not scoreobj.notifyon;
        if (scoreobj.notifyon)
		    "[You will now be notified of score changes]";
		else
		    "[You will no longer be notified of score changes.
		    If you change your mind and want to be notified
		    again, just type NOTIFY again]";
		abort;
	}
;

/*
 * it is worth points to reach some rooms.
 */
modify class room 

	// on first visit to this room, we score 'room_value' if its non-nil
	firstseen = {
		if ( self.room_value <> nil)
			scoreobj.adjust( self.room_value );
		pass firstseen;
	}
;
