/* This program is Copyright (c) 1994 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore Ave,
   Maynard, MA 01754. */


/* This is a more sophisticated implementation of "ask about", which allows
   you to establish a kind of "knowledge base" about objects which other
   actors in the game can tell you about.  See the files askdemo.t and
   askdemo2.t for examples of how it can be used; see askabout.doc for
   more information about what it does. */


/* Version history
    3/25/94 DaveA
            * initial version
    3/29/94 Ron Hale-Evans (evans@binah.cc.brandeis.edu)
            * fixed bug on "follower" class
    3/30/94 DaveA:
            * aaTell is now boolean; return nil to allow repeats
            * added optional aaActorsKnowing and aaClassKnowing to make 
              "available" functions easier to write
            * added code to make actors remember the last place they saw an 
              object and tell you when you ask
*/


/*** ASKABOUT FUNCTIONS ***/


/* Attach factoids to their owners and get initial information about item
   locations: call this from preinit () */
aaInit: function {
   local a, c, i, j;

   /* Attach factoids */
   for (i:=firstobj (aaFactoid); i<>nil; i:=nextobj (i, aaFactoid))
      i.aaFactLoc.aaFactoids += i;

   /* Get initial location information for all actors */
   for (a:=firstobj (Actor); a <> nil; a := nextobj (a, Actor)) {
      /* Actor knows about all objects at actor's start location */  
      if (a.location <> nil) 
         for (i:=1; i<=length(a.location.contents); i++)
            aaActorSees (a, a.location.contents[i], a.location);
      /* If actor has a turf, loop over all objects in that class and add
         their contents to actor's known list */
      if (defined (a, &aaTurfClass)) {
         c := a.aaTurfClass;
         for (i := firstobj (c); i <> nil; i := nextobj (i, c))
            for (j:=1; j <= length (i.contents); j++)
               aaActorSees (a, i.contents[j], i); } } }


/* Common function to choose one from a list of possible responses */
aaOneOf: function (stringlist) {
   say (stringlist [(rand (length (stringlist)))]); }


/* Function to return all basic factoids on an object, plus the location
   factoid if it is known.  This would be a method on "thing", except that 
   when objects override the method, they almost always need to do this too.
   Less repeated code this way. */
aaGetBasicFacts: function (obj, actor) {
   local i; local l := [];
   for (i:=1; i<=length(obj.aaFactoids); i++)
      if (obj.aaFactoids[i].aaAvail (actor)) {
         l += obj.aaFactoids[i]; l += &aaTell; }
   if (obj <> actor)
      for (i:=1; i <= length (obj.aaActorSeen); i+=2)
         if (obj.aaActorSeen[i] = actor) {
            l += obj; l += &aaSeenFact; return (l); }
   return (l); }


/* For each item in a room with an actor, or each actor who enters the room
   with an actor, store the actor and the present location on a list of actors
   who have seen the item. */   
aaActorSees: function (actor, obj, loc) {
   local i;

   /* I should already know where I am */
   if (actor = obj) return;

   /* If actor already on list of items seen, update location */
   for (i:=1; i <= length (obj.aaActorSeen); i+=2)
      if (obj.aaActorSeen[i] = actor) {
         obj.aaActorSeen[i+1] := loc; return; }

   /* Not already seen: append actor and location to list of actors seen */
   obj.aaActorSeen += actor; obj.aaActorSeen += loc; }


/*** ASKABOUT CLASSES ***/


/* This is the basic object you will use to create the knowledge base.  You
   will write an aaTheFact method for each factoid; factoids about actors
   should have an aaMyFact method.  Every factoid must have an aaFactLoc
   method, like the location method for items.  Optionally, you may define
   an aaAvail method to limit the times when a factoid could be told; you
   might want a factoid to only be available to certain actors, or only if
   they have already told you some other factoid. */
class aaFactoid: object

   /* If you want to restrict knowledge of the factoid to a list of actors, 
      set the aaActorsKnowing list; actors not on the list will not know the
      factoid.  If you want to restrict to a class of actors, set the 
      aaClassKnowing property to the name of a class; actors not of that 
      class will not know the factoid. */
   aaActorsKnowing = [] aaClassKnowing = nil

   /* The normal job of the tell function is to print the text in "aaTheFact",
      which is the basic text.  However, if you have asked an actor about
      him/herself, the text printed is "aaMyFact" instead.  Return "true"
      from this function to allow repeated tellings of the factoid. */
   aaTell (actor) = {
      if (actor = self.aaFactLoc) { "\""; self.aaMyFact; "\""; }
      else { "\""; self.aaTheFact; "\""; }
      return (nil); }

   /* Normal factoids are always available.  This is what you want to override
      if the factoid is only available to certain actors or at certain times. */
   aaAvail (actor) = { 
      local o := self.aaFactLoc;
      if (length (self.aaActorsKnowing) > 0) {
         if ((find (self.aaActorsKnowing, actor)) <> nil) return (true);
         else return (nil); }
      if (self.aaClassKnowing <> nil) {
         if (isclass (actor, self.aaClassKnowing)) return (true);
         else return (nil); }
      return (true); };


/* A "topic" is a word for a group of things, like "suspects" in a mystery.
   When you ask about "suspects", you are really asking for information about
   either the suspects in general, or information about any one of the people
   in the class of "suspects".  To define a topic, you declare an object of
   this class; give it a noun ('suspects'), and set its aaFactClass to the
   name of a class.  Each of the things in the topic group should have this
   class.  For example, if Jane is a suspect, you might write:
      class suspectClass: Actor;
   and then set the aaFactClass of the 'suspects' object to suspectClass.
   This will group together all of the factoids for all of the members.  You
   may also want to define some factoids about the suspect group, like "All
   of the suspects went to grade school with the victim.". */
class aaTopic: thing
   aaGetFactoids (actor) = {
      local i, l, m, subclass;
      /* Factoids for topic in general */
      l := aaGetBasicFacts (self, actor);

      /* Loop over all items in given class adding factoids to list */
      subclass := self.aaFactClass;
      for (i:=firstobj (subclass); i<>nil; i:=nextobj (i, subclass)) {
         m := i.aaGetFactoids (actor);
         if (m <> nil) l += m; }
      return (l); };


/*** MODIFICATIONS TO EXISTING CLASSES ***/


modify class thing
   /* Place to store factoids and actors who have seen this thing */
   aaFactoids = [] aaActorSeen = []

   /* Return basic factoids, plus location factoid if any */
   aaGetFactoids (actor) = { return (aaGetBasicFacts (self, actor)); }

   /* This is the factoid which describes the last location this actor saw it.
      It would only be called if aaGetFactoids determined this actor knows
      where it is. */
   aaSeenFact (actor) = { 
      local i;
      /* It's easy if the actor is here */
      if (actor.location = self.location) {
         "\"\^"; self.thedesc; " is right here.\" "; }
      /* Dig through list of actors on the object to find the right one */      
      else for (i:=1; i <= length (self.aaActorSeen); i+=2)
         if (self.aaActorSeen[i] = actor) {
            "\"I last saw "; self.thedesc; 
            if (self.aaActorSeen[i+1] = actor.location) " right here";
            else { " in "; self.aaActorSeen[i+1].thedesc; }
            ".\" "; }
      /* This factoid is retellable */
      return (true); };


/* All actors need some code to handle "ask about" */
modify class Actor
   /* When a factoid is told, place on this list; this list is checked to
      make sure a given actor never tells you a fact twice */
   aaFactsTold = []

   /* Add code to beginning of moveInto which ensures that all items in the
      room record the fact they have been seen by this actor */
   replace moveInto (obj) = {
      local i;
      for (i:=1; i<=length (obj.contents); i++) {
         aaActorSees (self, obj.contents[i], obj);
         if (isclass (obj.contents[i], Actor))
            aaActorSees (obj.contents[i], self, obj); }
      pass moveInto; }

   /* Function to see if the factoid under consideration has been told before */
   aaNeverTold (f, m) = { local never, i;
      never := true;
      for (i:=1; i<=length(self.aaFactsTold); i:=i+2)
         if ((f = self.aaFactsTold[i]) and (m = self.aaFactsTold[i+1])) {
            never := nil; break; }
      return (never); }

   /* This function does all the work.  First, build a list of all factoids
      about the object.  Then, shorten the list by removing facts already told.
      Finally, pick one at random and tell it. */
   replace doAskAbout (actor, iobj) = {
      local allfacts:=[], possfacts:=[], i, n;
      allfacts := iobj.aaGetFactoids (self);
      n := length (allfacts);
      /* If there are no factoids at all, told or not, "I don't know". */
      if (n = 0) self.aaDontKnow (iobj);
      else {
         /* Build shorter list possfacts from factoids not yet told */
         for (i:=1; i<=n; i:=i+2)
            if (self.aaNeverTold (allfacts[i], allfacts[i+1])) {
               possfacts += allfacts[i]; possfacts += allfacts[i+1]; }
         n := length (possfacts);
         /* If all factoids already on told list, "I told you everything". */
         if (n = 0) self.aaToldAll (iobj);
         /* Pick and tell the factoid.  Add to told list if return is nil */
         else {
            n := rand (length (possfacts) / 2);
            if ((possfacts[n*2-1].(possfacts[n*2]) (self)) = nil) {
               self.aaFactsTold += possfacts[n*2-1];
               self.aaFactsTold += possfacts[n*2]; } } } }

   /* These are things you can customize for specific personalities */
   aaMaybeMe (obj) = {
      /* Fix required for "follower" class supplied by RWHE */
      if ((obj = self) or (obj = myfollower) or (obj = myactor)) "myself";
      else obj.thedesc; }
   aaToldAll (obj) =
      "\"I've told you everything I know about <<self.aaMaybeMe (obj)>>.\" "
   aaDontKnow (obj) =
      "\"I don't know much about <<self.aaMaybeMe (obj)>>.\" ";


/* Fix for "follower" class supplied by RWHE */
modify class follower
  aaFactsTold       = myactor.aaFactsTold
  aaNeverTold (f,m) = myactor.aaNeverTold (f,m)
  aaFactoids        = myactor.aaFactoids
  aaMaybeMe (obj)   = myactor.aaMaybeMe(obj)
  aaToldAll (obj)   = myactor.aaToldAll (obj)
  aaDontKnow (obj)  = myactor.aaDontKnow (obj);
