// BEGINNING OF THE PLOTDAG.T MODULE
/*
    Welcome to the PLOTDAG.T module, an add-on to the
    ADV.T and STD.T files which are bundled with
    TADS v2.1

    Written by Paul Munn, April 1994 as Senior Project,
    advised by Dr. Parsons (Hofstra College of Liberal
    Arts and Sciences, Computer Science Department)
    and Professor Orr (New College of Hofstra University),
    Hofstra University
    Hempstead, NY  11550-1090
    Syntax assistance by Dave Baggett,
      email: dmb@ai.mit.edu
      An experienced T.A.D.S. game author for ADVENTIONS.
*/

#include <adv.t>
#include <pdstd.t>
/*
    PDSTD.T is not part of this distribution.
    The only required changes between PDSTD.T and STD.T are

    plotDagDaemon: function;      
        added in with the other forward declarations of
        functions and
    setdaemon(plotDagDaemon, nil);
        added on to the init() function.

    You must make those changes.
*/


/*
    The preinit is executed at compile time, before
    any game is started.

    This preinit replaces the standard one by having
    the standard's lamp list creation along with
    the pd_nodes list creation needed by PLOTDAG.T.
*/
replace preinit: function
{
    local o,p,q,r;
    /*
        These are local variables that are dynamically
        type-defined.  That is, their type can be
        changed simply by making an assignment
        statement of a variable of a different type
        to it.
    */

    // list of lightsources (from ccr-std.t)
    /*
        This routine makes a list of objects that are
        of the class light source.  In our sample
        adventure, DAGE1.T there are no light sources.
        It is assumed that there is light available.
        Or that the player has night vision.  It is
        included here to minimize code writing in
        the future.
    */
    global.lamplist := [];
    o := firstobj(lightsource);
    while (o <> nil)
    {
        global.lamplist := global.lamplist + o;
        o := nextobj(o, lightsource);
    }

    // list of pd_nodes
    /*
        These statements develop the list of pd_nodes
        stored in the global "catch-all" object.
    */
    global.pd_nodes := [];
    o := firstobj(pd_node);
    while (o <> nil)
    {
        global.pd_nodes := global.pd_nodes + o;
        o := nextobj(o, pd_node);
    }

    // set each pd_node's parents_max.
    // set each pd_node's done_actions to all zeroes.
    /*
        These statements initialize some of the
        pd_node's necessary values.  It automates
        the setting of parents_max values for
        each set by analyzing the structure of
        the given DAG, freeing the programmer
        from having to worry about it.
    */
    o := length(global.pd_nodes);
    for(p := o; p >= 1; --p)  //For all pd_nodes.
    {
        /*
            Initialize the values for this pd_node before
            filling them in.
        */
        global.pd_nodes[p].done_actions := [];
        global.pd_nodes[p].parents_cur := 0;
        global.pd_nodes[p].parents_max := 0;

        q := length(global.pd_nodes[p].child);
                            //Num of children.
        for(r := 1; r <= q; ++r)  //For all child pd_nodes
        {
            global.pd_nodes[p].child[r].parents_max :=
            global.pd_nodes[p].child[r].parents_max + 1;

            global.pd_nodes[p].done_actions :=
            global.pd_nodes[p].done_actions + 0;
        }
    }

    /*
        Initialize the removal list, reset each
        time it is emptied.

        Every turn the daemon executes, any pd_nodes
        that have had all exit paths taken no longer
        need to be checked, so they are placed on
        this list and deleted after the turn.
    */
    global.pd_nodes[1].remove_list := [];

    // Load first pd_node on frontier.
    /*
        The frontier, which stands for next pd_node list,
        will hold the current pd_nodes on the "frontier"
        of the DAG.  That is, these pd_nodes are the
        only ones that need to be checked for progress
        across the DAG.
    */
    global.pd_nodes[1].frontier := [];
    global.pd_nodes[1].frontier := global.pd_nodes[1].frontier
      + global.pd_nodes[1];

    /*
        Much more processing can be added here.
        Suggestions many have overlooked is counting up
        the point values for all treasure types and
        setting the appropriate global values...
    */
}


/*
    Sets global.debug_dag to true, referenced by the deamon
    and the check_paths method of each pd_node.

    When true, the daemon and check_paths methods display
    very detailed information about their runs between
    turns.
*/
modify global
    debug_dag = true
;


/*
    This function was written out of a need to display
    the pd_nodes in a list of them repeatedly.  This
    is used by the debugging messages allowed by
    setting global.debug_dag to true
*/
dumpStates: function(list)
{
    local a,b;

    a := length(list);
    for(b := 1; b <= a; ++b)
        "<<list[b].sdesc>>\ ";
}


// The PlotDagDaemon
plotDagDaemon: function(parm)
{
    local g,h,i,j,addOk;
    // variables local to this function

    g := global.debug_dag;  //The debugging flag...

    /*
        These statements go through the frontier.
        Note that we do not assign the length
        to a variable and then use that fixed
        length as our watch criterion.
        It is important that we have length(frontier)
        re-evaluated for each iteration of the loop as
        the frontier[i].check_paths may add items to
        that list during the loop's iterations.
    */
    for(i := 1; i <= length(global.pd_nodes[1].frontier);
     ++i)
    {
        // This is one of the debugging messages.
        if(g)
        "\n#DAEMON# Checking pd_node
        \"<<global.pd_nodes[1].frontier[i].sdesc>>\"...";

        /*
            We have check_paths return a boolean (true
            or nil) value that will tell us if the
            pd_node must be removed from the frontier
            (if all of its ways out have been done).
        */
        if(global.pd_nodes[1].frontier[i].check_paths(i))
        {
            // Add to the remove_list
            global.pd_nodes[1].remove_list += global.
            pd_nodes[1].frontier[i];

            if(g)
                "\n#DAEMON# All done_actions finished.
                State will be deleted from frontier.";
        }
        else
            if(g)
                "\n#DAEMON# All done_actions not done.
                Keep it on frontier.";

        if(g)
        "\n#DAEMON# Finished checking pd_node
        \"<<global.pd_nodes[1].frontier[i].sdesc>>\".";
    }

    /*
        Now remove all items in remove_list from the
        frontier
    */
    if(length(global.pd_nodes[1].remove_list) <> 0)
    {
        if(g)
        {
            "\n#DAEMON# Removing done pd_nodes from the
            frontier.";

            "\n#DAEMON# frontier: ";
            dumpStates(global.pd_nodes[1].frontier);

            "\n#DAEMON# remove_list: ";
            dumpStates(global.pd_nodes[1].remove_list);
        }

        h := length(global.pd_nodes[1].remove_list);
        for(i := 1; i <= h; ++i)
            global.pd_nodes[1].frontier -= global.
            pd_nodes[1].remove_list[i];

        //Clean out remove list for next deletion go.
        global.pd_nodes[1].remove_list := [];

        if(g)
        {
            "\n#DAEMON# after removal...";

            "\n#DAEMON# frontier: ";
            dumpStates(global.pd_nodes[1].frontier);
        }
    }// end if anything in remove_list
    else
        if(g)
            "\n#DAEMON# Nothing in remove_list to
             remove.";

    /*
        Now that we're done removing things, we should
        process the frontier so there are no doubles.
        We don't prevent doubles during the above loop
        as new conditions may prompt review of the same
        pd_node again in the turn.
    */
    global.pd_nodes[1].temp := global.pd_nodes[1].frontier;
    global.pd_nodes[1].temp := [];
    //Load first pd_node of frontier onto temporary list.
    global.pd_nodes[1].temp += global.pd_nodes[1].frontier[1];

    if(g)
    {
        "\n#DAEMON# Compacting frontier to remove
        duplicates...";
        "\n#DAEMON# frontier before: ";
        dumpStates(global.pd_nodes[1].frontier);
    }

    h := length(global.pd_nodes[1].frontier);
    /*
        We're looking at the 2nd to the end of the
        frontier to see if we'll add it on.
    */
    for(i := 2; i <= h; ++i)
    {
        /*
            If next pd_node is already in temp,
            don't add it
        */
        addOk := true;
        for(j := 1; j <= length(global.pd_nodes[1].
        temp); ++j) // Check thru temp
        {
            if(global.pd_nodes[1].frontier[i] =
            global.pd_nodes[1].temp[j])
            {
                addOk := nil;
                break;
            }
        }

        if(addOk)  //We add it on to temp.
        {
            global.pd_nodes[1].temp +=
            global.pd_nodes[1].frontier[i];
        }
    }

    /*
        At this point, temp is the next interation's
        frontier.  Make it so.
    */
    global.pd_nodes[1].frontier := global.pd_nodes[1].temp;

    if(g)
    {
        "\n#DAEMON# frontier after: ";
        dumpStates(global.pd_nodes[1].frontier);
    }
}


/*
    The definition of the pd_node class
*/
class pd_node: object
    /*
        The notation "Reset by author" means the default
        values set by this class definition must be
        reset by the author of the pd_nodes for each game.
        "Set by preinit" signifies that the preinit
        method, defined above, will take care of
        scanning through the DAG at compile time and
        making the appropriate calculations.
    */

    noun = 'pd_noden'
    sdesc = "pd_noden"
    /*
        The pd_nodes are given a noun and a ShortDESCription
        to allow their use with the new verb, STATE,
        detailed later on.  For a player to get
        information on the plot DAG's specific pd_node,
        he/she need only type STATE <NOUN> to get that
        information.

        This new STATE verb would be commented out of use
        when an adventure works fine.  For this test
        program, however, the verb is kept.

        Reset by author.
    */

    child_conditions = []
    /*
        A list of pointers to methods in the pd_node the
        author defines.  These child conditions will only
        be evaluated for a pd_node if all of the edges
        leading to it have been traveled.  Each method
        will return a value of true or nil when called.
        If this value is true, the check_paths may
        move to the child of that same edge.
        Reset by author.
    */

    child_actions = []
    /*
        A list of pointers to methods in the pd_node the
        author defines.  Upon travel to the new pd_node
        child[i], child_actions[i] is executed, causing
        a set of actions to be taken.
        Reset by author.
    */

    done_actions = []
    /*
        A list of integers, initialized by the preinit
        function to zeros.  When child_actions[i] are
        executed, done_actions[i] is set to 1.  This
        tells check_paths that we can skip evaluation
        of this path.  It is also used to determine
        if the pd_node has had all children traveled
        to so it can now be removed from the frontier
        (as it's no longer a frontier node).
    */

    child = []
    /*
        A list of identifiers of the child pd_node objects.
        These are not pointers to methods but actual
        object names.  Once all critera for moving to
        the pd_node are satisfied, this identifier is loaded
        on the frontier (stored in the first pd_node) by
        check_paths (if, we'll find later, it's not
        already later on the list).  Set by author.
    */

    parents_cur = 0
    /*
        This integer is the total number of parent nodes
        that have allowed entry to this node at this
        point in time.  Set by preinit.
    */

    parents_max = 0
    /*
        This integer is the total number of parents of
        this node.  When parents_cur = parents_max,
        this node can be exited.  Set by preinit.
    */

    /*
        The following method checks the paths of this
        pd_node and updates its status.
    */
    check_paths(place) =
    {
        local g,i,j,k,l,wantAdd,m;  //local variables.
        g := global.debug_dag;  //The debug variable.

        /*
            We can evaluate child_conditions to try
            and leave this pd_node only when ALL of the
            edges TO the node have been traveled.
        */
        if(self.parents_cur = self.parents_max)
        {
            if(g)
            {
                "\n#CP#  We can leave:  all edges to
                this pd_node are satisfied.";
            }

            j := length(self.child_conditions);
                            //Num of ways out of pd_node.

            if(g)
            {
                "\n#CP# State \"<<self.sdesc>>\" has
                \ <<j>> child-conditions.";
            }

            for(i := 1; i <= j; ++i)  //For all ways out:
            {
                /*
                    The default is that we do NOT want
                    to add the child on to frontier yet.
                */
                wantAdd := nil;
                if(g)
                    "\n#CP# Examining edge <<i>>...";

                if(self.done_actions[i] = 0)  
                /* If actions haven't been done yet. */
                {
                    if(g)
                        "\n#CP# checking
                        conditions...";
                    if(self.(self.child_conditions[i]))
                    // child conditions satisfied?
                    {
                        if(g)
                            "\n#CP# **CONDITIONS NOW
                            SATISFIED!**";

                        if(g)
                            "\n#CP# Executing edge exit
                            actions...";

                        //This executes exit actions.
                        self.(self.child_actions[i]);

                        /*
                            Show we're traveling out
                            this edge by setting its
                            corresponding done_actions
                            entry to 1.
                        */
                        self.done_actions[i] := 1;

                        //Add 1 to child node's p_cur.
                        self.child[i].parents_cur :=
                          self.child[i].parents_cur + 1;

                        if(g)
                            "\n#CP# ...finished
                            executing edge exit actions";

                        // We want to add child on...
                        wantAdd := true;
                    }
                    else
                        if(g)
                            "\n#CP#    conditions not
                            satisfied.";
                }
                else
                {
                    /*
                        We DON'T want to add this child
                        on the frontier as it was added
                        on earlier.
                    */
                    if(g)
                        "\n#CP# Actions done already.
                        Skip conditions check.  Don't add
                        on again.";
                }

                 /**************************************
                 * Check if we  1) want to add on       *
                 *           2) should add on only if it*
                 *              is not already on list. *
                  **************************************/
                if(wantAdd)
                {
                    /*
                        Only put child on frontier if it
                        is not current pd_node and not
                        later on list.
                    */
                    if(g)
                        "\n#CP# Can we add child \"<<
                        self.child[i].sdesc>>\" to
                         list?";

                    k := length(global.pd_nodes[1].frontier);

                    if(place < k) /* Not currently at end
                                     of list. */
                    {
                        m := true;  //default not add

                        if(g)
                            "\n#CP# ?-currently at item
                            <<place>> out of <<k>>.";

                        /* This not only prevents adding
                           a pd_node existing later on on the
                           list but also prevents a pd_node
                           from adding itself and entering
                           an infinite loop.  l = place to
                           start, not place + 1. */

                        for(l := place; l <= k; l++)
                        {
                            if(global.pd_nodes[1].frontier[l]
                               = self.child[i])
                            {
                                m := nil;
                                if(g)
                                    "\n#CP# ?-found at
                                    place <<l>>.";
                                break;
                            }
                            else
                                if(g)
                                    "\n#CP# ?-not found at
                                    place <<l>>...";
                        } // end for
                    } // end if place < k
                    else  // we're at end of list
                    {
                        /* We are at list's end so we
                           should add this new pd_node
                           on to be processed ONLY if
                           it is not the current one.*/
                        if(g)
                            "\n#CP# ?-We're at end of
                            list...";
                        if(self.child[i] = global.
                        pd_nodes[1].frontier[place])
                        {
                            m := nil;
                            if(g)
                                "\n#CP# ?-Can't add
                                itself to list.";
                        }
                        else
                            m := true;
                    }

                    if(m)
                    {
                        global.pd_nodes[1].frontier :=
                            global.pd_nodes[1].frontier +
                            self.child[i];
                        if(g)
                            "\n#CP# Yes, child
                            \"<<self.child[i].sdesc>>\"
                            added to frontier.";
                    }
                    else
                        if(g)
                            "\n#CP# No, child
                              \"<<self.child[i].sdesc>>\"
                            not added.  It is
                            already later on list.";
                } //end if wantAdd
            } //end for i
        } //end if parents_cur = parents_max
        else
        {
            if(g)
                "\n#CP#  Can't leave pd_node:  all edges
                to it are not satisfied.";
            return(nil);
        }

        /*
            Return true if all done_actions have been
            done.
        */
        for(i := 1; i <= j; ++i)
            if(self.done_actions[i] = 0)
                return(nil);  //All done_actions not done.
        //All done_actions done.
        return(true);
    } //end check_paths

    /*
        Adjustments for the STATE verb.
    */
    verDoState( actor ) = {}
    /*
        verify-direct-object-of-State's blank entry
        indicates that this pd_node can be a direct
        object of the STATE verb.
    */
    doState( actor ) =
    /*
        What to do when the pd_node is a direct object
        of the STATE verb.
    */
    {
        local i,j;
        j := length(self.done_actions);
        "\bStatus of the <<j>> done_actions of pd_node
         \"<<self.sdesc>>\" is: ";
        for(i := 1; i <= j; ++i)
            "<<self.done_actions[i]>>";

        j := length(self.child_conditions);
        "\nThere are <<j>> conditions and they return:";
        for(i := 1; i <= j; ++i)
        {
            switch(self.(self.child_conditions[i]))
            {
                case nil:
                    "\nFALSE";
                    break;
                case true:
                    "\nTRUE";
                    break;
                default:
                    "\nNot False or True?!";
            }
        }

        "\nParents_cur = <<self.parents_cur>>,
         parents_max = <<self.parents_max>>";
    } //end of doState

; //end of pd_node definition

/*
    Add-On Verb:
    state <pd_nodename>
    Display the status of its done_actions array.
*/
stateVerb: sysverb
    /*
        System verb status allows it to be executed
        without time passing.  No real effect on the
        game.
    */
    verb = 'state'
    sdesc = "state"
    doAction = 'State'
    validDo( actor, obj, seqno ) = { return( true ); }
    validDoList( actor, prep, dobj ) = { return( true ); }
;

//END OF PLOTDAG.T MODULE
