/*				C P P M A K E
 *
 *			Author: C. E. Chew
 *			Date:   October 1989
 *
 * (C) Copyright C E Chew
 *
 * Feel free to copy, use and distribute this software provided:
 *
 *	1. you do not pretend that you wrote it
 *	2. you leave this copyright notice intact.
 *
 * Patchlevel 1.5
 *
 * Edit History:
 * 03-Nov-1989	Oops mixed up WSTOPSIG and WTERMSIG. Patches from Bruce
 *		Evans about running in background and add pid_t for
 *		_MINIX. Scan CPPMAKECPP and CPPMAKEMAKE to break up
 *		into word sized morsels. Leave comments intact.
 * 01-Nov-1989	Don't use @$ since some systems use $ as alphabetic.
 *		More problems with getline. Add CPPOPT. Add -M and -C.
 * 31-Oct-1989	Quote with printable characters.
 * 23-Oct-1989	Fixed getline() looping bug. Fix single argument problem.
 */

#include <ctype.h>
#include <stdio.h>
#include <signal.h>

#ifndef	_MSDOS
#include <sys/types.h>
#endif

#ifdef	_BSD
#include <strings.h>
#else
#include <string.h>
#endif

#ifdef	_BSD
#include <sys/wait.h>
#endif

#ifdef	_MSDOS
#include <process.h>
#endif

/* Varargs handling */

#ifdef		__STDC__
# include <stdarg.h>
# define VA_START(n,l)	va_start(n,l)
# define VA_ALIST	...
# define VA_LIST	va_list
# define VA_END(n)	va_end(n)
# define VA_ARG(n,t)	va_arg(n,t)
# define VA_DCL
#else
# include <varargs.h>
# define VA_START(n,l)	va_start(n)
# define VA_ALIST	va_alist
# define VA_LIST	va_list
# define VA_END(n)	va_end(n)
# define VA_ARG(n,t)	va_arg(n,t)
# define VA_DCL		va_dcl
#endif

/* Function prototypes */

#ifdef	__STDC__
#define	P(x)		x
#define T(x,y)		x
#define D(x)
#else
#define P(x)		()
#define T(x,y)		y
#define D(x)		x;
#endif

/* Local definitions */

#define DEFINE		"-D"		/* define */
#define INCLUDE		"-I"		/* include */
#define UNDEF		"-U"		/* undefine */

#define INCFILE		".i"		/* extension for include file */

#define MAXARGS		20		/* maximum args in process descriptor */

#define GETLINELENGTH	80		/* initial allocation for getline */
#define GETLINEINC	20		/* increment for getline */

#define CATHEADERWIDTH	60		/* header width for cat */

#define CPPINPUT	"/tmp/cppiXXXXXX"/* cpp input */
#define CPPOUTPUT	"/tmp/cppoXXXXXX"/* cpp output */
#define MAKEFILE	"/tmp/makeXXXXXX"/* make file */

/* System deficiencies */

#ifdef	_BSD
typedef	int pid_t;			/* process id */
typedef union wait wait_t;		/* wait return structure */
#endif

#ifdef	_MINIX
typedef	int pid_t;			/* process id */
typedef int wait_t;			/* wait return structure */
#endif

#ifdef	INTSIGNAL
typedef	int signal_t;			/* signal handler return */
#else
typedef void signal_t;			/* signal handler return */
#endif

#ifdef	CHARMALLOC
typedef char *malloc_t;			/* malloc return type */
typedef int free_t;			/* free return type */
#else
typedef void *malloc_t;			/* malloc return type */
typedef void free_t;			/* free return type */
#endif

#ifndef	_MSDOS

#ifndef	WIFSTOPPED
#define WIFSTOPPED(x)	((*((unsigned int *) (&(x))) & 0xff) == 0x7f)
#endif

#ifndef	WIFEXITED
#define WIFEXITED(x)	((*((unsigned int *) (&(x))) & 0xff) == 0)
#endif

#ifndef	WIFSIGNALED
#define WIFSIGNALED(x)	((*((unsigned int *) (&(x))) - 1 & 0xffff) <  0xff)
#endif

#ifndef	WSTOPSIG
#define WSTOPSIG(x)	((*((unsigned int *) (&(x))) >> 8) & 0xff)
#endif

#ifndef	WEXITSTATUS
#define WEXITSTATUS(x)	((*((unsigned int *) (&(x))) >> 8) & 0xff)
#endif

#ifndef	WTERMSIG
#define WTERMSIG(x)	(*((unsigned int *) (&(x))) & 0x7f)
#endif

#endif

#ifdef	_BSD
#define strrchr rindex
#endif

/* Library function prototypes */

malloc_t malloc P((unsigned int));	/* raw malloc */
malloc_t realloc P((malloc_t, unsigned int));/* raw realloc */
free_t free P((void *));		/* raw free */
signal_t (*signal P((int, signal_t (*)(int)))) P((int)); /* signal handler */
char *mktemp P((char *));		/* make a temporary filename */
int unlink P((char *));			/* remove file */
int dup P((int));			/* duplicate handle */
int dup2 P((int, int));			/* duplicate handle */
int close P((int));			/* close handle */

#ifndef		_MSDOS
int wait P((wait_t *));			/* wait for child */
pid_t wait P((int *));			/* wait for child */
pid_t fork P((void));			/* fork a child */
int execvp P((char *, char **));	/* execute process */
#endif

int getopt P((int, char **, char *));	/* get options */
char *getenv P((char *));		/* get environment */

/* Library externals */

extern char *optarg;			/* option argument */
extern int opterr;			/* error processing */
extern int optind;			/* argc index */

/* Local types */

typedef struct process {
  int argc;
  char *argv[MAXARGS+1];
} PROCESS;

/* Local function prototypes */

char *getline P((FILE *));		/* read a line */
FILE *ftemp P((char *, char **));	/* create a temporary file */
char *smalloc P((unsigned int));	/* safe malloc */
char *srealloc P((char *, unsigned int));/* safe realloc */
char *stringdup P((char *));		/* duplicate a string */
int procreate P((PROCESS *, FILE *, FILE *)); /* doctors and nurses */
void appendarg P((PROCESS *, ...));	/* append an argument */
void appendlist P((PROCESS *, char **));/* append an argument vector */
void setarg P((PROCESS *, char *));	/* set arguments */
void cpppreamble P((FILE *));		/* insert cpp preamble */
void cppmaketocpp P((FILE *, FILE *));	/* cppmake to cpp */
void cpptomake P((FILE *, FILE *));	/* cpp to make */
signal_t wrapup P((int));		/* signal wrapup */
void purge P((void));			/* purge temporary files */
void done P((int));			/* purge then exit */
void cat P((char *, FILE *, FILE *));	/* dump a file */
void dsync P((FILE *));			/* sync a stream */

/* Process descriptors */

PROCESS cpp =  {1, {"/lib/cpp", 0}};
PROCESS make = {1, {"make",     0}};
PROCESS cppargs  = {0, {0}};
PROCESS makeargs = {0, {0}};

/* File names */

char *CppInputName    = 0;		/* name of cpp input */
char *CppOutputName   = 0;		/* name of cpp output */
char *MakefileName    = 0;		/* name of makefile */
char *CppMakefileName = 0;		/* name of cppmake file */
char *CppMakeMake     = 0;		/* name of make program */
char *CppMakeCpp      = 0;		/* name of cpp program */

/* Option switches  */

char verbose     = 0;			/* verbose */
char nomake      = 0;			/* don't start make */
char compileonly = 0;			/* compile only */

/* Main program */

int main(T(int argc, argc), T(char *argv[], argv))

D(int argc)				/* number of arguments */
D(char *argv[])				/* argument vector */

{
  FILE *CppMakefile;			/* cppmake file */
  FILE *CppInput;			/* cpp input */
  FILE *CppOutput;			/* cpp output */
  FILE *Makefile;			/* makefile */
  int sw;				/* option switch */
  int status;				/* exit status */
  char *env;				/* environment */
  char *DefineArg;			/* -D argument */
  char *UndefArg;			/* -U argument */
  char *IncludeArg;			/* -I argument */
  char *OutfileName;			/* name of output file */
  int lastopt;				/* last option */

/* Cleanup */
#ifdef	SIGINT
  if (signal(SIGINT,  SIG_IGN) != SIG_IGN)
    (void) signal(SIGINT,  wrapup);
#endif
#ifdef	SIGHUP
  if (signal(SIGHUP,  SIG_IGN) != SIG_IGN)
    (void) signal(SIGHUP,  wrapup);
#endif
#ifdef	SIGQUIT
  if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
    (void) signal(SIGQUIT, wrapup);
#endif

/* Check for environment */
  if ((env = getenv("CPPMAKEMAKE")) != 0)
    CppMakeMake = stringdup(env);
  if ((env = getenv("CPPMAKECPP")) != 0)
    CppMakeCpp = stringdup(env);

/* Do options */
  lastopt = 1;
  sw = opterr = 0;
  OutfileName = 0;
  while (sw != '?' && (sw = getopt(argc, argv, "o:f:D:I:U:M:C:cv")) != EOF) {
    switch (sw) {
    case 'v':
      verbose = 1;
      break;

    case 'c':
      compileonly = 1;
      break;

    case 'f':
      CppMakefileName = optarg;
      break;

    case 'M':
      CppMakeMake = optarg;
      break;

    case 'C':
      CppMakeCpp = optarg;
      break;

    case 'D':
      DefineArg = smalloc((unsigned int) (strlen(optarg) + sizeof(DEFINE)));
      (void) strcpy(DefineArg, DEFINE);
      (void) strcat(DefineArg, optarg);
      appendarg(&cppargs, DefineArg, (char *) 0);
      break;

    case 'I':
      IncludeArg = smalloc((unsigned int) (strlen(optarg) + sizeof(INCLUDE)));
      (void) strcpy(IncludeArg, INCLUDE);
      (void) strcat(IncludeArg, optarg);
      appendarg(&cppargs, IncludeArg, (char *) 0);
      break;

    case 'U':
      UndefArg = smalloc((unsigned int) (strlen(optarg) + sizeof(UNDEF)));
      (void) strcpy(UndefArg, UNDEF);
      (void) strcat(UndefArg, optarg);
      appendarg(&cppargs, UndefArg, (char *) 0);
      break;

    case 'o':
      OutfileName = optarg;
      break;

    case '?':
      break;
    }

    if (sw != '?')
      lastopt = optind;
  }

/* Move argument indicator back to last argument */
  if (optind < argc)
    lastopt = optind;

/* Set the process names */
  if (CppMakeMake != 0)
    setarg(&make, CppMakeMake);
  if (CppMakeCpp != 0)
    setarg(&cpp, CppMakeCpp);

/* Add cpp and make options */
#ifdef CPPOPT
  appendarg(&cpp, CPPOPT, (char *) 0);
#endif
  appendlist(&cpp,  &cppargs.argv[0]);
  appendlist(&make, &makeargs.argv[0]);

/* Set the name of the makefile and the name of the cppmake file */
  if (! compileonly) {

/* Makefile name */
    if (OutfileName != 0) {
      nomake       = 1;
      MakefileName = OutfileName;
    }

/* Cppmake file name */
    if (CppMakefileName != 0)
      CppMakefile = fopen(CppMakefileName, "r");
    else {
      CppMakefileName = "Makefile.cpp";
      if ((CppMakefile = fopen(CppMakefileName, "r")) == 0) {
	CppMakefileName = "makefile.cpp";
	CppMakefile = fopen(CppMakefileName, "r");
      }
    }
  }

/* Set the name of the cppmake file and the output file */
  else {
    if (lastopt == argc && CppMakefileName == 0) {
      (void) fputs("No cppmake file specified.\n", stderr);
      exit(1);
    }
    else if (CppMakefileName == 0 && lastopt < argc-1 ||
	     CppMakefileName != 0 && lastopt < argc) {
      (void) fputs("Too many cppmake files specified.\n", stderr);
      exit(1);
    }

    if (CppMakefileName == 0) {
      CppMakefileName = argv[lastopt];
      lastopt = argc;
    }

    CppMakefile = fopen(CppMakefileName, "r");

    if (OutfileName != 0)
      CppInputName = OutfileName;
    else {
      if ((OutfileName = strrchr(CppMakefileName, '.')) == 0) {
	(void) fprintf(stderr, "Cannot make output file name from %s.\n",
		       CppMakefileName);
        exit(1);
      }
      sw = OutfileName - CppMakefileName;
      CppInputName = smalloc((unsigned int) (sw+sizeof(INCFILE)));
      (void) strncpy(CppInputName, CppMakefileName, sw);
      (void) strcpy(CppInputName+sw, INCFILE);
    }
  }

/* Check that the cppmake file is ok */
  if (CppMakefile == 0) {
    (void) fputs("Cannot open makefile\n", stderr);
    exit(1);
  }

/* Create the temporary files */
  if (compileonly) {
    if ((CppInput = fopen(CppInputName, "w")) == 0) {
      (void) fprintf(stderr, "Cannot open output file %s.\n", CppInputName);
      exit(1);
    }
  }
  else {

/* Open the temporary files */
    CppInput  = ftemp(CPPINPUT,  &CppInputName);
    CppOutput = ftemp(CPPOUTPUT, &CppOutputName);
    if (MakefileName == 0)
      Makefile = ftemp(MAKEFILE, &MakefileName);
    else if ((Makefile = fopen(MakefileName, "w")) == 0) {
      (void) fprintf(stderr, "Cannot open makefile %s.\n", MakefileName);
      done(1);
    }

/* Insert the cpp preamble */
    cpppreamble(CppInput);
  }

/* Convert cppmake file for cpp input */
  cppmaketocpp(CppMakefile, CppInput);
  dsync(CppInput);

/* Compile only so don't erase output file */
  if (compileonly)
    CppInputName = 0;

  else {

/* Run cpp on the cpp input file */
    if (verbose)
      cat("Preprocessor Input", CppInput, stderr);
    if ((status = procreate(&cpp, CppInput, CppOutput)) != 0)
      done(status);
    dsync(CppOutput);

/* Convert cpp output to makefile */
    if (verbose)
      cat("Preprocessor Output", CppOutput, stderr);
    cpptomake(CppOutput, Makefile);
    dsync(Makefile);

/* Run make on the makefile */
    if (verbose)
      cat("Makefile Input", Makefile, stderr);
    if (nomake) {
      (void) fclose(Makefile);
      MakefileName = 0;
    }
    else {
      appendarg(&make, "-f", MakefileName, (char *) 0);
      for (; lastopt < argc; lastopt++)
	appendarg(&make, argv[lastopt], (char *) 0);

      if ((status = procreate(&make, stdin, stdout)) != 0)
	done(status);
    }
  }

/* Complete */
  done(0);

/* Don't trust return from main on some machines */
  return 0;
}

/* Insert cpp preamble
 *
 * Insert a preamble into the cpp input file. This preamble will
 * provide the names of the modules that make up the cppmake file.
 */

void cpppreamble(f)

FILE *f;				/* cpp input file */

{
  (void) fprintf(f, "#define CPPMAKEMAKE %s\n",     make.argv[0]);
  (void) fprintf(f, "#define CPPMAKECPP %s\n",      cpp.argv[0]);
  (void) fprintf(f, "#define CPPMAKEFILE %s\n",     CppMakefileName);
  (void) fprintf(f, "#define CPPMAKEMAKEFILE %s\n", MakefileName);
}

/* Change an cppmake file into a cpp input file
 *
 * A cppmake file is changed into a makefile. The following transformations
 * are made:
 *
 *	1. Preprocessor lines are retained
 *	2. Comments are deleted
 *	3. @ is replaced by @@ except in cpp commands
 *	4. ' ' is replaced by @% except in cpp commands
 *	5. \t is replaced by @^ except in cpp commands
 *	6. \n is replaced by @! except in cpp commands
 *	7. # is replaced by @& except in cpp commands
 *	8. Empty lines have @* inserted.
 *	9. Leading whitespace in macros definitions are removed
 */

void cppmaketocpp(T(FILE *cppmake, cppmake), T(FILE *cpp, cpp))

D(FILE *cppmake)			/* cppmake file */
D(FILE *cpp)				/* cpp file */

{
  char *lp;				/* line pointer */
  char *token;				/* preprocessor token */
  int toklen;				/* length of token */
  char *p, *q;				/* scanner */
  char *qf;				/* quote from here */
  char *rp;				/* replacement for quote */

  for (; (lp = getline(cppmake)) != 0; ) {

/* Remove makefile comments */
    for (qf = p = lp; isspace(*p); p++)
      ;

/* Comment or cpp command */
    if (*p++ == '#') {
      qf = lp = p-1;

/* Join lines by removing quoted newlines */
      for (q = p = lp; *p; p++) {
	if (p[0] != '\\' || p[1] != '\n')
	  *q++ = *p;
	else
	  p++;
      }
      *q = 0;

/* Scan for command */
      for (p = lp + 1; isspace(*p); p++)
	;
      token = p;

      for (; isalpha(*p); p++)
	;

      toklen = p - token;

      if (toklen != 0) {

/* Define macros */
	if (strncmp("define", token, toklen) == 0) {

/* Quote spaces and tabs within macros */
	  for (; isspace(*p); p++)
	    ;
	  for (; isalnum(*p) || *p == '_'; p++)
	    ;
	  if (*p == '(') {
	    for (; *p && *p != ')'; p++)
	      ;
	    if (*p == ')')
	      p++;
	  }
	  else if (isspace(*p))
	    *p++ = ' ';

/* Note start of definition */
	  q = p;

/* Quote characters in macro definition only */
	  qf = p;

/* Kill leading spaces in macro definition */
	  if (isspace(*p)) {
	    for (; isspace(*p); p++)
	      ;
	    for (; *p; )
	      *q++ = *p++;
	    *q = 0;
	  }
	}

/* Other cpp directives are left alone */
	else if (strncmp("undef",   token, toklen) == 0 ||
	         strncmp("include", token, toklen) == 0 ||
	         strncmp("line",    token, toklen) == 0 ||
	         strncmp("endif",   token, toklen) == 0 ||
	         strncmp("if",      token, toklen) == 0 ||
	         strncmp("else",    token, toklen) == 0 ||
	         strncmp("elsif",   token, toklen) == 0 ||
	         strncmp("ifndef",  token, toklen) == 0 ||
	         strncmp("ifdef",   token, toklen) == 0)
	  qf = lp + strlen(lp);
      }
    }

/* Quote blank lines */
    if (*lp == 0)
      (void) fputs("@*\n", cpp);

    else {
/* Output the current line and */
      for (q = lp, p = qf; *p; p++) {
	switch (*p) {
	case '@':  rp = "@@"; break;
	case ' ':  rp = "@%"; break;
	case '\t': rp = "@^"; break;
	case '\n': rp = "@!"; break;
	case '#':  rp = "@&"; break;
	default  : rp = 0;    break;
	}
	if (rp != 0) {
	  (void) fwrite(q, sizeof(char), p-q, cpp);
	  (void) fputs(rp, cpp);
	  q = p + 1;
	}
      }
      if (q < p)
	(void) fwrite(q, sizeof(char), p-q, cpp);
      (void) putc('\n', cpp);
    }
  }
}

/* Change a cpp output file into a makefile
 *
 * Read the cpp output file and make the following transformations:
 *
 *	1. Leading spaces are removed
 *	2. Lines beginning with # are discarded
 *	3. @@ is converted to @
 *	4. @% is converted to ' '
 *	5. @^ is converted to \t
 *	6. @! is converted to \n
 *	7. @& is converted to #
 *	8. @* is removed
 *	9. ## and surrounding whitespace is removed
 *     10. @@ in the original source is converted to \n
 *     11. Trailing whitespace is removed
 *     12. Blank lines are removed
 */

void cpptomake(T(FILE *cpp, cpp), T(FILE *make, make))

D(FILE *cpp)				/* cpp output */
D(FILE *make)				/* makefile */

{
  char *lp;				/* line */
  char *p, *q;				/* line scanners */

  for (; (lp = getline(cpp)) != 0; ) {

/* Ignore cpp line control and empty lines */
    if (*lp == '#' || *lp == 0)
      continue;

/* Skip leading whitespace */
    for (; isspace(*lp); lp++)
      ;

/* Replace quoted characters */
    for (q = p = lp; *p; p++) {
      if (p[0] != '@')
	*q++ = *p;
      else {
	switch (p[1]) {
	default:  *q++ = p[0]; *q++ = p[1]; break;
	case '@': *q++ = '@';  break;
	case '%': *q++ = ' ';  break;
	case '^': *q++ = '\t'; break;
	case '!': *q++ = '\n'; break;
	case '&': *q++ = '#';  break;
	case '*':              break;
	}
	p++;
      }
    }
    *q = 0;

/* Scan original source */
    for (q = p = lp; *p; p++) {

/* Concatenate adjacent symbols */
      if (p[0] == '#' && p[1] == '#') {
	for (; q > lp; ) {
	  if (! isspace(*--q)) {
	    q++;
	    break;
	  }
	}
	for (p += 2; isspace(*p); p++)
	  ;
	p--;
      }

/* Visible newline */
      else if (p[0] == '@' && p[1] == '@') {
	for (; q > lp; ) {
	  if (! isspace(*--q)) {
	    q++;
	    break;
	  }
	}
	*q++ = '\n';
	p++;
      }

/* Default is to leave it alone */
      else
	*q++ = *p;
    }

/* Eat trailing whitespace */
    for (; q > lp; ) {
      if (! isspace(*--q)) {
	q++;
	break;
      }
    }
    *q = 0;

/* Output the line */
    (void) fputs(lp, make);
    (void) putc('\n', make);
  }
}

/* Get a line
 *
 * Read a line from a stream. Return a null terminated string
 * containing the line. Return null on end of file. A dynamically
 * allocated static buffer is used to hold the line. Lines with
 * continuation marks are joined with the continuation mark and
 * the \n left intact.
 */

char *getline(T(FILE *f, f))

D(FILE *f)				/* stream */

{
  static char *buf;			/* line buffer */
  static int buflen = 0;		/* size of buffer */
  char *bp;				/* buffer pointer */
  int size;				/* size of buffer */

  if (buflen == 0) {
    buflen = GETLINELENGTH * sizeof(char);
    buf = smalloc((unsigned int) buflen);
  }
  for (*(bp = buf) = 0; ; ) {
    if (fgets(bp, buflen - (bp - buf), f) == 0)
      break;
    bp += (size = strlen(bp));
    if (bp[-1] == '\n') {
      if (bp - buf == 1 || bp[-2] != '\\') {
	bp[-1] = 0;
	break;
      }
    }
    if (bp - buf + 1 >= buflen) {
      buflen += GETLINEINC * sizeof(char);
      size = bp - buf;
      buf = srealloc(buf, (unsigned int) buflen);
      bp = buf + size;
    }
  }
  return bp == buf ? 0 : buf;
}

/* Duplicate a string
 *
 * Return a copy of the string. The copy is made in space malloc'ed
 * for the string.
 */

char *stringdup(T(char *s, s))

D(char *s)				/* string */

{
  return strcpy(smalloc((unsigned int) (strlen(s)+1)), s);
}

/* Safe malloc
 *
 * Malloc with error checking.
 */

char *smalloc(T(unsigned int n, n))

D(unsigned int n)			/* size of malloc */

{
  char *p;				/* area */

  if ((p = (char *) malloc(n)) == 0) {
    (void) fputs("Insufficient memory\n", stderr);
    done(1);
  }
  return p;
}

/* Safe realloc
 *
 * Realloc and check for error.
 */

char *srealloc(T(char *p, p), T(unsigned int n, n))

D(char *p)				/* area */
D(unsigned int n)			/* size */

{
  if ((p = (char *) realloc((malloc_t) p, n)) == 0) {
    (void) fputs("Insufficient memory\n", stderr);
    done(1);
  }
  return p;
}

/* Open a temporary file
 *
 * A temporary file is opened using the given template. The
 * function returns a stream pointer to the temporary file.
 */

FILE *ftemp(T(char *template, template), T(char **name, name))

D(char *template)			/* name template */
D(char **name)				/* real name */

{
  FILE *fp;				/* file */

  for (;;) {
    (void) mktemp(*name = stringdup(template));

    if ((fp = fopen(*name, "w+")) != NULL)
      return fp;
    free(*name);
  }
}

/* Start a child
 *
 * Start a child process with the specified argument list and
 * the named streams as stdin and stdout.
 */

int procreate(T(PROCESS *child, child), T(FILE *in, in), T(FILE *out, out))

D(PROCESS *child)			/* arguments */
D(FILE *in)				/* input */
D(FILE *out)				/* output */

{
  int i;				/* argv index */
#ifdef	_MSDOS
  int Stdin;				/* saved stdin */
  int Stdout;				/* saved stdout */
  int status;				/* return status */
#else
  pid_t pid;				/* child id */
  wait_t status;			/* child exit status */
#endif

  if (verbose) {
    (void) fprintf(stderr, "Spawning child: ");
    for (i = 0; i < child->argc; i++)
      (void) fprintf(stderr, "%s ", child->argv[i]);
    (void) putc('\n', stderr);
  }

#ifdef	_MSDOS

  if (fileno(in) != 0) {
    if ((Stdin = dup(0)) < 0) {
      (void) fputs("Cannot save stdin\n", stderr);
      done(1);
    }
    if (dup2(fileno(in), 0) < 0) {
      (void) fputs("Cannot set stdin\n", stderr);
      done(1);
    }
  }

  if (fileno(out) != 1) {
    if ((Stdout = dup(1)) < 0) {
      (void) fputs("Cannot save stdout\n", stderr);
      done(1);
    }
    if (dup2(fileno(out), 1) < 0) {
      (void) fputs("Cannot set stdout\n", stderr);
      done(1);
    }
  }

  status = spawnvp(P_WAIT, child->argv[0], child->argv);

  if (fileno(in) != 0) {
    if (dup2(Stdin, 0) < 0) {
      (void) fputs("Cannot restore stdin\n", stderr);
      done(1);
    }
    (void) close(Stdin);
  }

  if (fileno(out) != 1) {
    if (dup2(Stdout, 1) < 0) {
      (void) fputs("Cannot restore stdout\n", stderr);
      done(1);
    }
    (void) close(Stdout);
  }

  if (status < 0) {
    (void) fprintf(stderr, "Cannot spawn %s.\n", child->argv[0]);
    return 1;
  }

  return status;

#else
  switch (pid = fork()) {
  case -1:
    (void) fprintf(stderr, "Cannot fork %s.\n", child->argv[0]);
    done(1);

  case 0:
#ifdef	SIGINT
    if (signal(SIGINT,  SIG_IGN) != SIG_IGN)
      (void) signal(SIGINT,  SIG_DFL);
#endif
#ifdef	SIGHUP
    if (signal(SIGHUP,  SIG_IGN) != SIG_IGN)
      (void) signal(SIGHUP,  SIG_DFL);
#endif
#ifdef	SIGQUIT
    if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
      (void) signal(SIGQUIT, SIG_DFL);
#endif

    if ((fileno(in)  != 0 && dup2(fileno(in),  0) < 0) ||
        (fileno(out) != 1 && dup2(fileno(out), 1) < 0)) {
      (void) fputs("Cannot set stdin and stdout\n", stderr);
      exit(1);
    }
    execvp(child->argv[0], child->argv);
    (void) fprintf(stderr, "Cannot exec %s.\n", child->argv[0]);
    exit(1);

  default:
    do ;
    while (pid != wait(&status) || WIFSTOPPED(status));

    if (WIFSIGNALED(status)) {
      (void) fprintf(stderr, "%s killed by signal %d.\n",
		     child->argv[0], WTERMSIG(status));
      return 1;
    }

    return WEXITSTATUS(status);
  }
#endif
}

/* Append arguments to the process
 *
 * Add null terminated list of arguments to the list of arguments
 * for the process.
 */

void appendarg(T(PROCESS *p, p), VA_ALIST)

D(PROCESS *p)				/* process */
VA_DCL

{
  VA_LIST ap;				/* point at argument list */
  char *q;				/* point at argument */

  VA_START(ap, p);

  for (; (q = VA_ARG(ap, char *)) != 0; ) {
    if (p->argc >= MAXARGS) {
      (void) fprintf(stderr, "Too many arguments for %s.\n", p->argv[0]);
      done(1);
    }
    p->argv[p->argc++] = q;
    p->argv[p->argc]   = 0;
  }

  VA_END(ap);
}

/* Set the arguments for a process
 *
 * This resets the process argument list and sets it to the
 * one named in the parameter list. The parameter list will
 * be scanned and separated into space separated words. Note
 * that the string will be written into.
 */

void setarg(T(PROCESS *p, p), T(char *s, s))

D(PROCESS *p)				/* process */
D(char *s)				/* arguments */

{
  char *a;				/* point at argument */

  p->argc    = 0;
  p->argv[0] = 0;

  for (;;) {
    for (; isspace(*s); s++)
      ;
    if (*s == 0)
      break;
    for (a = s; *s != 0 && !isspace(*s); s++)
      ;
    if (*s != 0)
      *s++ = 0;
    appendarg(p, a, (char *) 0);
  }
}

/* Append an argument vector to the list of arguments
 *
 * The vector of arguments is appended to the list of
 * arguments for the process.
 */

void appendlist(T(PROCESS *p, p), T(char **argv, argv))

D(PROCESS *p)				/* process descriptor */
D(char **argv)				/* argument vector */

{
  for (; *argv != 0; argv++)
    appendarg(p, *argv, (char *) 0);
}

/* Wrap up with exit status
 *
 * Delete all temporary files and exit with the specified status.
 */

void done(T(int status, status))

D(int status)				/* exit status */

{
  purge();
  exit(status);
}

/* Wrap up
 *
 * Delete all temporary files and exit with error status.
 */

signal_t wrapup(T(int sig, sig))

D(int sig)				/* signal */

{
  sig = sig;
  purge();
  exit(1);
}

/* Remove all temporary files
 *
 * Check for temporary files and remove them.
 */

void purge(T(void,))

{
  if (CppInputName != 0)
    (void) unlink(CppInputName);
  if (CppOutputName != 0)
    (void) unlink(CppOutputName);
  if (MakefileName != 0)
    (void) unlink(MakefileName);
}

/* Cat a file
 *
 * Copy the named stream to the named output stream. On completion
 * the input stream is rewound.
 */

void cat(T(char *name, name), T(FILE *in, in), T(FILE *out, out))

D(char *name)					/* heading to use */
D(FILE *in)					/* input stream */
D(FILE *out)				/* output stream */

{
  int ch;				/* copy */
  int len;				/* length of name */
  int left;				/* left of header */
  int right;				/* right of header */
  int lastch;				/* last character */

  len = strlen(name);
  left = (CATHEADERWIDTH - len) / 2;
  right = CATHEADERWIDTH - len - left;

  for (len = left; len--; (void) putc('-', out))
    ;
  (void) fprintf(out, " %s ", name);
  for (len = right; len--; (void) putc('-', out))
    ;
  (void) putc('\n', out);

  for (lastch = '\n'; (ch = getc(in)) != EOF; lastch = ch)
    (void) putc(ch, out);
  
  if (lastch != '\n')
    (void) putc('\n', out);

  for (len = left; len--; (void) putc('+', out))
    ;
  (void) fprintf(out, " %s ", name);
  for (len = right; len--; (void) putc('+', out))
    ;
  (void) putc('\n', out);

  rewind(in);
}

/* Synchronise disk data blocks and directory
 *
 * Make sure that the data in the stream has been written to
 * the disk and that the directory has been updated.
 */

void dsync(T(FILE *f, f))

D(FILE *f)				/* stream */

{
#ifdef	_MSDOS
  int sf;				/* save descriptor */
#endif

  (void) fflush(f);
  rewind(f);

#ifdef	_MSDOS
  if ((sf = dup(fileno(f))) < 0) {
    (void) fputs("Cannot sync file.\n", stderr);
    done(1);
  }
  (void) close(sf);
#endif
}
