/*
 * keys.c: Does command line parsing, etc 
 *
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */
#include <stdio.h>
#include <ctype.h>
#include "irc.h"
#include "keys.h"
#include "names.h"
#include "ircaux.h"
#include "window.h"


extern	void	input_add_character();
extern	void	input_backward_word();
extern	void	input_forward_word();
extern	void	input_delete_previous_word();
extern	void	input_delete_next_word();
extern	void	forward_character();
extern	void	backward_character();
extern	void	input_clear_to_bol();
extern	void	input_clear_line();
extern	void	input_beginning_of_line();
extern	void	input_end_of_line();
extern	void	input_clear_to_eol();
extern	void	refresh_inputline();
extern	SIGTYP	refresh_screen();
extern	void	input_delete_character();
extern	void	input_backspace();
extern	void	backward_history();
extern	void	forward_history();
extern	void	toggle_insert_mode();
extern	void	input_transpose_characters();
extern	void	input_yank_cut_buffer();
extern	void	send_line();
extern	void	meta1_char();
extern	void	meta2_char();
extern	void	meta3_char();
extern	void	meta4_char();
extern	void	irc_quit();
extern	void	term_pause();
extern	void	quote_char();
extern	void	type_text();
extern	void	parse_text();
extern	void	toggle_stop_screen();
extern	void	command_completion();
extern	void	clear_screen();
extern	void	enter_digraph();
extern	void	scrollback_backwards();
extern	void	scrollback_forwards();
extern	void	scrollback_end();
extern	void	enter_menu();

/*
 * lookup_function: looks up an irc function by name, and returns the
 * number of functions that match the name, and sets where index points
 * to to be the index of the (first) function found.
 */
int	lookup_function(name, index)
char	*name;
int	*index;
{
	int	len,
		cnt,
		i;

	if (name)
	{
		upper(name);
		len = strlen(name);
		cnt = 0;
		*index = -1;
		for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
		{
			if (strnicmp(name, key_names[i].name, len) == 0)
			{
				cnt++;
				if (*index == -1)
					*index = i;
			}
		}
		if (*index == -1)
			return (0);
		if (stricmp(name, key_names[*index].name) == 0)
			return (1);
		else
			return (cnt);
	}
	return (0);
}

/*
 * display_key: converts the character c to a displayable form and returns
 * it.  Very simple indeed 
 */
unsigned char *display_key(c)
unsigned char c;
{
	static	unsigned char key[3];

	key[2] = null(char);
	if (c < 32)
	{
		key[0] = '^';
		key[1] = c + 64;
	}
	else if (c == '\177')
	{
		key[0] = '^';
		key[1] = '?';
	}
	else
	{
		key[0] = c;
		key[1] = null(char);
	}
	return (key);
}

/*
 * show_binding: given the ascii value of a key and a meta key status (1 for
 * meta1 keys, 2 for meta2 keys, anything else for normal keys), this will
 * display the key binding for the key in a nice way
 */
void	show_binding(c, meta)
char	c;
int	meta;
{
	KeyMap	*map;
	char	*meta_str;

	switch (meta)
	{
	case 1:
		map = meta1_keys;
		meta_str = "META1-";
		break;
	case 2:
		map = meta2_keys;
		meta_str = "META2-";
		break;
	case 3:
		map = meta3_keys;
		meta_str = "META3-";
		break;
	case 4:
		map = meta4_keys;
		meta_str = "META4-";
		break;
	default:
		map = keys;
		meta_str = empty_string;
		break;
	}
	say("%s%s is bound to %s %s", meta_str, display_key(c),
		key_names[map[c].index].name, (map[c].stuff &&
		(*(map[c].stuff))) ? map[c].stuff : "");
}

/*
 * parse_key: converts a key string. Accepts any key, or ^c where c is any
 * key (representing control characters), or META1- or META2- for meta1 or
 * meta2 keys respectively.  The string itself is converted to true ascii
 * value, thus "^A" is converted to 1, etc.  Meta key info is removed and
 * returned as the function value, 0 for no meta key, 1 for meta1, and 2 for
 * meta2.  Thus, "META1-a" is converted to "a" and a 1 is returned.
 * Furthermore, if ^X is bound to META2_CHARACTER, and "^Xa" is passed to
 * parse_key(), it is converted to "a" and 2 is returned.  Do ya understand
 * this? 
 */
int	parse_key(key_str)
char	*key_str;
{
	char	*ptr1,
	*ptr2,
	c;
	int	meta = 0;

	ptr2 = ptr1 = key_str;
	while (*ptr1)
	{
		if (*ptr1 == '^')
		{
			ptr1++;
			switch (*ptr1)
			{
			case 0:
				*(ptr2++) = '^';
				break;
			case '?':
				*(ptr2++) = '\177';
				ptr1++;
				break;
			default:
				c = *(ptr1++);
				if (islower(c))
					c = toupper(c);
				if (c < 64)
				{
					say("Illegal key sequence: ^%c", c);
					return (-1);
				}
				*(ptr2++) = c - 64;
			}
		}
		else
			*(ptr2++) = *(ptr1++);
	}
	*ptr2 = null(char);
	if (strlen(key_str) > 1)
	{
		if (strnicmp(key_str, "META1-", 6) == 0)
		{
			strcpy(key_str, key_str + 6);
			meta = 1;
		}
		else if (strnicmp(key_str, "META2-", 6) == 0)
		{
			strcpy(key_str, key_str + 6);
			meta = 2;
		}
		else if (strnicmp(key_str, "META3-", 6) == 0)
		{
			strcpy(key_str, key_str + 6);
			meta = 3;
		}
		else if (strnicmp(key_str, "META4-", 6) == 0)
		{
			strcpy(key_str, key_str + 6);
			meta = 4;
		}
		else if (keys[*key_str].index == META1_CHARACTER)
		{
			meta = 1;
			strcpy(key_str, key_str + 1);
		}
		else if (keys[*key_str].index == META2_CHARACTER)
		{
			meta = 2;
			strcpy(key_str, key_str + 1);
		}
		else if (keys[*key_str].index == META3_CHARACTER)
		{
			meta = 3;
			strcpy(key_str, key_str + 1);
		}
		else if (keys[*key_str].index == META4_CHARACTER)
		{
			meta = 4;
			strcpy(key_str, key_str + 1);
		}
		else
		{
			say("Illegal key sequence: %s is not a meta-key",
				display_key(*key_str));
			return (-1);
		}
	}
	return (meta);
}

/*
 * bind_it: does the actually binding of the function to the key with the
 * given meta modifier
 */
static	void	bind_it(function, string, key, meta)
char	key;
char	*function,
	*string;
int	meta;
{
	KeyMap	*km;
	int	cnt,
	index,
	i;

	switch (meta)
	{
	case 0:
		km = keys;
		break;
	case 1:
		km = meta1_keys;
		break;
	case 2:
		km = meta2_keys;
		break;
	case 3:
		km = meta3_keys;
		break;
	case 4:
		km = meta4_keys;
		break;
	default:
		km = keys;
	}
	if (*string == null(char))
		string = null(char *);
	switch (cnt = lookup_function(function, &index))
	{
	case 0:
		say("No such function: %s", function);
		break;
	case 1:
		if (! km[key].changed)
		{
			if ((km[key].index != index) ||
					((string == null(char *)) &&
					km[key].stuff) ||
					((km[key].stuff == null(char *)) &&
					string) || (string && km[key].stuff &&
					strcmp(km[key].stuff,string)))
				km[key].changed = 1;
		}
		km[key].index = index;
		malloc_strcpy(&(km[key].stuff), string);
		show_binding(key, meta);
		break;
	default:
		say("Ambiguous function name: %s", function);
		for (i = 0; i < cnt; i++, index++)
			put_it("%s", key_names[index].name);
		break;
	}
}

/* parsekeycmd: does the PARSEKEY command.  */
void	parsekeycmd(command, args)
char	*command,
	*args;
{
	int	i;
	char	*arg;

	if (arg = next_arg(args, &args))
	{
		switch (lookup_function(arg, &i))
		{
		case 0:
			say("No such function %s", arg);
			return;
		case 1:
			key_names[i].func();
			break;
		default:
			say("Ambigious function %S", arg);
			break;
		}
	}
}

/*
 * bindcmd: the bind command, takes a key sequence followed by a function
 * name followed by option arguments (used depending on the function) and
 * binds a key.  If no function is specified, the current binding for that
 * key is shown 
 */
/*ARGSUSED*/
void	bindcmd(command, args)
char	*command,
	*args;
{
	char	*key,
		*function;
	int	meta;

	if (key = next_arg(args, &args))
	{
		if ((meta = parse_key(key)) == -1)
			return;
		if (strlen(key) > 1)
		{
			say("Key sequences may not contain more than two keys");
			return;
		}
		if (function = next_arg(args, &args))
			bind_it(function, args, *key, meta);
		else
			show_binding(*key, meta);
	}
	else
	{
		int	i;

		for (i = 0; i < 128; i++)
		{
			if ((keys[i].index != NOTHING) && (keys[i].index !=
					SELF_INSERT))
				show_binding(i, 0);
		}
		for (i = 0; i < 128; i++)
		{
			if ((meta1_keys[i].index != NOTHING) &&
					(meta1_keys[i].index != SELF_INSERT))
				show_binding(i, 1);
		}
		for (i = 0; i < 128; i++)
		{
			if ((meta2_keys[i].index != NOTHING) &&
					(meta2_keys[i].index != SELF_INSERT))
				show_binding(i, 2);
		}
		for (i = 0; i < 128; i++)
		{
			if ((meta3_keys[i].index != NOTHING) &&
					(meta3_keys[i].index != SELF_INSERT))
				show_binding(i, 3);
		}
		for (i = 0; i < 128; i++)
		{
			if ((meta4_keys[i].index != NOTHING) &&
					(meta4_keys[i].index != SELF_INSERT))
				show_binding(i, 4);
		}
	}
}

void	(* get_send_line())()
{
	return(key_names[SEND_LINE].func);
}

/*
 * change_send_line: Allows you to change the everything bound to SENDLINE in
 * one fell swoop.  Used by the various functions that gather input using the
 * normal irc interface but dont wish to parse it and send it to the server.
 * Sending NULL resets it to send_line()
 */
void	change_send_line(func)
void	(*func) ();
{
	if (func)
		key_names[SEND_LINE].func = func;
	else
		key_names[SEND_LINE].func = send_line;
}

/*
 * type: The TYPE command.  This parses the given string and treats each
 * character as tho it were typed in by the user.  Thus key bindings are used
 * for each character parsed.  Special case characters are control character
 * sequences, specified by a ^ follow by a legal control key.  Thus doing
 * "/TYPE ^B" will be as tho ^B were hit at the keyboard, probably moving the
 * cursor backward one character.
 */
/*ARGSUSED*/
void	type(command, args)
char	*command,
*args;
{
	int	c;
	char	key;

	while (*args)
	{
		if (*args == '^')
		{
			switch (*(++args))
			{
			case '?':
				key = '\177';
				args++;
				break;
			default:
				c = *(args++);
				if (islower(c))
					c = toupper(c);
				if (c < 64)
				{
					say("Illegal key sequence: ^%c", c);
					return;
				}
				key = c - 64;
				break;
			}
		}
		else if (*args == '\\')
		{
			key = *++args;
			args++;
		}
		else
			key = *(args++);
		edit_char(key);
	}
}

KeyMapNames key_names[] =
{
	"BACKSPACE",		input_backspace,
	"BACKWARD_CHARACTER",	backward_character,
	"BACKWARD_HISTORY",	backward_history,
	"BACKWARD_WORD",	input_backward_word,
	"BEGINNING_OF_LINE",	input_beginning_of_line,
	"CLEAR_SCREEN",		clear_screen,
	"COMMAND_COMPLETION",	command_completion,
	"DELETE_CHARACTER",	input_delete_character,
	"DELETE_NEXT_WORD",	input_delete_next_word,
	"DELETE_PREVIOUS_WORD",	input_delete_previous_word,
	"END_OF_LINE",		input_end_of_line,
	"ENTER_DIGRAPH",	enter_digraph,
	"ENTER_MENU",		enter_menu,
	"ERASE_LINE",		input_clear_line,
	"ERASE_TO_BEG_OF_LINE",	input_clear_to_bol,
	"ERASE_TO_END_OF_LINE",	input_clear_to_eol,
	"FORWARD_CHARACTER",	forward_character,
	"FORWARD_HISTORY",	forward_history,
	"FORWARD_WORD",		input_forward_word,
	"META1_CHARACTER",	meta1_char,
	"META2_CHARACTER",	meta2_char,
	"META3_CHARACTER",	meta3_char,
	"META4_CHARACTER",	meta4_char,
	"NEXT_WINDOW",		next_window,
	"NOTHING",		NULL,
	"PARSE_COMMAND",	parse_text,
	"PREVIOUS_WINDOW",	previous_window,
	"QUIT_IRC",		irc_quit,
	"QUOTE_CHARACTER",	quote_char,
	"REFRESH_INPUTLINE",	refresh_inputline,
	"REFRESH_SCREEN",	(void (*) ()) refresh_screen,
	"SCROLL_BACKWARD",	scrollback_backwards,
	"SCROLL_END",		scrollback_end,
	"SCROLL_FORWARD",	scrollback_forwards,
	"SELF_INSERT",		input_add_character,
	"SEND_LINE",		send_line,
	"STOP_IRC",		term_pause,
	"SWAP_LAST_WINDOW",	swap_last_window,
	"SWAP_NEXT_WINDOW",	swap_next_window,
	"SWAP_PREVIOUS_WINDOW",	swap_previous_window,
	"SWITCH_CHANNELS",	switch_channels,
	"TOGGLE_INSERT_MODE",	toggle_insert_mode,
	"TOGGLE_STOP_SCREEN",	toggle_stop_screen,
	"TRANSPOSE_CHARACTERS",	input_transpose_characters,
	"TYPE_TEXT",		type_text,
	"UNSTOP_ALL_WINDOWS",	unstop_all_windows,
	"YANK_FROM_CUTBUFFER",	input_yank_cut_buffer
};


KeyMap	keys[] =
{
	SELF_INSERT,		0,	null(char *),	/* 0 */
	BEGINNING_OF_LINE,	0,	null(char *),
	BACKWARD_CHARACTER,	0,	null(char *),
	QUIT_IRC,		0,	null(char *),
	DELETE_CHARACTER,	0,	null(char *),
	END_OF_LINE,		0,	null(char *),
	FORWARD_CHARACTER,	0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	BACKSPACE,		0,	null(char *),	/* 8 */
	TOGGLE_INSERT_MODE,	0,	null(char *),
	SEND_LINE,		0,	null(char *),
	ERASE_TO_END_OF_LINE,	0,	null(char *),
	REFRESH_SCREEN,		0,	null(char *),
	SEND_LINE,		0,	null(char *),
	FORWARD_HISTORY,	0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	BACKWARD_HISTORY,	0,	null(char *),	/* 16 */
	QUOTE_CHARACTER,	0,	null(char *),
	ENTER_MENU,		0,	null(char *),
	TOGGLE_STOP_SCREEN,	0,	null(char *),
	TRANSPOSE_CHARACTERS,	0,	null(char *),
	ERASE_LINE,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	META2_CHARACTER,	0,	null(char *),	/* 24 */
	YANK_FROM_CUTBUFFER,	0,	null(char *),
			/* And I moved STOP_IRC to META1 26 */
	ENTER_DIGRAPH,		0,	null(char *),
	META1_CHARACTER,	0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 32 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 40 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 48 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 56 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 64 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 72 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 80 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 88 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 96 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 104 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 112 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),

	SELF_INSERT,		0,	null(char *),	/* 120 */
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	SELF_INSERT,		0,	null(char *),
	BACKSPACE,		0,	null(char *)
};


KeyMap	meta1_keys[] =
{
	NOTHING,		0,	null(char *),	/* 0 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	
	NOTHING,		0,	null(char *),	/* 8 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 16 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 24 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	COMMAND_COMPLETION,	0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 32 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 40 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	CLEAR_SCREEN,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 48 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 56 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 64 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 72 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 80 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 88 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	META3_CHARACTER,	0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 96 */
	NOTHING,		0,	null(char *),
	BACKWARD_WORD,		0,	null(char *),
	NOTHING,		0,	null(char *),
	DELETE_NEXT_WORD,	0,	null(char *),
	SCROLL_END,		0,	null(char *),
	FORWARD_WORD,		0,	null(char *),
	NOTHING,		0,	null(char *),

	DELETE_PREVIOUS_WORD,	0,	null(char *),	/* 104 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	SCROLL_FORWARD,		0,	null(char *),
	NOTHING,		0,	null(char *),

	SCROLL_BACKWARD,	0,	null(char *),	/* 112 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 120 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *)
};

KeyMap	meta2_keys[] =
{
	NOTHING,		0,	null(char *),	/* 0 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 8 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 16 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 24 */
	NOTHING,		0,	null(char *),
#ifdef ALLOW_STOP_IRC
	STOP_IRC,		0,	null(char *),
#else
	NOTHING,		0,	null(char *),
#endif
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 32 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 40 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 48 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 56 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 64 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 72 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 80 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 88 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 96 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 104 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NEXT_WINDOW,		0,	null(char *),
	NOTHING,		0,	null(char *),

	PREVIOUS_WINDOW,	0,	null(char *),	/* 112 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 120 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *)
};

KeyMap	meta3_keys[] =
{
	NOTHING,		0,	null(char *),	/* 0 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 8 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 16 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 24 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 32 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 40 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 48 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 56 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 64 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 72 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 80 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 88 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 96 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 104 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 112 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 120 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *)
};

KeyMap	meta4_keys[] =
{
	NOTHING,		0,	null(char *),	/* 0 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	BACKWARD_CHARACTER,	0,	null(char *),	/* 8 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 16 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 24 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	FORWARD_CHARACTER,	0,	null(char *),	/* 32 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 40 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 48 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 56 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 64 */
	META4_CHARACTER,	0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	BACKWARD_CHARACTER,	0,	null(char *),	/* 72 */
	META4_CHARACTER,	0,	null(char *),
	FORWARD_HISTORY,	0,	null(char *),
	BACKWARD_HISTORY,	0,	null(char *),
	FORWARD_CHARACTER,	0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 80 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	DELETE_CHARACTER,	0,	null(char *),	/* 88 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 96 */
	META4_CHARACTER,	0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	BACKWARD_CHARACTER,	0,	null(char *),	/* 104 */
	META4_CHARACTER,	0,	null(char *),
	FORWARD_HISTORY,	0,	null(char *),
	BACKWARD_HISTORY,	0,	null(char *),
	FORWARD_CHARACTER,	0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	NOTHING,		0,	null(char *),	/* 112 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),

	DELETE_CHARACTER,	0,	null(char *),	/* 120 */
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *),
	NOTHING,		0,	null(char *)
};


/*
 * write_binding: This will write to the given FILE pointer the information
 * about the specified key binding.  The format it writes it out is such that
 * it can be parsed back in later using LOAD or with the -l switch 
 */
static	void	write_binding(c, meta, fp)
char	c,
	meta;
FILE	*fp;
{
	KeyMap	*map;
	char	*meta_str;

	if (c == 32)
		return;
	switch (meta)
	{
	case 1:
		map = meta1_keys;
		meta_str = "META1-";
		break;
	case 2:
		map = meta2_keys;
		meta_str = "META2-";
		break;
	case 3:
		map = meta3_keys;
		meta_str = "META3-";
		break;
	case 4:
		map = meta4_keys;
		meta_str = "META4-";
		break;
	default:
		map = keys;
		meta_str = empty_string;
		break;
	}
	if (map[c].changed)
	{
		fprintf(fp, "BIND %s%s %s", meta_str, display_key(c),
			key_names[map[c].index].name);
		if (map[c].stuff && (*(map[c].stuff)))
		{
			fprintf(fp, " %s\n", map[c].stuff);
		}
		else
			fprintf(fp, "\n");
	}
}

/*
 * save_bindings: this writes all the keys bindings for IRCII to the given
 * FILE pointer using the write_binding function 
 */
void	save_bindings(fp)
FILE *fp;
{
	int	i;

	for (i = 0; i < 128; i++)
		write_binding(i, 0, fp);
	for (i = 0; i < 128; i++)
		write_binding(i, 1, fp);
	for (i = 0; i < 128; i++)
		write_binding(i, 2, fp);
	for (i = 0; i < 128; i++)
		write_binding(i, 3, fp);
	for (i = 0; i < 128; i++)
		write_binding(i, 4, fp);
}
