;**************************************************************************
;** base.lisp                                                            **
;**                                                                      **
;**  *psy-db* est la base de connaissances courante.                     **
;**  Il existe aussi une liste de proprietes *psy-kb* qui contient tou-  **
;** tes les instances des `flavors' representant les bases de connais-   **
;** sances, on y accede par le nom.                                      **
;**  Ce programme a toutes les declarations, macros, methodes et fonc-   **
;** tions impliquees dans la manipulation des bases de connaissances.    **
;**                                                                      **
;** Claudia Coiteux-Rosu                                    Juillet 1989 **
;**************************************************************************
;** Fonctions, macros et methodes:                                       **                    **
;**                                                                      **
;** Parallelisme:                                                        **
;**  serveur                etablir-liaison          lirefaits           **
;**  ajoufaits                                          	         **
;**                                                                      **
;** Base de connaissance:                                                **
;**  define-kb              undefine-kb              init-base           **
;**  use-kb                 all-kbs                  erase-trase-kb      **
;**                                                                      **
;** Hierarchies:                                                         **
;**  racine                 retour-hierarchie        racinep             **
;**  perep                  cherche-hier             desc                **
;**  fils                   go-to-kb                 fin-experts         **
;**  exec-base              ret-hier                                     **
;**                                                                      **
;** Regles de production:                                                **
;**  p                      defp                     p-do-work           **
;**  installp               buildp                   rp-inst             **
;**  rp                     removep                  printp              **
;**  printrule              printp-inst              allp                **
;**  allp-inst              allprior                                     **
;**                                                                      **
;** Faits:                                                               **
;**  fact                   add-fact                 fact-to-cons        **
;**  add-fact-to-cons       faits-argum              loadfacts           **
;**  db                     rmf                      rm                  **
;**  remove-fact            remove-all-facts         refresh             **
;**  ref                                                                 **
;**                                                                      **
;** Execution:                                                           **
;**  explain-exec-base      run                      act-reg-exec        **
;**                                                                      **
;** Options d'execution:                                                 **
;**  autorun                no-autorun               psy-trace           **
;**  psy-untrace            permet-reexec            no-reexec           **
;**  detecte-cycles         no-detecte-cycles        detecter-cycle      **
;**                                                                      **
;** Points d'arret:                                                      **
;**  halt                   checkpbreak              pbreak              **
;**  unbreak                prbreaks                                     **
;**                                                                      **
;** Conflict-set:                                                        **
;**  cs                     rmcs                     chcs                **
;**  matches                verify-cs                resort-conflict-set **
;**  add-conflict-set       remove-conflict-set                          **
;**                                                                      **
;** Strategies:                                                          **
;**  all-kb-str             str-jeune-fait           str-vieux-fait      **
;**  str-max-faits          str-min-faits            str-moins-exec      **
;**  str-random             strategy                 prstrategy          **
;**  get-ages               cmp-y-ages               cmp-y-lsts          **
;**  cmp-o-ages             cmp-o-lsts               cmp+poids           **
;**  cmp-poids              cmp-exec                 cmp-recence         **
;**  count-exec             cmp-random                                   **
;**                                                                      **
;** Reexecution:                                                         **
;**  reexec-repl-rule       recons-db-exec           act-ordre           **
;**  rec-db-exec-sp         reexec-h                 rec-db-exec-h       **
;**  cherche-ne-rac         ch-ne-pc-pere            nettoyer-desc       **
;**  garde-d-init                                                        **
;**                                                                      **
;**************************************************************************

(proclaim '(special *error-output*))

(eval-when (compile)(load "varenv")
                    (load "ps-util")
                    (load "psmacs")
                    (load "justif")
                    (load "rules"))

;**************************************************************************
;** Manipulation de Sockets - paralellisme -                             **
;**************************************************************************
;** Cette fonction cree une fenetre sur chaque machine enumeree, demarre **
;** lisp et appelle le programme lisp *mel-chemin*/psy/bases/<x>, etant  **
;** <x> dans la liste bases dans la meme position relative que la machi- **
;** ne dans la liste machines. Elle comunique aussi le nom du serveur aux**
;** processus clients et cree une liste de proprietes utilise pour recon-**
;** naitre le canal de comunnication de chaque client.                   **
(defun serveur (bases machines)
 (let ((l (length bases)))
  (cond ((and (< l 6) (eq (length machines) l))
          (init-socket)
          (for b in bases
               m in machines
               bind (n 0) 
               do (excl:run-shell-command (concatenate 'string  
                    "xterm -display $DISPLAY -bg MistyRose -title " 
                    (string m) ":" (string b) " -sb -sl 300 -e rsh " 
                    (string m) " cl " *mel-chemin* "/psy/bases/" 
                    (string b) " " (nstring-downcase (string (hostname)))
                    "&"))
                  (conect n)
                  (putprop '*clients* n b)
                  (setq n (1+ n)))
           t)
        (t nil))))


;** Cette fonction est utilisee par un client pour etablir la liaison    **
;** avec le serveur, machine doit contenir le nom de la machine serveur  **
;** Remarquer que la fonction `serveur' anterieur comunique le nom du    **
;** serveur a chaque processus client comme le deuxieme argument lisp    **
(defun etablir-liaison(machine)
   (cond ((ouvrir-liaison machine) 
          (setf *base-en-parallel* t) t)
         (t (etablir-liaison machine))))       
 
  
;** Lit les faits provenants d'un canal de communication d'un socket.    **
;** Si la variable qui est omise le lecteur est un client et les faits   **
;** proviennent du serveur, sinon le lecteur est le serveur et il speci- **
;** fie dans qui du quel client proviennent les faits a lire, si le cli- **
;** ent a ete cree sous la forme d'une base par la fonction serveur `qui'**
;** sera egale a (get '*clients* <base>). Elle retourne une liste conte- **
;** nant les faits                                                       **
(defun lirefaits(&optional qui)
  (let ((r (recevoir qui)))
    (cond ((equal r "fin") nil)
          (t (envoyer "ok" qui)
             (cons (read-from-string r) (lirefaits qui))))))


;** Cette fonction est complementaire a `lirefaits', elles ajoute les    **
;** faits lu a la base de connaissance courante                          **
(defun ajoufaits(&optional (qui nil))
  (loadfacts(lirefaits qui)))


;**************************************************************************
;** Base de connaissances                                                **
;**************************************************************************
;** Sorte de frame qui define une base de connaissances                  **
(defflavor 
        psy-database (
         name                ; nom de la base
	(facts nil)	     ; liste de faits dans la base, ordonnes par
                             ; numero de cycle en ordre descendant
	(cycle 0)	     ; numero du fait courant
	(rules nil)	     ; regles actives dans la base
	(strategy 1)         ; strategie selectione
                             ; 1 le fait le plus recente
                             ; 2 le fait le plus age
                             ; 3 la regle avec le plus des faits
                             ; 4 la regle avec le moins des faits
                             ; 5 la regle la moins utilisee
			     ; 6 random
	(conflict-set nil)   ; liste de liens aux regles (bindings) apparies
                             ; completement ordonnes selon la strategie
	(cur-binding nil)    ; le lien (binding) courante
	(pbreaks nil)	     ; regles qui causent une interruption
        (root (make-instance 'net-node :path 'car))            
                             ; entree au reseau de comparaison de la base
        (nombre-exec 0)      ; quantite d'executions de regles
        (passage-ctl nil)    ; liste (n b n1 ) utilisee seulement dans les
                             ; bases hierachiques indique quand s'est fait
                             ; le passage de controle a une autre base
                             ;  n = numero d'exec dans la base
                             ;  b = base qui prend le controle
                             ;  n1 = numero d'exec dans l'autre base
        (permet-reexec nil)  ; flag qui indique si on permet la reexecution
        (detecte-cycles nil) ; flag qui indique si on detecte les cycles
                             ; d'execution  
        (reg-cycles nil)     ; regles recyclantes   
        (reg-exec nil)       ; trace d'execution des regles
                             ; (ne r (j cs) cy)
                             ; ne = numero d'execution
                             ; r  = regle executee
                             ; j  = justification
                             ; cs = reste du conflict-set apres la regle
                             ; cy = regles recyclantes
        (ordre-reexec)       ; changement d'ordre dans le conflict-set
                             ; introduit par des reexecutions
                             ; (l1 l2 l3 ...)
                             ; li = (ne nrcs1 nrcs2 ....) 
                             ; ne= numero d'execution
                             ; nrcsj =numero de regle choisi du cs
                             ; le changement nrcsh est arrive avant nrcsh+1
        (just nil)           ; justification des faits (faits justification)
        (hierarchie nil)     ; flag qui indique si la base appartient a
                             ; une hierarchie s'il est different de nil
                             ; donne la propriete de la liste *psy-controle*
                             ; contenant le controle du systeme
	)
  nil
 :initable-instance-variables)


;** Define d'une base de connaissances                                   **
(defun define-kb (name &optional hierar)
  (cond ((get '*psy-kb* name) (msg "Base deja existente" #\N))
        (t (putprop '*psy-kb* 
            (make-instance 'psy-database :name name :hierarchie hierar)
             name))))


;** Efface la definition d'une base de connaissances                     **
(defun undefine-kb (name)
  (send (get '*psy-kb* name) 'undefine-kb))
(defmethod (psy-database undefine-kb)() 
  (let* ((*psy-trace* nil))
   (send self 'rm '*)
   (for r in rules
        bind rulename
        do (setq rulename (symeval-in-instance r 'name))
           (remprop '*rules* rulename)
           (fmakunbound (concat 'rhs- rulename))
           (send r 'remove))
   (and hierarchie
        (progn ()
         (remprop '*psy-controle* hierarchie)
         (msg "Attention la base " name "appartennait a la hierarchie "
              hierarchie #\N)))
   (remprop '*psy-kb* name))
  (and (eq self *psy-db*)
       (progn()
          (msg "***** Il n'y a plus de base courante ****" #\N)
          (setq *psy-db* nil)
          (setq *psy-root* nil) 
           T)))


;** Reinitialise une base de connaissances en la laissant prete a une    **
;** eventuelle reexecution en partant de zero                            **
(defun init-base(name)
  (send (get '*psy-db* name) 'init-base))
(defmethod (psy-database init-base)()
  (and hierarchie
      (cond ((racinep name hierarchie) (for d in (desc hierarchie name)
                                            do (init-base d))
                                     (remprop '*psy-controle* hierarchie))
            (t (msg "Base appartennant a une hierarchie, reinitialiser la"
                    "racine" #\N))))
  (send self 'rm '*)
  (setq facts nil)
  (setq cycle 0)
  (setq conflict-set nil)
  (setq cur-binding nil)
  (setq pbreaks nil)
  (setq nombre-exec 0)
  (setq passage-ctl nil)
  (setq reg-cycles nil)
  (setq reg-exec nil)
  (setq ordre-reexec nil)
  (setq just nil))
 
;** Affiche toutes les bases de connaissances du systeme                 **
(defun all-kbs()
   (msg #\N)
   (for i in (symbol-plist '*psy-kb*)
        when (not (typep i 'psy-database))
        do (msg i #\T))
   (msg #\N))

;** Define la base de connaissances courante                             **
(defun use-kb (name)
  (let ((db (get '*psy-kb* name)))
   (cond ((null db) (msg name " n'est pas une base de connaissances" #\N))
         (*psy-runbase*
             ; *psy-db* tourne dans le moment
             (msg "Il y a une base qui tourne dans le moment" #\N)
             (if *psy-var-fen* (ArretExecution) (lispbreak)))
         ((and (boundp '*psy-db*) (eq db *psy-db*)) t)
         (t (and *psy-trace* 
                (msg #\N "*********Base : " name "*********" #\N))
            (setq *psy-db* db)
            (setq *psy-name-db* name)
            (setq *psy-root* (symeval-in-instance db 'root))
             t))))

;** Efface tout la trace d'execution d'une base de connaissances         **
(defun erase-trace-kb(name)
   (send (get '*psy-kb* name) 'erase-trace))
(defmethod (psy-database erase-trace)()
  (let (au-bases)
   (setq nombre-exec 0)
   (setq reg-exec nil)
   (setq ordre-reexec nil)
   (setq just nil)
   (and hierarchie 
        (progn ()
           (setq passage-ctl nil)
           (setq au-bases (desc hierarchie))
           (setf (symbol-plist '*psy-controle*) nil)
           (for b in au-bases
                when (not (eq b name))
                do (send (get '*psy-kb* b) 'erase-trace))
        ))))

;**************************************************************************
;** Systemes hierarchiques                                               **
;**************************************************************************
;** Retourne la racine de la  hierarchie `h'                             **
(defun racine(h)
  (caar (get '*psy-controle* h)))

;** Retourne l'arbre de la hierarchie `h'                                **
(defun retour-hierarchie(h)
  (cherche-hier (racine h) h))

;** Teste si la base `b' est la racine de la hierarchie `h'              **
(defun racinep (b h)
   (eq b (racine h)))

;** Teste si `pere' est le pere de `fils' dans la hierarchie `hier'      **
(defun perep (pere fils hier)
   (member (list pere fils) (get '*psy-controle* hier) :test 'equal))

;** Retourne l'arbre de la hierarchie `h' a partir du noeud `e'          **
(defun cherche-hier (e h)
    (cons e (for (i j) in (get '*psy-controle* h)
                 when (eq i e)
                 collect (cherche-hier j h))))
  
;** Retourne une liste avec les noeuds de la hierarchie `hierar' qui     **
;** descendent du noeud `node' en excluant le noeud `excl' et ses descen-**
;** dants. Si `node' n'est pas specifie elle part de la racine           **
(defun desc (hierar &optional node excl)
  (let ((ens-excl nil)
         fils 
         desc-fils
        (noeud (if node node (racine hierar))))
   (and excl (setq ens-excl (cons excl (desc hierar excl))))
   (setq fils (fils noeud hierar ens-excl))
   (setq desc-fils 
        (for f in fils
             when (member f (get '*psy-controle* hierar) :key 'car)
             join (desc hierar f excl)))
   (cond (desc-fils (nconc fils desc-fils))
         (t fils))))

;** Retourne les noeuds fils du noeud `noeud' dans la hierarchie `h' qui **
;** ne sont pas presents dans l'ensemble `ens'                           **
(defun fils (noeud h &optional ens)
   (for (p f) in (get '*psy-controle* h)
        when (eq p noeud)
        unless (member f ens)
        collect f))

;** Passe le controle a une autre base de connaissances, cette fonction  **
;** doit etre utilisee seulement dans la partie droite des regles        **
(defun go-to-kb (name parall &rest lst)
  (let* ((db (get '*psy-kb* name))
         (old-db   *psy-db*)
         (old-name *psy-name-db*)
         (old-n-exec (symeval-in-instance old-db 'nombre-exec))
         (hierar (symeval-in-instance old-db 'hierarchie))
          fact-list
         (lst1 (subst@ '* (reverse (symeval-in-instance old-db 'facts))
                lst)))
   (cond ((null db) (msg name " n'est pas une base de connaissances" #\N)
                    (if *psy-var-fen* (ArretExecution) (lispbreak)))
         ((and *psy-runbase*
               hierar
              (cond ((member `(,name ,old-name) 
                              (get '*psy-controle* hierar) :test 'equal)
                         (msg "Utilisez `ret-hier' pour passer des faits"
                           " a la hierarchie superieur" #\N "Le controle"
                           " est passe automatiquement" #\N) nil)
                    ((member name (desc hierar nil old-name))
                      (msg "La hierarchie n'est pas respecte" #\N)
                      (msg "La base " name "depend d'une autre base" #\N)
                      (if *psy-var-fen* (ArretExecution) (lispbreak)) nil)
                    ((member name (for f in (fils old-name hierar)
                                       join (desc hierar f)))
                      (msg "La hierarchie n'est pas respecte" #\N "La base "
                           name "depend d'un autre descendant de "
                           old-name #\N)
                      (if *psy-var-fen* (ArretExecution) (lispbreak)) nil)
                    ((not (eq hierar (symeval-in-instance db 'hierarchie)))
                       (msg "La base " name " n'appartient pas a la "
                         "hierarchie de la base " old-name #\N))
                    (t t)))
             (set-in-instance db 'permet-reexec 
                             (symeval-in-instance old-db 'permet-reexec))
             (and (not (perep old-name name hierar))
                  (setf (get '*psy-controle* hierar)
                        (append (get '*psy-controle* hierar)
                               `((,old-name ,name)))))
             (setq fact-list(symeval-in-instance *psy-binding* 'fact-list))
             (and *psy-trace*
                  (msg #\N "*** Passe le controle a la base " name " ***"
                       #\N))
             (cond ((numberp parall) ; Execution parallele
                       (for f in (faits-argum lst1 fact-list "go-to-kb")
                            do (envoyer (write-to-string f) parall)
                               (recevoir parall))
                           (envoyer "fin" parall))
                   (parall ; Execution en multitraitement
                       (putprop '*psy-exp-proc* 
                          (mp:process-run-function
                             `(:name ,(write-to-string name)
                               :initial-bindings ,*psy-var-env*)
                             #'exec-base db name old-db old-n-exec
                             fact-list lst1 nil)
                           name ))
                   (t  ;Execution en monotraitement
                       (exec-base db name old-db old-n-exec
                                  fact-list lst1 *psy-trace*)))
             (and *psy-trace*
                  (msg #\N "*** Le controle est revenu a la base "
                        old-name " ***" #\N)))
         (t (msg " Fonction utilisee seulement dans les regles, "
                 "utiliser use-kb dehors de regles" #\N)))))

;** Attend la fin de l'execution parallele des bases lance par go-to-kb **
(defun fin-experts (&rest l)
   (let ((procs (for i in l 
                     collect (get '*psy-exp-proc* i))))
       (processus-inactives procs)
       (for i in l
            do (remprop '*psy-exp-proc* i))))


;** Execute une base, fonction utilisee dans go-to-kb                   **
(defun exec-base (db name-db old-db old-nom-exec fact-list lst 
                  &optional trace)
  (let ( *psy-binding*
        (*psy-db* db)
        (*psy-trace* trace)
        (*psy-name-db* name-db)
         *psy-execreg* 
         *psy-halt*
         *psy-just-point*
         *psy-runbase*
         *psy-paths*
        (*psy-root* (symeval-in-instance db 'root))
        (nom-exec (symeval-in-instance db 'nombre-exec))
        (old-name (symeval-in-instance old-db 'name))
        (faits (faits-argum lst fact-list "go-to-kb")))
     (and faits
          (progn()
            (msg "Passage de faits de la base " old-name " a la base "
                  name-db #\N)
            (for f in faits
                 do (send db 'fact f))))
     (and (not *psy-autorun*) (run))
     (cond ((eq (symeval-in-instance db 'nombre-exec) nom-exec)
              (set-in-instance old-db 'passage-ctl
                (cons (list old-nom-exec *psy-name-db* '0)
                      (symeval-in-instance old-db 'passage-ctl))))
           (t (set-in-instance old-db 'passage-ctl
                (cons (list old-nom-exec name-db (1+ nom-exec))
                      (symeval-in-instance old-db 'passage-ctl)))
              (set-in-instance db  'passage-ctl
                (cons (list (symeval-in-instance db 'nombre-exec) old-name
                            (1+ old-nom-exec))
                      (symeval-in-instance *psy-db* 'passage-ctl)))))))

;** Passe des faits a la base pere dans la hierachie, utilisee seulement**
;** dans la partie droite des regles d'une base hierachique             **
(defun ret-hier (flagp &rest lst)
 (let ((hierar (symeval-in-instance *psy-db* 'hierarchie))
       (*old-psy-execreg* *psy-execreg*)
       (autorun *psy-autorun*)
        db-fils name-fils name-pere db-pere faits)
 (cond ((not *psy-runbase*) (msg "`ret-hier' peut etre utilisee seulement"
                                 " dans la partie gauche de regles" #\N))
       ((not hierar)  (msg "`ret-hier' peut etre utilisee seulement par "
                           "les regles d'une base hierarchique" #\N))
       (t (setq db-fils   *psy-db*)
          (setq name-fils *psy-name-db*)
          (setq name-pere (for (np nf) in (get '*psy-controle* hierar)
                                until (eq nf name-fils)
                                finally np))
          (setq db-pere   (get '*psy-kb* name-pere))
          (setq faits (faits-argum 
                         (subst@ '* (reverse 
                              (symeval-in-instance db-fils 'facts)) lst)
                         (symeval-in-instance *psy-binding* 'fact-list)
                         "ret-hier"))
          (and faits
              (cond (flagp (for f in faits
                                do (envoyer (write-to-string f))
                                   (recevoir))
                           (and *psy-trace*
                                (msg #\N "*** Passage de faits a la base "
                                      name-pere "***" #\N)))
                    (t (setq *psy-execreg* nil)
                       (setq *psy-autorun* nil)
                       (msg "Copie faits de la base " name-fils " a la base "
                             name-pere #\N)
                       (for f in faits do (send db-pere 'fact f))
                       (setq *psy-execreg* *old-psy-execreg*)
                       (setq *psy-autorun* autorun)
                       (and  *psy-trace*
                             (msg #\N "*** Faits passes a la base "
                                  name-pere " ***" #\N)))))))))

;**************************************************************************
;** Manipulation des regles                                              **
;**************************************************************************
;** define une regle dans la base de connaissances courante              **
(defmacro p (name &rest rule)
   `(cond (*psy-db* (p-do-work ,name ,rule *psy-db*))
          (t (msg "Il n'y a pas de base courante" #\N))))

;** define une regle a installer plus tard dans quelque base             **
(defmacro defp (name &rest rule)
   `(p-do-work ,name ,rule nil))

;** definition d'une regle : utilise par p, defp et buildp               **
(defmacro p-do-work (name rule db)
  (let (lhs fns nots desc tests rhs vars-fcn vars-ign)
    (and (attempt 
	  (progn (and (not (atom name)) 
                      (psy-error "Regle sans nom") 
                      (fail nil))
		 (setq rhs (cdr (memq '--> rule)))
		 (and (null rhs) 
                      (psy-error "Regle sans partie droite") 
                      (fail nil))
		 (setq tests (car (for x in (cdr (memq 'test rule))
				       until (or (eq x '-->)(eq x 'desc))
				       tcollect x))

                       desc  (car (for x in (cdr (memq 'desc rule))
                                       until (or (eq x 'test)(eq x '-->))
                                       tcollect x))
		       lhs (car 
                            (for x in rule
				 unless 
                                   (and (dtpr x)
                                        (cond ((eq (car x) '|~|)
                                                       (push1 nots x))
                                              ((eq (car x) '|?|)
                                                       (push1 fns x))))
				 until (or (eq x '-->)
                                           (eq x 'test)
                                           (eq x 'desc))
				 tcollect x)))
		 t))
         (progn
           (setq vars-fcn (union (get-vars fns)(get-vars lhs)))
           (setq vars-ign (set-difference vars-fcn (get-vars rhs)))
           (setq rhs (cons `(declare (ignore ,@vars-ign)) rhs))
            t)
         (progn 'compile
             (cond (db `(progn
                          (putprop '*rules*
                                (add-rule (symeval-in-instance ,db 'name)
                                          ',name ',lhs ',fns ',nots ',desc 
                                          ',tests ,db)
                                  (concat (symeval-in-instance ,db 'name)
                                          '- ',name))
                          (defun ,(concat 'rhs-
                                      (symeval-in-instance (eval db) 'name)
                                      '- name)
                              ,vars-fcn
                              ,@rhs)
	                  (and *psy-trace* 
                               (msg #\T "definition de la partie droite"
                                    #\N))))
                   (t `(progn 
                         (putprop '*add-rule*
                              (list '',lhs '',fns '',nots '',desc '',tests)
                                 ',name)
                         (defun ,(concat 'rhs- name)
                              ,vars-fcn
                              ,@rhs)
	                 (and *psy-trace* 
                         (msg #\T "definition de la partie droite" 
                              #\N)))))))))


;** Installe une regle dans la base courante                             **
(defun installp (&rest rule-names)
    (send *psy-db* 'installp rule-names))

;** Methode associee a installp                                          **
(defmethod (psy-database installp) (&rest rule-names)
  (and (dtpr (car rule-names)) 
       (setq rule-names (car rule-names)))
  (for name-rule in rule-names
       bind list-rule
       unless (if (null (setq list-rule (get '*add-rule* name-rule)))
                (progn
                (msg name-rule 
                     " ce n'est pas une regle, elle ne sera pas installe" 
                     #\N) t))
       do (and *psy-trace* (msg "Installation de la regle " name-rule 
                                " dans la base " name ":" #\N))
          (let ((new-name (concat name '- name-rule)))
             (putprop '*rules*
                      (eval
                       `(add-rule ',name ',name-rule ,@list-rule ',self))
                       new-name)
             (setq new-name (concat 'rhs- new-name))
             (setf (symbol-function new-name)
                       (symbol-function (concat 'rhs- name-rule))))))


;** Facilite la construction des regles dans d'autres regles             **
(defmacro buildp (name rule)
  `(cond (*psy-db* (send *psy-db* 'buildp ,name ,rule))
         (t (msg "Il n'y a pas de base courante" #\N))))
(defmethod (psy-database buildp) (nom rule)
    (eval `(p-do-work ,nom ,rule ',self)))



;** Efface une regle a installer                                         **
(defun rp-inst (&rest lst)
  (and (listp (car lst)) 
       (onep (length lst)) 
       (setq lst (car lst)))
  (for rule-name in lst
       bind prop-rule
       do (setq prop-rule  (get '*add-rule* rule-name))
          (remprop '*add-rule* rule-name)
          (cond (prop-rule (fmakunbound (concat 'rhs- rule-name)))
          (t (msg rule-name " ce n'est pas une regle a installer" #\N)))))


;** Efface les regles nommees de la base courante                        **
(defun rp (&rest lst) (send *psy-db* 'rp lst))
(defmethod (psy-database rp) (&rest lst)
  (and (listp (car lst)) 
       (eq (length lst) 1) 
       (setq lst (car lst)))
  (for bind prop-rule rulename
       rule-name in lst 
       do (setq rulename (concat name '- rule-name))
          (setq prop-rule  (get '*rules* rulename))
          (remprop '*rules* rulename)
          (cond (prop-rule (setq rules (remove prop-rule rules))
                           (fmakunbound (concat 'rhs- rulename))
                           (send prop-rule 'remove))
          (t (msg  rule-name " is not a rule in db " self #\N)))))


;** Efface une instance de regle de la base recevant le message          **
(defmethod (psy-database removep) (rule)
  (for b in conflict-set
       while (eq rule (symeval-in-instance b 'back-link))
       do (pop conflict-set))
  (for bind (back conflict-set)
       while (cdr back)
       do (cond ((eq rule (symeval-in-instance (cadr back) 'back-link))
		   (rplacd back (cddr back)))
		(t (setq back (cdr back)))))
  (setq rules (delete rule rules :test 'eq :count 1))) 


;** Affiche les regles nommees                                           **
(defun printp (&rest names)
 (for r in names
      bind rule 
      do (setq rule 
           (get '*rules*
                (concat *psy-name-db* '- r)))    
         (if rule 
              (send rule 'rprint r)
	      (msg r " n'est pas une regle dans la base " 
                     *psy-name-db* #\N))))


;** Affiche la regle decrite par nom, pg et pd (p. gauche et p. droite   **
(defun printrule (nom pg rhs)
  (let ((lhs (cadar pg))
        (fns (cadadr pg))
        (nots (cadr (caddr pg)))
        (desc (cadr (cadddr pg)))
        (tests (cadar (cddddr pg))))
  (msg #\N "   Regle: " nom #\N)
  (for initially   (msg "     Partie gauche  :" #\N)
       x in lhs do (msg "                     " x #\N))
  (and fns  (for  initially
                   (msg "     Fonctions PG   :" #\N)
                  x in fns do
                   (msg "                     " x #\N)))
  (and nots (for  initially
                   (msg "     Negations      :" #\N)
                  x in nots do
                   (msg "                     " x #\N)))
  (and desc (for  initially
                   (msg "     Description    :" #\N)
                  x in desc do
                   (msg "                     " x #\N)))
  (and tests (for initially
                   (msg "     Tests          :" #\N)
                  x in tests do
                   (msg "                     " x #\N)))
  (cond ((typep rhs 'compiled-function)
                   (msg "     La partie droite est compilee" #\N))
        (t    (setq rhs (cddar (cdddr rhs)))
              (msg "     Partie droite  :" #\N)
              (for x in rhs do
              (msg "                     " x #\N))))))
  

;** Affiche une ou plusieures regles a installer                         **  
(defun printp-inst(&rest names)
   (for r in names
        bind pg
        do (setq pg (get '*add-rule* r))
           (if pg (printrule r pg (symbol-function (concat 'rhs- r)))
                  (msg r " ce n'est pas une regle a installer" #\N))))


;** Affiche les noms des regles de la base courante                      **
(defun allp nil (send *psy-db* 'allp))
(defmethod (psy-database allp) ()
  (msg "Regles de production actives dans la base :" name #\N)
  (for r in rules 
       do (msg #\T (take-rule-name (symeval-in-instance  r 'name)) #\N)))


;**  Affiche les noms des regles a installer                             **
(defun allp-inst()
  (for bind (l (symbol-plist '*add-rule*) (cddr l))
       until (null l)
       do (msg #\T (car l) #\N)))


;** Affiche les priorites des regles de la base courante                 **
(defun allprior nil (send *psy-db* 'allprior))

;** Methode associe a allprior                                           **
(defmethod (psy-database allprior) ()
  (let ((reg-prior (sort (for r in rules
                              collect (list(take-rule-name 
                                           (symeval-in-instance  r 'name))
                                           (symeval-in-instance r 'prior)))
                         #'(lambda (a b) (< (cadr a) (cadr b))))))
   (msg "Priorites des regles dans la base :" name #\N)
   (msg "Priorite" #\T "Regle" #\N)
   (for r in reg-prior
        do (msg (cadr r)"     "  #\T (car r) #\N))))


;**************************************************************************
;** Manipulation des faits                                               **
;**************************************************************************

;** Ajoute un fait                                                       **
(defmacro fact (&rest x)
  `(add-fact ,(quotify x)))

;** Fonction associee a la macro fact                                    **
(defun add-fact (x &optional ref) (send *psy-db* 'fact x ref))

;** Methode associe a la fonction add-fact                               **
(defmethod (psy-database fact) (x &optional ref)
  (let ((*psy-root* root))
    (setq cycle (add1 cycle))
    (and *psy-trace*
         (msg #\T "Addition du fait :" #\T cycle ". " x #\N))
    (push1 facts (make-instance 'fact :cycle cycle :value x))
    (send root 'adds-fact (car facts))
    (for bn in (symeval-in-instance root 'bind-nodes)
         when (eq (car (symeval-in-instance bn 'pattern)) '|?|)
         do (send bn 'verify-fcn facts))
    (agrega (car(symeval-in-instance self 'facts))(cond (*psy-execreg* "r")
                                                         (ref ref)
                                                         (t nil)) self)
    (cond (cur-binding)	; already running database
	  (conflict-set
		(and *psy-autorun* (send self 'run))))
     (car facts)))


;** Construit la liste `(fact ...), utilisee pour ajoute un fait dans    **
;** une autre base en combinaison avec la fonction go-to-kb              **
(defmacro fact-to-cons (&rest x)
   `(add-fact-to-cons ,(quotify x)))
(defun add-fact-to-cons (x) 
   (cons 'fact x))


;** Recherche des faits explicites dans un go-to-kb ou dans ret-hier     **
;** excepte le symbol '*                                                 **
(defun faits-argum(lst-reg faits-bdg op)
 (for x in lst-reg
      bind (len (length faits-bdg))
      unless (and (numberp x)
                  (> x len)
                  (progn ()
                   (princ " il n'y a pas de fait ")
                   (print x)
                   (error (concatenate 'string " numero de fait: " op))))
      collect (cond ((numberp x) (symeval-in-instance
                                        (nth (- len x) faits-bdg) 'value))
                    ((typep x 'fact) (symeval-in-instance x 'value))
                    ((and (listp x)(eq (car x) 'fact)) (cdr x))
                    (t (msg x " n'est pas un fait")
                       (error  (concatenate 'string "fait: " op))))))




;** Ajoute les faits presentes dans la liste `lst' a la base courante    **
(defun loadfacts (lst)
  (for x in lst do (send *psy-db* 'fact x)))
(defmethod (psy-database loadfacts) (lst)
  (for x in lst do (send self 'fact x)))


;** Affiche tous les faits de la base courante                           **
(defun db nil (send *psy-db* 'db))

;** Methode associee a db                                                **
(defmethod (psy-database db) ()
  (msg "Faits dans la base : " name #\N)
  (for initially (msg "Cycle" #\T "Fait" #\N)
       f in (reverse facts)
       do (msg (symeval-in-instance f 'cycle) "." #\T 
               (symeval-in-instance f 'value) #\N)))

;** Efface les faits enumerees par `lst' , fonction utilisee seulement   **
;** dans la partie gauche des regles                                     **
(defun rmf (&rest lst)
 (let (fact-list)
  (cond (*psy-runbase*
	   (setq fact-list (symeval-in-instance *psy-binding* 'fact-list))
	   (for x in lst
		bind f (len (length fact-list))
		do (setq f (nth (- len x) fact-list))
		   (if (null f)
		       (progn (msg x " n'est pas un numero de 'pattern'" #\N)
		              (error " ")))
		   (send *psy-db* 'remove-fact f)))
	(t (msg "Utiliser 'rm' en dehors des regles" #\N)
	   nil))))

;** Efface les fait enumereex par `lst' (utilisee en dehors des regles)  **
(defun rm (&rest lst) (send *psy-db* 'rm lst))
(defmethod (psy-database rm) (&rest lst)
  (and (listp (car lst)) 
       (onep (length lst)) 
       (setq lst (car lst)))
  (if (eq (car lst) '*)
     (send self 'remove-all-facts)
     (for x in lst
	  do (or (for f in facts
		      thereis (if (eq (symeval-in-instance f 'cycle) x)
				 (progn 
                                    (send self 'remove-fact f)
				     t )))
		    (msg "Il n'y a pas de fait du cycle " x
			 " dans cette base." #\N)))))

;** Methode associee qui efface un fait                                  **
(defmethod (psy-database remove-fact) (fact)
       (setq facts (delete  fact facts :test 'eq :count 1))
       (and *psy-execreg* (elimina fact))
       (send fact 'remove root)
       (for bdg in conflict-set
	    when (memq fact (symeval-in-instance bdg 'fact-list))
	    do (send self 'remove-conflict-set bdg)))

;** Methode qui efface tous les faits de la base recevante le message    **
(defmethod (psy-database remove-all-facts) nil
  (for f in facts 
       do (send f 'remove root))
  (setq facts nil)
  (setq conflict-set nil)
  (and *psy-trace* (msg #\T "Tous les faits sont effaces de la base " 
                             name #\N)))

;** Efface et ajoute tous les faits qui ont ete apparies avec les modeles**
;** qui ont les numeros d'ordre de la liste `lst' dans la regle. Si `lst'**
;** est vide tous les faits de la base courante sont refraichis. Utiliser**
;** seulement dans la partie gauche des regles                           **
(defun refresh (&rest lst)
 (let (lst2 flst)
  (cond ((null *psy-runbase*)
	   (msg "Utiliser 'ref' en dehors des regles" #\N))
	(lst
	 (setq flst (symeval-in-instance *psy-binding* 'fact-list))
	 (setq lst2
	       (car (for x in lst
			 bind (len (length flst)) f
			 tcollect (setq f (nth (- len x) flst))
				  (if (null f)
                                   (progn
				     (msg x " n'est pas un numero de 'pattern'"
                                       #\N)
				     (error " ")))
				  (send *psy-db* 'remove-fact f)
				  (symeval-in-instance f 'value)))))
	(t (setq lst2 (reverse 
                         (car (for f in (symeval-in-instance *psy-db* 'facts)
		                 tcollect (symeval-in-instance f 'value)))))
	   (send *psy-db* 'remove-all-facts)))
   (for	f in lst2 do (send *psy-db* 'fact f (trouve-justif f)))))

;** Efface et ajoute tous les faits enumeres par `lst'. Si `lst' est vide**
;** tous les faits de la base courante sont refraichis.                  **
(defun ref (&rest lst) (send *psy-db* 'ref lst))

;** Methode associee a ref                                               **
(defmethod (psy-database ref) (&rest lst)
  (and (listp (car lst)) (onep (length lst)) (setq lst (car lst)))
  (let (lst2)
    (cond (lst
	     (for x in lst
		  do (or (for f in facts
			     thereis (if (eq (symeval-in-instance f 'cycle) x)
				       (progn
                                         (send self 'remove-fact f)
					 (push1 lst2 
                                            (symeval-in-instance f 'value)))))
			 (msg "le numero " x " n'est fait de la base" #\N))))
	  (t (setq lst2 (reverse 
                           (car (for f in facts
                                   tcollect (symeval-in-instance f 'value)))))
	     (send self 'remove-all-facts)))
    (for f in lst2 do (add-fact f (trouve-justif f)))))


;**************************************************************************
;** Execution du systeme                                                 **
;**************************************************************************

;** Methode qui affiche la trace d'execution de la base recevant le msg  **
(defmethod (psy-database explain-exec-base)(&optional (num-exec 0)(ppal t)(sl t))
 (let (dernier-exec
       fils)
  (declare (special lignes))
  (declare (special tabs))
  (setq dernier-exec
   (for r in reg-exec
        bind pass-ctl
        when (>= (car r) num-exec)
        until (for h in passage-ctl
                   thereis (and (> (car r) (car h))
                                (>= (car h) num-exec)
                                (perep (cadr h) name hierarchie)))  
        do (setq lignes (1+ lignes))
           (saut-ligne lignes tabs sl)
           (msg "  " (car r) ". "
                (take-rule-name (symeval-in-instance (cadr r) 'name)) #\N)
           (setq pass-ctl
                 (for ctl in passage-ctl
                      when (eq (car r) (car ctl))
                      collect ctl))
           (and pass-ctl 
                (setq fils
                     (for h in pass-ctl
                          when (not (perep (cadr h) name hierarchie))
                      collect h)) 
                (setq tabs (1+ tabs))                
                (progn()
                (setq lignes (1+ lignes))
                (saut-ligne lignes tabs sl)
                (msg  "*** Transfert de controle ****" #\N)
                (for ctl in (reverse fils)
                     do (setq lignes (1+ lignes))
                        (saut-ligne lignes tabs sl)
                        (msg "Base " (cadr ctl) #\N)
                        (cond ((eq (caddr ctl) 0)
                                   (setq lignes (1+ lignes))
                                   (saut-ligne lignes tabs sl)
                                  (msg #\T "Cette base n'a pas execute" #\N))
                              (t (send (get '*psy-kb* (cadr ctl)) 
                                       'explain-exec-base (caddr ctl) nil sl)))
                        )
                (setq tabs (1- tabs))
                t))
        finally (car r)))
  (if (and ppal dernier-exec)
           (progn () 
             (setq lignes (1+ lignes))
             (saut-ligne lignes tabs sl)
             (msg "*** Changement de controle ***" #\N)
             (send self 'explain-exec-base dernier-exec t sl)))))

;** Mise en marche du systeme, si n est > 0  execute n regles au maximun **
(defun run (&optional (n -1))(send *psy-db* 'run n))

;** Methode associee a run                                               **
(defmethod (psy-database run) (&optional (n -1))
  (let (halt)
    (unwind-protect
     (progn
        (setq *psy-runbase* name)
	(for bind rule-name rule f-util
	     while (and conflict-set 
                        (not (eq n 0)) 
                        (not *psy-halt*)
                        (eq  *psy-runbase* name)
                        (not (and (member nombre-exec pbreaks)
                                  (or (null hierarchie)
                                      (racinep name hierarchie)))))

             unless  (and detecte-cycles
                          (send *psy-db* 'detecter-cycle  (car conflict-set) 
                                (eq detecte-cycles 'val))
                          (setq reg-cycles
                               (append reg-cycles
                                        `(,(symeval-in-instance 
                                            (pop conflict-set) 'back-link)))))

	     do (and (member nombre-exec pbreaks)
                      hierarchie
                     (not (racinep name hierarchie))
                     (let ((db *psy-db*))
                        (setq *psy-db* self)
                        (msg "Vous etes arrete maintenant dans l'execution "
                              (1+ nombre-exec) " de la base " name #\N
                             "Tapez :cont quand vous voudrez continuer "
                             "l'execution" #\N)
                        (if *psy-var-fen* (ArretExecution) (lispbreak))
                        (setq *psy-db* db))) 
                (send self 'verify-cs) 
                (setq   nombre-exec   (1+ nombre-exec))
                (setq   n	      (1- n)
                       *psy-binding*  (pop conflict-set)
                        rule          (symeval-in-instance *psy-binding* 
                                                           'back-link))
                (setq   rule-name     (symeval-in-instance rule 'name)
                        f-util        (reverse (symeval-in-instance
                                                    *psy-binding* 'fact-list))
		        cur-binding   *psy-binding*)
                (setq *psy-just-point* (make-instance 'just :rule rule
                                                            :util f-util))
                (send self 'act-reg-exec rule)
                (setq *psy-execreg* t)
		                (and *psy-trace* 
                     (for initially (msg "Execution de la regle "
                                        (take-rule-name rule-name) #\N)
			            (msg  #\T "Faits utilises  :" #\N)
			  f in (reverse (symeval-in-instance 
                                                *psy-binding* 'fact-list))
			  do (msg #\T #\T (symeval-in-instance f 'cycle)
                                  ". " (symeval-in-instance f 'value) #\N)))
		                (send self 'checkpbreak 
                          (symeval-in-instance *psy-binding* 'back-link))
		                (apply (concat '|RHS-| rule-name) 
                       (symeval-in-instance *psy-binding* 'values))
		 (and *psy-trace* (terpri))
	     finally (setq halt *psy-halt*)
                     (setq *psy-execreg* nil))
        (and *base-en-parallel* 
             (null conflict-set)
             (not *psy-halt*)
             (envoyer "fin"))
	(and *psy-trace*
	     (null conflict-set)
             (setq *psy-execreg* nil)
	     (msg "No rules currently in the conflict set" #\N))
	(if (eq halt 'break)
          (progn
	    (msg "Halt of *psy-db*: " self #\N)
	    (if *psy-var-fen* (ArretExecution)(lispbreak)))))
     (setq cur-binding nil)
     (setq *psy-runbase* nil))))

;** Methode qui actualice la case reg-exec de la base recevant le message**
(defmethod (psy-database act-reg-exec)(i-reg)
  (let (j-cs) 
    (and permet-reexec
        (setq j-cs (list *psy-just-point*
                        (for bdg in conflict-set
                             collect (symeval-in-instance bdg 'back-link)))))
    (setq reg-exec 
         (append reg-exec `((,nombre-exec ,i-reg ,j-cs ,reg-cycles))))
    (setq reg-cycles nil)))


;**************************************************************************
;** Modes d'execution du systeme                                         **
;**************************************************************************

;** Mise en marche automatique apres qu'un fait est rentre               **
(defun autorun ()
   (setq *psy-autorun* t))

;** Mise en marche manuel                                                **
(defun no-autorun()
   (setq *psy-autorun* nil))

;** Indique qu'il doit avoir une trace d'execution                       **
(defun psy-trace ()
  (setq *psy-trace* t))

;** Indique qu'il ne doit pas avoir de trace d'execution                 **
(defun psy-untrace ()
  (setq *psy-trace* nil))
 
;** Determine qu'on peut reexecuter les systemes indiques par `base'     **
;** `base' = '*    - toutes les bases existentes sont reexecutables      **
;** `base' = nom   - la base nomee est reexecutable                      **
;** `base' = nil   - (sans argument) la base courante est reexecutable   **
;** `base' = liste - toutes les bases figurant dans `liste' sont reexec. **
(defun permet-reexec (&optional (base nil))
   (let (inst-base)
   (cond ((eq base '*) (for b in (symbol-plist '*psy-kb*)
                            when (typep b 'psy-database)
                            do (set-in-instance b 'permet-reexec t)))
         ((setq inst-base (get '*psy-kb* base))
                   (set-in-instance inst-base 'permet-reexec t))
         ((null base)
                   (set-in-instance *psy-db* 'permet-reexec t))
         ((listp base) (for b in base
                            do (set-in-instance (get '*psy-kb* b) 
                                  'permet-reexec t))))))

;** Indique qu'on ne peut pas reexecuter le systeme                      **
(defun no-reexec (&optional base)
   (let ((inst-base (cond (base (get '*psy-kb* base))
                          (t *psy-db*))))
      (send inst-base 'no-reexec)))

;** Methode associee a no-reexec                                         **
(defmethod (psy-database no-reexec)()
    (and (not detecte-cycles)
         (setq permet-reexec nil))
      T)

;** Determine la detection de cycles d'execution, c'est a dire si une re-**
;** gle est execute avec les memes appariements que dans une autre execu-**
;** tion 1) `value' = t     => les faits apparies sont compares par eq   **
;**      2) `value' = 'val  => les faits apparies sont compares par equal**
(defun detecte-cycles(&optional value base)
  (let ((b (cond (base (get '*psy-kb* base))
                 (t *psy-db*))))
    (send b 'detecte-cycles value)))

;** Methode associe a detecte-cycles                                     **
(defmethod (psy-database detecte-cycles)(val)
    (setq detecte-cycles val
          reg-cycles nil
          permet-reexec t))

;** Indique qu'on ne peut pas detecter les cycles                        **
(defun no-detecte-cycles (&optional base)
  (let ((b (cond (base (get '*psy-kb* base))
                 (t *psy-db*))))
   (set-in-instance b 'detecte-cycles nil)))

;** Detecte les cycles d'execution                                       **
(defmethod (psy-database detecter-cycle) (bdg value)
 (let ((j-cycle
          (for (ne r (j cs)) in reg-exec
               until (and
                      (eq (symeval-in-instance bdg 'back-link) r)
                      (cond (value (mapcan
                                    #'(lambda (e1 e2)
                                       (equal (symeval-in-instance e1 'value)
                                              (symeval-in-instance e2 'value)))
                                     (symeval-in-instance j 'util)
                                     (symeval-in-instance bdg 'fact-list)))
                           (t (equal (symeval-in-instance j 'util)
                                     (symeval-in-instance bdg 'fact-list)))))
               finally j)))
  (and j-cycle
      (progn ()
         (msg "******Attention *******" #\N)
         (msg "Cycle detecte : " #\N)
         (msg "       Regle " 
                  (symeval-in-instance 
                      (symeval-in-instance bdg 'back-link) 'name) #\N)
         (msg #\T "                " #\T "Faits utilisees" #\N)
	 (for f in (symeval-in-instance bdg 'fact-list)
	       do (msg #\T #\T (symeval-in-instance f 'cycle) ". " 
                       (symeval-in-instance f 'value) #\N))
         t))))
 

;**************************************************************************
;** Points d'arret du systeme                                            **
;**************************************************************************

;** Arrete l'execution du systeme si on l'appelle dans une regle avec    **
;** `flg'= 'break                                                        **
(defun halt (&optional (flg t)) (setq *psy-halt* flg))

;** Verifie s'il y a des points d'arret pour une regle a executer        **
(defmethod (psy-database checkpbreak) (rule)
 (cond ((memq rule pbreaks)
	  (msg "Point d'arret dans la regle: " 
               (take-rule-name (symeval-in-instance rule 'name))
               " dans la base: " name #\N)
	  (apply `(lambda ,(symeval-in-instance 
                             (symeval-in-instance rule 'out-node) 'vars)
                           (if *psy-var-fen* (ArretExecution)(lispbreak)))
		 (symeval-in-instance *psy-binding* 'values)))))

;** Marque un point d'arret pour chaque regle de la liste `lst'          **
(defun pbreak (&rest lst) (send *psy-db* 'pbreak lst))

;** Methode associee a pbreak                                            **
(defmethod (psy-database pbreak) (&rest lst)
  (and (listp (car lst)) (onep (length lst)) (setq lst (car lst)))
  (for rname in lst
       bind rule 
       when (cond ((setq rule (get '*rules* (concat name '- rname))))
	      	  (t (msg rname " n'est pas une regle" #\N)  nil))
       unless (memq rule pbreaks)
       do (push1 pbreaks rule)))

;** Efface le point d'arret pour chaque regle de la liste `lst'          **
(defun unbreak (&rest lst) (send *psy-db* 'unbreak lst))

;** Methode associee a unbreak                                           **
(defmethod (psy-database unbreak) (&rest lst)
  (and (listp (car lst)) (onep (length lst)) (setq lst (car lst)))
  (for rname in lst
       bind rule
       when (setq rule (get '*rules* (concat name '- rname)))
       do (setq pbreaks (remove rule pbreaks :test 'eq))))

;** Affiche quelles sont les regles qui on de points d'arret             **
(defun prbreaks nil (send *psy-db* 'prbreaks))
(defmethod (psy-database prbreaks) nil
  (cond ((eq self *psy-db*)
	   (msg "Points d'arret dans la base courante:" #\N))
	(t (msg "Points d'arret dans la base: " name #\N)))
  (for rule in pbreaks 
       do (msg #\T (take-rule-name (symeval-in-instance rule 'name)) #\N)))


;**************************************************************************
;** Manipulation du "conflict set"                                       **
;**************************************************************************

;** Affiche le "conflict set"                                            **
(defun cs (&optional (retour nil))  (send *psy-db* 'cs retour))

;** Methode associee a cs                                                **
(defmethod (psy-database cs) (retour)
 (let ((cs conflict-set)
       (val-retour nil))
  (send self 'verify-cs)
  (msg "'Conflict set' de la base " name #\N)
  (for b in conflict-set
       i from 1
       bind nom
       do (setq nom (take-rule-name (symeval-in-instance 
                                        (symeval-in-instance b 'back-link)
                                         'name)))
          (and retour (push nom val-retour))
          (msg i ". "  nom #\N))
  (setq conflict-set cs)
   val-retour))

;** Enleve les liens (bindings) aux regles du "conflict set"             **
(defun rmcs (&rest lst) (send *psy-db* 'rmcs lst))
(defmethod (psy-database rmcs) (&rest lst)
  (and (listp (car lst)) (onep (length lst)) (setq lst (car lst)))
  (for b in conflict-set
       i from 1
       when (memq i lst)
       do (setq conflict-set (delete b conflict-set :test 'eq :count 1))))

;** Change l'ordre du conflict-set, met comme premiere la regle numero n **
(defun chcs (n) (send *psy-db* 'chcs n))

;** Methode associee a chcs                                              **
(defmethod (psy-database chcs) (n)
  (let ((l (length conflict-set)))
     (cond ((= l 1) (msg "Impossible de changer l'ordre: le conflict-set a un" 
                         " seul element" #\N))
           ((or (not (numberp n))
                (< n 2)
                (> n l)) (msg "L'argument de cette fonction doit etre un" 
                              "nombre entre 2 et " l #\N))
           (t (setq ordre-reexec (cons `(,(1+ nombre-exec) ,(1- n))
                                         ordre-reexec))))))

;** Affiche les faits qui ont ete apparies aux regles de la liste `lst'  **
(defun matches (&rest lst) (send *psy-db* 'matches lst))

;** Methode associee a matches                                           **
(defmethod (psy-database matches) (&rest lst)
  (and (listp (car lst)) (onep (length lst)) (setq lst (car lst)))
  (msg "'Conflict set' de la base " name #\N)
  (msg #\T "Regles appariees" #\T "Faits utilisees" #\N)
  (for b in conflict-set
       i from 1
       when (or (null lst) 
                (memq (symeval-in-instance 
                         (symeval-in-instance b 'back-link) 'name) lst))
       do (msg "         " i ". " 
               (symeval-in-instance 
                   (symeval-in-instance b 'back-link) 'name))
	  (for f in (symeval-in-instance b 'fact-list)
	       do (msg #\T #\T (symeval-in-instance f 'cycle) ". " 
                       (symeval-in-instance f 'value) #\N))))

;** Verifie l'ordre du conflict-set et le reordonne si necessaire        **
(defmethod (psy-database verify-cs)()
  (let ((ordre (assoc (1+ nombre-exec) ordre-reexec))
         bdg-reg)
    (cond ((= strategy 6) 
           (and ordre
               (progn ()
                  (setq bdg-reg 
                       (car (for c in conflict-set
                                 when (eq (cadr ordre)
                                          (symeval-in-instance c 'back-link))
                                 collect c)))
                  (if (null bdg-reg)
                      (progn() (msg "La strategie random de cette base pose"
                                    " des problemes de reexecution " #\N)
                               (if *psy-var-fen* (ArretExecution)(lispbreak))))
                  (setq conflict-set 
                   (cons bdg-reg (remove bdg-reg conflict-set)))
                  T)))
          (t (for n in (cdr ordre)
                  bind bdg
                  do (setq bdg (nth n conflict-set))
                     (setq conflict-set 
                          (cons bdg (remove bdg conflict-set)))
                     )))))
 
;** Ordonne le "conflict-set" selon la strategie courante                **
(defmethod (psy-database resort-conflict-set) ()
  (for b in conflict-set
       do (set-in-instance b 'age-list 
                          (get-ages (symeval-in-instance b 'fact-list) 
                                    strategy)))
  (setq conflict-set(case strategy (1 (sort conflict-set 'cmp-y-ages))
                                   (2 (sort conflict-set 'cmp-o-ages))
                                   (3 (sort conflict-set 'cmp+poids))
                                   (4 (sort conflict-set 'cmp-poids))
                                   (5 (sort conflict-set 'cmp-exec))
                                   (6 (sort conflict-set 'cmp-random)))))

;** Ajoute un bindings au conclict-set                                   **
(defmethod (psy-database add-conflict-set) (b)
 (set-in-instance b 'age-list 
                 (get-ages (symeval-in-instance b 'fact-list) strategy))
 (setq conflict-set (insert b conflict-set (case strategy (1 'cmp-y-ages)
                                                          (2 'cmp-o-ages)
                                                          (3 'cmp+poids)
                                                          (4 'cmp-poids)
                                                          (5 'cmp-exec)
					                  (6 'cmp-random)))))

;** Efface un lien (binding) du "conflict-set"                           **
(defmethod (psy-database remove-conflict-set) (binding)
  (setq conflict-set (delete binding conflict-set :test 'eq :count 1)))


;**************************************************************************
;** Strategies d'execution                                               **
;**************************************************************************

;** Affiche les strategies de toutes les bases du systeme                **
(defun all-kb-str ()
   (for b in (symbol-plist '*psy-kb*)
        when (typep b 'psy-database)
        do (send b 'prstrategy)))

;** Strategie du fait le plus recente                                    **
(defun str-jeune-fait () (send *psy-db* 'strategy 1))

;** Strategie du fait le plus age                                        **
(defun str-vieux-fait () (send *psy-db* 'strategy 2))

;** Strategie de la regle avec le plus de faits                          **
(defun str-max-faits () (send *psy-db* 'strategy 3))

;** Strategie de la regle avec le moins de faits                         **
(defun str-min-faits () (send *psy-db* 'strategy 4))

;** Strategie de la regle la moins executee                              **
(defun str-moins-exec () (send *psy-db* 'strategy 5))

;** Strategie random                                                     **
(defun str-random () (send *psy-db* 'strategy 6))

;** Methode qui met la strategie `x' a la base recevant le message       **
(defmethod (psy-database strategy) (x)
  (and (or (and (= strategy 6)
                (< x 6))
           (and (< strategy 6)
                (= x 6)))
        reg-exec
       (progn()
          (msg "Attention la reexecution de cette base n'est plus fiable" #\N)
          (setq ordre-reexec nil)))
  (setq strategy x)
  (send self 'resort-conflict-set)
  nil)

;** Affiche la strategie de la base courante                             **
(defun prstrategy nil (send *psy-db* 'prstrategy))
(defmethod (psy-database prstrategy) nil
  (msg "La base " name " utilise la strategie ")
  (case strategy (1 (msg " du fait le plus recente " #\N))
                 (2 (msg " du fait le plus age " #\N))
                 (3 (msg " de la regle avec plus des faits" #\N))
                 (4 (msg " de la regle avec le moins des faits" #\N))
                 (5 (msg " de la regle la moins execute" #\N))
                 (6 (msg " random" #\N)))
   nil)

;** Ajoute l'age des faits aux bindings selon la strategie               **
(defun get-ages (fact-list strategy)
  (case strategy (1 (sort (car (for f in fact-list 
                                    tcollect (symeval-in-instance f 'cycle)))
		           'greaterp))
                 (2 (sort (car (for f in fact-list 
                                    tcollect (symeval-in-instance f 'cycle)))
		           'lessp))))

;** Teste si les faits relies a b1 sont plus jeunes que ceux relies a b2 **
(defun cmp-y-ages (b1 b2)
  (let ((prio1 (symeval-in-instance (symeval-in-instance b1 'back-link)
                                   'prior))
        (prio2 (symeval-in-instance (symeval-in-instance b2 'back-link)
                                   'prior)))
   (cond ((eq prio1 prio2) (cmp-y-lsts (symeval-in-instance b1 'age-list)
	                   (symeval-in-instance b2 'age-list)))
         (t (lessp prio1 prio2)))))

;** Compare si les ages de `lst1' sont plus jeunes que les ages de `lst2'**
(defun cmp-y-lsts (lst1 lst2)
 (for n1 in lst1
      n2 in lst2
      do (cond ((greaterp n1 n2) (return t))
	       ((greaterp n2 n1) (return nil)))
      finally (cond (n1 t)
		    (n2 nil)
		    (t 'eq))))

;** Teste si les faits relies a b1 sont plus vieux que ceux relies a b2  **
(defun cmp-o-ages (b1 b2)
  (let ((prio1 (symeval-in-instance (symeval-in-instance b1 'back-link)
                                   'prior))
        (prio2 (symeval-in-instance (symeval-in-instance b2 'back-link)
                                   'prior)))
   (cond ((eq prio1 prio2) (cmp-o-lsts (symeval-in-instance b1 'age-list)
	                               (symeval-in-instance b2 'age-list)))
         (t (lessp prio1 prio2)))))

;** Compare si les ages de `lst1' sont plus vieux que les ages de `lst2' **
(defun cmp-o-lsts (lst1 lst2)
 (for n1 in lst1
      n2 in lst2
      do (cond ((lessp n1 n2) (return t))
	       ((lessp n2 n1)))
      finally (cond (n1 t)
		    (n2 nil)
		    (t 'eq))))

;** Compare si b1 a plus de poids que b2                                 **
(defun cmp+poids (b1 b2)
  (let ((prio1 (symeval-in-instance (symeval-in-instance b1 'back-link)
                                   'prior))
        (prio2 (symeval-in-instance (symeval-in-instance b2 'back-link)
                                   'prior)))
   (cond ((eq prio1 prio2) 
             (greaterp (length (symeval-in-instance b1 'fact-list))
                       (length (symeval-in-instance b2 'fact-list))))
         (t (lessp prio1 prio2)))))

;** Compare si b1 moins de poids que b2                                  **
(defun cmp-poids (b1 b2)
  (let ((prio1 (symeval-in-instance (symeval-in-instance b1 'back-link)
                                   'prior))
        (prio2 (symeval-in-instance (symeval-in-instance b2 'back-link)
                                   'prior)))
   (cond ((eq prio1 prio2) 
             (lessp (length (symeval-in-instance b1 'fact-list))
                       (length (symeval-in-instance b2 'fact-list))))
         (t (lessp prio1 prio2)))))

;** Compare si la regle relie a b1 qui a ete execute moins des fois que  **
;** celle relie a b2                                                     **
(defun cmp-exec (b1 b2)
  (let* ((r1 (symeval-in-instance b1 'back-link)) 
         (r2 (symeval-in-instance b2 'back-link)) 
         (prio1 (symeval-in-instance r1 'prior))
         (prio2 (symeval-in-instance r2 'prior)) 
         (exec1 (count-exec r1))
         (exec2 (count-exec r2)))
   (cond ((eq prio1 prio2) 
             (cond ((lessp exec1 exec2) t)
                   ((eq exec1 exec2) (cmp-recence r1 r2))
                   (t nil)))
         (t (lessp prio1 prio2)))))

;** Compare si la regle r1 a ete execute avant que la regle r2           **
(defun cmp-recence (r1 r2)
   (let* ((reg-exec (reverse (symeval-in-instance *psy-db* 'reg-exec)))
          (exec1 (caar (member r1 reg-exec :test 'eq :key 'cadr)))
          (exec2 (caar (member r2 reg-exec :test 'eq :key 'cadr))))
      (and exec1
           exec2
          (lessp exec1 exec2))))

;** Compte les executions d'une regle                                    **
(defun count-exec (regle)
   (let ((nombre 0))
      (mapc #'(lambda (x) (and (eq (cadr x) regle)
                               (setq nombre (1+ nombre))))
              (symeval-in-instance *psy-db* 'reg-exec))
       nombre))

;** Compare en forme aleatoire b1 et b2                                  **
(defun cmp-random (b1 b2)
  (oddp (random (+ (length (symeval-in-instance b1 'fact-list))
                   (length (symeval-in-instance b2 'fact-list))))))
              

;**************************************************************************
;** Reexecution du systeme                                               **
;**************************************************************************

;** Reexecute la base `nom-base'  en reemplacant la regle execute dans   **
;** l'execution numero exec par la regle numero rnum du conflict set du  **
;** moment                                                               **
(defun reexec-repl-rule (exec nom-base rnum)
  (let ((db (get '*psy-kb* nom-base))
        (old-autorun *psy-autorun*)
         hier
         faits-non-aj)
    (cond (db (setq hier (symeval-in-instance db 'hierarchie))
              (cond ((or (null hier)
                         (eq nom-base (racine hier)))
                       (setq faits-non-aj
                            (send db 'recons-db-exec exec rnum))
                       (send db 'run)
                       (autorun)
                       (for f in faits-non-aj
                            do (send db 'fact f)))
                     ((send db 'reexec-h exec rnum)))
              (setq *psy-autorun* old-autorun))
          (t  (msg "Base inexistente " #\N)))))

;** Reconstruit la base `nombase' (base courante si `nombase' est vide)  **
;** au moment de l'execution numero `num-exec'. Si `rule-number' n'est   **
;** pas vide change l'ordre du conflict set avec cette numero de regle   **
(defun recons-db-exec (num-exec &optional nombase rule-number)
  (let ((base (cond (nombase (get  '*psy-kb* nombase))
                    (t *psy-db*)))
         faits)
     (cond (base (setq faits
                       (send base 'recons-db-exec num-exec rule-number))
                 (if faits 
                     (progn ()
                       (msg "Faits initiales non rentres :")
                       (for f in faits
                            do (msg f #\N))))
                 faits)
           (t (msg "Base inexistente " #\N)))))

;** Methode auxiliere de recons-db-exec                                  **
(defmethod (psy-database recons-db-exec) (num-exec rule-number)
 (let (base-break)
  (cond ((and permet-reexec
             (<= num-exec nombre-exec))
          (cond (hierarchie 
                 (cond ((racinep name hierarchie)
                             (nettoyer-desc self hierarchie)
                             (send self 'rec-db-exec-sp 
                                         num-exec rule-number))
                       (t  (setq base-break 
                               (send self 'rec-db-exec-h num-exec rule-number))
                           (setq pbreaks (delete base-break pbreaks))
                            nil)))
                (t (send self 'rec-db-exec-sp num-exec rule-number)))) 
        ((> num-exec nombre-exec)
           (msg "Cette base a eu seulement " nombre-exec " executions" #\N))
        (t (msg "Le systeme n'a pas tourne en mode reexecutable" #\N)
           nil))))

;** Methode qui actualize la case `ordre-reexec' de la base recevant le  **
;** message. Si la strategie est random `n-regle' est le nom de la regle **
;** a executer dans l'execution numero `num-exec' sinon `n-regle' est le **
;** numero d'ordre de la regle a executer dans le conflict set           **
(defmethod (psy-database act-ordre)(num-exec n-regle)
  (let ((ordre (assoc num-exec ordre-reexec)))
    (cond ((and ordre 
               (< strategy 6)) ;strategie differente de random
                 (setq ordre-reexec
                      (substitute (append ordre `(,n-regle)) 
                                   ordre ordre-reexec :test 'equal)))
          (t     (push `(,num-exec ,n-regle) ordre-reexec)))
     (setq ordre-reexec (for o in ordre-reexec
                             when (<= (car o) num-exec)
                             collect o))))

;** Methode qui reconstruit la base recevant le message dans un numero   **
;** d'execution en alterant l'ordre du conflict set selon `rule-number'  **
;** Cette methode est utilisee seulement pour les bases simples ou pour  **
;** la racine d'une hierarchie                                           **
(defmethod (psy-database rec-db-exec-sp) (num-exec &optional rule-number)
 (let* ((old-trace   *psy-trace*)
        (old-just (reverse just))
        (exec  (assoc num-exec reg-exec)) 
        (c-set (carcdrs 'cadaddr exec))
        (just-exec (caaddr exec))
        (nombre-regle     ; nombre-regle = rule-number + quantite de cycles
          (and rule-number
              (+ rule-number (length (cadddr exec)))))
        (faits (for (f j) in old-just
                    when (equal j "d")
                    until (eq just-exec j)
                    collect (symeval-in-instance f 'value)))
        (faits-non-aj (for (f j) in (member just-exec old-just :key 'cadr)
                                   when (equal j "d")
                                   collect (symeval-in-instance f 'value))))

  (cond ((and rule-number
             (> rule-number (length c-set)))
               (msg "Il n'y a pas de regle numero " rule-number 
                    "dans le conflict set de l'execution " num-exec #\N))
        (t (and (eq strategy 6)
                (progn()
                   (msg "**** Attention *****" #\N)
                   (msg "Cette base a comme strategie random, donc la "
                        " reexecution pourrait l'etre aussi ..." #\N)
                   (read)
                   (for  (n r l) in reg-exec
                         until (eq n num-exec)
                         do (send self 'act-ordre n r))
                    T))     
           (setq cycle 0)
           (setq just nil)
           (psy-untrace)
           (send self 'remove-all-facts)
           (setq passage-ctl nil)
           (setq nombre-exec 0)
           (setq reg-cycles nil)
           (setq reg-exec nil)
           (and rule-number
               (cond ((eq strategy 6) 
                        (send self 'act-ordre num-exec 
                                             (nth (1- rule-number) c-set)))
                     (t (send self 'act-ordre num-exec nombre-regle))))
           (setq pbreaks (cons (1- num-exec) pbreaks))
           (loadfacts faits)
           (run)
           (setq *psy-trace* old-trace)
           (setq pbreaks (delete (1- num-exec) pbreaks))
           faits-non-aj))))

;** Reexecution d'une base hierarchique differente de la racine          ** 
(defmethod (psy-database reexec-h) (num-exec rule-number)
 (let* ((old-trace *psy-trace*)
        (nombre-regle ; nombre-regle = rule-number + quantite de cycles
          (and rule-number
              (+ rule-number (length (cadddr (assoc num-exec reg-exec))))))
        (racine (racine hierarchie))
        (inst-rac (get '*psy-kb* racine))
        (ne-rac (cherche-ne-rac hierarchie name num-exec))
         faits)
    (psy-untrace)
    (nettoyer-desc inst-rac hierarchie)
    (setq faits (send inst-rac 'rec-db-exec-sp ne-rac))
    (setq *psy-trace* old-trace)
    (and rule-number (send self 'act-ordre num-exec nombre-regle))
    (send inst-rac 'run)
    (autorun)
    (for f in faits
         do (send inst-rac 'fact f))))

;** Methode qui reconstruit la base recevant le message dans un numero   **
;** d'execution en alterant l'ordre du conflict set selon `rule-number'  **
;** Cette methode est utilisee seulement pour les bases d'une hierarchie **
;** sauf la  racine                                                      **
(defmethod (psy-database rec-db-exec-h) (num-exec &optional rule-number)
 (let* ((old-trace *psy-trace*)
        (nombre-regle ; nombre-regle = rule-number + quantite de cycles
          (and rule-number
              (+ rule-number (length (cadddr (assoc num-exec reg-exec))))))
        (racine (racine hierarchie))
        (inst-rac (get '*psy-kb* racine))
        (ne-rac (cherche-ne-rac hierarchie name num-exec))
         faits)
    (psy-untrace)
    (setq pbreaks (cons (1- num-exec) pbreaks))
    (nettoyer-desc inst-rac hierarchie)
    (setq faits (send inst-rac 'rec-db-exec-sp ne-rac))
    (setq *psy-trace* old-trace)
    (and rule-number (send self 'act-ordre num-exec nombre-regle))
    (send inst-rac 'run)
    (autorun)
    (for f in faits
         do (send inst-rac 'fact f))
    (1- num-exec)))

;** Fonction qui cherche le numero d'execution de la base racine de la   **
;** hierarchie `h' qui a declenche l'execution `num-exec' dans la base   **
;** appellee `nbase' de la meme hierarchie                               **
(defun cherche-ne-rac (h nbase num-exec)
  (let ((pere (caar (member nbase (get '*psy-controle* h) :key 'cadr))))
    (cond (pere (cherche-ne-rac h pere (send (get '*psy-kb* nbase)
                                              'ch-ne-pc-pere num-exec)))
          (t num-exec))))

;** Methode auxiliere de la fonction cherche-ne. Cherche le numero d'exe-**
;** cution de la base pere dans la hierarchie qui a passe le controle a  **
;** la base recevant le message en declenchant l'execution `num-exec'    **
(defmethod (psy-database ch-ne-pc-pere)(num-exec)
 (let* ((ip (get '*psy-kb*
                 (caar (member name (get '*psy-controle* hierarchie) 
                              :key 'cadr))))
         ctl-pere)
    (setq ctl-pere (for c in (symeval-in-instance ip 'passage-ctl)
                        when (eq name (cadr c))
                        collect c))
    (caar (member num-exec ctl-pere :test '>= :key 'caddr))))

;** Efface les tout trace d'execution des bases qui descendent de la base**
;** `base-pere' dans la hierarchie `hierar' en gardant les faits inits.  **
(defun nettoyer-desc (base-pere hierar)
    (let* ((name (symeval-in-instance base-pere 'name))
           (desc (desc hierar name))
           (autorun *psy-autorun*))
      (for d in desc 
           do (send (get '*psy-kb* d) 'garde-d-init))
       (setq *psy-autorun* autorun)))

;** Efface tout trace d'execution de la base recevant le message en gar- **
;** dant les faits initiales. Utilisee par nettoyer-desc.                **
(defmethod (psy-database garde-d-init)()
  (let ((old-just (reverse just)) 
        (old-trace *psy-trace*))
       (psy-untrace)
       (send self 'remove-all-facts)
       (setq cycle        0
             nombre-exec  0
             passage-ctl  nil
             reg-cycles   nil
             just         nil
             reg-exec     nil)
    (for (f j) in old-just
         when (equal j "d")
         do (send self 'fact (symeval-in-instance f 'value)))
    (setq *psy-trace* old-trace)))

