/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */

// Tests functionality of the mozIAsyncLivemarks interface.

const FEED_URI = NetUtil.newURI("http://feed.rss/");
const SITE_URI = NetUtil.newURI("http://site.org/");

function run_test()
{
  run_next_test();
}

add_test(function test_addLivemark_noArguments_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark();
    do_throw("Invoking addLivemark with no arguments should throw");
  } catch (ex) {
    // The error is actually generated by XPConnect.
    do_check_eq(ex.result, Cr.NS_ERROR_XPC_NOT_ENOUGH_ARGS);
  }
  run_next_test();
});

add_test(function test_addLivemark_emptyObject_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({});
    do_throw("Invoking addLivemark with empty object should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_badParentId_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: "test" });
    do_throw("Invoking addLivemark with a bad parent id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_invalidParentId_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: -2 });
    do_throw("Invoking addLivemark with an invalid parent id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_noIndex_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId });
    do_throw("Invoking addLivemark with no index should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_badIndex_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: "test"
                                      });
    do_throw("Invoking addLivemark with a bad index should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_invalidIndex_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: -2
                                      });
    do_throw("Invoking addLivemark with an invalid index should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_noFeedURI_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                      });
    do_throw("Invoking addLivemark with no feedURI should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_badFeedURI_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                      , feedURI: "test"
                                      });
    do_throw("Invoking addLivemark with a bad feedURI should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_badSiteURI_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                      , feedURI: FEED_URI
                                      , siteURI: "test"
                                      });
    do_throw("Invoking addLivemark with a bad siteURI should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_badGuid_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                      , feedURI: FEED_URI
                                      , guid: "123456"
                                      });
    do_throw("Invoking addLivemark with a bad guid should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_addLivemark_badCallback_throws()
{
  try {
    PlacesUtils.livemarks.addLivemark({ parentId: PlacesUtils.unfiledBookmarksFolderId
                                      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                      , feedURI: FEED_URI
                                      }, "test");
    do_throw("Invoking addLivemark with a bad callback should throw");
  } catch (ex) {
    // The error is actually generated by XPConnect.
    do_check_eq(ex.result, Cr.NS_ERROR_XPC_BAD_CONVERT_JS);
  }
  run_next_test();
});

add_test(function test_addLivemark_noCallback_succeeds()
{
  PlacesUtils.bookmarks.addObserver({
    onItemAdded: function onItemAdded(aItemId, aParentId, aIndex, aItemType,
                                      aURI, aTitle)
    {
      PlacesUtils.bookmarks.removeObserver(this);
      do_check_eq(aParentId, PlacesUtils.unfiledBookmarksFolderId);
      do_check_eq(aIndex, 0);
      do_check_eq(aItemType, Ci.nsINavBookmarksService.TYPE_FOLDER);
      do_check_eq(aTitle, "test");
      run_next_test();
    },
    onBeginUpdateBatch: function onBeginUpdateBatch() {},
    onEndUpdateBatch: function onEndUpdateBatch() {},
    onBeforeItemRemoved: function onBeforeItemRemoved() {},
    onItemRemoved: function onItemRemoved() {},
    onItemChanged: function onItemChanged() {},
    onItemVisited: function onItemVisited() {},
    onItemMoved: function onItemMoved() {},
  }, false);
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    });
});

add_test(function test_addLivemark_noSiteURI_callback_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    do_check_true(aLivemark.id > 0);
    do_check_valid_places_guid(aLivemark.guid);
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_eq(aLivemark.lastModified, PlacesUtils.bookmarks.getItemLastModified(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_eq(aLivemark.siteURI, null);
    run_next_test();
  });
});

add_test(function test_addLivemark_callback_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    , siteURI: SITE_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    do_check_true(aLivemark.id > 0);
    do_check_valid_places_guid(aLivemark.guid);
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_eq(aLivemark.lastModified, PlacesUtils.bookmarks.getItemLastModified(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_true(aLivemark.siteURI.equals(SITE_URI));

    do_check_true(PlacesUtils.annotations
                             .itemHasAnnotation(aLivemark.id,
                                                PlacesUtils.LMANNO_FEEDURI));
    do_check_true(PlacesUtils.annotations
                             .itemHasAnnotation(aLivemark.id,
                                                PlacesUtils.LMANNO_SITEURI));
    run_next_test();
  });
});

add_test(function test_addLivemark_bogusid_callback_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ id: 100 // Should be ignored.
                                    , title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    , siteURI: SITE_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    do_check_true(aLivemark.id > 0);
    do_check_neq(aLivemark.id, 100);

    run_next_test();
  });
});

add_test(function test_addLivemark_bogusParent_callback_fails()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: 187
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_false(Components.isSuccessCode(aStatus));
    do_check_eq(aLivemark, null);
    run_next_test();
  });
});

add_test(function test_addLivemark_intoLivemark_callback_fails()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    
    PlacesUtils.livemarks.addLivemark({ title: "test"
                                      , parentId: aLivemark.id
                                      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                      , feedURI: FEED_URI
                                      }, function (aStatus, aLivemark)
    {
      do_check_false(Components.isSuccessCode(aStatus));
      do_check_eq(aLivemark, null);
      run_next_test();
    });
  });
});

add_test(function test_addLivemark_forceGuid_callback_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    , guid: "1234567890AB"
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    do_check_eq(aLivemark.guid, "1234567890AB");
    do_check_guid_for_bookmark(aLivemark.id, "1234567890AB");

    run_next_test();
  });
});

add_test(function test_addLivemark_lastModified_callback_succeeds()
{
  let now = Date.now() * 1000;
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    , lastModified: now
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    do_check_eq(aLivemark.lastModified, now);

    run_next_test();
  });
});

add_test(function test_removeLivemark_emptyObject_throws()
{
  try {
    PlacesUtils.livemarks.removeLivemark({});
    do_throw("Invoking removeLivemark with empty object should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_removeLivemark_noValidId_throws()
{
  try {
    PlacesUtils.livemarks.removeLivemark({ id: -10, guid: "test"});
    do_throw("Invoking removeLivemark with no valid id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_removeLivemark_nonExistent_fails()
{
  PlacesUtils.livemarks.removeLivemark(
    { id: 1337 },
    function (aStatus, aLivemark) {
      do_check_false(Components.isSuccessCode(aStatus));
      do_check_eq(aLivemark, null);
      run_next_test();
    }
  );
});

add_test(function test_removeLivemark_guid_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    , guid: "234567890ABC"
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    do_check_eq(aLivemark.guid, "234567890ABC");
    // invalid id to check the guid wins.
    PlacesUtils.livemarks.removeLivemark(
      { id: 789, guid: "234567890ABC" },
      function (aStatus, aRemovedLivemark) {
        do_check_true(Components.isSuccessCode(aStatus));
        do_check_eq(PlacesUtils.bookmarks.getItemIndex(aLivemark.id), -1);
        do_check_eq(aRemovedLivemark, null);
        run_next_test();
      }
    );
  });
});

add_test(function test_removeLivemark_id_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));
    PlacesUtils.livemarks.removeLivemark(
      { id: aLivemark.id },
      function (aStatus, aRemovedLivemark) {
        do_check_true(Components.isSuccessCode(aStatus));
        do_check_eq(PlacesUtils.bookmarks.getItemIndex(aLivemark.id), -1);
        do_check_eq(aRemovedLivemark, null);
        run_next_test();
      }
    );
  });
});

add_test(function test_getLivemark_emptyObject_throws()
{
  try {
    PlacesUtils.livemarks.getLivemark({}, function () {});
    do_throw("Invoking getLivemark with empty object should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_getLivemark_noValidId_throws()
{
  try {
    PlacesUtils.livemarks.getLivemark({ id: -10, guid: "test"}, function () {});
    do_throw("Invoking getLivemark with no valid id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
  run_next_test();
});

add_test(function test_getLivemark_nonExistentId_fails()
{
  PlacesUtils.livemarks.getLivemark({ id: 1234 },
    function (aStatus, aLivemark){
      do_check_false(Components.isSuccessCode(aStatus));
      do_check_eq(aLivemark, null);
      run_next_test();
    }
  );
});

add_test(function test_getLivemark_nonExistentGUID_fails()
{
  PlacesUtils.livemarks.getLivemark({ guid: "34567890ABCD" },
    function (aStatus, aLivemark){
      do_check_false(Components.isSuccessCode(aStatus));
      do_check_eq(aLivemark, null);
      run_next_test();
    }
  );
});

add_test(function test_getLivemark_guid_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    , guid: "34567890ABCD"
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));

    // invalid id to check the guid wins.
    PlacesUtils.livemarks.getLivemark({ id: 789, guid: "34567890ABCD" },
      function(aStatus, aLivemark) {
        do_check_true(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark.title, "test");
        do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
        do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
        do_check_true(aLivemark.feedURI.equals(FEED_URI));
        do_check_eq(aLivemark.siteURI, null);
        do_check_eq(aLivemark.guid, "34567890ABCD");
        run_next_test();
      }
    );
  });
});

add_test(function test_getLivemark_id_succeeds()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function (aStatus, aLivemark)
  {
    do_check_true(Components.isSuccessCode(aStatus));

    // invalid id to check the guid wins.
    PlacesUtils.livemarks.getLivemark({ id: aLivemark.id },
      function(aStatus, aLivemark) {
        do_check_true(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark.title, "test");
        do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
        do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
        do_check_true(aLivemark.feedURI.equals(FEED_URI));
        do_check_eq(aLivemark.siteURI, null);
        do_check_guid_for_bookmark(aLivemark.id, aLivemark.guid);
        run_next_test();
      }
    );
  });
});

add_test(function test_getLivemark_removeItem_contention()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    });
  PlacesUtils.bookmarks.removeFolderChildren(PlacesUtils.unfiledBookmarksFolderId);
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    });
  let id = PlacesUtils.bookmarks.getIdForItemAt(PlacesUtils.unfiledBookmarksFolderId,
                                                PlacesUtils.bookmarks.DEFAULT_INDEX);
  
  PlacesUtils.livemarks.getLivemark(
    { id: id },
    function(aStatus, aLivemark) {
      do_check_true(Components.isSuccessCode(aStatus));
      do_check_eq(aLivemark.title, "test");
      do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
      do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
      do_check_true(aLivemark.feedURI.equals(FEED_URI));
      do_check_eq(aLivemark.siteURI, null);
      do_check_guid_for_bookmark(aLivemark.id, aLivemark.guid);
      run_next_test();
    }
  );
});

add_test(function test_title_change()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function(aStatus, aLivemark) {
    PlacesUtils.bookmarks.setItemTitle(aLivemark.id, "test2");
    do_check_eq(aLivemark.title, "test2");
    run_next_test();
  });
});

add_test(function test_livemark_move()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    }, function(aStatus, aLivemark) {
    PlacesUtils.bookmarks.moveItem(aLivemark.id,
                                   PlacesUtils.toolbarFolderId,
                                   PlacesUtils.bookmarks.DEFAULT_INDEX);
    do_check_eq(aLivemark.parentId, PlacesUtils.toolbarFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    run_next_test();
  });
});
