/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright (C) 2000 Edgar Denny (e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "recipe_win.h"
#include "support.h"
#include "food_srch_dlg.h"
#include "food_srch_res.h"

static gint selected_row = -1;
static GladeXML *xml = NULL;

/* Show the food search dialog. food_res_glist contains a
 * list of Food_t*. */
void
gnutr_show_search_result_dlg( GList *food_res_glist)
{
	GList *measure_text_glist = NULL, *ptr;
	static gboolean loaded_xml = FALSE;
	GtkWidget *search_dialog;
	Food_t *food;
	GtkCList *clist;
	gint n_row;

	/* load the glade interface. */
	if ( !loaded_xml)
	{
		xml = glade_xml_new( GNUTRITION_GLADEDIR "/search_result_dlg.glade",
				NULL);
		loaded_xml = TRUE;

		if (  xml != NULL)
		{
			glade_xml_signal_autoconnect( xml);
		}
		else
		{
			g_log( "Gnutrition", G_LOG_LEVEL_ERROR,
					"gnutr_show_srch_dlg: cannot load glade file\n");
			return;
		}
	}

	search_dialog = glade_xml_get_widget( xml, "gnutr_srch_result_dlg");
	clist = (GtkCList *) glade_xml_get_widget( xml, "gnutr_srch_result_clist");

	/* Reset the widgets. */
	measure_text_glist = g_list_append( measure_text_glist, "");

	gtk_combo_set_popdown_strings( GTK_COMBO( glade_xml_get_widget( 
			xml, "gnutr_srch_result_msre_combo")), measure_text_glist);
	gtk_entry_set_text( GTK_ENTRY( glade_xml_get_widget( 
			xml, "gnutr_srch_result_amount_entry") ), "");
	gtk_entry_set_text( GTK_ENTRY( glade_xml_get_widget( 
			xml, "gnutr_srch_result_sel_food_entry")), "");
	gtk_clist_clear( clist);

	/* create the clist of foods. */
	gtk_clist_freeze( clist);

	for ( ptr = g_list_first( food_res_glist), n_row = 0; ptr; 
		ptr = g_list_next( ptr), n_row++)
	{
		food = ( Food_t *)ptr->data;
		
		/* append visible text to recipie clist. */
		gtk_clist_append( clist, &food->desc->fd_desc);

		/* append invisible food gpointer to recipie clist. */
		gtk_clist_set_row_data( clist, n_row, ( gpointer)food);
	}
	gtk_clist_thaw( clist);

	/* show the search dialog. */
	gtk_widget_show( search_dialog);
}

/*  When a row in the clist is selected, add the selected food to the 
 *  text entry below it. */
void
on_gnutr_srch_result_clist_select_row( GtkCList  *clist,
                                       gint       row,
                                       gint       column,
                                       GdkEvent  *event,
                                       gpointer   user_data)
{
	gchar *food_desc;

	gtk_clist_get_text( clist, row, 0, &food_desc);

	/* store the selected_row - used in several other functions. */
	selected_row = row;

	gtk_entry_set_text( GTK_ENTRY( glade_xml_get_widget(
			xml, "gnutr_srch_result_sel_food_entry")), food_desc);
}

/* When the selected food is changed, 
 * also change the measure_combo to list the measures for the new food. */
void
on_gnutr_srch_result_sel_food_entry_changed( GtkEditable *editable,
                                             gpointer     user_data)
{
	gchar *food_name = gtk_entry_get_text( GTK_ENTRY( glade_xml_get_widget( 
			xml, "gnutr_srch_result_sel_food_entry")));

	if ( strcmp( food_name, "") == 0) return;

	{
		gint i, no_measures;
		gchar *measure = NULL;
		GList *measure_text_glist;
		GtkCList *clist = ( GtkCList *)glade_xml_get_widget(
				xml, "gnutr_srch_result_clist");
		Food_t *food = (Food_t *)gtk_clist_get_row_data( clist, selected_row);

		measure_text_glist = NULL;
		no_measures = food->no_msre;

		for( i=0; i<no_measures; i++)
		{
			measure = food->msre[i]->msre_desc;
			measure_text_glist = g_list_append( measure_text_glist, measure);
		}

		gtk_combo_set_popdown_strings( GTK_COMBO( glade_xml_get_widget(
				xml, "gnutr_srch_result_msre_combo")), measure_text_glist);

		g_list_free( measure_text_glist);
	}
}

/* When the measure is changed, reset the amount. */
void
on_gnutr_srch_result_msre_combo_entry_changed( GtkEditable *editable,
                                               gpointer     user_data)
{
	GtkWidget *entry = glade_xml_get_widget( 
			xml, "gnutr_srch_result_amount_entry");
	gtk_entry_set_text( GTK_ENTRY( entry), "1\0");
}

/* Exit the Search Result Dialog with the "OK" button. Gather the necessary
 * data for the food clist in the recipe window. */
void
on_gnutr_srch_result_ok_button_released( GtkButton *button,
                                         gpointer   user_data)
{
	gchar *string, *text[3];
	gint i, row = -1;
	GtkCList *clist = ( GtkCList *)glade_xml_get_widget(
			xml, "gnutr_srch_result_clist");
	Food_t *food = (Food_t *)gtk_clist_get_row_data( clist, selected_row);
	GtkWidget *result_dialog = glade_xml_get_widget( 
			xml, "gnutr_srch_result_dlg");

	Recipe_t *recipe = gnutr_ret_recipe();

	/* food name. */
	string = gtk_entry_get_text( GTK_ENTRY( glade_xml_get_widget( xml, 
			"gnutr_srch_result_sel_food_entry")));

	/* check that a food has been selected to add to the recipe
	 * clist. If not just return. */
	if ( strcmp( string, "") == 0)
	{
		return;
	}
	else
	{
		text[2] = g_strdup( string);
	}

	/* amount. */
	string = gtk_entry_get_text( GTK_ENTRY( glade_xml_get_widget( xml, 
			"gnutr_srch_result_amount_entry")));
	text[0] = g_strdup( string);

	/* measure. */
	string = gtk_entry_get_text( GTK_ENTRY( glade_xml_get_widget( xml,
			"gnutr_srch_result_msre_combo_entry")));
	text[1] = g_strdup( string);

	/* find the selected measure number from its name. */
	food->sel_msre = get_food_measure_num( food, string);

	/* Does the selected_food already exist in the recipe clist? 
	 * If so, overwrite, else, append. */
	row = gtk_clist_find_row_from_data( recipe->ingredient_clist,
			( gpointer)food);

	if ( row == -1 ) /* does not exist - APPEND */
	{
		/* append visible text to recipie clist. */
		gtk_clist_append( recipe->ingredient_clist, text);

		/* append invisible selected_food gpointer data to recipie clist. */
		gtk_clist_set_row_data( recipe->ingredient_clist, 
				recipe->no_ingredients++,
				( gpointer)food);
	}
	else /* does exist - OVERWRITE */
	{
		/* insert visible text. */
		gtk_clist_insert( recipe->ingredient_clist, row, text);
		/* insert hidden data. */
		gtk_clist_set_row_data( recipe->ingredient_clist, row,
				( gpointer)food);

		/* remove old ingredient from the recipe clist. */
		gtk_clist_remove( recipe->ingredient_clist, row+1);
	}

	/* Free allocated memory. */
	for ( i=0; i<3; i++) g_free( text[i]);

	/* clear the food result clist. */
	gtk_clist_clear( (GtkCList *) glade_xml_get_widget( xml,
			"gnutr_srch_result_clist"));

	/* Hide the result dialog.  If the search dialog is visible,
	 * also hide it as well. */
	gnutr_hide_srch_dlg();
	gtk_widget_hide( result_dialog);
}

/* exit the Search Result Dialog with the "Cancel" button. */
void
on_gnutr_srch_result_cancel_button_released( GtkButton *button,
                                             gpointer   user_data)
{	
	/* clear the food result clist. */
	gtk_clist_clear( (GtkCList *) glade_xml_get_widget( xml,
			"gnutr_srch_result_clist"));

	/* Hide the result dialog. */
	gtk_widget_hide( glade_xml_get_widget( xml, "gnutr_srch_result_dlg"));
}

/* Hide the result dialog if it is visible. */
void
gnutr_hide_srch_result_dialog()
{
	if ( xml != NULL)    /* check that the interface has been loaded. */
	{
		GtkWidget *dlg = glade_xml_get_widget( xml, "gnutr_srch_result_dlg");

		if ( GTK_WIDGET_VISIBLE( dlg)) gtk_widget_hide( dlg);
	}
}
