/* PSPP - computes sample statistics.
   Copyright (C) 2004 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301, USA. */

#include <config.h>
#include <libpspp/message.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <data/settings.h>
#include <language/command.h>
#include <libpspp/message.h>
#include <language/lexer/lexer.h>
#include <libpspp/misc.h>
#include <libpspp/str.h>

#include "gettext.h"
#define _(msgid) gettext (msgid)

enum PER {PER_RO, PER_RW};

int change_permissions(const char *file_name, enum PER per);


/* Parses the PERMISSIONS command. */
int
cmd_permissions (struct lexer *lexer, struct dataset *ds UNUSED)
{
  char  *fn = 0;

  lex_match (lexer, '/');

  if (lex_match_id (lexer, "FILE"))
    lex_match (lexer, '=');

  fn = ds_xstrdup (lex_tokstr (lexer));
  lex_force_match (lexer, T_STRING);


  lex_match (lexer, '/');
  
  if ( ! lex_match_id (lexer, "PERMISSIONS"))
    goto error;

  lex_match (lexer, '=');

  if ( lex_match_id (lexer, "READONLY"))
    {
      if ( ! change_permissions(fn, PER_RO ) ) 
	goto error;
    }
  else if ( lex_match_id (lexer, "WRITEABLE"))
    {
      if ( ! change_permissions(fn, PER_RW ) ) 
	goto error;
    }
  else
    {
      msg (SE, _("Expecting %s or %s."), "WRITEABLE", "READONLY");
      goto error;
    }

  free(fn);

  return CMD_SUCCESS;

 error:

  free(fn);

  return CMD_FAILURE;
}



int
change_permissions(const char *file_name, enum PER per)
{
  struct stat buf;
  mode_t mode;

  if (get_safer_mode ())
    {
      msg (SE, _("This command not allowed when the SAFER option is set."));
      return CMD_FAILURE;
    }


  if ( -1 == stat(file_name, &buf) ) 
    {
      const int errnum = errno;
      msg (SE, _("Cannot stat %s: %s"), file_name, strerror(errnum));
      return 0;
    }

  if ( per == PER_RW )
    mode = buf.st_mode | S_IWUSR ;
  else
    mode = buf.st_mode & ~( S_IWOTH | S_IWUSR | S_IWGRP );

  if ( -1 == chmod(file_name, mode))

    {
      const int errnum = errno;
      msg (SE, _("Cannot change mode of %s: %s"), file_name, strerror(errnum));
      return 0;
    }

  return 1;
}
