/* PSPPIRE - a graphical user interface for PSPP.
   Copyright (C) 2007, 2009  Free Software Foundation

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */


/* This file is a rubbish bin where stuff gets put when it doesn't seem to
   belong anywhere else.
*/
#include <config.h>

#include	<glib-object.h>

#include <glib.h>
#include "helper.h"
#include "message-dialog.h"
#include <data/format.h>
#include <data/data-in.h>
#include <data/data-out.h>
#include <data/dictionary.h>
#include <data/casereader-provider.h>
#include <libpspp/message.h>
#include "psppire-syntax-window.h"
#include <gtk/gtkbuilder.h>
#include <libpspp/i18n.h>

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <data/settings.h>

#include "psppire-data-store.h"

#include "xalloc.h"

#include <gettext.h>

/* Formats a value according to FORMAT
   The returned string must be freed when no longer required */
gchar *
value_to_text (union value v, struct fmt_spec format)
{
  gchar *s = 0;

  s = g_new (gchar, format.w + 1);
  data_out (&v, &format, s);
  s[format.w]='\0';
  g_strchug (s);

  return s;
}



gboolean
text_to_value (const gchar *text, union value *v,
	      struct fmt_spec format)
{
  bool ok;

  if ( format.type != FMT_A)
    {
      if ( ! text ) return FALSE;

      {
	const gchar *s = text;
	while (*s)
	  {
	    if ( !isspace (*s))
	      break;
	    s++;
	  }

	if ( !*s) return FALSE;
      }
    }

  msg_disable ();
  ok = data_in (ss_cstr (text), LEGACY_NATIVE, format.type, 0, 0, 0,
                v, fmt_var_width (&format));
  msg_enable ();

  return ok;
}


GtkBuilder *
builder_new_real (const gchar *name)
{
  GtkBuilder *builder = gtk_builder_new ();

  GError *err = NULL;
  if ( ! gtk_builder_add_from_file (builder, name,  &err))
    {
      g_critical ("Couldnt open user interface  file %s: %s", name, err->message);
      g_clear_error (&err);
    }

  return builder;
}


GObject *
get_object_assert (GtkBuilder *builder, const gchar *name, GType type)
{
  GObject *o = NULL;
  g_assert (name);

  o = gtk_builder_get_object (builder, name);

  if ( !o )
    g_critical ("Object \"%s\" could not be found\n", name);

  if ( ! g_type_is_a (G_OBJECT_TYPE (o), type))
   {
     g_critical ("Object \"%s\" was expected to have type %s, but in fact has type %s", 
	name, g_type_name (type), G_OBJECT_TYPE_NAME (o));
   }

  return o;
}


GtkAction *
get_action_assert (GtkBuilder *builder, const gchar *name)
{
  return GTK_ACTION (get_object_assert (builder, name, GTK_TYPE_ACTION));
}

GtkWidget *
get_widget_assert (GtkBuilder *builder, const gchar *name)
{
  return GTK_WIDGET (get_object_assert (builder, name, GTK_TYPE_WIDGET));
}

/* This function must be used whenever a filename generated by glib,
   (eg, from gtk_file_chooser_get_filename) and passed to the C library,
   (eg through a pspp syntax string).
*/
gchar *
convert_glib_filename_to_system_filename (const gchar *fname, GError **err)
{
  gchar *output_name;

#ifdef G_OS_WIN32
  const gchar *target_encoding;
  gchar *utf8_name = NULL;

  g_get_charset (&target_encoding);

  output_name = g_convert (fname, -1, target_encoding,
			"UTF-8", NULL, NULL, err);
#else
  output_name = xstrdup (fname);
#endif

  return output_name;
}



#define _(msgid) gettext (msgid)
#define N_(msgid) msgid


static void
give_help (void)
{
  GtkWidget *dialog;

  dialog = gtk_message_dialog_new (NULL,
                                   GTK_DIALOG_MODAL,
                                   GTK_MESSAGE_INFO,
                                   GTK_BUTTONS_CLOSE,
                                   _("Sorry. The help system hasn't yet "
                                     "been implemented."));
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

void
connect_help (GtkBuilder *xml)
{
  GSList *helps = gtk_builder_get_objects (xml);

  GSList *i;
  for ( i = helps; i ; i = g_slist_next (i))
    {
      GObject *o = i->data;
      if ( GTK_IS_WIDGET (o) )
	{
	  gchar *name = NULL;
	  gchar s[12] = {0};
	  g_object_get (o, "name", &name, NULL);

	  if ( name)
	    strncpy (s, name, 11);
	  s[11] = '\0';


	  if ( 0 == strcmp ("help_button", s))
	    {
	    g_signal_connect (o, "clicked", give_help, 0);
	    }
	}
    }

  g_slist_free (helps);
}


void
reference_manual (GtkMenuItem *menu, gpointer data)
{
  GError *err = NULL;
  gchar *cmd = g_strdup_printf ("yelp file://%s", relocate (DOCDIR "/pspp.xml"));

  if ( ! g_spawn_command_line_async (cmd, &err) )
    {
      msg (ME, _("Cannot open reference manual: %s"), err->message);
    }

  g_free (cmd);
  g_clear_error (&err);
}


/* Create a deep copy of SRC */
GtkListStore *
clone_list_store (const GtkListStore *src)
{
  GtkTreeIter src_iter;
  gboolean ok;
  gint i;
  const gint n_cols =  gtk_tree_model_get_n_columns (GTK_TREE_MODEL (src));
  GType *types = g_malloc (sizeof (*types) *  n_cols);

  int row = 0;
  GtkListStore *dest;

  for (i = 0 ; i < n_cols; ++i )
    types[i] = gtk_tree_model_get_column_type (GTK_TREE_MODEL (src), i);

  dest = gtk_list_store_newv (n_cols, types);

  for (ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (src),
					   &src_iter);
       ok;
       ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (src), &src_iter))
    {
      GtkTreeIter dest_iter;
      gtk_list_store_append  (dest, &dest_iter);

      for (i = 0 ; i < n_cols; ++i )
	{
	  GValue val = {0};

	  gtk_tree_model_get_value (GTK_TREE_MODEL (src), &src_iter, i, &val);
	  gtk_list_store_set_value (dest, &dest_iter, i, &val);

	  g_value_unset (&val);
	}
      row++;
    }

  g_free (types);

  return dest;
}


void
paste_syntax_in_new_window (const gchar *syntax)
{
  GtkWidget *se = psppire_syntax_window_new ();

  gtk_text_buffer_insert_at_cursor (PSPPIRE_SYNTAX_WINDOW (se)->buffer, syntax, -1);

  gtk_widget_show (se);
}
