/* -*- c++ -*- */
/*
 * Copyright 2003,2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * ----------------------------------------------------------------------
 * Mid level interface to the Universal Software Radio Peripheral (Rev 1)
 *
 * These classes implement the basic functionality for talking to the
 * USRP.  They try to be as independent of the signal processing code
 * in FPGA as possible.  They implement access to the low level
 * peripherals on the board, provide a common way for reading and
 * writing registers in the FPGA, and provide the high speed interface
 * to streaming data across the USB.
 *
 * It is expected that subclasses will be derived that provide
 * access to the functionality to a particular FPGA configuration.
 * ----------------------------------------------------------------------
 */

#ifndef INCLUDED_USRP_BASIC_H
#define INCLUDED_USRP_BASIC_H


struct usb_dev_handle;
class  fusb_devhandle;
class  fusb_ephandle;

/*!
 * \brief base class for usrp operations
 */
class usrp_basic {
private:
  // NOT IMPLEMENTED
  usrp_basic (const usrp_basic &rhs);			// no copy constructor
  usrp_basic &operator= (const usrp_basic &rhs);	// no assignment operator

protected:
  struct usb_dev_handle	*d_udh;

  usrp_basic (int which_board,
	      struct usb_dev_handle *open_interface (struct usb_device *dev));

  // called after construction in base class to derived class order
  bool initialize ();
  
public:
  virtual ~usrp_basic ();

  /*!
   * \brief return frequency of master oscillator on USRP
   */
  long  fpga_master_clock_freq () const { return 128000000; }

  //
  // Low level implementation routines.
  //
  bool _set_led (int which, bool on);
#if 0
  bool _set_fpga_reset (bool on);
#endif
  bool _write_fpga_reg (int regno, int value);	// 7-bit regno, 32-bit value
  bool _read_fpga_reg (int regno, int *value);	// 7-bit regno, 32-bit value
  bool _write_9862 (int which_codec, int regno, unsigned char value);
  bool _read_9862 (int which_codec, int regno, unsigned char *value);
};

/*!
 * \brief class for accessing the receive side of the USRP
 */
class usrp_basic_rx : public usrp_basic {
private:
  fusb_devhandle	*d_devhandle;
  fusb_ephandle		*d_ephandle;
  int			 d_bytes_per_poll;	// how often to poll for underruns
  int			 d_bytes_seen;		// how many bytes we've seen
  bool			 d_first_read;
  bool			 d_rx_enable;

  void setup_status_polling ();

 protected:
  usrp_basic_rx (int which_board);	// throws if trouble

  // called after construction in base class to derived class order
  bool initialize ();

  bool set_rx_enable (bool on);
  bool rx_enable () const { return d_rx_enable; }

public:
  ~usrp_basic_rx ();

  /*!
   * \brief invokes constructor, returns instance or 0 if trouble
   */
  static usrp_basic_rx *make (int which_board);


  // MANIPULATORS

  /*!
   * \brief tell the fpga the rate rx samples are coming from the A/D's
   *
   * div = fpga_master_clock_freq () / sample_rate
   *
   * sample_rate is determined by a myriad of registers
   * in the 9862.  That's why you have to tell us, so
   * we can tell the fpga.
   */
  bool set_fpga_rx_sample_rate_divisor (unsigned int div);

  /*!
   * \brief read data from the D/A's via the FPGA.
   * \p len must be a multiple of 512 bytes.
   *
   * \returns the number of bytes read, or -1 on error.
   *
   * If overrun is non-NULL it will be set true iff an RX overrun is detected.
   */
  int read (void *buf, int len, bool *overrun);

  // ACCESSORS

  //! sampling rate of A/D converter
  virtual long adc_freq () const { return fpga_master_clock_freq () / 2; } // 64M

};

/*!
 * \brief class for accessing the transmit side of the USRP
 */
class usrp_basic_tx : public usrp_basic {
private:
  fusb_devhandle	*d_devhandle;
  fusb_ephandle		*d_ephandle;
  int			 d_bytes_per_poll;	// how often to poll for overruns
  int			 d_bytes_seen;		// how many bytes we've seen
  bool			 d_first_write;
  bool			 d_tx_enable;

  void setup_status_polling ();

 protected:
  usrp_basic_tx (int which_board);	// throws if trouble

  // called after construction in base class to derived class order
  bool initialize ();

  bool set_tx_enable (bool on);
  bool tx_enable () const { return d_tx_enable; }

public:

  ~usrp_basic_tx ();

  /*!
   * \brief invokes constructor, returns instance or 0 if trouble
   */
  static usrp_basic_tx *make (int which_board);


  // MANIPULATORS

  /*!
   * \brief tell the fpga the rate tx samples are going to the D/A's
   *
   * div = fpga_master_clock_freq () / sample_rate
   *
   * sample_rate is determined by a myriad of registers
   * in the 9862.  That's why you have to tell us, so
   * we can tell the fpga.
   */
  bool set_fpga_tx_sample_rate_divisor (unsigned int div);

  /*!
   * \brief Write data to the A/D's via the FPGA.
   *
   * \p len must be a multiple of 512 bytes.
   * \returns number of bytes written or -1 on error.
   *
   * if \p underrun is non-NULL, it will be set to true iff
   * a transmit underrun condition is detected.
   */
  int write (const void *buf, int len, bool *underrun);

  /*
   * Block until all outstanding writes have completed.
   * This is typically used to assist with benchmarking
   */
  void wait_for_completion ();

  // ACCESSORS

  //! sampling rate of D/A converter
  virtual long dac_freq () const { return fpga_master_clock_freq (); } // 128M

};

#endif
